<?php

namespace Bitrix\Crm\Kanban\Entity;

use Bitrix\Crm\Filter;
use Bitrix\Crm\Item;
use Bitrix\Crm\Kanban\Entity;
use Bitrix\Crm\PhaseSemantics;
use Bitrix\Crm\Service;
use Bitrix\Crm\Service\Container;
use Bitrix\Crm\Settings\QuoteSettings;

class Quote extends Entity
{
	use DynamicInlineEditorFieldsTrait;

	public function getTypeName(): string
	{
		return \CCrmOwnerType::QuoteName;
	}

	public function getItemsSelectPreset(): array
	{
		return [
			'ID',
			'STATUS_ID',
			'TITLE',
			'DATE_CREATE',
			'BEGINDATE',
			'CLOSEDATE',
			'OPPORTUNITY',
			'OPPORTUNITY_ACCOUNT',
			'CURRENCY_ID',
			'ACCOUNT_CURRENCY_ID',
			'CONTACT_ID',
			'COMPANY_ID',
			'MODIFY_BY_ID',
			'ASSIGNED_BY',
			'QUOTE_NUMBER',
			Item::FIELD_NAME_LAST_ACTIVITY_TIME,
			Item::FIELD_NAME_LAST_ACTIVITY_BY,
			'ACTUAL_DATE'
		];
	}

	protected function getDetailComponentName(): ?string
	{
		return 'bitrix:crm.quote.details';
	}

	protected function getInlineEditorConfiguration(\CBitrixComponent $component): array
	{
		/** @var \CrmQuoteDetailsComponent $component */
		return $component->prepareKanbanConfiguration();
	}

	protected function getPersistentFilterFields(): array
	{
		return [ 'ACTIVITY_COUNTER', 'ACTIVITY_RESPONSIBLE_IDS' ];
	}

	public function getFilterPresets(): array
	{
		return (new Filter\Preset\Quote())
			->setDefaultValues($this->getFilter()->getDefaultFieldIDs())
			->getDefaultPresets()
		;
	}

	public function isCustomPriceFieldsSupported(): bool
	{
		return false;
	}

	public function isInlineEditorSupported(): bool
	{
		return QuoteSettings::getCurrent()->isFactoryEnabled();
	}

	public function isEntitiesLinksInFilterSupported(): bool
	{
		return true;
	}

	public function isActivityCountersSupported(): bool
	{
		return $this->factory->isCountersEnabled();
	}

	public function isNeedToRunAutomation(): bool
	{
		return false;
	}

	public function getCloseDateFieldName(): ?string
	{
		return 'CLOSEDATE';
	}

	public function prepareFilterField(string $field): string
	{
		if(in_array(
			$field,
			['PRODUCT_ID', 'STATUS_ID', 'COMPANY_ID', 'LEAD_ID','DEAL_ID', 'CONTACT_ID', 'MYCOMPANY_ID']
		))
		{
			return '=' . $field;
		}

		return $field;
	}

	public function prepareItemCommonFields(array $item): array
	{
		$item['PRICE'] = $item['OPPORTUNITY'];
		if ($item['BEGINDATE'])
		{
			$item['FORMAT_TIME'] = false;
			$item['DATE'] = $item['BEGINDATE'];
		}
		else
		{
			$item['DATE'] = $item['DATE_CREATE'];
		}

		if (empty($item['TITLE']))
		{
			$item['TITLE'] = Item\Quote::getTitlePlaceholderFromData($item);
		}

		$item = parent::prepareItemCommonFields($item);

		// emulating crm element user field to render value properly
		if (isset($item[Item\Quote::FIELD_NAME_LEAD_ID]) && $item[Item\Quote::FIELD_NAME_LEAD_ID] > 0)
		{
			$item[Item\Quote::FIELD_NAME_LEAD_ID] = 'L_' . $item[Item\Quote::FIELD_NAME_LEAD_ID];
		}

		if (isset($item[Item\Quote::FIELD_NAME_DEAL_ID]) && $item[Item\Quote::FIELD_NAME_DEAL_ID] > 0)
		{
			$item[Item\Quote::FIELD_NAME_DEAL_ID] = 'D_' . $item[Item\Quote::FIELD_NAME_DEAL_ID];
		}

		return $item;
	}

	public function getRequiredFieldsByStages(array $stages): array
	{
		$factory = Service\Container::getInstance()->getFactory($this->getTypeId());
		return static::getRequiredFieldsByStagesByFactory(
			$factory,
			$this->getRequiredUserFieldNames(),
			$stages
		);
	}

	protected function getPopupFieldsBeforeUserFields(): array
	{
		return [];
	}

	public function getTypeInfo(): array
	{
		return array_merge(
			parent::getTypeInfo(),
			[
				'hasPlusButtonTitle' => true,
				'useFactoryBasedApproach' => true,
				'canUseCallListInPanel' => true,
				'showPersonalSetStatusNotCompletedText' => true,
				'kanbanItemClassName' => 'crm-kanban-item',
				'canUseMergeInPanel' => true,
			]
		);
	}

	public function canAddItemToStage(string $stageId, \CCrmPerms $userPermissions, string $semantics = PhaseSemantics::UNDEFINED): bool
	{
		return Container::getInstance()->getUserPermissions()->checkAddPermissions(
			$this->getTypeId(),
			$this->getCategoryId(),
			$stageId
		);
	}

	public function getAllowStages(array $filter = []): array
	{
		if (isset($filter['=STATUS_ID']))
		{
			$allowStages = [];
			$list = \Bitrix\Crm\StatusTable::getList([
				'order' => [
					'SORT' => 'ASC',
				],
				'filter' => [
					'=STATUS_ID' => $filter['=STATUS_ID'],
					'=ENTITY_ID' => 'QUOTE_STATUS',
				],
			]);
			while ($status = $list->fetch())
			{
				$allowStages[] = $status['STATUS_ID'];
			}
			return $allowStages;
		}
		return parent::getAllowStages($filter); // TODO: Change the autogenerated stub
	}

	final protected function isItemsAssignedNotificationSupported(): bool
	{
		return true;
	}

	public function getPopupFields(string $viewType): array
	{
		$fields = parent::getPopupFields($viewType);
		foreach ($fields as $i => $field)
		{
			if (mb_strpos($field['NAME'], 'ACTIVITY_FASTSEARCH_') === 0)
			{
				unset($fields[$i]);
			}
		}

		return $fields;
	}

	protected function getHideSumForStagePermissionType(string $stageId, \CCrmPerms $userPermissions): ?string
	{
		return $userPermissions->GetPermType(
			$this->getTypeName(),
			'HIDE_SUM',
			["STAGE_ID{$stageId}"]
		);
	}
}
