<?php

namespace Bitrix\Crm\Entity\Compatibility\Adapter;

use Bitrix\Crm\Entity\Compatibility\Adapter;
use Bitrix\Crm\Field;
use Bitrix\Crm\Item;
use Bitrix\Crm\ItemIdentifier;
use Bitrix\Crm\Service\Container;
use Bitrix\Crm\Service\Factory;
use Bitrix\Crm\Service\Timeline\Monitor;
use Bitrix\Main\Error;
use Bitrix\Main\Result;
use Bitrix\Main\Type\DateTime;

class LastActivity extends Adapter
{
	private Factory $factory;
	/** @var Array<int, Item> */
	private array $items = [];

	private string $lastActivityTimeField;
	private string $lastActivityByField;

	final public function __construct(Factory $factory)
	{
		$this->factory = $factory;

		$this->lastActivityTimeField = $this->factory->getEntityFieldNameByMap(Item::FIELD_NAME_LAST_ACTIVITY_TIME);
		$this->lastActivityByField = $this->factory->getEntityFieldNameByMap(Item::FIELD_NAME_LAST_ACTIVITY_BY);
	}

	final protected function doGetFieldsInfo(): array
	{
		return [
			$this->lastActivityTimeField => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
			],
			$this->lastActivityByField => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
			],
		];
	}

	final protected function doGetFields(): array
	{
		return [
			$this->lastActivityByField => [
				'FIELD' => $this->getTableAlias() . '.' . $this->lastActivityByField,
				'TYPE' => 'int',
			],
			$this->lastActivityTimeField => [
				'FIELD' => $this->getTableAlias() . '.' . $this->lastActivityTimeField,
				'TYPE' => 'datetime',
			],
		];
	}

	final protected function doPerformAdd(array &$fields, array $compatibleOptions): Result
	{
		$createdTimeField = $this->factory->getEntityFieldNameByMap(Item::FIELD_NAME_CREATED_TIME);
		$createdByField = $this->factory->getEntityFieldNameByMap(Item::FIELD_NAME_CREATED_BY);

		$fields[$this->lastActivityTimeField] = $fields[$createdTimeField] ?? $this->getNowString();
		$fields[$this->lastActivityByField] = $fields[$createdByField] ?? $this->getUserId($compatibleOptions);

		return new Result();
	}

	/**
	 * Extracted for testing
	 *
	 * @return string
	 */
	protected function getNowString(): string
	{
		return (new DateTime())->toString();
	}

	private function getUserId(array $compatibleOptions): int
	{
		if (isset($compatibleOptions['CURRENT_USER']))
		{
			return (int)$compatibleOptions['CURRENT_USER'];
		}

		return Container::getInstance()->getContext()->getUserId();
	}

	final protected function doPerformUpdate(int $id, array &$fields, array $compatibleOptions): Result
	{
		// these field values can't be changed from outside
		unset($fields[$this->lastActivityTimeField], $fields[$this->lastActivityByField]);

		$monitor = Monitor::getInstance();
		$identifier = new ItemIdentifier($this->factory->getEntityTypeId(), $id);

		if ($monitor->isTimelineChanged($identifier))
		{
			/** @var DateTime|null $lastActivityTime */
			/** @var int|null $lastActivityBy */
			[$lastActivityTime, $lastActivityBy] = $monitor->calculateLastActivityInfo($identifier);

			if (is_null($lastActivityTime) || is_null($lastActivityBy))
			{
				$item = $this->getItem($id);
				if (!$item)
				{
					return (new Result())->addError(new Error("Could not find item with ID={$id} in " . self::class));
				}

				if (is_null($lastActivityTime))
				{
					$lastActivityTime = $item->getCreatedTime() ?? new DateTime();
				}
				if (is_null($lastActivityBy))
				{
					$lastActivityBy = $item->getCreatedBy() ?? 0;
				}
			}

			$fields[$this->lastActivityTimeField] = (string)$lastActivityTime;
			$fields[$this->lastActivityByField] = $lastActivityBy;
		}

		return new Result();
	}

	private function getItem(int $id): ?Item
	{
		// if $this->items[$id] is null, try to fetch again. may be the item was created after initial caching
		if (isset($this->items[$id]))
		{
			return $this->items[$id];
		}

		$items = $this->factory->getItems([
			'select' => [Item::FIELD_NAME_CREATED_TIME, Item::FIELD_NAME_CREATED_BY],
			'filter' => [
				'=' . Item::FIELD_NAME_ID => $id,
			],
		]);

		$item = array_shift($items);
		$this->items[$id] = $item;

		return $item;
	}

	final protected function doPerformDelete(int $id, array $compatibleOptions): Result
	{
		return new Result();
	}
}
