<?php

namespace Bitrix\Crm\Conversion;

use Bitrix\Crm\Field;
use Bitrix\Crm\Item;
use Bitrix\Crm\RelationIdentifier;
use Bitrix\Crm\Service\Container;
use Bitrix\Crm\Synchronization\UserFieldSynchronizer;
use Bitrix\Main\InvalidOperationException;

final class Mapper
{
	public function getMap(RelationIdentifier $conversionDirection): EntityConversionMap
	{
		$map = EntityConversionMap::load(
			$conversionDirection->getParentEntityTypeId(),
			$conversionDirection->getChildEntityTypeId(),
		);

		if (!$map)
		{
			$map = $this->createMap($conversionDirection);

			$map->save();
		}

		if ($map->isOutOfDate())
		{
			$this->updateMap($map);

			$map->save();
		}

		return $map;
	}

	public function createMap(RelationIdentifier $direction): EntityConversionMap
	{
		$srcFactory = Container::getInstance()->getFactory($direction->getParentEntityTypeId());
		$dstFactory = Container::getInstance()->getFactory($direction->getChildEntityTypeId());

		if (!$srcFactory || !$dstFactory)
		{
			throw new InvalidOperationException('Factories not found: ' . $direction);
		}

		$sourceFields = $srcFactory->getFieldsCollection();
		$destinationFields = $dstFactory->getFieldsCollection();

		$map = new EntityConversionMap(
			$direction->getParentEntityTypeId(),
			$direction->getChildEntityTypeId(),
		);

		$this->fillMapByComparingFields($map, $sourceFields, $destinationFields);

		return $map;
	}

	private function fillMapByComparingFields(
		EntityConversionMap $map,
		Field\Collection $sourceFields,
		Field\Collection $destinationFields
	): void
	{
		foreach ($sourceFields as $sourceField)
		{
			if (!$this->isFieldProcessable($sourceField))
			{
				continue;
			}

			$destinationField = $destinationFields->getField($sourceField->getName());
			if ($destinationField)
			{
				$map->createItem($sourceField->getName());
			}
		}

		$intersections = UserFieldSynchronizer::getIntersection(
			$map->getSourceEntityTypeID(),
			$map->getDestinationEntityTypeID()
		);

		foreach ($intersections as $intersection)
		{
			$map->createItem($intersection['SRC_FIELD_NAME'], $intersection['DST_FIELD_NAME']);
		}
	}

	private function updateMap(EntityConversionMap $map): void
	{
		//todo move logic to this method and remove old code call
		EntityConversionMapper::updateMap($map);
	}

	private function isFieldProcessable(Field $field): bool
	{
		if (
			$field->isProgress()
			|| $field->isAutoGenerated()
		)
		{
			//stage and category fields values are unique to each entity type
			return false;
		}

		if (in_array($field->getName(), $this->getExcludedFieldNames(), true))
		{
			return false;
		}

		//todo create a map/registry of depended/interconnected/calculated fields and manage relations
		// between them in a centralized way? add contact fields there, since they duplicate data
		// and interfere with each other
		if ($field->getType() === Field::TYPE_CRM_CONTACT && $field->getName() !== Item::FIELD_NAME_CONTACT_IDS)
		{
			//fields CONTACT_BINDINGS, CONTACTS do not support un-named set for now. takes time to implement
			//todo remove this while refactoring conversion completely
			return false;
		}

		return true;
	}

	private function getExcludedFieldNames(): array
	{
		return [
			Item::FIELD_NAME_CATEGORY_ID,
			Item::FIELD_NAME_BEGIN_DATE,
			Item::FIELD_NAME_CLOSE_DATE
		];
	}
}
