<?php

namespace Bitrix\Crm\Controller;

use Bitrix\Crm\AutomatedSolution\Action\Read\Fetch;
use Bitrix\Crm\AutomatedSolution\AutomatedSolutionManager;
use Bitrix\Crm\AutomatedSolution\Entity\AutomatedSolutionTable;
use Bitrix\Crm\Field;
use Bitrix\Crm\Integration\Analytics\Builder\Automation\AutomatedSolution\CreateEvent;
use Bitrix\Crm\Integration\Analytics\Builder\Automation\AutomatedSolution\DeleteEvent;
use Bitrix\Crm\Integration\Analytics\Builder\Automation\AutomatedSolution\EditEvent;
use Bitrix\Crm\Integration\Analytics\Dictionary;
use Bitrix\Crm\Restriction\RestrictionManager;
use Bitrix\Crm\Service\Container;
use Bitrix\Crm\Service\Converter;
use Bitrix\Crm\Service\DynamicTypesMap;
use Bitrix\Crm\Service\UserPermissions;
use Bitrix\Main\Engine\Response\DataType\Page;
use Bitrix\Main\Result;
use Bitrix\Main\Type\ArrayHelper;
use Bitrix\Main\UI\PageNavigation;

final class AutomatedSolution extends Base
{
	private UserPermissions $userPermissions;
	private AutomatedSolutionManager $manager;
	private DynamicTypesMap $dynamicTypesMap;
	private Converter\AutomatedSolution $converter;

	protected function init(): void
	{
		parent::init();

		$this->userPermissions = Container::getInstance()->getUserPermissions();
		$this->manager = Container::getInstance()->getAutomatedSolutionManager();
		$this->dynamicTypesMap = Container::getInstance()->getDynamicTypesMap();
		$this->converter = Container::getInstance()->getAutomatedSolutionConverter();
	}

	private function checkPermissions(): bool
	{
		if (!$this->userPermissions->canEditAutomatedSolutions())
		{
			$this->addError(ErrorCode::getAccessDeniedError());

			return false;
		}

		return true;
	}

	public function fieldsAction(): ?array
	{
		if (!$this->checkPermissions())
		{
			return null;
		}

		return $this->prepareFieldsInfo($this->getFieldsInfo());
	}

	private function getFieldsInfo(): array
	{
		return [
			'ID' => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::AutoGenerated, \CCrmFieldInfoAttr::ReadOnly],
				'NORMALIZER' => intval(...),
			],

			'TITLE' => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Required, \CCrmFieldInfoAttr::CanNotBeEmptied],
				'NORMALIZER' => strval(...),
			],

			'TYPE_IDS' => [
				'TYPE' => Field::TYPE_CRM_DYNAMIC_TYPE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Multiple],
				'NORMALIZER' => static function (mixed $value): array {
					$array = (array)$value;

					ArrayHelper::normalizeArrayValuesByInt($array);

					return $array;
				},
			],
		];
	}

	public function getAction(int $id): ?array
	{
		if (!$this->checkPermissions())
		{
			return null;
		}

		$solution = $this->manager->getAutomatedSolution($id);
		if (!$solution)
		{
			$this->addError(ErrorCode::getNotFoundError());

			return null;
		}

		return [
			'automatedSolution' => $this->converter->toJson($solution),
		];
	}

	public function addAction(array $fields = []): ?array
	{
		$builder = (new CreateEvent())
			->setSection(Dictionary::SECTION_REST)
		;

		if ($this->isRest())
		{
			$builder
				->setStatus(Dictionary::STATUS_ATTEMPT)
				->buildEvent()
				->send()
			;
		}

		$result = $this->add($fields);

		if ($this->isRest())
		{
			if (isset($result['automatedSolution']['id']))
			{
				$builder->setId($result['automatedSolution']['id']);
			}
			if (isset($result['automatedSolution']['typeIds']))
			{
				$builder->setTypeIds($result['automatedSolution']['typeIds']);
			}

			if ($this->getErrors())
			{
				$builder->setStatus(Dictionary::STATUS_ERROR);
			}
			else
			{
				$builder->setStatus(Dictionary::STATUS_SUCCESS);
			}

			$builder
				->buildEvent()
				->send()
			;
		}

		return $result;
	}

	private function add(array $fields = []): ?array
	{
		if (!$this->checkPermissions())
		{
			return null;
		}

		$limitCheckResult = RestrictionManager::getAutomatedSolutionLimitRestriction()->check();
		if (!$limitCheckResult->isSuccess())
		{
			$this->addErrors($limitCheckResult->getErrors());

			return null;
		}

		$fieldsToAdd = $this->prepareFieldsFromRequest($fields);
		if (!$this->checkAccessToTypeIds($fieldsToAdd['TYPE_IDS'] ?? []))
		{
			return null;
		}

		$addResult = $this->manager->addAutomatedSolution($fieldsToAdd);
		if (!$addResult->isSuccess())
		{
			$this->addErrors($addResult->getErrors());

			return null;
		}

		$solutionId = $addResult->getData()['fields']['ID'];

		$setTypesResult = $this->setTypes($solutionId, $fieldsToAdd['TYPE_IDS'] ?? []);
		if (!$setTypesResult->isSuccess())
		{
			$this->addErrors($setTypesResult->getErrors());
		}

		return $this->getAction($solutionId);
	}

	public function updateAction(int $id, array $fields = []): ?array
	{
		$builder = (new EditEvent())
			->setSection(Dictionary::SECTION_REST)
			->setId($id)
		;

		if ($this->isRest())
		{
			$builder
				->setStatus(Dictionary::STATUS_ATTEMPT)
				->buildEvent()
				->send()
			;
		}

		$result = $this->update($id, $fields);

		if ($this->isRest())
		{
			if (isset($result['automatedSolution']['typeIds']))
			{
				$builder->setTypeIds($result['automatedSolution']['typeIds']);
			}

			if ($this->getErrors())
			{
				$builder->setStatus(Dictionary::STATUS_ERROR);
			}
			else
			{
				$builder->setStatus(Dictionary::STATUS_SUCCESS);
			}

			$builder
				->buildEvent()
				->send()
			;
		}

		return $result;
	}

	private function update(int $id, array $fields = []): ?array
	{
		if (!$this->checkPermissions())
		{
			return null;
		}

		$currentAutomatedSolutionResponse = $this->getAction($id);
		if (!is_array($currentAutomatedSolutionResponse) || !empty($this->getErrors()))
		{
			return null;
		}

		// fill not provided fields with current values. we don't want unprovided fields to change
		$fieldsToUpdate = $this->prepareFieldsFromRequest(
			$fields + $currentAutomatedSolutionResponse['automatedSolution'],
		);
		if (!$this->checkAccessToTypeIds($fieldsToUpdate['TYPE_IDS'] ?? []))
		{
			return null;
		}

		$updateResult = $this->manager->updateAutomatedSolution($id, $fieldsToUpdate);
		if (!$updateResult->isSuccess())
		{
			$this->addErrors($updateResult->getErrors());
		}

		$setTypesResult = $this->setTypes($id, $fieldsToUpdate['TYPE_IDS'] ?? []);
		if (!$setTypesResult->isSuccess())
		{
			$this->addErrors($setTypesResult->getErrors());
		}

		return $this->getAction($id);
	}

	public function deleteAction(int $id): void
	{
		$analyticsBuilder = (new DeleteEvent())
			->setSection(Dictionary::SECTION_REST)
			->setId($id)
		;

		if ($this->isRest())
		{
			$analyticsBuilder
				->setStatus(Dictionary::STATUS_ATTEMPT)
				->buildEvent()
				->send()
			;
		}

		$this->delete($id);

		if ($this->isRest())
		{
			if ($this->getErrors())
			{
				$analyticsBuilder->setStatus(Dictionary::STATUS_ERROR);
			}
			else
			{
				$analyticsBuilder->setStatus(Dictionary::STATUS_SUCCESS);
			}

			$analyticsBuilder
				->buildEvent()
				->send()
			;
		}
	}

	private function delete(int $id): void
	{
		if (!$this->checkPermissions())
		{
			return;
		}

		$deleteResult = $this->manager->deleteAutomatedSolution($id);

		if (!$deleteResult->isSuccess())
		{
			$this->addErrors($deleteResult->getErrors());
		}
	}

	/**
	 * @param array $fields
	 *
	 * @return array filtered and normalized SNAKE_CASE fields
	 */
	private function prepareFieldsFromRequest(array $fields): array
	{
		$snakeFields = $this->convertKeysToUpper($fields);

		$filteredFields = $this->filterOutUnknownFields($snakeFields);

		return $this->normalizeFieldValues($filteredFields);
	}

	private function filterOutUnknownFields(array $snakeCaseFields): array
	{
		$knownFields = array_keys($this->getFieldsInfo());

		return array_filter(
			$snakeCaseFields,
			fn(mixed $key) => in_array($key, $knownFields, true),
			ARRAY_FILTER_USE_KEY,
		);
	}

	private function normalizeFieldValues(array $snakeCaseFields): array
	{
		$result = [];
		foreach ($snakeCaseFields as $fieldName => $value)
		{
			$normalizer = $this->getFieldsInfo()[$fieldName]['NORMALIZER'];

			$result[$fieldName] = $normalizer($value);
		}

		return $result;
	}

	private function setTypes(int $automatedSolutionId, array $typeIds): Result
	{
		$this->dynamicTypesMap->load([
			'isLoadStages' => false,
			'isLoadCategories' => false,
		]);

		$types = $this->dynamicTypesMap->getBunchOfTypesByIds($typeIds);

		return $this->manager->setTypeBindingsInAutomatedSolution(
			$types,
			$automatedSolutionId,
		);
	}

	public function listAction(array $order = null, array $filter = null, PageNavigation $pageNavigation = null): ?Page
	{
		if (!$this->checkPermissions())
		{
			return null;
		}

		$order ??= ['id' => 'ASC'];
		if (!$this->validateOrder($order, ['id', 'title']))
		{
			return null;
		}

		$filter ??= [];
		if (!$this->validateFilter($filter, ['id', 'title']))
		{
			return null;
		}

		$snakeCaseOrder = $this->convertKeysToUpper($order);
		$snakeCaseFilter = $this->convertKeysToUpper($filter);

		$action = new Fetch(
			$snakeCaseFilter,
			$snakeCaseOrder,
			$pageNavigation?->getOffset(),
			$pageNavigation?->getLimit(),
		);

		$result = $action->execute();
		if (!$result->isSuccess())
		{
			$this->addErrors($result->getErrors());

			return null;
		}

		$solutions = $result->getData()['automatedSolutions'] ?? [];

		return new Page(
			'automatedSolutions',
			array_map($this->converter->toJson(...), $solutions),
			fn() => AutomatedSolutionTable::getCount($snakeCaseFilter),
		);
	}

	private function checkAccessToTypeIds(array $typeIds): bool
	{
		$this->dynamicTypesMap->load([
			'isLoadStages' => false,
			'isLoadCategories' => false,
		]);

		$types = $this->dynamicTypesMap->getBunchOfTypesByIds($typeIds);
		foreach ($types as $type)
		{
			$hasPermission = false;
			if (!$type->getCustomSectionId())
			{
				$hasPermission = $this->userPermissions->isCrmAdmin();
			}
			else{
				$hasPermission = $this->userPermissions->canEditAutomatedSolutions();
			}
			if (!$hasPermission)
			{
				$this->addError(ErrorCode::getAccessDeniedError());

				return false;
			}
		}

		return true;
	}
}
