<?php

namespace Bitrix\Crm\Service\Operation;

use Bitrix\Crm\Component\EntityDetails\Files\CopyFilesOnItemClone;
use Bitrix\Crm\Item;
use Bitrix\Crm\Service\Container;
use Bitrix\Crm\Service\Factory;
use Bitrix\Crm\Service\Operation;
use Bitrix\Main\Error;
use Bitrix\Main\Localization\Loc;
use Bitrix\Main\Result;

class Copy extends Operation
{
	public const ERROR_CODE_ITEM_IS_NEW = 'CRM_ITEM_IS_NEW';

	/**
	 * @return CopyResult
	 */
	public function checkAccess(): Result
	{
		$userPermissions = Container::getInstance()->getUserPermissions($this->getContext()->getUserId());

		$result = new CopyResult();

		if (!$userPermissions->canReadItem($this->item))
		{
			$result->addError(
				new Error(Loc::getMessage('CRM_COMMON_READ_ACCESS_DENIED'), static::ERROR_CODE_ITEM_READ_ACCESS_DENIED)
			);
		}

		if (
			!$userPermissions->canAddItem($this->item)
		)
		{
			$entityDescription = \CCrmOwnerType::GetDescription($this->item->getEntityTypeId());
			$result->addError(
				new Error(
					Loc::getMessage('CRM_COMMON_ADD_ACCESS_DENIED', ['#ENTITY_DESCRIPTION#' => $entityDescription]),
					static::ERROR_CODE_ITEM_ADD_ACCESS_DENIED
				)
			);
		}

		return $result;
	}

	/**
	 * @return CopyResult
	 */
	public function launch(): Result
	{
		return $this->changeResultClass(parent::launch(), CopyResult::class);
	}

	public function isFieldProcessionEnabled(): bool
	{
		// fields will be processed in a copy add operation
		return false;
	}

	public function isCheckFieldsEnabled(): bool
	{
		return false;
	}

	/**
	 * @return CopyResult
	 */
	protected function save(): Result
	{
		$result = new CopyResult();

		if ($this->item->isNew())
		{
			return $result->addError(new Error(
				'You can not copy a new not saved item',
				static::ERROR_CODE_ITEM_IS_NEW
			));
		}

		$factory = Container::getInstance()->getFactory($this->item->getEntityTypeId());
		if (!$factory)
		{
			return $result->addError(new Error('A factory for the item was not found'));
		}

		$copy = $factory->createItem();
		$result->setCopy($copy);

		$this->copyData($this->item, $copy);

		$copyAddResult = $this->getCopyAddOperation($factory, $copy)->launch();
		if (!$copyAddResult->isSuccess())
		{
			$result->addErrors($copyAddResult->getErrors());
		}

		return $result;
	}

	protected function copyData(Item $original, Item $copy): void
	{
		$originalData = $original->getCompatibleData();
		if (!is_null($this->fieldsCollection))
		{
			foreach ($this->fieldsCollection as $field)
			{
				if ($field->isAutoGenerated())
				{
					unset($originalData[$field->getName()]);
				}
			}
		}

		if (isset($originalData[Item::FIELD_NAME_PRODUCTS]) && is_array($originalData[Item::FIELD_NAME_PRODUCTS]))
		{
			foreach ($originalData[Item::FIELD_NAME_PRODUCTS] as &$originalProductRow)
			{
				//do not copy primary key, it will cause sql error
				unset($originalProductRow['ID']);
			}
			unset($originalProductRow);
		}

		$copy->setFromCompatibleData($originalData);

		// if stages are disabled, it doesn't present in a compatible data
		$copy->setStageId($original->getStageId());

		$factory = Container::getInstance()->getFactory($this->item->getEntityTypeId());
		if ($factory)
		{
			CopyFilesOnItemClone::getInstance()->execute($copy, $factory);
		}
	}

	protected function getCopyAddOperation(Factory $factory, Item $copy): Operation\Add
	{
		$operation = $factory->getAddOperation($copy, $this->getContext());

		$operation->importSettings($this->exportSettings());

		return $operation;
	}
}
