<?php

namespace Bitrix\Crm\Service\Factory;

use Bitrix\Crm\Binding\QuoteContactTable;
use Bitrix\Crm\Category\Entity\Category;
use Bitrix\Crm\Conversion\EntityConversionConfig;
use Bitrix\Crm\Field;
use Bitrix\Crm\Item;
use Bitrix\Crm\QuoteTable;
use Bitrix\Crm\Restriction\RestrictionManager;
use Bitrix\Crm\Service\Container;
use Bitrix\Crm\Service\Context;
use Bitrix\Crm\Service\EditorAdapter;
use Bitrix\Crm\Service\EventHistory\TrackedObject;
use Bitrix\Crm\Service\Factory;
use Bitrix\Crm\Service\Operation;
use Bitrix\Crm\Settings\QuoteSettings;
use Bitrix\Main\IO\Path;
use Bitrix\Main\Localization\Loc;
use Bitrix\Main\NotSupportedException;
use Bitrix\Main\ORM\Objectify\EntityObject;

class Quote extends Factory
{
	protected $itemClassName = Item\Quote::class;

	public function __construct()
	{
		Loc::loadMessages(Path::combine(__DIR__, '..', '..', '..', 'classes', 'general', 'crm_quote.php'));
	}

	/**
	 * @inheritDoc
	 */
	public function getDataClass(): string
	{
		return QuoteTable::class;
	}

	protected function configureItem(Item $item, EntityObject $entityObject): void
	{
		parent::configureItem($item, $entityObject);

		$fieldNameMap =
			(new Item\FieldImplementation\Binding\FieldNameMap())
				->setSingleId(Item::FIELD_NAME_CONTACT_ID)
				->setMultipleIds(Item::FIELD_NAME_CONTACT_IDS)
				->setBindings(Item::FIELD_NAME_CONTACT_BINDINGS)
				->setBoundEntities(Item::FIELD_NAME_CONTACTS)
		;

		$item->addImplementation(
			new Item\FieldImplementation\Binding(
				$entityObject,
				\CCrmOwnerType::Contact,
				$fieldNameMap,
				QuoteContactTable::getEntity(),
				Container::getInstance()->getContactBroker(),
			)
		);
	}

	/**
	 * @inheritDoc
	 */
	public function getEntityTypeId(): int
	{
		return \CCrmOwnerType::Quote;
	}

	/**
	 * @inheritDoc
	 */
	public function getFieldsSettings(): array
	{
		$info =  [
			Item::FIELD_NAME_ID => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
			],
			Item\Quote::FIELD_NAME_NUMBER => [
				'TYPE' => Field::TYPE_STRING,
				'CLASS' => Field\Number::class,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::AutoGenerated, \CCrmFieldInfoAttr::Unique],
				'SETTINGS' => [
					'numeratorType' => 'CRM_QUOTE',
					'numeratorIdSettings' => 'QUOTE_ID',
					'eventName' => 'OnBeforeCrmQuoteNumberSet',
					'tableClassName' => $this->getDataClass(),
					'fieldValueNotUniqueErrorMessage' => Loc::getMessage('CRM_SERVICE_FACTORY_QUOTE_NUMBER_NOT_UNIQUE_ERROR_MSGVER_1'),
				],
			],
			Item::FIELD_NAME_TITLE => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [],
			],
			Item::FIELD_NAME_CREATED_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\CreatedTime::class,
			],
			Item::FIELD_NAME_UPDATED_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\UpdatedTime::class,
			],
			Item::FIELD_NAME_CREATED_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\CreatedBy::class,
			],
			Item::FIELD_NAME_UPDATED_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\UpdatedBy::class,
			],
			Item::FIELD_NAME_LAST_ACTIVITY_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
			],
			//LAST_ACTIVITY_TIME should be processed after LAST_ACTIVITY_BY
			Item::FIELD_NAME_LAST_ACTIVITY_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\LastActivityTime::class,
			],
			Item::FIELD_NAME_ASSIGNED => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::CanNotBeEmptied, \CCrmFieldInfoAttr::HasDefaultValue],
				'CLASS' => Field\Assigned::class,
			],
			Item::FIELD_NAME_OPENED => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Required],
				'CLASS' => Field\Opened::class,
			],
			Item\Quote::FIELD_NAME_CONTENT => [
				'TYPE' => Field::TYPE_TEXT,
				'VALUE_TYPE' => Field::VALUE_TYPE_BB,
				'SETTINGS' => [
					'isFlexibleContentType' => true,
				],
				'ATTRIBUTES' => [],
				'CLASS' => Field\Comments::class,
			],
			Item\Quote::FIELD_NAME_TERMS => [
				'TYPE' => Field::TYPE_TEXT,
				'VALUE_TYPE' => Field::VALUE_TYPE_BB,
				'SETTINGS' => [
					'isFlexibleContentType' => true,
				],
				'ATTRIBUTES' => [],
				'CLASS' => Field\Comments::class,
			],
			Item::FIELD_NAME_COMMENTS => [
				'TYPE' => Field::TYPE_TEXT,
				'VALUE_TYPE' => Field::VALUE_TYPE_BB,
				'SETTINGS' => [
					'isFlexibleContentType' => true,
				],
				'ATTRIBUTES' => [],
				'CLASS' => Field\Comments::class,
			],
			Item\Quote::FIELD_NAME_DEAL_ID => [
				'TYPE' => Field::TYPE_CRM_DEAL,
				'SETTINGS' => [
					'parentEntityTypeId' => \CCrmOwnerType::Deal,
				],
			],
			Item::FIELD_NAME_LEAD_ID => [
				'TYPE' => Field::TYPE_CRM_LEAD,
				'SETTINGS' => [
					'parentEntityTypeId' => \CCrmOwnerType::Lead,
				],
			],
			Item\Quote::FIELD_NAME_STORAGE_TYPE => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item\Quote::FIELD_NAME_STORAGE_ELEMENTS => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::Multiple],
			],
			Item::FIELD_NAME_WEBFORM_ID => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
		];

		if ($this->isClientEnabled())
		{
			$info[Item::FIELD_NAME_COMPANY_ID] = [
				'TYPE' => Field::TYPE_CRM_COMPANY,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
				'SETTINGS' => [
					'parentEntityTypeId' => \CCrmOwnerType::Company,
				],
			];

			$info[Item::FIELD_NAME_CONTACT_ID] = [
				'TYPE' => Field::TYPE_CRM_CONTACT,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::Deprecated],
			];

			$info[Item::FIELD_NAME_CONTACT_IDS] = [
				'TYPE' => Field::TYPE_CRM_CONTACT,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::Multiple]
			];

			$info[Item::FIELD_NAME_CONTACTS] = [
				'TYPE' => Field::TYPE_CRM_CONTACT,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::Multiple]
			];

			$info[Item\Quote::FIELD_NAME_PERSON_TYPE_ID] = [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\PersonTypeId::class,
			];

			$locationAttributes = [];
			if (!Container::getInstance()->getAccounting()->isTaxMode())
			{
				$locationAttributes = [\CCrmFieldInfoAttr::NotDisplayed];
			}
			$info[Item\Quote::FIELD_NAME_LOCATION_ID] = [
				'TYPE' => Field::TYPE_LOCATION,
				'ATTRIBUTES' => $locationAttributes,
			];
		}

		if ($this->isLinkWithProductsEnabled())
		{
			$info[Item::FIELD_NAME_CURRENCY_ID] = [
				'TYPE' => Field::TYPE_CRM_CURRENCY,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::HasDefaultValue,
					\CCrmFieldInfoAttr::CanNotBeEmptied,
				],
				'CLASS' => Field\CurrencyId::class,
			];
			$info[Item::FIELD_NAME_EXCH_RATE] = [
				'TYPE' => Field::TYPE_DOUBLE,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::AutoGenerated,
				],
				'CLASS' => Field\ExchRate::class,
			];
			$info[Item::FIELD_NAME_ACCOUNT_CURRENCY_ID] = [
				'TYPE' => Field::TYPE_CRM_CURRENCY,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::ReadOnly,
					\CCrmFieldInfoAttr::HasDefaultValue,
				],
			];
			$info[Item::FIELD_NAME_IS_MANUAL_OPPORTUNITY] = [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			];
			$info[Item::FIELD_NAME_PRODUCTS] = [
				'TYPE' => Field::TYPE_CRM_PRODUCT_ROW,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Multiple, \CCrmFieldInfoAttr::Hidden, \CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\ProductRows::class,
			];
			$info[Item::FIELD_NAME_OPPORTUNITY] = [
				'TYPE' => Field::TYPE_DOUBLE,
				'CLASS' => Field\Opportunity::class,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			];
			$info[Item::FIELD_NAME_TAX_VALUE] = [
				'TYPE' => Field::TYPE_DOUBLE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\TaxValue::class,
			];
			$info[Item::FIELD_NAME_OPPORTUNITY_ACCOUNT] = [
				'TYPE' => Field::TYPE_DOUBLE,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::ReadOnly,
				],
				'CLASS' => Field\OpportunityAccount::class,
			];
			$info[Item::FIELD_NAME_TAX_VALUE_ACCOUNT] = [
				'TYPE' => Field::TYPE_DOUBLE,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::ReadOnly,
				],
				'CLASS' => Field\TaxValueAccount::class,
			];
		}

		if($this->isStagesSupported())
		{
			$info[Item::FIELD_NAME_STAGE_ID] = [
				'TYPE' => Field::TYPE_CRM_STATUS,
				'CRM_STATUS_TYPE' => $this->getStagesEntityId(),
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::HasDefaultValue, \CCrmFieldInfoAttr::Progress],
				'CLASS' => Field\Stage::class,
			];
		}

		$info[Item::FIELD_NAME_CLOSED] = [
			'TYPE' => Field::TYPE_BOOLEAN,
			'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly],
			'CLASS' => Field\Closed::class,
		];
		$info[Item::FIELD_NAME_BEGIN_DATE] = [
			'TYPE' => Field::TYPE_DATE,
			'ATTRIBUTES' => [\CCrmFieldInfoAttr::CanNotBeEmptied, \CCrmFieldInfoAttr::HasDefaultValue],
		];
		$info[Item::FIELD_NAME_CLOSE_DATE] = [
			'TYPE' => Field::TYPE_DATE,
			'ATTRIBUTES' => [\CCrmFieldInfoAttr::CanNotBeEmptied, \CCrmFieldInfoAttr::HasDefaultValue],
			'CLASS' => Field\CloseDate::class,
		];
		$info[Item\Quote::FIELD_NAME_ACTUAL_DATE] = [
			'TYPE' => Field::TYPE_DATE,
			'ATTRIBUTES' => [\CCrmFieldInfoAttr::CanNotBeEmptied, \CCrmFieldInfoAttr::HasDefaultValue],
		];

		if ($this->isMyCompanyEnabled())
		{
			$info[Item::FIELD_NAME_MYCOMPANY_ID] = [
				'TYPE' => Field::TYPE_CRM_COMPANY,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::HasDefaultValue],
				'SETTINGS' => [
					'isMyCompany' => true,
					'parentEntityTypeId' => \CCrmOwnerType::Company,
					'isEmbeddedEditorEnabled' => true,
				],
			];
		}

		if (!QuoteSettings::getCurrent()->isUseNumberInTitlePlaceholder())
		{
			$info[Item::FIELD_NAME_TITLE]['CLASS'] = Field\Title::class;
		}

		return $info;
	}

	/**
	 * @inheritDoc
	 */
	public function getFieldsMap(): array
	{
		return [
			Item::FIELD_NAME_CREATED_TIME => 'DATE_CREATE',
			Item::FIELD_NAME_UPDATED_TIME => 'DATE_MODIFY',
			Item::FIELD_NAME_CREATED_BY => 'CREATED_BY_ID',
			Item::FIELD_NAME_UPDATED_BY => 'MODIFY_BY_ID',
			Item::FIELD_NAME_STAGE_ID => 'STATUS_ID',
			Item::FIELD_NAME_COMPANY => 'COMPANY_BY',
			Item\Quote::FIELD_NAME_LEAD => 'LEAD_BY',
		];
	}

	/**
	 * @inheritDoc
	 */
	public function createCategory(array $data = []): Category
	{
		throw new NotSupportedException('Quote doesn\'t support categories');
	}

	protected function loadCategories(): array
	{
		throw new NotSupportedException('Quote doesn\'t support categories');
	}

	public function isBeginCloseDatesEnabled(): bool
	{
		return true;
	}

	public function isLinkWithProductsEnabled(): bool
	{
		return true;
	}

	public function isClientEnabled(): bool
	{
		return true;
	}

	public function isCrmTrackingEnabled(): bool
	{
		return true;
	}

	public function isMyCompanyEnabled(): bool
	{
		return true;
	}

	public function isNewRoutingForListEnabled(): bool
	{
		return false;
	}

	public function isNewRoutingForDetailEnabled(): bool
	{
		return QuoteSettings::getCurrent()->isFactoryEnabled();
	}

	public function getStagesEntityId(?int $categoryId = null): ?string
	{
		return 'QUOTE_STATUS';
	}

	protected function getTrackedFieldNames(): array
	{
		return [
			Item::FIELD_NAME_TITLE,
			Item::FIELD_NAME_ASSIGNED,
			Item::FIELD_NAME_CURRENCY_ID,
			Item::FIELD_NAME_STAGE_ID,
			Item::FIELD_NAME_IS_MANUAL_OPPORTUNITY,
		];
	}

	protected function getDependantTrackedObjects(): array
	{
		$objects = [];

		$productTrackedObject = new TrackedObject\Product();
		$productTrackedObject->makeThisObjectDependant(Item::FIELD_NAME_PRODUCTS);
		$objects[] = $productTrackedObject;

		return $objects;
	}

	protected function configureAddOperation(Operation $operation): void
	{
		$operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Update(
					'OnBeforeCrmQuoteAdd',
					'CRM_QUOTE_CREATION_CANCELED_MSGVER_1'
				)
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache('b_crm_quote')
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent('OnAfterCrmQuoteAdd')
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				$this->getProductRowsSaveEventAction()
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessSendNotification\WhenAddingEntity(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\QuoteAttachedFilesCreate()
			)
		;
	}

	public function getUpdateOperation(Item $item, Context $context = null): Operation\Update
	{
		$operation = parent::getUpdateOperation($item, $context);

		return $operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Update(
					'OnBeforeCrmQuoteUpdate',
					'CRM_QUOTE_UPDATE_CANCELED_MSGVER_1'
				)
			)
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\QuoteAttachedFilesUpdate()
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache(
					null,
					'crm_entity_name_' . $this->getEntityTypeId() . '_',
					[Item::FIELD_NAME_TITLE]
				)
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\FillEntityFieldsContext()
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent('OnAfterCrmQuoteUpdate')
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				$this->getProductRowsSaveEventAction()
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessSendNotification\WhenUpdatingEntity(),
			)
		;
	}

	protected function getProductRowsSaveEventAction(): Operation\Action
	{
		return new Operation\Action\Compatible\SendEvent\ProductRowsSave('OnAfterCrmQuoteProductRowsSave');
	}

	public function getDeleteOperation(Item $item, Context $context = null): Operation\Delete
	{
		$operation = parent::getDeleteOperation($item, $context);

		return $operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Delete('OnBeforeCrmQuoteDelete')
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache(
					'b_crm_quote',
					'crm_entity_name_' . $this->getEntityTypeId() . '_'
				)
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\DeleteQuoteFiles(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent\Delete('OnAfterCrmQuoteDelete')
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\DeleteEntityFieldsContext()
			)
		;
	}

	public function getConversionOperation(
		Item $item,
		EntityConversionConfig $configs,
		Context $context = null
	): Operation\Conversion
	{
		$operation = parent::getConversionOperation($item, $configs, $context);

		return $operation->addAction(
			Operation::ACTION_BEFORE_SAVE,
			new Operation\Action\CheckRestrictions(RestrictionManager::getConversionRestriction())
		);
	}

	protected function getFieldTitlesMap(): array
	{
		return array_merge(parent::getFieldTitlesMap(), [
			EditorAdapter::FIELD_FILES => Loc::getMessage('CRM_TYPE_QUOTE_FIELD_FILES'),
		]);
	}

	public function getEditorAdapter(): EditorAdapter
	{
		$adapter = parent::getEditorAdapter();

		$adapter->addEntityField(
			EditorAdapter::getUtmField(
				$this->getFieldCaption(EditorAdapter::FIELD_UTM)
			)
		);
		$locationField = $this->getFieldsCollection()->getField(Item\Quote::FIELD_NAME_LOCATION_ID);
		if ($locationField && $locationField->isDisplayed())
		{
			$adapter->addEntityField(EditorAdapter::getLocationFieldDescription($locationField));
		}

		$adapter->addEntityField([
			'name' => EditorAdapter::FIELD_FILES,
			'title' => $this->getFieldCaption(EditorAdapter::FIELD_FILES),
			'type' => 'file_storage',
			'editable' => true,
			'data' => [
				'diskFileInfos' => EditorAdapter::DATA_FILES,
				'storageElementIds' => Item\Quote::FIELD_NAME_STORAGE_ELEMENTS,
				'storageTypeId' => Item\Quote::FIELD_NAME_STORAGE_TYPE,
			]
		]);

		return $adapter;
	}

	public function getDependantFieldsMap(): array
	{
		return array_merge(parent::getDependantFieldsMap(), [
			EditorAdapter::FIELD_FILES => [
				Item\Quote::FIELD_NAME_STORAGE_ELEMENTS
			],
		]);
	}

	public function isAutomationEnabled(): bool
	{
		return true;
	}

	public function isBizProcEnabled(): bool
	{
		return true;
	}

	public function isCountersEnabled(): bool
	{
		return true;
	}

	public function isCommunicationRoutingSupported(): bool
	{
		return true;
	}
}
