<?php

namespace Bitrix\Crm\Service\Factory;

use Bitrix\Crm\Binding\DealContactTable;
use Bitrix\Crm\Category\Entity\Category;
use Bitrix\Crm\Category\Entity\DealCategory;
use Bitrix\Crm\Category\Entity\DealCategoryTable;
use Bitrix\Crm\Category\Entity\DealDefaultCategory;
use Bitrix\Crm\DealTable;
use Bitrix\Crm\Field;
use Bitrix\Crm\Item;
use Bitrix\Crm\Reservation;
use Bitrix\Crm\Service\Container;
use Bitrix\Crm\Service\Context;
use Bitrix\Crm\Service\EventHistory\TrackedObject;
use Bitrix\Crm\Service\Factory;
use Bitrix\Crm\Service\Operation;
use Bitrix\Crm\Settings\DealSettings;
use Bitrix\Crm\Settings\LayoutSettings;
use Bitrix\Crm\Statistics;
use Bitrix\Crm\StatusTable;
use Bitrix\Main\IO\Path;
use Bitrix\Main\Localization\Loc;
use Bitrix\Main\ORM\Objectify\EntityObject;
use CCrmSaleHelper;

final class Deal extends Factory
{
	protected $itemClassName = Item\Deal::class;

	public function __construct()
	{
		Loc::loadMessages(Path::combine(__DIR__, '..', '..', '..', 'classes', 'general', 'crm_deal.php'));
	}

	public function isSourceEnabled(): bool
	{
		return true;
	}

	public function isObserversEnabled(): bool
	{
		return true;
	}

	public function isClientEnabled(): bool
	{
		return true;
	}

	public function isBeginCloseDatesEnabled(): bool
	{
		return true;
	}

	public function isAutomationEnabled(): bool
	{
		return true;
	}

	public function isBizProcEnabled(): bool
	{
		return true;
	}

	public function isCategoriesSupported(): bool
	{
		return true;
	}

	public function isPaymentsEnabled(): bool
	{
		return true;
	}

	public function isInventoryManagementEnabled(): bool
	{
		return
			$this->isLinkWithProductsEnabled()
			&& LayoutSettings::getCurrent()->isSliderEnabled()
			&& CCrmSaleHelper::isProcessInventoryManagement()
		;
	}

	public function getDataClass(): string
	{
		return DealTable::class;
	}

	protected function configureItem(Item $item, EntityObject $entityObject): void
	{
		parent::configureItem($item, $entityObject);

		$fieldNameMap =
			(new Item\FieldImplementation\Binding\FieldNameMap())
				->setSingleId(Item::FIELD_NAME_CONTACT_ID)
				->setMultipleIds(Item::FIELD_NAME_CONTACT_IDS)
				->setBindings(Item::FIELD_NAME_CONTACT_BINDINGS)
				->setBoundEntities(Item::FIELD_NAME_CONTACTS)
		;

		$item->addImplementation(
			new Item\FieldImplementation\Binding(
				$entityObject,
				\CCrmOwnerType::Contact,
				$fieldNameMap,
				DealContactTable::getEntity(),
				Container::getInstance()->getContactBroker(),
			)
		);
	}

	/**
	 * @inheritDoc
	 */
	public function getFieldsMap(): array
	{
		return [
			Item::FIELD_NAME_CREATED_TIME => 'DATE_CREATE',
			Item::FIELD_NAME_UPDATED_TIME => 'DATE_MODIFY',
			Item::FIELD_NAME_CREATED_BY => 'CREATED_BY_ID',
			Item::FIELD_NAME_UPDATED_BY => 'MODIFY_BY_ID',
			Item::FIELD_NAME_MOVED_BY => 'MOVED_BY_ID',
			Item::FIELD_NAME_OBSERVERS => 'OBSERVER_IDS',
		];
	}

	public function isNewRoutingForDetailEnabled(): bool
	{
		return false;
	}

	public function isNewRoutingForListEnabled(): bool
	{
		return false;
	}

	public function isNewRoutingForAutomationEnabled(): bool
	{
		return false;
	}

	public function isUseInUserfieldEnabled(): bool
	{
		return true;
	}

	public function isCrmTrackingEnabled(): bool
	{
		return true;
	}

	public function isLinkWithProductsEnabled(): bool
	{
		return true;
	}

	public function isRecyclebinEnabled(): bool
	{
		return DealSettings::getCurrent()->isRecycleBinEnabled();
	}

	public function isDeferredCleaningEnabled(): bool
	{
		return DealSettings::getCurrent()->isDeferredCleaningEnabled();
	}

	public function getEntityTypeId(): int
	{
		return \CCrmOwnerType::Deal;
	}

	protected function getFieldsSettings(): array
	{
		$info = [
			Item::FIELD_NAME_ID => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
			],
			Item::FIELD_NAME_TITLE => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::HasDefaultValue, \CCrmFieldInfoAttr::CanNotBeEmptied],
				'CLASS' => Field\Title::class,
			],
			Item::FIELD_NAME_TYPE_ID => [
				'TYPE' => Field::TYPE_CRM_STATUS,
				'CRM_STATUS_TYPE' => StatusTable::ENTITY_ID_DEAL_TYPE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::HasDefaultValue],
			],
			Item::FIELD_NAME_CATEGORY_ID => [
				'TYPE' => Field::TYPE_CRM_CATEGORY,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::HasDefaultValue],
				'CLASS' => Field\Category::class,
				'SETTINGS' => [
					'eventNames' => [
						'onBeforeChange' => 'OnBeforeDealMoveToCategory',
						'onAfterChange' => 'OnAfterDealMoveToCategory',
					],
				],
			],
			Item::FIELD_NAME_STAGE_ID => [
				'TYPE' => Field::TYPE_CRM_STATUS,
				'CRM_STATUS_TYPE' => $this->getStagesEntityId(),
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Progress],
				'CLASS' => Field\Stage::class,
			],
			Item::FIELD_NAME_STAGE_SEMANTIC_ID => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\StageSemanticId::class,
			],
			Item\Deal::FIELD_NAME_IS_NEW => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\IsNew::class,
			],
			Item::FIELD_NAME_IS_RECURRING => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item::FIELD_NAME_IS_RETURN_CUSTOMER => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\IsReturnCustomer::class,
			],
			Item\Deal::FIELD_NAME_IS_REPEATED_APPROACH => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\IsRepeatedApproach::class,
			],
			Item\Deal::FIELD_NAME_PROBABILITY => [
				'TYPE' => Field::TYPE_INTEGER,
				'CLASS' => Field\Percentage::class,
			],
			Item::FIELD_NAME_CURRENCY_ID => [
				'TYPE' => Field::TYPE_CRM_CURRENCY,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::HasDefaultValue,
					\CCrmFieldInfoAttr::CanNotBeEmptied,
				],
				'CLASS' => Field\CurrencyId::class,
			],
			Item::FIELD_NAME_EXCH_RATE => [
				'TYPE' => Field::TYPE_DOUBLE,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::AutoGenerated,
				],
				'CLASS' => Field\ExchRate::class,
			],
			Item::FIELD_NAME_ACCOUNT_CURRENCY_ID => [
				'TYPE' => Field::TYPE_CRM_CURRENCY,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::ReadOnly,
					\CCrmFieldInfoAttr::HasDefaultValue,
				],
			],
			Item::FIELD_NAME_IS_MANUAL_OPPORTUNITY => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item::FIELD_NAME_PRODUCTS => [
				'TYPE' => Field::TYPE_CRM_PRODUCT_ROW,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Multiple, \CCrmFieldInfoAttr::Hidden, \CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\ProductRows::class,
			],
			Item::FIELD_NAME_OPPORTUNITY => [
				'TYPE' => Field::TYPE_DOUBLE,
				'CLASS' => Field\Opportunity::class,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item::FIELD_NAME_TAX_VALUE => [
				'TYPE' => Field::TYPE_DOUBLE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
				'CLASS' => Field\TaxValue::class,
			],
			Item::FIELD_NAME_OPPORTUNITY_ACCOUNT => [
				'TYPE' => Field::TYPE_DOUBLE,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::ReadOnly,
				],
				'CLASS' => Field\OpportunityAccount::class,
			],
			Item::FIELD_NAME_TAX_VALUE_ACCOUNT => [
				'TYPE' => Field::TYPE_DOUBLE,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::NotDisplayed,
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::ReadOnly,
				],
				'CLASS' => Field\TaxValueAccount::class,
			],
			Item::FIELD_NAME_COMPANY_ID => [
				'TYPE' => Field::TYPE_CRM_COMPANY,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
				'SETTINGS' => [
					'parentEntityTypeId' => \CCrmOwnerType::Company,
				],
			],
			Item::FIELD_NAME_CONTACT_ID => [
				'TYPE' => Field::TYPE_CRM_CONTACT,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::Deprecated],
			],
			Item::FIELD_NAME_CONTACT_IDS => [
				'TYPE' => Field::TYPE_CRM_CONTACT,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::Multiple]
			],
			Item::FIELD_NAME_CONTACTS => [
				'TYPE' => Field::TYPE_CRM_CONTACT,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::Multiple]
			],
			Item\Deal::FIELD_NAME_QUOTE_ID => [
				'TYPE' => Field::TYPE_CRM_QUOTE,
				'SETTINGS' => [
					'parentEntityTypeId' => \CCrmOwnerType::Quote,
				],
			],
			Item::FIELD_NAME_BEGIN_DATE => [
				'TYPE' => Field::TYPE_DATE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::CanNotBeEmptied, \CCrmFieldInfoAttr::HasDefaultValue],
			],
			Item::FIELD_NAME_CLOSE_DATE => [
				'TYPE' => Field::TYPE_DATE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::CanNotBeEmptied, \CCrmFieldInfoAttr::HasDefaultValue],
				'CLASS' => Field\CloseDate::class,
				'SETTINGS' => [
					'isSetCurrentDateOnCompletionEnabled' => DealSettings::getCurrent()->isCloseDateSyncEnabled(),
				],
			],
			Item::FIELD_NAME_OPENED => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Required],
				'CLASS' => Field\Opened::class,
			],
			Item::FIELD_NAME_CLOSED => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly],
				'CLASS' => Field\Closed::class,
			],
			Item::FIELD_NAME_COMMENTS => [
				'TYPE' => Field::TYPE_TEXT,
				'VALUE_TYPE' => Field::VALUE_TYPE_BB,
				'ATTRIBUTES' => [],
				'CLASS' => Field\Comments::class,
			],
			Item::FIELD_NAME_ASSIGNED => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::CanNotBeEmptied, \CCrmFieldInfoAttr::HasDefaultValue],
				'CLASS' => Field\Assigned::class,
			],
			Item::FIELD_NAME_CREATED_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\CreatedBy::class,
			],
			Item::FIELD_NAME_UPDATED_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\UpdatedBy::class,
			],
			Item::FIELD_NAME_MOVED_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\MovedBy::class,
			],
			Item::FIELD_NAME_LAST_ACTIVITY_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
			],
			Item::FIELD_NAME_CREATED_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\CreatedTime::class,
			],
			Item::FIELD_NAME_UPDATED_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\UpdatedTime::class,
			],
			Item::FIELD_NAME_MOVED_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\MovedTime::class,
			],
			Item::FIELD_NAME_LAST_ACTIVITY_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\LastActivityTime::class,
			],
			Item::FIELD_NAME_SOURCE_ID => [
				'TYPE' => Field::TYPE_CRM_STATUS,
				'CRM_STATUS_TYPE' => StatusTable::ENTITY_ID_SOURCE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::HasDefaultValue],
			],
			Item::FIELD_NAME_SOURCE_DESCRIPTION => [
				'TYPE' => Field::TYPE_TEXT,
			],
			Item::FIELD_NAME_LEAD_ID => [
				'TYPE' => Field::TYPE_CRM_LEAD,
				'SETTINGS' => [
					'parentEntityTypeId' => \CCrmOwnerType::Lead,
				],
			],
			Item\Deal::FIELD_NAME_ADDITIONAL_INFO => [
				'TYPE' => Field::TYPE_STRING,
			],
			Item::FIELD_NAME_ORIGINATOR_ID => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item::FIELD_NAME_ORIGIN_ID => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item::FIELD_NAME_OBSERVERS => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Multiple],
				'CLASS' => Field\Observers::class,
			],
		];

		$locationAttributes = [];
		if (!Container::getInstance()->getAccounting()->isTaxMode())
		{
			$locationAttributes = [\CCrmFieldInfoAttr::NotDisplayed];
		}
		$info[Item::FIELD_NAME_LOCATION_ID] = [
			'TYPE' => Field::TYPE_LOCATION,
			'ATTRIBUTES' => $locationAttributes,
		];

		return $info;
	}

	/**
	 * @inheritDoc
	 */
	public function getStagesEntityId(?int $categoryId = null): ?string
	{
		$categoryId = (int)$categoryId;

		if ($categoryId > 0)
		{
			return 'DEAL_STAGE_' . $categoryId;
		}

		return 'DEAL_STAGE';
	}

	protected function checkIfCategoryAvailable(int $categoryId): bool
	{
		if ($categoryId === 0)
		{
			//It's deal default category. Always is available
			return true;
		}

		return \Bitrix\Crm\Category\DealCategory::isEnabled($categoryId);
	}

	public function getCategoryFieldsInfo(): array
	{
		$fieldsInfo = parent::getCategoryFieldsInfo();

		$fieldsInfo['ENTITY_TYPE_ID']['ATTRIBUTES'][] = \CCrmFieldInfoAttr::ReadOnly;
		$fieldsInfo['IS_DEFAULT']['ATTRIBUTES'][] = \CCrmFieldInfoAttr::ReadOnly;

		$fieldsInfo['ORIGIN_ID'] = [
			'TYPE' => Field::TYPE_STRING,
		];
		$fieldsInfo['ORIGINATOR_ID'] = [
			'TYPE' => Field::TYPE_STRING,
		];

		unset($fieldsInfo['IS_SYSTEM'], $fieldsInfo['CODE']);

		return $fieldsInfo;
	}

	public function createCategory(array $data = []): Category
	{
		$object = DealCategoryTable::createObject($data);

		return new DealCategory($object);
	}

	protected function loadCategories(): array
	{
		$defaultCategory = new DealDefaultCategory(
			\Bitrix\Crm\Category\DealCategory::getDefaultCategoryName(),
			\Bitrix\Crm\Category\DealCategory::getDefaultCategorySort()
		);

		$result = [$defaultCategory];

		$categories = DealCategoryTable::getList([
			'order' => [
				'SORT' => 'ASC',
				'ID' => 'ASC',
			],
			'cache' => [
				'ttl' => 84600,
			],
		])->fetchCollection();
		foreach ($categories as $category)
		{
			$result[] = new DealCategory($category);
		}

		usort(
			$result,
			static function(Category $a, Category $b) {
				if ($a->getSort() === $b->getSort())
				{
					return 0;
				}

				return ($a->getSort() < $b->getSort()) ? -1 : 1;
			}
		);

		return $result;
	}

	protected function getTrackedFieldNames(): array
	{
		return [
			Item::FIELD_NAME_TITLE,
			Item::FIELD_NAME_ASSIGNED,
			Item::FIELD_NAME_CURRENCY_ID,
			Item::FIELD_NAME_CATEGORY_ID,
			Item::FIELD_NAME_STAGE_ID,
			Item::FIELD_NAME_OPPORTUNITY,
			Item::FIELD_NAME_IS_MANUAL_OPPORTUNITY,
			Item::FIELD_NAME_TYPE_ID,
			Item::FIELD_NAME_COMMENTS,
			Item::FIELD_NAME_SOURCE_ID,
			Item\Deal::FIELD_NAME_PROBABILITY,
			Item::FIELD_NAME_BEGIN_DATE,
			Item::FIELD_NAME_CLOSE_DATE,
			Item::FIELD_NAME_CLOSED,
			Item::FIELD_NAME_LOCATION_ID,
		];
	}

	protected function getDependantTrackedObjects(): array
	{
		$objects = [];

		$productTrackedObject = new TrackedObject\Product();
		$productTrackedObject->makeThisObjectDependant(Item::FIELD_NAME_PRODUCTS);
		$objects[] = $productTrackedObject;

		return $objects;
	}

	protected function getStatisticsFacade(): ?Statistics\OperationFacade
	{
		return new Statistics\OperationFacade\Deal();
	}

	protected function configureAddOperation(Operation $operation): void
	{
		$operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Update(
					'OnBeforeCrmDealAdd',
					'CRM_DEAL_CREATION_CANCELED',
				),
			)
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\ProductRowsSave('OnBeforeCrmDealProductRowsSave'),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache('b_crm_deal'),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessAdd(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessSendNotification\WhenAddingEntity(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent('OnAfterCrmDealAdd'),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent\ExternalAdd('OnAfterExternalCrmDealAdd'),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent\ProductRowsSave('OnAfterCrmDealProductRowsSave'),
			)
		;

		if ($this->isInventoryManagementEnabled())
		{
			$operation
				->addAction(
					Operation::ACTION_BEFORE_SAVE,
					new Reservation\Actions\CheckProductsOnAdd()
				)
				->addAction(
					Operation::ACTION_AFTER_SAVE,
					new Reservation\Actions\SynchronizeReservesOnAdd()
				)
				->addAction(
					Operation::ACTION_AFTER_SAVE,
					new Reservation\Actions\ProcessInventoryManagementOnAdd()
				)
			;
		}
	}

	public function getUpdateOperation(Item $item, Context $context = null): Operation\Update
	{
		$operation = parent::getUpdateOperation($item, $context);

		$operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Update(
					'OnBeforeCrmDealUpdate',
					'CRM_DEAL_UPDATE_CANCELED',
				),
			)
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\ProductRowsSave('OnBeforeCrmDealProductRowsSave'),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache(
					null,
					'crm_entity_name_' . $this->getEntityTypeId() . '_',
					[Item::FIELD_NAME_TITLE]
				)
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\FillEntityFieldsContext()
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessUpdate(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessSendNotification\WhenUpdatingEntity(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\UpdateMlScoring(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent('OnAfterCrmDealUpdate'),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent\ProductRowsSave('OnAfterCrmDealProductRowsSave'),
			)
		;

		if ($this->isInventoryManagementEnabled())
		{
			$synchronizeReserveOperation = new Reservation\Actions\SynchronizeReservesOnUpdate();

			$operation
				->addAction(
					Operation::ACTION_BEFORE_SAVE,
					$synchronizeReserveOperation
				)
				->addAction(
					Operation::ACTION_AFTER_SAVE,
					$synchronizeReserveOperation
				)
				->addAction(
					Operation::ACTION_BEFORE_SAVE,
					new Reservation\Actions\CheckProductsOnUpdate()
				)
				->addAction(
					Operation::ACTION_AFTER_SAVE,
					new Reservation\Actions\ProcessInventoryManagementOnUpdate()
				)
			;
		}

		$operation->addAction(
			Operation::ACTION_AFTER_SAVE,
			new Operation\Action\CreateFinalSummaryTimelineHistoryItem()
		);

		return $operation;
	}

	public function getDeleteOperation(Item $item, Context $context = null): Operation\Delete
	{
		$operation = parent::getDeleteOperation($item, $context);

		$operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Delete('OnBeforeCrmDealDelete')
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache(
					'b_crm_deal',
					'crm_entity_name_' . $this->getEntityTypeId() . '_'
				)
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessDelete(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\DeleteRecurringDealSchedule(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\DeleteEntityFieldsContext()
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent\Delete('OnAfterCrmDealDelete')
			)
		;

		return $operation;
	}

	public function getItemCategoryId(int $id): ?int
	{
		// here is some extra logic for backward compatibility
		if($id <= 0)
		{
			return 0;
		}
		$categoryId = parent::getItemCategoryId($id);

		return is_null($categoryId) ? -1 : $categoryId;
	}

	public function isCountersEnabled(): bool
	{
		return true;
	}

	public function isSmartActivityNotificationSupported(): bool
	{
		return true;
	}

	public function isCommunicationRoutingSupported(): bool
	{
		return true;
	}
}
