<?php

namespace Bitrix\Crm\Service\Factory;

use Bitrix\Crm\Binding\ContactCompanyTable;
use Bitrix\Crm\Category\Entity\Category;
use Bitrix\Crm\Category\Entity\ClientDefaultCategory;
use Bitrix\Crm\Category\Entity\ItemCategory;
use Bitrix\Crm\ContactTable;
use Bitrix\Crm\Field;
use Bitrix\Crm\Item;
use Bitrix\Crm\Model\ItemCategoryTable;
use Bitrix\Crm\Service;
use Bitrix\Crm\Service\Container;
use Bitrix\Crm\Service\Context;
use Bitrix\Crm\Service\Operation;
use Bitrix\Crm\Settings\ContactSettings;
use Bitrix\Crm\Statistics;
use Bitrix\Crm\StatusTable;
use Bitrix\Main\IO\Path;
use Bitrix\Main\Localization\Loc;
use Bitrix\Main\ORM\Objectify\EntityObject;

class Contact extends Service\Factory
{
	protected $itemClassName = Item\Contact::class;

	public function __construct()
	{
		Loc::loadMessages(Path::combine(__DIR__, '..', '..', '..', 'classes', 'general', 'crm_contact.php'));
	}

	public function isSourceEnabled(): bool
	{
		return true;
	}

	public function isNewRoutingForDetailEnabled(): bool
	{
		return false;
	}

	public function isNewRoutingForAutomationEnabled(): bool
	{
		return false;
	}

	public function isNewRoutingForListEnabled(): bool
	{
		return false;
	}

	public function isRecyclebinEnabled(): bool
	{
		return ContactSettings::getCurrent()->isRecycleBinEnabled();
	}

	public function isDeferredCleaningEnabled(): bool
	{
		return ContactSettings::getCurrent()->isDeferredCleaningEnabled();
	}

	public function isUseInUserfieldEnabled(): bool
	{
		return true;
	}

	public function isCrmTrackingEnabled(): bool
	{
		return true;
	}

	public function isStagesSupported(): bool
	{
		return false;
	}

	public function isBizProcEnabled(): bool
	{
		return true;
	}

	public function isObserversEnabled(): bool
	{
		return true;
	}

	public function isMultiFieldsEnabled(): bool
	{
		return true;
	}

	public function isCountersEnabled(): bool
	{
		return true;
	}

	public function getDataClass(): string
	{
		return ContactTable::class;
	}

	public function isFieldExists(string $commonFieldName): bool
	{
		if (
			$commonFieldName === Item\Contact::FIELD_NAME_COMPANIES
			|| $commonFieldName === Item\Contact::FIELD_NAME_COMPANY_IDS
		)
		{
			return parent::isFieldExists(Item\Contact::FIELD_NAME_COMPANY_BINDINGS);
		}

		return parent::isFieldExists($commonFieldName);
	}

	protected function prepareSelect(array $select): array
	{
		$select = parent::prepareSelect($select);

		if (in_array('*', $select, true) && $this->isFieldExists(Item\Contact::FIELD_NAME_COMPANIES))
		{
			$select[] = Item\Contact::FIELD_NAME_COMPANIES;
		}

		$selectWithoutCompanies = array_diff(
			$select,
			[Item\Contact::FIELD_NAME_COMPANIES, Item\Contact::FIELD_NAME_COMPANY_IDS],
		);
		$isCompaniesInSelect = ($select !== $selectWithoutCompanies);

		$select = $selectWithoutCompanies;

		if ($isCompaniesInSelect)
		{
			$select[] = Item\Contact::FIELD_NAME_COMPANY_BINDINGS;
			$select[] = Item::FIELD_NAME_COMPANY_ID;
		}

		return $select;
	}

	protected function configureItem(Item $item, EntityObject $entityObject): void
	{
		parent::configureItem($item, $entityObject);

		$fieldNameMap =
			(new Item\FieldImplementation\Binding\FieldNameMap())
				->setSingleId(Item::FIELD_NAME_COMPANY_ID)
				->setMultipleIds(Item\Contact::FIELD_NAME_COMPANY_IDS)
				->setBindings(Item\Contact::FIELD_NAME_COMPANY_BINDINGS)
				->setBoundEntities(Item\Contact::FIELD_NAME_COMPANIES)
		;

		$item->addImplementation(
			new Item\FieldImplementation\Binding(
				$entityObject,
				\CCrmOwnerType::Company,
				$fieldNameMap,
				ContactCompanyTable::getEntity(),
				Container::getInstance()->getCompanyBroker(),
			)
		);
	}

	/**
	 * @inheritDoc
	 */
	public function getFieldsMap(): array
	{
		return [
			Item::FIELD_NAME_CREATED_TIME => 'DATE_CREATE',
			Item::FIELD_NAME_UPDATED_TIME => 'DATE_MODIFY',
			Item::FIELD_NAME_CREATED_BY => 'CREATED_BY_ID',
			Item::FIELD_NAME_UPDATED_BY => 'MODIFY_BY_ID',
			Item::FIELD_NAME_OBSERVERS => 'OBSERVER_IDS',
		];
	}

	public function getEntityTypeId(): int
	{
		return \CCrmOwnerType::Contact;
	}

	protected function getFieldsSettings(): array
	{
		return [
			Item::FIELD_NAME_ID => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
			],
			Item::FIELD_NAME_HONORIFIC => [
				'TYPE' => Field::TYPE_CRM_STATUS,
				'CRM_STATUS_TYPE' => StatusTable::ENTITY_ID_HONORIFIC,
			],
			Item::FIELD_NAME_NAME => [
				'TYPE' => Field::TYPE_STRING,
			],
			Item::FIELD_NAME_SECOND_NAME => [
				'TYPE' => Field::TYPE_STRING,
			],
			Item::FIELD_NAME_LAST_NAME => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::HasDefaultValue],
				'CLASS' => Field\LastName::class
			],
			Item::FIELD_NAME_FULL_NAME => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::ReadOnly,
					\CCrmFieldInfoAttr::AutoGenerated,
				],
				'CLASS' => Field\FullName::class,
			],
			Item\Contact::FIELD_NAME_PHOTO => [
				'TYPE' => Field::TYPE_FILE,
				'VALUE_TYPE' => Field::VALUE_TYPE_IMAGE,
				'CLASS' => Field\Photo::class,
			],
			Item::FIELD_NAME_BIRTHDATE => [
				'TYPE' => Field::TYPE_DATE,
			],
			Item::FIELD_NAME_BIRTHDAY_SORT => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [
					\CCrmFieldInfoAttr::Hidden,
					\CCrmFieldInfoAttr::ReadOnly,
					\CCrmFieldInfoAttr::AutoGenerated,
				],
				'CLASS' => Field\BirthdaySort::class,
			],
			Item::FIELD_NAME_TYPE_ID => [
				'TYPE' => Field::TYPE_CRM_STATUS,
				'CRM_STATUS_TYPE' => StatusTable::ENTITY_ID_CONTACT_TYPE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::HasDefaultValue],
			],
			Item::FIELD_NAME_SOURCE_ID => [
				'TYPE' => Field::TYPE_CRM_STATUS,
				'CRM_STATUS_TYPE' => StatusTable::ENTITY_ID_SOURCE,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::HasDefaultValue],
			],
			Item::FIELD_NAME_SOURCE_DESCRIPTION => [
				'TYPE' => Field::TYPE_TEXT,
			],
			Item::FIELD_NAME_POST => [
				'TYPE' => Field::TYPE_STRING,
			],
			Item::FIELD_NAME_COMMENTS => [
				'TYPE' => Field::TYPE_TEXT,
				'VALUE_TYPE' => Field::VALUE_TYPE_BB,
				'ATTRIBUTES' => [],
				'CLASS' => Field\Comments::class,
			],
			Item::FIELD_NAME_OPENED => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Required],
				'CLASS' => Field\Opened::class,
			],
			Item\Contact::FIELD_NAME_EXPORT => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [],
			],
			Item::FIELD_NAME_HAS_PHONE => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\HasPhone::class,
			],
			Item::FIELD_NAME_HAS_EMAIL => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\HasEmail::class,
			],
			Item::FIELD_NAME_HAS_IMOL => [
				'TYPE' => Field::TYPE_BOOLEAN,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\HasImol::class,
			],
			Item::FIELD_NAME_ASSIGNED => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::CanNotBeEmptied, \CCrmFieldInfoAttr::HasDefaultValue],
				'CLASS' => Field\Assigned::class,
			],
			Item::FIELD_NAME_CREATED_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\CreatedBy::class,
			],
			Item::FIELD_NAME_UPDATED_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\UpdatedBy::class,
			],
			Item::FIELD_NAME_LAST_ACTIVITY_BY => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
			],
			Item::FIELD_NAME_CREATED_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\CreatedTime::class,
			],
			Item::FIELD_NAME_UPDATED_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\UpdatedTime::class,
			],
			Item::FIELD_NAME_LAST_ACTIVITY_TIME => [
				'TYPE' => Field::TYPE_DATETIME,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::ReadOnly, \CCrmFieldInfoAttr::AutoGenerated],
				'CLASS' => Field\LastActivityTime::class,
			],
			Item::FIELD_NAME_COMPANY_ID => [
				'TYPE' => Field::TYPE_CRM_COMPANY,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::Deprecated],
			],
			Item\Contact::FIELD_NAME_COMPANY_IDS => [
				'TYPE' => Field::TYPE_CRM_COMPANY,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Multiple],
			],
			Item\Contact::FIELD_NAME_COMPANIES => [
				'TYPE' => Field::TYPE_CRM_COMPANY,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Multiple]
			],
			Item::FIELD_NAME_LEAD_ID => [
				'TYPE' => Field::TYPE_CRM_LEAD,
				'SETTINGS' => [
					'parentEntityTypeId' => \CCrmOwnerType::Lead,
				],
			],
			Item::FIELD_NAME_ORIGINATOR_ID => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item::FIELD_NAME_ORIGIN_ID => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item::FIELD_NAME_ORIGIN_VERSION => [
				'TYPE' => Field::TYPE_STRING,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed],
			],
			Item::FIELD_NAME_FACE_ID => [
				'TYPE' => Field::TYPE_INTEGER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Hidden],
			],
			Item::FIELD_NAME_CATEGORY_ID => [
				'TYPE' => Field::TYPE_CRM_CATEGORY,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::NotDisplayed, \CCrmFieldInfoAttr::HasDefaultValue],
				'CLASS' => Field\Category::class,
			],
			Item::FIELD_NAME_FM => [
				'TYPE' => Field::TYPE_CRM_MULTIFIELD,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Multiple],
				'CLASS' => Field\Multifield::class,
			],
			Item::FIELD_NAME_OBSERVERS => [
				'TYPE' => Field::TYPE_USER,
				'ATTRIBUTES' => [\CCrmFieldInfoAttr::Multiple],
				'CLASS' => Field\Observers::class,
			],
		];
	}

	//region categories

	/**
	 * Returns true if this entity supports categories.
	 *
	 * @return bool
	 */
	public function isCategoriesSupported(): bool
	{
		return true;
	}

	public function createCategory(array $data = []): Category
	{
		$object = ItemCategoryTable::createObject($data);
		$object->setEntityTypeId($this->getEntityTypeId());

		return new ItemCategory($object);
	}

	protected function loadCategories(): array
	{
		$categories = ItemCategoryTable::getItemCategoriesByEntityTypeId($this->getEntityTypeId());

		Service\Container::getInstance()->getLocalization()->loadMessages();
		$defaultCategory = new ClientDefaultCategory(
			$this->getEntityTypeId(),
			Loc::getMessage("CRM_TYPE_CATEGORY_DEFAULT_NAME"),
			0
		);

		array_unshift($categories, $defaultCategory);

		return $categories;
	}

	//endregion

	protected function getTrackedFieldNames(): array
	{
		return [
			Item::FIELD_NAME_HONORIFIC,
			Item::FIELD_NAME_NAME,
			Item::FIELD_NAME_LAST_NAME,
			Item::FIELD_NAME_SECOND_NAME,
			Item::FIELD_NAME_FM,
			Item::FIELD_NAME_POST,
			Item::FIELD_NAME_COMMENTS,
			Item::FIELD_NAME_SOURCE_ID,
			Item::FIELD_NAME_SOURCE_DESCRIPTION,
			Item::FIELD_NAME_TYPE_ID,
			Item::FIELD_NAME_ASSIGNED,
			Item::FIELD_NAME_BIRTHDATE,
		];
	}

	protected function getDependantTrackedObjects(): array
	{
		return [];
	}

	protected function configureAddOperation(Operation $operation): void
	{
		$operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Update(
					'OnBeforeCrmContactAdd',
					'CRM_CONTACT_CREATION_CANCELED',
				),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache('b_crm_contact'),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessSendNotification\WhenAddingEntity(),
			)
		;

		if ($operation->getItem()->getCategoryId() === 0)
		{
			$operation
				->addAction(
					Operation::ACTION_AFTER_SAVE,
					new Operation\Action\Compatible\SocialNetwork\ProcessAdd(),
				)
			;
		}

		$operation
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent('OnAfterCrmContactAdd'),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent\ExternalAdd('OnAfterExternalCrmContactAdd'),
			)
		;
	}

	public function getUpdateOperation(Item $item, Context $context = null): Operation\Update
	{
		$operation = parent::getUpdateOperation($item, $context);

		$operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Update(
					'OnBeforeCrmContactUpdate',
					'CRM_CONTACT_UPDATE_CANCELED',
				),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache(
					null,
					'crm_entity_name_' . $this->getEntityTypeId() . '_',
					[Item::FIELD_NAME_NAME, Item::FIELD_NAME_LAST_NAME, Item::FIELD_NAME_SECOND_NAME]
				)
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\FillEntityFieldsContext()
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ResetEntityCommunicationSettingsInActivities(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessSendNotification\WhenUpdatingEntity(),
			)
		;

		if ($operation->getItem()->getCategoryId() === 0)
		{
			$operation
				->addAction(
					Operation::ACTION_AFTER_SAVE,
					new Operation\Action\Compatible\SocialNetwork\ProcessUpdate(),
				)
			;
		}

		$operation
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent('OnAfterCrmContactUpdate'),
			)
		;

		return $operation;
	}

	public function getDeleteOperation(Item $item, Context $context = null): Operation\Delete
	{
		$operation = parent::getDeleteOperation($item, $context);

		$operation
			->addAction(
				Operation::ACTION_BEFORE_SAVE,
				new Operation\Action\Compatible\SendEvent\WithCancel\Delete('OnBeforeCrmContactDelete')
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearCache(
					'b_crm_contact',
					'crm_entity_name_' . $this->getEntityTypeId() . '_'
				)
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SocialNetwork\ProcessDelete(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\ClearEntitySelectorPrepareRequisiteDataCache(),
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\Compatible\SendEvent\Delete('OnAfterCrmContactDelete')
			)
			->addAction(
				Operation::ACTION_AFTER_SAVE,
				new Operation\Action\DeleteEntityFieldsContext()
			)
		;

		return $operation;
	}

	protected function getStatisticsFacade(): ?Statistics\OperationFacade
	{
		return new Statistics\OperationFacade\Contact();
	}

	public function isCommunicationRoutingSupported(): bool
	{
		return true;
	}
}
