<?php

if (!defined('B_PROLOG_INCLUDED') || B_PROLOG_INCLUDED !== true)
{
	die();
}

use Bitrix\Bizproc\FieldType;
use Bitrix\Crm\ItemIdentifier;
use Bitrix\Crm\RelationIdentifier;
use Bitrix\Crm\Service\Container;
use Bitrix\Main\Error;
use Bitrix\Main\Localization\Loc;

class CBPCrmGetRelationsInfoActivity extends \Bitrix\Bizproc\Activity\BaseActivity
{
	protected static $requiredModules = ['crm'];

	public function __construct($name)
	{
		parent::__construct($name);

		$this->arProperties = [
			'Title' => '',
			'ParentTypeId' => 0,

			// return
			'ParentEntityFields' => null,
		];

		$this->SetPropertiesTypes([
			'ParentTypeId' => ['Type' => FieldType::INT],
		]);
	}

	protected function checkProperties(): \Bitrix\Main\ErrorCollection
	{
		$errors = parent::checkProperties();

		if (CCrmBizProcHelper::ResolveDocumentName($this->ParentTypeId) === '')
		{
			$errors->setError(new Error(Loc::getMessage('CRM_GRI_PARENT_TYPE_EXISTENCE_ERROR')));
		}

		return $errors;
	}

	protected function internalExecute(): \Bitrix\Main\ErrorCollection
	{
		$errors = parent::internalExecute();

		$searchResult = $this->findParentDocumentId();
		if ($searchResult->isSuccess())
		{
			$document = static::getDocumentService()->GetDocument($searchResult->getData());
			$documentFields = array_keys($this->ParentEntityFields);

			foreach ($documentFields as $fieldId)
			{
				$this->arProperties[$fieldId] = $document[$fieldId] ?? null;
			}
			$this->setPropertiesTypes($this->ParentEntityFields);
		}
		else
		{
			$errors->add($searchResult->getErrors());
		}

		return $errors;
	}

	protected function findParentDocumentId(): \Bitrix\Main\Result
	{
		$result = new \Bitrix\Main\Result();

		[$currentEntityTypeId, $currentEntityId] = CCrmBizProcHelper::resolveEntityId($this->GetDocumentId());
		$currentElement = new ItemIdentifier($currentEntityTypeId, $currentEntityId);

		$parentElements = $this->getParentElements($currentElement);

		if (is_array($parentElements))
		{
			if ($parentElements)
			{
				$parentTypeId = $parentElements[0]->getEntityTypeId();
				$parentId = $parentElements[0]->getEntityId();

				$result->setData(CCrmBizProcHelper::ResolveDocumentId($parentTypeId, $parentId));
			}
			else
			{
				$errorMessage = Loc::getMessage(
					'CRM_GRI_ENTITY_EXISTENCE_ERROR',
					['#ELEMENT_TYPE#' => CCrmOwnerType::GetDescription($this->ParentTypeId)]
				);
				$result->addError(new Error($errorMessage));
			}
		}
		else
		{
			$result->addError(new Error(Loc::getMessage('CRM_GRI_RELATION_EXISTENCE_ERROR')));
		}

		return $result;
	}

	protected function getParentElements(ItemIdentifier $child): ?array
	{
		if ($this->ParentTypeId === $child->getEntityTypeId())
		{
			return null;
		}

		$relationIdentifier = new RelationIdentifier($this->ParentTypeId, $child->getEntityTypeId());
		$relation = Container::getInstance()->getRelationManager()->getRelation($relationIdentifier);

		return isset($relation) ? $relation->getParentElements($child) : null;
	}

	protected static function extractPropertiesValues(\Bitrix\Bizproc\Activity\PropertiesDialog $dialog, array $fieldsMap): \Bitrix\Main\Result
	{
		$result = parent::extractPropertiesValues($dialog, $fieldsMap);

		if ($result->isSuccess())
		{
			$map = static::getPropertiesDialogMap($dialog);
			$properties = $result->getData();

			if (array_key_exists((int)$properties['ParentTypeId'], $map['ParentTypeId']['Options']))
			{
				$parentDocumentType = CCrmBizProcHelper::ResolveDocumentType($properties['ParentTypeId']);
				$properties['ParentEntityFields'] = static::filterAutoGeneratedPrintableFields(
					static::getDocumentService()->GetDocumentFields($parentDocumentType)
				);

				$result->setData($properties);
			}
			else
			{
				$result->addError(new Error(Loc::getMessage('CRM_GRI_PARENT_TYPE_ERROR')));
			}
		}

		return $result;
	}

	protected static function filterAutoGeneratedPrintableFields(array $fieldsMap): array
	{
		$autoGeneratedFieldTypes = [FieldType::BOOL, FieldType::FILE, FieldType::USER];

		foreach ($fieldsMap as $fieldId => $field)
		{
			$printablePostfixIndex = mb_strrpos($fieldId, '_PRINTABLE');
			if ($printablePostfixIndex !== false)
			{
				$originalFieldId = mb_substr($fieldId, 0, $printablePostfixIndex);
				if (!isset($fieldsMap[$originalFieldId]))
				{
					$originalFieldId = isset($fieldsMap["{$originalFieldId}_ID"]) ? $originalFieldId . '_ID' : '';
				}

				if (
					$originalFieldId
					&& in_array($fieldsMap[$originalFieldId]['Type'], $autoGeneratedFieldTypes, true)
				)
				{
					unset($fieldsMap[$fieldId]);
				}
			}
		}

		return $fieldsMap;
	}

	protected static function getFileName(): string
	{
		return __FILE__;
	}

	public static function getPropertiesDialogMap(?\Bitrix\Bizproc\Activity\PropertiesDialog $dialog = null): array
	{
		$map = [
			'ParentTypeId' => [
				'Name' => Loc::getMessage('CRM_GRI_BINDING'),
				'FieldName' => 'parent_type_id',
				'Type' => FieldType::SELECT,
				'Options' => [],
			],
		];

		if (isset($dialog))
		{
			$entityTypeId = CCrmOwnerType::ResolveID($dialog->getDocumentType()[2]);
			$map['ParentTypeId']['Options'] = static::getCustomParentBindingOptions($entityTypeId);
		}

		return $map;
	}

	protected static function getCustomParentBindingOptions(int $entityTypeId): array
	{
		$parentOptions = [];

		foreach (static::getCustomParentRelations($entityTypeId) as $relation)
		{
			$parentEntityTypeId = $relation->getParentEntityTypeId();
			$parentOptions[$parentEntityTypeId] = CCrmOwnerType::GetDescription($parentEntityTypeId);
		}

		return $parentOptions;
	}

	protected static function getCustomParentRelations(int $entityTypeId): \Bitrix\Crm\Relation\Collection
	{
		$relationManager = Container::getInstance()->getRelationManager();

		return $relationManager->getParentRelations($entityTypeId)->filterOutPredefinedRelations();
	}
}
