/**
 * Bitrix Im
 * Conference application
 *
 * @package bitrix
 * @subpackage mobile
 * @copyright 2001-2021 Bitrix
 */

// call
import * as Call from 'call.core';
import { Analytics } from 'call.lib.analytics';
import './css/view.css';

// im
import 'im.debug';
import 'im.application.launch';
import 'im.component.conference.conference-public';
import {DesktopApi} from 'im.v2.lib.desktop-api';
import { ConferenceModel, CallModel } from "im.model";
import { Controller } from 'im.controller';
import { Utils } from "im.lib.utils";
import { Cookie } from "im.lib.cookie";
import { LocalStorage } from "im.lib.localstorage";
import { Logger } from "im.lib.logger";
import { Clipboard } from 'im.lib.clipboard';
import { Desktop } from "im.lib.desktop";
import {
	EventType,
	ConferenceErrorCode,
	ConferenceRightPanelMode as RightPanelMode
} from "im.const";

//ui
import {Notifier, NotificationOptions} from 'ui.notification-manager';
import 'ui.notification';
import 'ui.buttons';
import 'ui.progressround';
import 'ui.viewer';
import { VueVendorV2 } from "ui.vue";
import { VuexBuilder } from "ui.vue.vuex";

// core
import { Loc, Tag, Dom, Text } from "main.core";
import "promise";
import 'main.date';
import {BaseEvent, EventEmitter} from 'main.core.events';

// pull and rest
import { PullClient } from "pull.client";
import { ImCallPullHandler } from "im.provider.pull";
import { CallRestClient } from "./utils/restclient"

class ConferenceApplication
{
	/* region 01. Initialize */
	constructor(params = {})
	{
		this.inited = false;
		this.hardwareInited = false;
		this.dialogInited = false;
		this.initPromise = new BX.Promise;

		this.params = params;
		this.params.userId = this.params.userId? parseInt(this.params.userId): 0;
		this.params.siteId = this.params.siteId || '';
		this.params.chatId = this.params.chatId? parseInt(this.params.chatId): 0;
		this.params.dialogId = this.params.chatId? 'chat'+this.params.chatId.toString(): '0';
		this.params.passwordRequired = !!this.params.passwordRequired;
		this.params.isBroadcast = !!this.params.isBroadcast;

		BX.Messenger.Lib.Logger.setConfig(params.loggerConfig);

		this.messagesQueue = [];

		this.template = null;
		this.rootNode = this.params.node || document.createElement('div');

		this.event = new VueVendorV2;
		this.callContainer = null;
		// this.callView = null;
		this.preCall = null;
		this.currentCall = null;
		this.videoStrategy = null;
		this.callDetails = {};
		this.showFeedback = true;

		this.featureConfig = {};
		(params.featureConfig || []).forEach(limit => {
			this.featureConfig[limit.id] = limit;
		});

		this.localVideoStream = null;

		// save reconnect camera
		this.lastUsedCameraId = null;
		this.reconnectingCameraId = null;

		this.conferencePageTagInterval = null;

		this.onCallUserInvitedHandler = this.onCallUserInvited.bind(this);
		this.onCallUserStateChangedHandler = this.onCallUserStateChanged.bind(this);
		this.onCallUserMicrophoneStateHandler = this.onCallUserMicrophoneState.bind(this);
		this.onCallUserCameraStateHandler = this.onCallUserCameraState.bind(this);
		this.onNeedResetMediaDevicesStateHandler = this.onNeedResetMediaDevicesState.bind(this);
		this.onCallUserVideoPausedHandler = this.onCallUserVideoPaused.bind(this);
		this.onCallLocalMediaReceivedHandler = BX.debounce(this.onCallLocalMediaReceived.bind(this), 1000);
		this.onCallRemoteMediaReceivedHandler = this.onCallRemoteMediaReceived.bind(this);
		this.onCallRemoteMediaStoppedHandler = this.onCallRemoteMediaStopped.bind(this);
		this.onCallUserVoiceStartedHandler = this.onCallUserVoiceStarted.bind(this);
		this.onCallUserVoiceStoppedHandler = this.onCallUserVoiceStopped.bind(this);
		this.onUserStatsReceivedHandler = this.onUserStatsReceived.bind(this);
		this.onCallUserScreenStateHandler = this.onCallUserScreenState.bind(this);
		this.onCallUserRecordStateHandler = this.onCallUserRecordState.bind(this);
		this.onCallUserFloorRequestHandler = this.onCallUserFloorRequest.bind(this);
		this.onMicrophoneLevelHandler = this.onMicrophoneLevel.bind(this);
		this._onCallJoinHandler = this.onCallJoin.bind(this);
		this.onCallFailureHandler = this.onCallFailure.bind(this);
		this.onCallLeaveHandler = this.onCallLeave.bind(this);
		this.onCallDestroyHandler = this.onCallDestroy.bind(this);
		this.onInputFocusHandler = this.onInputFocus.bind(this);
		this.onInputBlurHandler = this.onInputBlur.bind(this);
		this.onReconnectingHandler = this.onReconnecting.bind(this);
		this.onReconnectedHandler = this.onReconnected.bind(this);
		this.onUpdateLastUsedCameraIdHandler = this.onUpdateLastUsedCameraId.bind(this);
		this.onCallConnectionQualityChangedHandler = this.onCallConnectionQualityChanged.bind(this);
		this.onCallToggleRemoteParticipantVideoHandler = this.onCallToggleRemoteParticipantVideo.bind(this);
		this._onGetUserMediaEndedHandler = this.onGetUserMediaEnded.bind(this);
		this._onSwitchTrackRecordStatusHandler = this.onUpdateCallCopilotState.bind(this);

		this.onPreCallDestroyHandler = this.onPreCallDestroy.bind(this);
		this.onPreCallUserStateChangedHandler = this.onPreCallUserStateChanged.bind(this);

		this.waitingForCallStatus = false;
		this.waitingForCallStatusTimeout = null;
		this.callEventReceived = false;
		this.callRecordState = Call.View.RecordState.Stopped;
		this.callRecordInfo = null;

		this.floatingScreenShareWindow = null;
		this.webScreenSharePopup = null;
		this.screenShareStartTime = null;

		this.mutePopup = null;
		this.allowMutePopup = true;

		this.loopTimers = {};

		this.initDesktopEvents()
			.then(() => this.initRestClient())
			.then(() => this.subscribePreCallChanges())
			.then(() => this.subscribeNotifierEvents())
			.then(() => this.initPullClient())
			.then(() => this.initCore())
			.then(() => this.setModelData())
			.then(() => this.initComponent())
			.then(() => this.initCallInterface())
			.then(() => this.initHardware())
			.then(() => this.initUserComplete())
			.catch((error) => {
				console.error('Init error', error);
			})
		;
	}
		/* region 01. Initialize methods */
		initDesktopEvents()
		{
			if (!DesktopApi.isDesktop())
			{
				return new Promise((resolve, reject) => resolve());
			}

			this.floatingScreenShareWindow = new Call.FloatingScreenShare({
				onBackToCallClick: this.onFloatingScreenShareBackToCallClick.bind(this),
				onStopSharingClick: this.onFloatingScreenShareStopClick.bind(this),
				onChangeScreenClick: this.onFloatingScreenShareChangeScreenClick.bind(this)
			});

			if (this.floatingScreenShareWindow)
			{
				DesktopApi.subscribe("BXScreenMediaSharing", (id, title, x, y, width, height, app) =>
				{
					this.floatingScreenShareWindow.setSharingData({
						title: title,
						x: x,
						y: y,
						width: width,
						height: height,
						app: app
					}).then(() => {
						this.floatingScreenShareWindow.show();
					}).catch(error => {
						Logger.error('setSharingData error', error);
					});
				});

				window.addEventListener('focus', () => {
					this.onWindowFocus();
				});

				window.addEventListener('blur', () => {
					this.onWindowBlur();
				});
			}

			DesktopApi.subscribe('bxImUpdateCounterMessage', (counter) =>
			{
				if (!this.controller)
				{
					return false;
				}

				this.controller.getStore().commit('conference/common', {
					messageCount: counter
				});
			});

			EventEmitter.subscribe(EventType.textarea.focus, this.onInputFocusHandler);
			EventEmitter.subscribe(EventType.textarea.blur, this.onInputBlurHandler);
			EventEmitter.subscribe(EventType.conference.userRenameFocus, this.onInputFocusHandler);
			EventEmitter.subscribe(EventType.conference.userRenameBlur, this.onInputBlurHandler);

			return new Promise((resolve, reject) => resolve());
		}

		initRestClient()
		{
			this.restClient = new CallRestClient({endpoint: this.getHost()+'/rest'});
			this.restClient.setConfId(this.params.conferenceId);

			return new Promise((resolve, reject) => resolve());
		}

		subscribePreCallChanges()
		{
			BX.addCustomEvent(window, 'CallEvents::callCreated', this.onCallCreated.bind(this));
		}

		subscribeNotifierEvents()
		{
			Notifier.subscribe('click', (event: BaseEvent<NotifierClickParams>) => {
				const { id } = event.getData();
				if (id.startsWith('im-videconf'))
				{
					this.toggleChat();
				}
			});
		}

		initPullClient()
		{
			if (!this.params.isIntranetOrExtranet)
			{
				this.pullClient = new PullClient({
					serverEnabled: true,
					userId: this.params.userId,
					siteId: this.params.siteId,
					restClient: this.restClient,
					skipStorageInit: true,
					configTimestamp: 0,
					skipCheckRevision: true,
					getPublicListMethod: 'im.call.channel.public.list'
				});

				return new Promise((resolve, reject) => resolve());
			}
			else
			{
				this.pullClient = BX.PULL;

				return this.pullClient.start().then(() => {
					return new Promise((resolve, reject) => resolve());
				});
			}
		}

		initCore()
		{
			this.controller = new Controller({
				host: this.getHost(),
				siteId: this.params.siteId,
				userId: this.params.userId,
				languageId: this.params.language,
				pull: {client: this.pullClient},
				rest: {client: this.restClient},
				vuexBuilder: {
					database: !Utils.browser.isIe(),
					databaseName: 'imol/call',
					databaseType: VuexBuilder.DatabaseType.localStorage,
					models: [
						ConferenceModel.create(),
						CallModel.create()
					],
				}
			});

			window.BX.Messenger.Application.Core = {
				controller: this.controller
			};

			return new Promise((resolve, reject) => {
				this.controller.ready().then(() => resolve());
			});
		}

		setModelData()
		{
			this.controller.getStore().commit('application/set', {
				dialog: {
					chatId: this.getChatId(),
					dialogId: this.getDialogId()
				},
				options: {
					darkBackground: true
				}
			});

			//set presenters ID list
			const presentersIds = this.params.presenters.map(presenter => presenter['id']);
			this.controller.getStore().dispatch('conference/setBroadcastMode', {broadcastMode: this.params.isBroadcast});
			this.controller.getStore().dispatch('conference/setPresenters', {presenters: presentersIds});

			//set presenters info in users model
			this.params.presenters.forEach(presenter => {
				this.controller.getStore().dispatch('users/set', presenter);
			});

			if (this.params.passwordRequired)
			{
				this.controller.getStore().commit('conference/common', {
					passChecked: false,
				});
			}

			if (this.params.conferenceTitle)
			{
				this.controller.getStore().dispatch('conference/setConferenceTitle', {
					conferenceTitle: this.params.conferenceTitle,
				});
			}

			if (this.params.alias)
			{
				this.controller.getStore().commit('conference/setAlias', {
					alias: this.params.alias,
				});
			}

			return new Promise((resolve, reject) => resolve());
		}

		initComponent()
		{
			if (this.getStartupErrorCode())
			{
				this.setError(this.getStartupErrorCode());
			}

			return new Promise((resolve, reject) =>
			{
				this.controller.createVue(this, {
					el: this.rootNode,
					data: () =>
					{
						return {
							dialogId: this.getDialogId()
						};
					},
					template: `<bx-im-component-conference-public :dialogId="dialogId"/>`,
				}).then(vue =>
				{
					this.template = vue;
					resolve();
				}).catch(error => reject(error));
			});
		}

		initCallInterface()
		{
			return new Promise((resolve, reject) =>
			{
				try {
					this.callContainer = document.getElementById('bx-im-component-call-container');

					let hiddenButtons = ['document'];
					if (this.isViewerMode())
					{
						hiddenButtons = ['camera', 'microphone', 'screen', 'record', 'floorRequest', 'document'];
					}
					if (!this.params.isIntranetOrExtranet)
					{
						hiddenButtons.push('record');
					}

					if (!Call.Util.isConferenceChatEnabled())
					{
						hiddenButtons.push('chat');
					}

					this.callView = new Call.View({
						container: this.callContainer,
						showChatButtons: true,
						showUsersButton: true,
						showShareButton: this.getFeatureState('screenSharing') !== ConferenceApplication.FeatureState.Disabled,
						showRecordButton: this.getFeatureState('record') !== ConferenceApplication.FeatureState.Disabled,
						userLimit: Call.Util.getUserLimit(),
						isIntranetOrExtranet: !!this.params.isIntranetOrExtranet,
						language: this.params.language,
						layout: Utils.device.isMobile() ? Call.View.Layout.Mobile : Call.View.Layout.Centered,
						uiState: Call.View.UiState.Preparing,
						blockedButtons: ['camera', 'microphone', 'floorRequest', 'screen', 'record', 'copilot'],
						localUserState: Call.UserState.Idle,
						hiddenTopButtons: !this.isBroadcast() || this.getBroadcastPresenters().length > 1? []: ['grid'],
						hiddenButtons: hiddenButtons,
						broadcastingMode: this.isBroadcast(),
						broadcastingPresenters: this.getBroadcastPresenters(),
						isCopilotFeaturesEnabled: false,
						isCopilotActive: false,
					});

					this.callView.subscribe(Call.View.Event.onButtonClick, this.onCallButtonClick.bind(this));
					this.callView.subscribe(Call.View.Event.onReplaceCamera, this.onCallReplaceCamera.bind(this));
					this.callView.subscribe(Call.View.Event.onReplaceMicrophone, this.onCallReplaceMicrophone.bind(this));
					this.callView.subscribe(Call.View.Event.onReplaceSpeaker, this.onCallReplaceSpeaker.bind(this));
					this.callView.subscribe(Call.View.Event.onHasMainStream, this.onCallViewHasMainStream.bind(this));
					this.callView.subscribe(Call.View.Event.onChangeHdVideo, this.onCallViewChangeHdVideo.bind(this));
					this.callView.subscribe(Call.View.Event.onChangeMicAutoParams, this.onCallViewChangeMicAutoParams.bind(this));
					this.callView.subscribe(Call.View.Event.onChangeFaceImprove, this.onCallViewChangeFaceImprove.bind(this));
					this.callView.subscribe(Call.View.Event.onUserRename, this.onCallViewUserRename.bind(this));
					this.callView.subscribe(Call.View.Event.onUserPinned, this.onCallViewUserPinned.bind(this));
					this.callView.subscribe(Call.View.Event.onToggleSubscribe, this.onCallToggleSubscribe.bind(this));

					this.callView.blockAddUser();
					this.callView.blockHistoryButton();

					if (!Utils.device.isMobile())
					{
						this.callView.show();
					}

					resolve()
				} catch (error)
				{
					Logger.error('creating call interface conference', error);

					let errorCode;
					if (typeof (error) == "string")
					{
						errorCode = error;
					}
					else if (typeof (error) == "object" && error.code)
					{
						errorCode = error.code == 'access_denied' ? 'ACCESS_DENIED' : error.code
					}
					else
					{
						errorCode = 'UNKNOWN_ERROR';
					}

					this.onCallFailure({
						code: errorCode,
						message: error.message || "",
					})

					reject('call interface error');
				}
			})
		}

		initUserComplete()
		{
			return new Promise((resolve, reject) => {
				this.initUser()
					.then(() => this.startPageTagInterval())
					.then(() => this.tryJoinExistingCall())
					.then(() => this.initCall())
					.then(() => this.initPullHandlers())
					.then(() => this.subscribeToStoreChanges())
					.then(() => this.initComplete())
					.then(() => resolve)
					.catch((error) => reject(error));
			})
		}
		/* endregion 01. Initialize methods */

		/* region 02. initUserComplete methods */
		initUser()
		{
			return new Promise((resolve, reject) => {
				if (this.getStartupErrorCode() || !this.getConference().common.passChecked)
				{
					return reject();
				}

				if (this.params.userId > 0)
				{
					this.controller.setUserId(this.params.userId);

					if (this.params.isIntranetOrExtranet)
					{
						this.switchToSessAuth();

						this.controller.getStore().commit('conference/user', {
							id: this.params.userId
						});
					}
					else
					{
						let hashFromCookie = this.getUserHashCookie();
						if (hashFromCookie)
						{
							this.restClient.setAuthId(hashFromCookie);
							this.restClient.setChatId(this.getChatId());
							this.controller.getStore().commit('conference/user', {
								id: this.params.userId,
								hash: hashFromCookie
							});

							this.pullClient.start();
						}
					}

					this.controller.getStore().commit('conference/common', {
						inited: true
					});

					return resolve();
				}
				else
				{
					this.restClient.setAuthId('guest');
					this.restClient.setChatId(this.getChatId());

					if (typeof BX.SidePanel !== 'undefined')
					{
						BX.SidePanel.Instance.disableAnchorBinding();
					}

					return this.restClient.callMethod('im.call.user.register', {
						alias: this.params.alias,
						user_hash: this.getUserHashCookie() || '',
					}).then(result =>
					{
						BX.message['USER_ID'] = result.data().id;
						this.controller.getStore().commit('conference/user', {
							id: result.data().id,
							hash: result.data().hash
						});

						this.controller.setUserId(result.data().id);
						this.callView.setLocalUserId(result.data().id);

						if (result.data().created)
						{
							this.params.userCount++;
						}

						this.controller.getStore().commit('conference/common', {
							inited: true
						});

						this.restClient.setAuthId(result.data().hash);
						this.pullClient.start();

						return resolve();
					});
				}
			});
		}

		startPageTagInterval()
		{
			return new Promise((resolve) => {
				clearInterval(this.conferencePageTagInterval);
				this.conferencePageTagInterval = setInterval(() => {
					LocalStorage.set(this.params.siteId, this.params.userId, BX.CallEngine.getConferencePageTag(this.params.dialogId), "Y", 2);
				}, 1000);
				resolve();
			})
		}

		tryJoinExistingCall()
		{
			const provider = Call.Provider.Bitrix;

			this.restClient.callMethod("im.call.tryJoinCall", {
					entityType: 'chat',
					entityId: this.params.dialogId,
					provider: provider,
					type: Call.Type.Permanent
				})
				.then(result => {
					Logger.warn('tryJoinCall', result.data());
					if (result.data().success)
					{
						this.waitingForCallStatus = true;
						this.waitingForCallStatusTimeout = setTimeout(() => {
							this.waitingForCallStatus = false;
							if (!this.callEventReceived)
							{
								this.setConferenceStatus(false);
							}
							this.callEventReceived = false;
						}, 5000);
					}
					else
					{
						this.setConferenceStatus(false);
					}
				})
		}

		initCall()
		{
			Call.Engine.setRestClient(this.restClient);
			Call.Engine.setPullClient(this.pullClient);
			Call.Engine.setCurrentUserId(this.controller.getUserId());
			this.callView.unblockButtons(['chat']);
		}

		initPullHandlers()
		{
			this.pullClient.subscribe(
				new ImCallPullHandler({
					store: this.controller.getStore(),
					application: this,
					controller: this.controller,
				})
			);

			return new Promise((resolve, reject) => resolve());
		}

		subscribeToStoreChanges()
		{
			this.controller.getStore().subscribe((mutation, state) => {
				const { payload, type } = mutation;
				if (type === 'users/update' && payload.fields.name)
				{
					if (!this.callView)
					{
						return false;
					}

					this.callView.updateUserData(
						{[payload.id]: {name: payload.fields.name}}
					);
				}
				else if (type === 'dialogues/set')
				{
					if (payload[0].dialogId !== this.getDialogId())
					{
						return false;
					}

					if (!Utils.platform.isBitrixDesktop())
					{
						this.callView.setButtonCounter('chat', payload[0].counter);
					}
				}
				else if (type === 'dialogues/update')
				{
					if (payload.dialogId !== this.getDialogId())
					{
						return false;
					}

					if (typeof payload.fields.counter === 'number' && this.callView)
					{
						if (Utils.platform.isBitrixDesktop())
						{
							if (
								payload.actionName === "decreaseCounter"
								&& !payload.dialogMuted
								&& typeof payload.fields.previousCounter === 'number'
							)
							{
								let counter = payload.fields.counter;
								if (this.getConference().common.messageCount)
								{
									counter = this.getConference().common.messageCount - (payload.fields.previousCounter - counter);
									if (counter < 0)
									{
										counter = 0;
									}
								}
								this.callView.setButtonCounter('chat', counter);
							}
						}
						else
						{
							this.callView.setButtonCounter('chat', payload.fields.counter);
						}
					}

					if (typeof payload.fields.name !== 'undefined')
					{
						document.title = payload.fields.name.toString();
					}
				}
				else if (type === 'conference/common' && typeof payload.messageCount === 'number')
				{
					if (this.callView)
					{
						this.callView.setButtonCounter('chat', payload.messageCount);
					}
				}
			});
		}

		initComplete()
		{
			if (this.isExternalUser())
			{
				this.callView.localUser.userModel.allowRename = true;
			}

			if (this.getConference().common.inited)
			{
				this.inited = true;
				this.initPromise.resolve(this);
			}

			if (DesktopApi.isDesktop())
			{
				DesktopApi.emitToMainWindow('bxConferenceLoadComplete', []);
			}

			return new Promise((resolve, reject) => resolve());
		}
		/* endregion 02. initUserComplete methods */
	/* endregion 01. Initialize */

	/* region 02. Methods */

	/* region 01. Call methods */
	initHardware()
	{
		return new Promise((resolve, reject) =>
		{
			Call.Hardware.init().then(() => {
				if (this.hardwareInited)
				{
					resolve();
					return true;
				}

				if (Object.values(Call.Hardware.microphoneList).length === 0)
				{
					this.setError(ConferenceErrorCode.missingMicrophone);
				}

				if (!this.isViewerMode())
				{
					this.callView.unblockButtons(["camera", "microphone"]);
					this.callView.enableMediaSelection();
				}

				Call.Hardware.subscribe(Call.Hardware.Events.deviceChanged, this._onDeviceChange.bind(this));

				this.hardwareInited = true;
				resolve();
			}).catch(error => {
				if (error === 'NO_WEBRTC' && this.isHttps())
				{
					this.setError(ConferenceErrorCode.unsupportedBrowser);
				}
				else if (error === 'NO_WEBRTC' && !this.isHttps())
				{
					this.setError(ConferenceErrorCode.unsafeConnection);
				}
				Logger.error('Init hardware error', error);
				reject(error)
			})
		});
	}

	_onDeviceChange(e)
	{
		if (!this.currentCall || !this.currentCall.ready)
		{
			return;
		}

		const allAddedDevice = e.data.added;
		const allRemovedDevice = e.data.removed;
		const removed = Call.Hardware.getRemovedUsedDevices(
			e.data.removed,
			{
				microphoneId: this.currentCall.microphoneId,
				cameraId: this.currentCall.cameraId,
				speakerId: this.callView.speakerId,
			}
		);

		if (allAddedDevice)
		{
			setTimeout(() => this.useDevicesInCurrentCall(allAddedDevice), 500);
		}

		if (removed.length > 0)
		{
			BX.UI.Notification.Center.notify({
				content: BX.message("IM_CALL_DEVICES_DETACHED") + "<br><ul>" + removed.map(function (deviceInfo)
				{
					return "<li>" + deviceInfo.label
				}) + "</ul>",
				position: "top-right",
				autoHideDelay: 10000,
				closeButton: true,
				//category: "call-device-change",
				actions: [{
					title: BX.message("IM_CALL_DEVICES_CLOSE"),
					events: {
						click: function (event, balloon)
						{
							balloon.close();
						}
					}
				}]
			});
		}

		if (allRemovedDevice)
		{
			setTimeout(() => this.removeDevicesFromCurrentCall(allRemovedDevice), 500);
		}
	}

	useDevicesInCurrentCall(deviceList, isForceUse = false)
	{
		if (!this.currentCall || !this.currentCall.ready)
		{
			return;
		}

		for (let i = 0; i < deviceList.length; i++)
		{
			const deviceInfo = deviceList[i];

			switch (deviceInfo.kind)
			{
				case "audioinput":
					if (deviceInfo.deviceId === 'default' || isForceUse)
					{
						const newDeviceId = Call.Hardware.getDefaultDeviceIdByGroupId(deviceInfo.groupId, 'audioinput');
						this.currentCall.setMicrophoneId(newDeviceId);
						this.callView.setMicrophoneId(newDeviceId);
					}
					break;
				case "videoinput":
					if (deviceInfo.deviceId === 'default' || isForceUse)
					{
						this.currentCall.setCameraId(deviceInfo.deviceId);
					}

					if (this.reconnectingCameraId === deviceInfo.deviceId && !Call.Hardware.isCameraOn)
					{
					   this.updateCameraSettingsInCurrentCallAfterReconnecting(deviceInfo.deviceId)
					}
					break;
				case "audiooutput":
					if (this.callView && deviceInfo.deviceId === 'default' || isForceUse)
					{
						const newDeviceId = Call.Hardware.getDefaultDeviceIdByGroupId(deviceInfo.groupId, 'audiooutput');
						this.callView.setSpeakerId(newDeviceId);
					}
					break;
			}
		}
	}

	removeDevicesFromCurrentCall(deviceList)
	{
		if (!this.currentCall || !this.currentCall.ready)
		{
			return;
		}

		for (let i = 0; i < deviceList.length; i++)
		{
			const deviceInfo = deviceList[i];

			switch (deviceInfo.kind)
			{
				case "audioinput":
					if (this.currentCall.microphoneId == deviceInfo.deviceId)
					{
						const microphoneIds = Object.keys(Call.Hardware.microphoneList);
						let deviceId;

						if (microphoneIds.includes('default'))
						{
							const deviceGroup = Call.Hardware.getDeviceGroupIdByDeviceId('default', 'audioinput');
							deviceId = Call.Hardware.getDefaultDeviceIdByGroupId(deviceGroup, 'audioinput');
						}

						if (!deviceId)
						{
							deviceId = microphoneIds.length > 0 ? microphoneIds[0] : "";
						}

						this.currentCall.setMicrophoneId(deviceId);

						if (this.currentCall.provider === Call.Provider.Bitrix)
						{
							this.callView.setMicrophoneId(deviceId);
						}
					}
					break;
				case "videoinput":
					if (this.currentCall.cameraId == deviceInfo.deviceId)
					{
						const cameraIds = Object.keys(Call.Hardware.cameraList);
						this.currentCall.setCameraId(cameraIds.length > 0 ? cameraIds[0] : "");
					}
					break;
				case "audiooutput":
					if (this.callView && this.callView.speakerId == deviceInfo.deviceId)
					{
						const speakerIds = Object.keys(Call.Hardware.audioOutputList);
						let deviceId;

						if (speakerIds.includes('default'))
						{
							const deviceGroup = Call.Hardware.getDeviceGroupIdByDeviceId('default', 'audiooutput');
							deviceId = Call.Hardware.getDefaultDeviceIdByGroupId(deviceGroup, 'audiooutput');
						}

						if (!deviceId)
						{
							this.callView.setSpeakerId(speakerIds.length > 0 ? speakerIds[0] : "");
						}
					}
					break;
			}
		}
	}

	startCall(videoEnabled, viewerMode = false, withCopilot)
	{
		if (this.initCallPromise)
		{
			return;
		}

		const provider = Call.Provider.Bitrix;

		if (Utils.device.isMobile())
		{
			this.callView.show();
			this.callView.setButtonCounter('chat', this.getDialogData().counter);
		}
		else
		{
			this.callView.setLayout(Call.View.Layout.Grid);
		}

		this.callView.setUiState(Call.View.UiState.Calling);

		if (videoEnabled && !Call.Hardware.hasCamera())
		{
			this.showNotification(BX.message('IM_CALL_NO_CAMERA_ERROR'));
			videoEnabled = false;
		}

		if (this.localVideoStream)
		{
			if (!videoEnabled)
			{
				this.stopLocalVideoStream();
			}
		}
		this.controller.getStore().commit('conference/startCall');

		this.initCallPromise = Call.Engine.createCall({
			type: Call.Type.Permanent,
			entityType: 'chat',
			entityId: this.getDialogId(),
			provider: provider,
			videoEnabled: videoEnabled,
			enableMicAutoParameters: Call.Hardware.enableMicAutoParameters,
			joinExisting: true
		}).then(e => {
			Logger.warn('call created', e);

			this.currentCall = e.call;
			//this.currentCall.useHdVideo(Call.Hardware.preferHdQuality);
			this.currentCall.useHdVideo(true);
			this.onUpdateCallCopilotState({
				isTrackRecordOn: this.currentCall.isCopilotActive,
			});
			if(Call.Hardware.defaultMicrophone)
			{
				this.currentCall.setMicrophoneId(Call.Hardware.defaultMicrophone);
			}
			if(Call.Hardware.defaultCamera)
			{
				this.currentCall.setCameraId(Call.Hardware.defaultCamera);
			}

			if(!Utils.device.isMobile())
			{
				this.callView.setLayout(Call.View.Layout.Grid);
			}
			this.callView.appendUsers(this.currentCall.getUsers());
			Call.Util.getUsers(this.currentCall.id, this.getCallUsers(true)).then(userData => {
				this.controller.getStore().dispatch('users/set', Object.values(userData));
				this.controller.getStore().dispatch('conference/setUsers', {users: Object.keys(userData)});
				this.callView.updateUserData(userData)
			});
			this.releasePreCall();
			this.bindCallEvents();
			this.updateCallUser(this.currentCall.userId, {microphoneState: !Call.Hardware.isMicrophoneMuted});
			if(e.isNew)
			{
				Analytics.getInstance().onStartVideoconf({
					callId: this.currentCall?.id,
					withVideo: videoEnabled,
					mediaParams: {
						video: Call.Hardware.isCameraOn,
						audio: !Call.Hardware.isMicrophoneMuted,
					},
					status: Analytics.AnalyticsStatus.success,
					isCopilotActive: this.currentCall.isCopilotActive,
				});

				this.currentCall.inviteUsers();
			}
			else
			{
				this.currentCall.answer({
					joinAsViewer: viewerMode
				});
				Analytics.getInstance().onJoinVideoconf({
					callId: this.currentCall?.id,
					withVideo: videoEnabled,
					mediaParams: {
						video: Call.Hardware.isCameraOn,
						audio: !Call.Hardware.isMicrophoneMuted,
					},
					status: Analytics.AnalyticsStatus.success,
				});
			}

			this.onUpdateLastUsedCameraId();

		}).catch(error => {
			Logger.error('creating call error', error);
			let errorCode;
			if (typeof (error) == "string")
			{
				errorCode = error;
			}
			else if (typeof (error) == "object" && error.code)
			{
				errorCode = error.code == 'access_denied' ? 'ACCESS_DENIED' : error.code
			}
			else
			{
				errorCode = 'UNKNOWN_ERROR';
			}
			this.onCallFailure({
				code: errorCode,
				message: error.message || "",
			})
			this.initCallPromise = null;
		});
	}

	/**
	 * @param {int} callId
	 * @param {object} options
	 */
	joinCall(callId, options)
	{
		if (this.initCallPromise)
		{
			return;
		}

		let video = BX.prop.getBoolean(options, "video", false);
		let joinAsViewer = BX.prop.getBoolean(options, "joinAsViewer", false);
		if (!!video)
		{
			Call.Hardware.isCameraOn = false;
		}

		if (Utils.device.isMobile())
		{
			this.callView.show();
		}
		else
		{
			this.callView.setLayout(Call.View.Layout.Grid);
		}

		if (joinAsViewer)
		{
			this.callView.setLocalUserDirection(Call.EndpointDirection.RecvOnly);
		}
		else
		{
			this.callView.setLocalUserDirection(Call.EndpointDirection.SendRecv);
		}

		this.callView.setUiState(Call.View.UiState.Calling);
		this.initCallPromise = Call.Engine.getCallWithId(callId).then((result) =>
		{
			this.currentCall = result.call;
			this.releasePreCall();
			this.bindCallEvents();

			this.controller.getStore().commit('conference/startCall');

			this.callView.appendUsers(this.currentCall.getUsers());
			Call.Util.getUsers(this.currentCall.id, this.getCallUsers(true)).then(userData => {
				this.controller.getStore().dispatch('users/set', Object.values(userData));
				this.controller.getStore().dispatch('conference/setUsers', {users: Object.keys(userData)});
				this.callView.updateUserData(userData)
			});

			if (!joinAsViewer)
			{
				//this.currentCall.useHdVideo(Call.Hardware.preferHdQuality);
				this.currentCall.useHdVideo(true);
				if (Call.Hardware.defaultMicrophone)
				{
					this.currentCall.setMicrophoneId(Call.Hardware.defaultMicrophone);
				}
				if (Call.Hardware.defaultCamera)
				{
					this.currentCall.setCameraId(Call.Hardware.defaultCamera);
				}
				this.updateCallUser(this.currentCall.userId, {microphoneState: !Call.Hardware.isMicrophoneMuted});
			}

			this.currentCall.answer({
				joinAsViewer: joinAsViewer
			});
			this.onUpdateLastUsedCameraId();
		}).catch((error) =>
		{
			console.error(error);
			this.initCallPromise = null;
		});
	}

	endCall(finishCall = false)
	{
		if (this.currentCall)
		{
			this.showFeedback = this.currentCall.wasConnected;
			this.callDetails = {
				id: this.currentCall.id,
				provider: this.currentCall.provider,
				userCount: this.currentCall.users.length,
				browser: Call.Util.getBrowserForStatistics(),
				isMobile: BX.browser.IsMobile(),
				isConference: true
			}

			this.removeCallEvents();
			this.currentCall.hangup(false, '', finishCall);
		}

		if (this.isRecording())
		{
			Analytics.getInstance().onRecordStop({
				callId: this.currentCall.id,
				callType: Analytics.AnalyticsType.videoconf,
				subSection: finishCall ? Analytics.AnalyticsSubSection.contextMenu : Analytics.AnalyticsSubSection.window,
				element: finishCall ? Analytics.AnalyticsElement.finishForAllButton : Analytics.AnalyticsElement.disconnectButton,
				recordTime: Call.Util.getRecordTimeText(this.callRecordInfo),
			});
			this.callRecordInfo = null;

			BXDesktopSystem.CallRecordStop();
		}
		this.callRecordState = Call.View.RecordState.Stopped;

		if (Utils.platform.isBitrixDesktop())
		{
			if (this.floatingScreenShareWindow)
			{
				this.floatingScreenShareWindow.destroy();
				this.floatingScreenShareWindow = null;
			}

			window.close();
			// if the conference was opened incorrectly, then "window.close();" may not work in some cases
			// as a workaround, we can redirect the user back to the portal's front page.
			location.href = '/';
		}
		else
		{
			this.callView.releaseLocalMedia();
			this.callView.close();
			this.closeReconnectionBaloon();
			this.setError(ConferenceErrorCode.userLeftCall);
			this.controller.getStore().commit('conference/endCall');
		}

		EventEmitter.unsubscribe(EventType.textarea.focus, this.onInputFocusHandler);
		EventEmitter.unsubscribe(EventType.textarea.blur, this.onInputBlurHandler);
		EventEmitter.unsubscribe(EventType.conference.userRenameFocus, this.onInputFocusHandler);
		EventEmitter.unsubscribe(EventType.conference.userRenameBlur, this.onInputBlurHandler);
	}

	restart()
	{
		console.trace(" restart");
		if(this.currentCall)
		{
			this.removeCallEvents();
			this.currentCall = null;
		}

		if(this.callView)
		{
			this.callView.releaseLocalMedia();
			this.callView.close();
			this.closeReconnectionBaloon();
			this.callView.destroy();
			this.callView = null;
		}
		this.initCallInterface();
		this.initCall();
		this.controller.getStore().commit('conference/endCall');
	}

	kickFromCall()
	{
		this.setError(ConferenceErrorCode.kickedFromCall);
		this.pullClient.disconnect();
		this.endCall();
	}

	getCallUsers(includeSelf)
	{
		if (!this.currentCall)
		{
			return [];
		}

		let result = Object.keys(this.currentCall.getUsers());
		if (includeSelf)
		{
			result.push(this.currentCall.userId);
		}
		return result;
	}

	getActiveCallUsers()
	{
		const userStates = this.currentCall.getUsers();
		let activeUsers = [];

		for (let userId in userStates)
		{
			if (userStates.hasOwnProperty(userId))
			{
				if (
					userStates[userId] === Call.UserState.Connected
					|| userStates[userId] === Call.UserState.Connecting
					|| userStates[userId] === Call.UserState.Calling
				)
				{
					activeUsers.push(userId);
				}
			}
		}
		return activeUsers;
	}

	setLocalVideoStream(stream)
	{
		this.localVideoStream = stream;
	}

	updateMediaDevices() {
		Call.Hardware.getCurrentDeviceList()
		console.log('updateMediaDevices')
	}

	stopLocalVideoStream()
	{
		if (this.localVideoStream)
		{
			this.localVideoStream.getTracks().forEach(tr => tr.stop());
		}
		this.localVideoStream = null;
	}

	setSelectedCamera(cameraId)
	{
		if (this.callView)
		{
			this.callView.setCameraId(cameraId)
		}
	}

	setSelectedMic(micId)
	{
		if (this.callView)
		{
			this.callView.setMicrophoneId(micId);
		}
	}

	getFeature(id)
	{
		if (typeof this.featureConfig[id] === 'undefined')
		{
			return {
				id,
				state: ConferenceApplication.FeatureState.Enabled,
				articleCode: ''
			}
		}

		return this.featureConfig[id];
	}

	getFeatureState(id)
	{
		return this.getFeature(id).state;
	}

	canRecord()
	{
		return Utils.platform.isBitrixDesktop() && Utils.platform.getDesktopVersion() >= 54;
	}

	isRecording()
	{
		return this.canRecord() && this.callRecordState != Call.View.RecordState.Stopped;
	}

	showFeatureLimitSlider(id)
	{
		const articleCode = this.getFeature(id).articleCode;
		if (!articleCode || !window.BX.UI.InfoHelper)
		{
			console.warn('Limit article not found', id);
			return false;
		}

		window.BX.UI.InfoHelper.show(articleCode);

		return true;
	}

	showNotification(notificationText, actions)
	{
		if (!actions)
		{
			actions = [];
		}
		BX.UI.Notification.Center.notify({
			content: Text.encode(notificationText),
			position: "top-right",
			autoHideDelay: 5000,
			closeButton: true,
			actions: actions
		});
	}

	showMicMutedNotification()
	{
		if (this.mutePopup || !this.callView)
		{
			return;
		}

		this.mutePopup = new Call.Hint({
			bindElement: this.callView.buttons.microphone.elements.icon,
			targetContainer: this.callView.container,
			buttons: [
				this.createUnmuteButton()
			],
			onClose: () =>
			{
				this.allowMutePopup = false;
				this.mutePopup.destroy();
				this.mutePopup = null;
			},
		});
		this.mutePopup.show();
	}
	createUnmuteButton()
	{
		return new BX.UI.Button({
			baseClass: "ui-btn ui-btn-icon-mic",
			text: BX.message("IM_CALL_UNMUTE_MIC"),
			size: BX.UI.Button.Size.EXTRA_SMALL,
			color: BX.UI.Button.Color.LIGHT_BORDER,
			noCaps: true,
			round: true,
			events: {
				click: () =>
				{
					this.onCallViewToggleMuteButtonClick({
						data: {
							muted: false
						}
					});
					this.mutePopup.destroy();
					this.mutePopup = null;
				}
			}
		})
	}

	showWebScreenSharePopup()
	{
		if (this.webScreenSharePopup)
		{
			this.webScreenSharePopup.show();

			return;
		}

		this.webScreenSharePopup = new Call.WebScreenSharePopup({
			bindElement: this.callView.buttons.screen.elements.root,
			targetContainer: this.callView.container,
			onClose: function ()
			{
				this.webScreenSharePopup.destroy();
				this.webScreenSharePopup = null;
			}.bind(this),
			onStopSharingClick: function ()
			{
				this.onCallViewToggleScreenSharingButtonClick();
				this.webScreenSharePopup.destroy();
				this.webScreenSharePopup = null;
			}.bind(this)
		});
		this.webScreenSharePopup.show();
	}

	isViewerMode()
	{
		let viewerMode = false;
		const isBroadcast = this.isBroadcast();
		if (isBroadcast)
		{
			const presenters = this.getBroadcastPresenters();
			const currentUserId = this.controller.getStore().state.application.common.userId;
			const isCurrentUserPresenter = presenters.includes(currentUserId);
			viewerMode = isBroadcast && !isCurrentUserPresenter;
		}
		return viewerMode;
	}

	onCallCreated(e)
	{
		Logger.warn('we got event onCallCreated', e);
		if(this.preCall || this.currentCall)
		{
			return;
		}
		let call = e.call;
		if (call.associatedEntity.type === 'chat' && call.associatedEntity.id === this.params.dialogId)
		{
			this.preCall = e.call;
			this.updatePreCallCounter();
			this.preCall.addEventListener(Call.Event.onUserStateChanged, this.onPreCallUserStateChangedHandler);
			this.preCall.addEventListener(Call.Event.onDestroy, this.onPreCallDestroyHandler);

			if (this.waitingForCallStatus)
			{
				this.callEventReceived = true;
			}
			this.setConferenceStatus(true);
			this.setConferenceStartDate(e.call.startDate);
		}

		const userReadyToJoin = this.getConference().common.userReadyToJoin;
		if (userReadyToJoin)
		{
			let viewerMode = this.isViewerMode();

			const videoEnabled = this.getConference().common.joinWithVideo;
			Logger.warn('ready to join call after waiting', videoEnabled, viewerMode);
			setTimeout(() => {
				Call.Hardware.init().then(() => {
					if (viewerMode && this.preCall)
					{
						this.joinCall(this.preCall.id, {
							joinAsViewer: true
						})
					}
					else
					{
						this.startCall(videoEnabled);
					}
				});
			}, 1000);
		}
	}

	releasePreCall()
	{
		if(this.preCall)
		{
			this.preCall.removeEventListener(Call.Event.onUserStateChanged, this.onPreCallUserStateChangedHandler);
			this.preCall.removeEventListener(Call.Event.onDestroy, this.onPreCallDestroyHandler);
			this.preCall = null;
		}
	}

	onPreCallDestroy(e)
	{
		if (this.waitingForCallStatusTimeout)
		{
			clearTimeout(this.waitingForCallStatusTimeout);
		}
		this.setConferenceStatus(false);

		this.releasePreCall();
	}

	onPreCallUserStateChanged(e)
	{
		this.updatePreCallCounter();
	}

	updatePreCallCounter()
	{
		if(this.preCall)
		{
			this.controller.getStore().commit('conference/common', {
				userInCallCount: this.preCall.getParticipatingUsers().length
			});
		}
		else
		{
			this.controller.getStore().commit('conference/common', {
				userInCallCount: 0
			});
		}
	}

	createVideoStrategy()
	{
		if (this.videoStrategy)
		{
			this.videoStrategy.destroy();
		}

		var strategyType = Utils.device.isMobile() ? VideoStrategy.Type.OnlySpeaker : VideoStrategy.Type.AllowAll;

		this.videoStrategy = new VideoStrategy({
			call: this.currentCall,
			callView: this.callView,
			strategyType: strategyType
		});
	}

	removeVideoStrategy()
	{
		if (this.videoStrategy)
		{
			this.videoStrategy.destroy();
		}
		this.videoStrategy = null;
	}

	onCallReplaceCamera(event)
	{
		let cameraId = event.data.deviceId;

		if (this.reconnectingCameraId) {
			this.setReconnectingCameraId(null);
		}

		Call.Hardware.defaultCamera = cameraId;
		if (this.currentCall)
		{
			this.currentCall.setCameraId(cameraId);
		}
		else
		{
			this.template.$emit('cameraSelected', cameraId);
		}
	}

	onCallReplaceMicrophone(event)
	{
		let microphoneId = event.data.deviceId;
		Call.Hardware.defaultMicrophone = microphoneId.deviceId;
		if (this.callView)
		{
			this.callView.setMicrophoneId(microphoneId);
		}
		if (this.currentCall)
		{
			this.currentCall.setMicrophoneId(microphoneId);
		}
		else
		{
			this.template.$emit('micSelected', event.data.deviceId);
		}
	}

	onCallReplaceSpeaker(event)
	{
		Call.Hardware.defaultSpeaker = event.data.deviceId;
	}

	onCallViewHasMainStream(event)
	{
		if (this.currentCall && this.currentCall.provider === Call.Provider.Bitrix)
		{
			this.currentCall.setMainStream(event.data.userId);
		}
	}

	onCallViewChangeHdVideo(event)
	{
		Call.Hardware.preferHdQuality = event.data.allowHdVideo;
	}

	onCallViewChangeMicAutoParams(event)
	{
		Call.Hardware.enableMicAutoParameters = event.data.allowMicAutoParams;
	}

	onCallViewChangeFaceImprove(event)
	{
		if (!DesktopApi.isDesktop())
		{
			return;
		}

		DesktopApi.setCameraSmoothingStatus(event.data.faceImproveEnabled);
	}

	onCallViewUserRename(event)
	{
		const newName = event.data.newName;

		if (!this.isExternalUser())
		{
			return false;
		}

		if (Utils.device.isMobile())
		{
			this.renameGuestMobile(newName)
		}
		else
		{
			this.renameGuest(newName);
		}
	}

	onCallViewUserPinned(event)
	{
		if (event.data.userId)
		{
			this.updateCallUser(event.data.userId, {pinned: true});

			return true;
		}

		this.controller.getStore().dispatch('call/unpinUser');

		return true;
	}

	onCallToggleSubscribe(e) {
		if (this.currentCall && this.currentCall.provider === Call.Provider.Bitrix && e.data)
		{
			this.currentCall.toggleRemoteParticipantVideo(e.data.participantIds, e.data.showVideo, true)
		}
	}

	renameGuest(newName)
	{
		this.callView.localUser.userModel.renameRequested = true;
		this.setUserName(newName).then(() => {
			this.callView.localUser.userModel.wasRenamed = true;
			Logger.log('setting name to', newName);
		}).catch(error => {
			Logger.error('error setting name', error);
		});
	}

	renameGuestMobile(newName)
	{
		this.setUserName(newName).then(() => {
			Logger.log('setting mobile name to', newName);
			if (this.callView.renameSlider)
			{
				this.callView.renameSlider.close();
			}
		}).catch(error => {
			Logger.error('error setting name', error);
		});
	}

	onCallButtonClick(event)
	{
		const buttonName = event.data.buttonName;
		Logger.warn('Button clicked!', buttonName);

		const handlers = {
			hangup: this.onCallViewHangupButtonClick.bind(this),
			hangupOptions: this._onCallViewHangupOptionsButtonClick.bind(this),
			close: this.onCallViewCloseButtonClick.bind(this),
			//inviteUser: this.onCallViewInviteUserButtonClick.bind(this),
			toggleMute: this.onCallViewToggleMuteButtonClick.bind(this),
			toggleScreenSharing: this.onCallViewToggleScreenSharingButtonClick.bind(this),
			record: this.onCallViewRecordButtonClick.bind(this),
			toggleVideo: this.onCallViewToggleVideoButtonClick.bind(this),
			toggleSpeaker: this.onCallViewToggleSpeakerButtonClick.bind(this),
			showChat: this.onCallViewShowChatButtonClick.bind(this),
			toggleUsers: this.onCallViewToggleUsersButtonClick.bind(this),
			share: this.onCallViewShareButtonClick.bind(this),
			fullscreen: this.onCallViewFullScreenButtonClick.bind(this),
			floorRequest: this.onCallViewFloorRequestButtonClick.bind(this),
			feedback: this.onCallViewFeedbackButtonClick.bind(this),
			onUserClick: this.onCallUserClick.bind(this),
			copilot: this.onCallCopilotButtonClick.bind(this),
		};

		if(handlers[buttonName])
		{
			handlers[buttonName](event);
		}
		else
		{
			Logger.error('Button handler not found!', buttonName);
		}
	}

	onCallViewHangupButtonClick(e)
	{
		Analytics.getInstance().onDisconnectCall({
			callId: this.currentCall?.id,
			callType: Analytics.AnalyticsType.videoconf,
			subSection: Analytics.AnalyticsSubSection.finishButton,
			mediaParams: {
				video: Call.Hardware.isCameraOn,
				audio: !Call.Hardware.isMicrophoneMuted,
			},
		});
		this.stopLocalVideoStream();
		this.endCall();
	}

	_onCallViewHangupOptionsButtonClick()
	{
		if (this.hangupOptionsMenu)
		{
			this.hangupOptionsMenu.destroy();
			return;
		}

		const targetNodeWidth = this.callView.buttons.hangupOptions.elements.root.offsetWidth;

		let menuItems = [
			{
				text: BX.message("CALL_M_BTN_HANGUP_OPTION_FINISH"),
				onclick: () => {
					Analytics.getInstance().onFinishCall({
						callId: this.currentCall?.id,
						callType: Analytics.AnalyticsType.videoconf,
						status: Analytics.AnalyticsStatus.finishedForAll,
						chatId: this.currentCall?.associatedEntity.id,
						callUsersCount: this.getCallUsers(true).length,
						callLength: Call.Util.getTimeText(this.currentCall?.startDate),
					});

					this.stopLocalVideoStream();
					this.endCall(true);
				},
			},
			{
				text: BX.message("CALL_M_BTN_HANGUP_OPTION_LEAVE"),
				onclick: () => {
					Analytics.getInstance().onDisconnectCall({
						callId: this.currentCall?.id,
						callType: Analytics.AnalyticsType.videoconf,
						subSection: Analytics.AnalyticsSubSection.contextMenu,
						mediaParams: {
							video: Call.Hardware.isCameraOn,
							audio: !Call.Hardware.isMicrophoneMuted,
						},
					});
					this.stopLocalVideoStream();
					this.endCall(false);
				},
			},
		];

		this.hangupOptionsMenu = new BX.PopupMenuWindow({
			className: 'bx-messenger-videocall-hangup-options-container',
			background: '#22272B',
			contentBackground: '#22272B',
			darkMode: true,
			contentBorderRadius: '6px',
			angle: false,
			bindElement: this.callView.buttons.hangupOptions.elements.root,
			targetContainer: this.container,
			offsetTop: -15,
			bindOptions: {position: "top"},
			cacheable: false,
			subMenuOptions: {
				maxWidth: 450
			},
			events: {
				onShow: (event) =>
				{
					const popup = event.getTarget();
					popup.getPopupContainer().style.display = 'block'; // bad hack

					const offsetLeft = (targetNodeWidth / 2) - popup.getPopupContainer().offsetWidth / 2;
					popup.setOffset({offsetLeft: offsetLeft + 40, offsetTop: 0});
					popup.setAngle({offset: popup.getPopupContainer().offsetWidth / 2 - 17});
				},
				onDestroy: () => this.hangupOptionsMenu = null
			},
			items: menuItems,
		});

		this.hangupOptionsMenu.show();
	}

	onCallViewCloseButtonClick(e)
	{
		this.stopLocalVideoStream();
		this.endCall();
	}

	onCallViewToggleMuteButtonClick(event)
	{
		Analytics.getInstance().onToggleMicrophone({
			muted: event.data.muted,
			callId: this.currentCall ? this.currentCall.id : 0,
			callType: Analytics.AnalyticsType.videoconf,
		});

		if (this.currentCall && !this.currentCall.microphoneId && !event.data.muted)
		{
			this.currentCall.setMicrophoneId(Call.Hardware.defaultMicrophone);
		}

		Call.Hardware.isMicrophoneMuted = event.data.muted;
		if (!this.currentCall)
		{
			this.template.$emit('setMicState', !event.data.muted);
		}

		if (this.isRecording())
		{
			BXDesktopSystem.CallRecordMute(event.data.muted);
		}

		if (this.currentCall?.userId)
		{
			this.updateCallUser(this.currentCall.userId, {microphoneState: !event.data.muted});
		}
	}

	onCallViewToggleScreenSharingButtonClick()
	{
		Analytics.getInstance().onScreenShareBtnClick({
			callId: this.currentCall.id,
			callType: Analytics.AnalyticsType.videoconf,
		});

		if (this.getFeatureState('screenSharing') === ConferenceApplication.FeatureState.Limited)
		{
			this.showFeatureLimitSlider('screenSharing');
			return;
		}

		if (this.getFeatureState('screenSharing') === ConferenceApplication.FeatureState.Disabled)
		{
			return;
		}

		if (this.currentCall.isScreenSharingStarted())
		{
			this.currentCall.stopScreenSharing();

			if (this.isRecording())
			{
				BXDesktopSystem.CallRecordStopSharing();
			}

			if (this.floatingScreenShareWindow)
			{
				this.floatingScreenShareWindow.close();
			}

			if (this.webScreenSharePopup)
			{
				this.webScreenSharePopup.close();
			}
		}
		else
		{
			this.restClient.callMethod("im.call.onShareScreen", {callId: this.currentCall.id});
			this.currentCall.startScreenSharing();
		}
	}

	onCallViewRecordButtonClick(event)
	{
		Analytics.getInstance().onRecordBtnClick({
			callId: this.currentCall.id,
			callType: Analytics.AnalyticsType.videoconf,
		});

		if (event.data.recordState === Call.View.RecordState.Started)
		{
			if (this.getFeatureState('record') === ConferenceApplication.FeatureState.Limited)
			{
				this.showFeatureLimitSlider('record');
				return;
			}

			if (this.getFeatureState('record') === ConferenceApplication.FeatureState.Disabled)
			{
				return;
			}

			if (this.canRecord())
			{
				// TODO: create popup menu with choice type of record - im/install/js/im/call/controller.js:1635
				// Call.View.RecordType.Video / Call.View.RecordType.Audio

				this.callView.setButtonActive('record', true);
			}
			else
			{
				if (window.BX.Helper)
				{
					window.BX.Helper.show("redirect=detail&code=22079566");
				}

				return;
			}
		}
		else if (event.data.recordState === Call.View.RecordState.Paused)
		{
			if (this.canRecord())
			{
				BXDesktopSystem.CallRecordPause(true);
			}
		}
		else if (event.data.recordState === Call.View.RecordState.Resumed)
		{
			if (this.canRecord())
			{
				BXDesktopSystem.CallRecordPause(false);
			}
		}
		else if (event.data.recordState === Call.View.RecordState.Stopped)
		{
			this.callView.setButtonActive('record', false);
		}

		this.currentCall.sendRecordState({
			action: event.data.recordState,
			date: new Date()
		});

		this.callRecordState = event.data.recordState;
	}

	onCallViewToggleVideoButtonClick(event)
	{
		Analytics.getInstance().onToggleCamera({
			video: event.data.video,
			callId: this.currentCall ? this.currentCall.id : 0,
			callType: Analytics.AnalyticsType.videoconf,
		});

		Call.Hardware.isCameraOn = event.data.video;
		if (this.currentCall)
		{
			if (!Call.Hardware.initialized)
			{
				return;
			}
			if (event.data.video && Object.values(Call.Hardware.cameraList).length === 0)
			{
				return;
			}
			if(!event.data.video)
			{
				this.callView.releaseLocalMedia();
			}

			if (!this.currentCall.cameraId && event.data.video)
			{
				this.currentCall.setCameraId(Call.Hardware.defaultCamera);
			}
		}
		else
		{
			this.template.$emit('setCameraState', event.data.video);
		}
	}

	onCallViewToggleSpeakerButtonClick(event)
	{
		this.callView.muteSpeaker(!event.data.speakerMuted);

		if (event.data.fromHotKey)
		{
			BX.UI.Notification.Center.notify({
				content: BX.message(this.callView.speakerMuted? 'IM_M_CALL_MUTE_SPEAKERS_OFF': 'IM_M_CALL_MUTE_SPEAKERS_ON'),
				position: "top-right",
				autoHideDelay: 3000,
				closeButton: true
			});
		}
	}

	onCallViewShareButtonClick()
	{
		let notifyWidth = 400;
		if (Utils.device.isMobile() && document.body.clientWidth < 400)
		{
			notifyWidth = document.body.clientWidth - 40;
		}

		BX.UI.Notification.Center.notify({
			content: Loc.getMessage('BX_IM_VIDEOCONF_LINK_COPY_DONE'),
			autoHideDelay: 4000,
			width: notifyWidth
		});

		Clipboard.copy(this.getDialogData().public.link);
	}

	onCallViewFullScreenButtonClick()
	{
		this.toggleFullScreen();
	}

	onFloatingScreenShareBackToCallClick()
	{
		BXDesktopWindow.ExecuteCommand('show.active')
		if (this.floatingScreenShareWindow)
		{
			this.floatingScreenShareWindow.hide();
		}
	}

	onFloatingScreenShareStopClick()
	{
		BXDesktopWindow.ExecuteCommand('show.active')
		this.onCallViewToggleScreenSharingButtonClick();
	}

	onFloatingScreenShareChangeScreenClick()
	{
		if (this.currentCall)
		{
			this.currentCall.startScreenSharing(true);
		}
	}

	onWindowFocus()
	{
		if (this.floatingScreenShareWindow)
		{
			this.floatingScreenShareWindow.hide();
		}
	}

	onWindowBlur()
	{
		if(this.floatingScreenShareWindow && this.currentCall && this.currentCall.isScreenSharingStarted())
		{
			this.floatingScreenShareWindow.show();
		}
	}

	isFullScreen ()
	{
		if ("webkitFullscreenElement" in document)
		{
			return (!!document.webkitFullscreenElement);
		}
		else if ("fullscreenElement" in document)
		{
			return (!!document.fullscreenElement);
		}
		return false;
	}

	toggleFullScreen ()
	{
		if(this.isFullScreen())
		{
			this.exitFullScreen();
		}
		else
		{
			this.enterFullScreen();
		}
	}

	enterFullScreen ()
	{
		if (BX.browser.IsChrome() || BX.browser.IsSafari())
		{
			document.body.webkitRequestFullScreen();
		}
		else if (BX.browser.IsFirefox())
		{
			document.body.requestFullscreen();
		}
	}

	exitFullScreen()
	{
		if (document.cancelFullScreen)
		{
			document.cancelFullScreen();
		}
		else if (document.mozCancelFullScreen)
		{
			document.mozCancelFullScreen();
		}
		else if (document.webkitCancelFullScreen)
		{
			document.webkitCancelFullScreen();
		}
		else if (document.document.exitFullscreen())
		{
			document.exitFullscreen()
		}
	}

	onCallViewShowChatButtonClick()
	{
		Analytics.getInstance().onShowChat({
			callId: this.currentCall?.id,
			callType: Analytics.AnalyticsType.videoconf,
		});

		this.toggleChat();
	}

	onCallViewToggleUsersButtonClick()
	{
		this.toggleUserList();
	}

	onCallViewFeedbackButtonClick()
	{
		BX.loadExt('ui.feedback.form').then(() => {
			BX.UI.Feedback.Form.open({
				id: `call_feedback_${this.currentCall.id}-${this.currentCall.instanceId}-${Math.random()}`,
				forms: [
					{ zones: ['ru', 'by', 'kz'], id: 406, sec: '9lhjhn', lang: 'ru' },
					{ zones: ['de'], id: 754, sec: '6upe49', lang: 'de' },
					{ zones: ['es'], id: 750, sec: 'whk4la', lang: 'es' },
					{ zones: ['com.br'], id: 752, sec: 'is01cs', lang: 'com.br' },
					{ zones: ['en'], id: 748, sec: 'pds0h6', lang: 'en' },
				],
				presets: {
					sender_page: 'call',
					call_type: this.currentCall.provider,
					call_amount: this.currentCall.users.length + 1,
					call_id: `id: ${this.currentCall.id}, instanceId: ${this.currentCall.instanceId}`,
					id_of_user: this.currentCall.userId,
					from_domain: location.origin
				},
			});
		})
	}

	onCallUserClick(e)
	{
		Analytics.getInstance().onClickUser({
			callId: this.currentCall.id,
			callType: Analytics.AnalyticsType.videoconf,
			layout: Object.keys(Call.View.Layout).find(key => Call.View.Layout[key] === e.layout),
		});
	}

	onCallCopilotButtonClick()
	{
		this.onChangeStateCopilot();

		/* if (!Call.Util.isAIServiceEnabled())
		{
			BX.SidePanel.Instance.open(CallAI.serviceEnabled, {
				cacheable: false
			});
			return;
		}

		this.copilotPopup = new Call.CopilotPopup({
			isCopilotActive: this.currentCall.isCopilotActive,
			isCopilotFeaturesEnabled: this.currentCall.isCopilotFeaturesEnabled,
			updateCopilotState: () => {
				this.onChangeStateCopilot();
			},
			onClose: () => {
				this.copilotPopup = null;
			}
		});

		if (this.copilotPopup)
		{
			this.copilotPopup.toggle();
		} */
	}

	onUpdateCallCopilotState({ isTrackRecordOn })
	{
		this.currentCall.isCopilotActive = isTrackRecordOn;
		this.callView.updateCopilotState(this.currentCall.isCopilotActive);
	}

	onChangeStateCopilot()
	{
		const action = !this.currentCall.isCopilotActive ? 'call.Track.start' : 'call.Track.stop';
		BX.ajax.runAction(action, {
			data: { callId: this.currentCall.id }
		}).then(() => {
			this.onUpdateCallCopilotState({
				isTrackRecordOn: !this.currentCall.isCopilotActive,
			});
		});
	}

	onCallViewFloorRequestButtonClick()
	{
		Analytics.getInstance().onFloorRequest({
			callId: this.currentCall.id,
			callType: Analytics.AnalyticsType.videoconf,
		});

		const floorState = this.callView.getUserFloorRequestState(Call.Engine.getCurrentUserId());
		const talkingState = this.callView.getUserTalking(Call.Engine.getCurrentUserId());

		this.callView.setUserFloorRequestState(Call.Engine.getCurrentUserId(), !floorState);

		if (this.currentCall)
		{
			this.currentCall.requestFloor(!floorState);
		}

		clearTimeout(this.callViewFloorRequestTimeout);
		if (talkingState && !floorState)
		{
			this.callViewFloorRequestTimeout = setTimeout(() =>
			{
				if (this.currentCall)
				{
					this.currentCall.requestFloor(false);
				}
			}, 1500);
		}
	}

	bindCallEvents()
	{
		this.currentCall.addEventListener(Call.Event.onUserInvited, this.onCallUserInvitedHandler);
		this.currentCall.addEventListener(Call.Event.onDestroy, this.onCallDestroyHandler);
		this.currentCall.addEventListener(Call.Event.onUserStateChanged, this.onCallUserStateChangedHandler);
		this.currentCall.addEventListener(Call.Event.onUserMicrophoneState, this.onCallUserMicrophoneStateHandler);
		this.currentCall.addEventListener(Call.Event.onUserCameraState, this.onCallUserCameraStateHandler);
		this.currentCall.addEventListener(Call.Event.onNeedResetMediaDevicesState, this.onNeedResetMediaDevicesStateHandler);
		this.currentCall.addEventListener(Call.Event.onUserVideoPaused, this.onCallUserVideoPausedHandler);
		this.currentCall.addEventListener(Call.Event.onLocalMediaReceived, this.onCallLocalMediaReceivedHandler);
		this.currentCall.addEventListener(Call.Event.onRemoteMediaReceived, this.onCallRemoteMediaReceivedHandler);
		this.currentCall.addEventListener(Call.Event.onRemoteMediaStopped, this.onCallRemoteMediaStoppedHandler);
		this.currentCall.addEventListener(Call.Event.onUserVoiceStarted, this.onCallUserVoiceStartedHandler);
		this.currentCall.addEventListener(Call.Event.onUserVoiceStopped, this.onCallUserVoiceStoppedHandler);
		this.currentCall.addEventListener(Call.Event.onUserStatsReceived, this.onUserStatsReceivedHandler);
		this.currentCall.addEventListener(Call.Event.onUserScreenState, this.onCallUserScreenStateHandler);
		this.currentCall.addEventListener(Call.Event.onUserRecordState, this.onCallUserRecordStateHandler);
		this.currentCall.addEventListener(Call.Event.onUserFloorRequest, this.onCallUserFloorRequestHandler);
		this.currentCall.addEventListener(Call.Event.onMicrophoneLevel, this.onMicrophoneLevelHandler);
		//this.currentCall.addEventListener(Call.Event.onDeviceListUpdated, this._onCallDeviceListUpdatedHandler);
		this.currentCall.addEventListener(Call.Event.onCallFailure, this.onCallFailureHandler);
		this.currentCall.addEventListener(Call.Event.onJoin, this._onCallJoinHandler);
		this.currentCall.addEventListener(Call.Event.onLeave, this.onCallLeaveHandler);
		this.currentCall.addEventListener(Call.Event.onReconnecting, this.onReconnectingHandler);
		this.currentCall.addEventListener(Call.Event.onReconnected, this.onReconnectedHandler);
		this.currentCall.addEventListener(Call.Event.onUpdateLastUsedCameraId, this.onUpdateLastUsedCameraIdHandler);
		this.currentCall.addEventListener(Call.Event.onConnectionQualityChanged, this.onCallConnectionQualityChangedHandler);
		this.currentCall.addEventListener(Call.Event.onToggleRemoteParticipantVideo, this.onCallToggleRemoteParticipantVideoHandler);
		this.currentCall.addEventListener(Call.Event.onGetUserMediaEnded, this._onGetUserMediaEndedHandler);
		this.currentCall.addEventListener(Call.Event.onSwitchTrackRecordStatus, this._onSwitchTrackRecordStatusHandler);
	}

	removeCallEvents()
	{
		this.currentCall.removeEventListener(Call.Event.onUserInvited, this.onCallUserInvitedHandler);
		this.currentCall.removeEventListener(Call.Event.onDestroy, this.onCallDestroyHandler);
		this.currentCall.removeEventListener(Call.Event.onUserStateChanged, this.onCallUserStateChangedHandler);
		this.currentCall.removeEventListener(Call.Event.onUserMicrophoneState, this.onCallUserMicrophoneStateHandler);
		this.currentCall.removeEventListener(Call.Event.onUserCameraState, this.onCallUserCameraStateHandler);
		this.currentCall.removeEventListener(Call.Event.onNeedResetMediaDevicesState, this.onNeedResetMediaDevicesStateHandler);
		this.currentCall.removeEventListener(Call.Event.onUserVideoPaused, this.onCallUserVideoPausedHandler);
		this.currentCall.removeEventListener(Call.Event.onLocalMediaReceived, this.onCallLocalMediaReceivedHandler);
		this.currentCall.removeEventListener(Call.Event.onRemoteMediaReceived, this.onCallRemoteMediaReceivedHandler);
		this.currentCall.removeEventListener(Call.Event.onRemoteMediaStopped, this.onCallRemoteMediaStoppedHandler);
		this.currentCall.removeEventListener(Call.Event.onUserVoiceStarted, this.onCallUserVoiceStartedHandler);
		this.currentCall.removeEventListener(Call.Event.onUserVoiceStopped, this.onCallUserVoiceStoppedHandler);
		this.currentCall.removeEventListener(Call.Event.onUserStatsReceived, this.onUserStatsReceivedHandler);
		this.currentCall.removeEventListener(Call.Event.onUserScreenState, this.onCallUserScreenStateHandler);
		this.currentCall.removeEventListener(Call.Event.onUserRecordState, this.onCallUserRecordStateHandler);
		this.currentCall.removeEventListener(Call.Event.onUserFloorRequest, this.onCallUserFloorRequestHandler);
		this.currentCall.removeEventListener(Call.Event.onMicrophoneLevel, this.onMicrophoneLevelHandler);
		this.currentCall.removeEventListener(Call.Event.onConnectionQualityChanged, this.onCallConnectionQualityChangedHandler);
		this.currentCall.removeEventListener(Call.Event.onToggleRemoteParticipantVideo, this.onCallToggleRemoteParticipantVideoHandler);
		//this.currentCall.removeEventListener(Call.Event.onDeviceListUpdated, this._onCallDeviceListUpdatedHandler);
		this.currentCall.removeEventListener(Call.Event.onCallFailure, this.onCallFailureHandler);
		this.currentCall.removeEventListener(Call.Event.onLeave, this.onCallLeaveHandler);
		this.currentCall.removeEventListener(Call.Event.onReconnecting, this.onReconnectingHandler);
		this.currentCall.removeEventListener(Call.Event.onReconnected, this.onReconnectedHandler);
		this.currentCall.addEventListener(Call.Event.onUpdateLastUsedCameraId, this.onUpdateLastUsedCameraIdHandler);
		this.currentCall.removeEventListener(Call.Event.onGetUserMediaEnded, this._onGetUserMediaEndedHandler);
		this.currentCall.removeEventListener(Call.Event.onSwitchTrackRecordStatus, this._onSwitchTrackRecordStatusHandler);
	}

	onCallUserInvited(e)
	{
		this.callView.addUser(e.userId);

		Call.Util.getUsers(this.currentCall.id, [e.userId]).then(userData => {
			this.controller.getStore().dispatch('users/set', Object.values(userData));
			this.controller.getStore().dispatch('conference/setUsers', {users: Object.keys(userData)});
			this.callView.updateUserData(userData)
		});
	}

	onCallUserStateChanged(e)
	{
		if (e.state !== Call.UserState.Connected && this.loopTimers[e.userId] === undefined)
		{
			this.loopConnectionQuality(e.userId, 1);
		}

		if (e.state === Call.UserState.Connected)
		{
			this.clearConnectionQualityTimer(e.userId);
			this.callView.setUserConnectionQuality(e.userId, 5);

			if (false && CallAI.serviceEnabled)
			{
				this.callView.unblockButtons(['copilot']);
			}
		}

		if (
			e.state === Call.UserState.Ready
			&& e.previousState === Call.UserState.Connected
			&& !Object.keys(this.currentCall).includes(e.userId)
		)
		{
			e.state = Call.UserState.Idle;
		}

		this.callView.setUserState(e.userId, e.state);
		this.updateCallUser(e.userId,{state: e.state});

		if (!this.isRecording())
		{
			this.callView.getConnectedUserCount(false)
				? this.callView.unblockButtons(['record'])
				: this.callView.blockButtons(['record'])
			;
		}
		/*if (e.direction)
		{
			this.callView.setUserDirection(e.userId, e.direction);
		}*/
	}

	onCallUserMicrophoneState(e)
	{
		if (e.userId == this.currentCall.userId)
		{
			Call.Hardware.isMicrophoneMuted = !e.microphoneState;
		}
		else
		{
			this.callView.setUserMicrophoneState(e.userId, e.microphoneState);
			this.updateCallUser(e.userId, {microphoneState: e.microphoneState});
		}
	}

	onCallUserCameraState(e)
	{
		this.callView.setUserCameraState(e.userId, e.cameraState);
		this.updateCallUser(e.userId, {cameraState: e.cameraState});
	}

	onNeedResetMediaDevicesState(e)
	{
		Call.Hardware.isMicrophoneMuted = true;
		Call.Hardware.isCameraOn = false;
	}

	onCallUserVideoPaused(e)
	{
		this.callView.setUserVideoPaused(e.userId, e.videoPaused);
	}

	onCallLocalMediaReceived(e)
	{
		console.log("Received local media stream " + e);
		if (this.callView)
		{
			const flipVideo = (e.tag == "main" || e.mediaRenderer) ? Call.Hardware.enableMirroring : false;

			this.callView.setLocalStream(e);
			this.callView.flipLocalVideo(flipVideo);

			this.callView.setButtonActive("screen", this.currentCall.isScreenSharingStarted());
			if (this.currentCall.isScreenSharingStarted())
			{
				this.screenShareStartTime = new Date();
				Analytics.getInstance().onScreenShareStarted({
					callId: this.currentCall.id,
					callType: Analytics.AnalyticsType.videoconf,
				});

				if (!DesktopApi.isDesktop())
				{
					this.showWebScreenSharePopup();
				}

				this.callView.updateButtons();
			}
			else
			{
				Analytics.getInstance().onScreenShareStopped({
					callId: this.currentCall.id,
					callType: Analytics.AnalyticsType.videoconf,
					status: Analytics.AnalyticsStatus.success,
					screenShareLength: Call.Util.getTimeText(this.screenShareStartTime),
				});
				this.screenShareStartTime = null;

				if (this.floatingScreenShareWindow)
				{
					this.floatingScreenShareWindow.close();
				}

				if (this.webScreenSharePopup)
				{
					this.webScreenSharePopup.close();
				}
			}

			if(!this.currentCall.callFromMobile && !this.isViewerMode())
			{
				this.callView.unblockSwitchCamera();
				this.callView.updateButtons();
			}
		}

		if (this.currentCall && Call.Hardware.isCameraOn && e.tag === 'main' && e.stream.getVideoTracks().length === 0)
		{
			Call.Hardware.isCameraOn = false;
		}
	}

	onCallRemoteMediaReceived(e)
	{
		const getStreamType = (stream) =>
		{
			if (stream?.getVideoTracks()?.length)
			{
				return 'video';
			}

			if (stream?.getAudioTracks()?.length)
			{
				return 'audio';
			}

			return null;
		}

		if (this.callView)
		{
			if ('track' in e)
			{
				this.callView.setUserMedia(e.userId, e.kind, e.track)
			}
			if ('mediaRenderer' in e && e.mediaRenderer.kind === 'audio' && getStreamType(e.mediaRenderer.stream) === 'audio')
			{
				this.callView.setUserMedia(e.userId, 'audio', e.mediaRenderer.stream.getAudioTracks()[0]);
			}
			if ('mediaRenderer' in e && e.mediaRenderer.kind === 'sharing' && getStreamType(e.mediaRenderer.stream) === 'audio')
			{
				this.callView.setUserMedia(e.userId, 'sharingAudio', e.mediaRenderer.stream.getAudioTracks()[0]);
			}
			if ('mediaRenderer' in e && (e.mediaRenderer.kind === 'video' || e.mediaRenderer.kind === 'sharing') && getStreamType(e.mediaRenderer.stream) === 'video')
			{
				this.callView.setVideoRenderer(e.userId, e.mediaRenderer);
			}
		}
	}

	onCallRemoteMediaStopped(e)
	{
		if (this.callView)
		{
			if ('mediaRenderer' in e)
			{
				if (e.kind === 'video' || e.kind === 'sharing')
				{
					e.mediaRenderer.stream = null;
					this.callView.setVideoRenderer(e.userId, e.mediaRenderer);
				}
			}
			else
			{
				this.callView.setUserMedia(e.userId, e.kind, null);
			}
		}
	}

	loopConnectionQuality(userId, quality, timeout = 200)
	{
		if (this.callView)
		{
			this.loopTimers[userId] = setTimeout(() => {
				this.callView.setUserConnectionQuality(userId, quality);
				const newQuality = quality >= 4 ? 1 : quality + 1;
				this.loopConnectionQuality(userId, newQuality, timeout);
			}, timeout);
		}
	}

	clearConnectionQualityTimer(userId)
	{
		if (this.loopTimers[userId] !== undefined)
		{
			clearTimeout(this.loopTimers[userId]);
			delete this.loopTimers[userId];
		}
	}

	onCallConnectionQualityChanged(e)
	{
		this.clearConnectionQualityTimer(e.userId);

		this.callView.setUserConnectionQuality(e.userId, e.score);
	}

	onGetUserMediaEnded()
	{
		this.updateMediaDevices();
	}

	onCallToggleRemoteParticipantVideo(e)
	{
		if (this.toogleParticipantsVideoBaloon)
		{
			if (e.isVideoShown)
			{
				this.toogleParticipantsVideoBaloon.close();
			}

			return;
		}

		if (!e.isVideoShown)
		{
			this.toogleParticipantsVideoBaloon = BX.UI.Notification.Center.notify({
				content: Text.encode(BX.message('IM_M_CALL_REMOTE_PARTICIPANTS_VIDEO_MUTED')),
				autoHide: false,
				position: "top-right",
				closeButton: false,
			})
		}
	}

	onCallUserVoiceStarted(e)
	{
		if (e.local)
		{
			if (this.currentCall.muted && this.allowMutePopup)
			{
				this.showMicMutedNotification();
			}
			return;
		}

		this.callView.setUserTalking(e.userId, true);
		this.callView.setUserFloorRequestState(e.userId, false);
		this.updateCallUser(e.userId, {talking: true, floorRequestState: false});
	}

	onCallUserVoiceStopped(e)
	{
		this.callView.setUserTalking(e.userId, false);
		this.updateCallUser(e.userId, {talking: false});
	}

	onUserStatsReceived(e)
	{
		if (this.callView)
		{
			this.callView.setUserStats(e.userId, e.report);
		}
	}

	onCallUserScreenState(e)
	{
		if(this.callView)
		{
			this.callView.setUserScreenState(e.userId, e.screenState);
		}
		this.updateCallUser(e.userId, {screenState: e.screenState});
	}

	onCallUserRecordState(event)
	{
		this.callRecordState = event.recordState.state;
		this.callView.setRecordState(event.recordState);

		if (!this.canRecord() || event.userId != this.controller.getUserId())
		{
			return true;
		}

		if (event.recordState.state !== Call.View.RecordState.Stopped)
		{
			this.callRecordInfo = event.recordState;
		}

		if (
			event.recordState.state === Call.View.RecordState.Started
			&& event.recordState.userId == this.controller.getUserId()
		)
		{
			const windowId = window.bxdWindowId || window.document.title;
			let fileName = BX.message('IM_CALL_RECORD_NAME');
			let dialogId = this.currentCall.associatedEntity.id;
			let dialogName = this.currentCall.associatedEntity.name;
			let callId = this.currentCall.id;
			let callDate = BX.Main.Date.format(this.params.formatRecordDate || 'd.m.Y');

			if (fileName)
			{
				fileName = fileName
					.replace('#CHAT_TITLE#', dialogName)
					.replace('#CALL_ID#', callId)
					.replace('#DATE#', callDate)
				;
			}
			else
			{
				fileName = "call_record_"+this.currentCall.id;
			}

			Call.Engine.getRestClient().callMethod("im.call.onStartRecord", {callId: this.currentCall.id});

			Analytics.getInstance().onRecordStart({
				callId: this.currentCall.id,
				callType: Analytics.AnalyticsType.videoconf,
			});

			BXDesktopSystem.CallRecordStart({
				windowId,
				fileName,
				callId,
				callDate,
				dialogId,
				dialogName,
				muted: Call.Hardware.isMicrophoneMuted,
				cropTop: 72,
				cropBottom: 90,
				shareMethod: 'im.disk.record.share'
			});
		}
		else if (event.recordState.state === Call.View.RecordState.Stopped)
		{
			Analytics.getInstance().onRecordStop({
				callId: this.currentCall.id,
				callType: Analytics.AnalyticsType.videoconf,
				subSection: Analytics.AnalyticsSubSection.window,
				element: Analytics.AnalyticsElement.recordButton,
				recordTime: Call.Util.getRecordTimeText(this.callRecordInfo),
			});
			this.callRecordInfo = null;

			BXDesktopSystem.CallRecordStop();
		}

		return true;
	}

	onCallUserFloorRequest(e)
	{
		this.callView.setUserFloorRequestState(e.userId, e.requestActive);
		this.updateCallUser(e.userId, {floorRequestState: e.requestActive});
	}

	onMicrophoneLevel(e)
	{
		this.callView.setMicrophoneLevel(e.level);
	}

	onCallJoin(e)
	{
		if (!e.local)
		{
			return;
		}

		if (!this.isViewerMode())
		{
			this.callView.unblockButtons(['camera', 'floorRequest', 'screen']);
		}

		if (this.callView.getConnectedUserCount(false))
		{
			this.callView.unblockButtons(['record']);
		}

		this.callView.setUiState(Call.View.UiState.Connected);
	}

	onCallFailure(e)
	{
		this.setConferenceHasErrorInCall(true);
		const errorCode = e.code || e.name || e.error;

		let errorMessage;

		if (e.name == "VoxConnectionError" || e.name == "AuthResult")
		{
			Call.Util.reportConnectionResult(e.call.id, false);
		}

		if (e.name == "AuthResult" || errorCode == "AUTHORIZE_ERROR")
		{
			errorMessage = BX.message("IM_CALL_ERROR_AUTHORIZATION");
		}
		else if (e.name == "Failed" && errorCode == 403)
		{
			errorMessage = BX.message("IM_CALL_ERROR_HARDWARE_ACCESS_DENIED");
		}
		else if (errorCode == "ERROR_UNEXPECTED_ANSWER")
		{
			errorMessage = BX.message("IM_CALL_ERROR_UNEXPECTED_ANSWER");
		}
		else if (errorCode == "BLANK_ANSWER_WITH_ERROR_CODE")
		{
			errorMessage = BX.message("IM_CALL_ERROR_BLANK_ANSWER");
		}
		else if (errorCode == "BLANK_ANSWER")
		{
			errorMessage = BX.message("IM_CALL_ERROR_BLANK_ANSWER");
		}
		else if (errorCode == "ACCESS_DENIED")
		{
			errorMessage = BX.message("IM_CALL_ERROR_ACCESS_DENIED");
		}
		else if (errorCode == "NO_WEBRTC")
		{
			errorMessage = this.isHttps ? BX.message("IM_CALL_NO_WEBRT") : BX.message("IM_CALL_ERROR_HTTPS_REQUIRED");
		}
		else if (errorCode == "UNKNOWN_ERROR")
		{
			errorMessage = BX.message("IM_CALL_ERROR_UNKNOWN");
		}
		else if (errorCode == "NETWORK_ERROR")
		{
			errorMessage = BX.message("IM_CALL_ERROR_NETWORK");
		}
		else if (errorCode == "NotAllowedError")
		{
			errorMessage = BX.message("IM_CALL_ERROR_HARDWARE_ACCESS_DENIED");
		}
		else
		{
			//errorMessage = BX.message("IM_CALL_ERROR_HARDWARE_ACCESS_DENIED");
			errorMessage = BX.message("IM_CALL_ERROR_UNKNOWN_WITH_CODE").replace("#ERROR_CODE#", errorCode);
		}

		if (this.callView)
		{
			this.callView.showFatalError({text: errorMessage});
		}
		else
		{
			this.showNotification(errorMessage);
		}

		this.autoCloseCallView = false;
		if (this.currentCall)
		{
			this.removeVideoStrategy();
			this.removeCallEvents();

			if (this.currentCallIsNew)
			{
				BX.CallEngine.getRestClient().callMethod('im.call.interrupt', {callId: this.currentCall.id});
			}

			this.currentCall.destroy();
			this.currentCall = null;
			this.currentCallIsNew = false;
		}

		Call.Hardware.isMicrophoneMuted = false;
	}

	onCallLeave(e)
	{
		if (!e.local)
		{
			return;
		}

		if (this.webScreenSharePopup)
		{
			this.webScreenSharePopup.close();
		}

		if (!this.getActiveCallUsers().length)
		{
			Analytics.getInstance().onFinishCall({
				callId: this.currentCall?.id,
				callType: Analytics.AnalyticsType.videoconf,
				status: Analytics.AnalyticsStatus.lastUserLeft,
				chatId: this.currentCall?.associatedEntity.id,
				callUsersCount: this.getCallUsers(true).length,
				callLength: Call.Util.getTimeText(this.currentCall.startDate),
			});
		}

		this.endCall();
	}

	onCallDestroy(e)
	{
		this.currentCall = null;

		if (this.floatingScreenShareWindow)
		{
			this.floatingScreenShareWindow.close;
		}

		if (this.webScreenSharePopup)
		{
			this.webScreenSharePopup.close();
		}

		this.restart();
	}

	onCheckDevicesSave(changedValues)
	{
		if (changedValues['camera'])
		{
			Call.Hardware.defaultCamera = changedValues['camera'];
		}

		if (changedValues['microphone'])
		{
			Call.Hardware.defaultMicrophone = changedValues['microphone'];
		}

		if (changedValues['audioOutput'])
		{
			Call.Hardware.defaultSpeaker = changedValues['audioOutput'];
		}

		if (changedValues['preferHDQuality'])
		{
			Call.Hardware.preferHdQuality = changedValues['preferHDQuality'];
		}

		if (changedValues['enableMicAutoParameters'])
		{
			Call.Hardware.enableMicAutoParameters = changedValues['enableMicAutoParameters'];
		}
	}

	setCameraState(state)
	{
		Call.Hardware.isCameraOn = state;
	}
	/* endregion 01. Call methods */

	/* region 02. Component methods */
		/* region 01. General actions */
		isChatShowed()
		{
			return this.getConference().common.showChat;
		}

		toggleChat()
		{
			const rightPanelMode = this.getConference().common.rightPanelMode;
			if (rightPanelMode === RightPanelMode.hidden)
			{
				this.controller.getStore().dispatch('conference/changeRightPanelMode', {mode: RightPanelMode.chat});
				this.callView.setButtonActive('chat', true);
			}
			else if (rightPanelMode === RightPanelMode.chat)
			{
				this.controller.getStore().dispatch('conference/changeRightPanelMode', {mode: RightPanelMode.hidden});
				this.callView.setButtonActive('chat', false);
			}
			else if (rightPanelMode === RightPanelMode.users)
			{
				this.controller.getStore().dispatch('conference/changeRightPanelMode', {mode: RightPanelMode.split});
				this.callView.setButtonActive('chat', true);
			}
			else if (rightPanelMode === RightPanelMode.split)
			{
				this.controller.getStore().dispatch('conference/changeRightPanelMode', {mode: RightPanelMode.users});
				this.callView.setButtonActive('chat', false);
			}
		}

		toggleUserList()
		{
			const rightPanelMode = this.getConference().common.rightPanelMode;
			if (rightPanelMode === RightPanelMode.hidden)
			{
				this.controller.getStore().dispatch('conference/changeRightPanelMode', {mode: RightPanelMode.users});
				this.callView.setButtonActive('users', true);
			}
			else if (rightPanelMode === RightPanelMode.users)
			{
				this.controller.getStore().dispatch('conference/changeRightPanelMode', {mode: RightPanelMode.hidden});
				this.callView.setButtonActive('users', false);
			}
			else if (rightPanelMode === RightPanelMode.chat)
			{
				this.controller.getStore().dispatch('conference/changeRightPanelMode', {mode: RightPanelMode.split});
				this.callView.setButtonActive('users', true);
			}
			else if (rightPanelMode === RightPanelMode.split)
			{
				this.controller.getStore().dispatch('conference/changeRightPanelMode', {mode: RightPanelMode.chat});
				this.callView.setButtonActive('users', false);
			}
		}

		pinUser(user)
		{
			if (!this.callView)
			{
				return false;
			}
			this.callView.pinUser(user.id);
			this.callView.setLayout(Call.View.Layout.Centered);
		}

		unpinUser()
		{
			if (!this.callView)
			{
				return false;
			}
			this.callView.unpinUser();
		}

		changeBackground()
		{
			if (!Call.Hardware)
			{
				return false;
			}
			Call.BackgroundDialog.open();
		}

		openChat(user)
		{
			DesktopApi.emitToMainWindow('bxConferenceOpenChat', [user.id]);
		}

		openProfile(user)
		{
			DesktopApi.emitToMainWindow('bxConferenceOpenProfile', [user.id]);
		}

		setDialogInited()
		{
			this.dialogInited = true;
			let dialogData = this.getDialogData();
			document.title = dialogData.name;
		}

		changeVideoconfUrl(newUrl)
		{
			window.history.pushState("", "", newUrl);
		}

		sendNewMessageNotify(params)
		{
			const MAX_LENGTH = 40;
			const AUTO_HIDE_TIME = 4000;

			if (!this.checkIfMessageNotifyIsNeeded(params))
			{
				return false;
			}
			const text = Utils.text.purify(params.message.text, params.message.params, params.files);
			let avatar = '';
			let userName = '';

			// avatar and username only for non-system messages
			if (params.message.senderId > 0 && params.message.system !== 'Y')
			{
				const messageAuthor = this.controller.getStore().getters['users/get'](params.message.senderId, true);
				userName = messageAuthor.name;
				avatar = messageAuthor.avatar;
			}

			Notifier.notify({
				id: `im-videconf-${params.message.id}`,
				title: userName,
				icon: avatar,
				text
			});

			return true;
		}

		checkIfMessageNotifyIsNeeded(params)
		{
			if (!Call.Util.isConferenceChatEnabled())
			{
				return false;
			}

			const rightPanelMode = this.getConference().common.rightPanelMode;
			return !Utils.device.isMobile()
				&& params.chatId === this.getChatId()
				&& (rightPanelMode !== RightPanelMode.chat || rightPanelMode !== RightPanelMode.split)
				&& params.message.senderId !== this.controller.getUserId()
				&& !this.getConference().common.error;
		}

		onInputFocus(e)
		{
			this.callView.setHotKeyTemporaryBlock(true);
		}

		onInputBlur(e)
		{
			this.callView.setHotKeyTemporaryBlock(false);
		}

	closeReconnectionBaloon()
	{
		if (this.reconnectionBaloon)
		{
			this.reconnectionBaloon.close();
			this.reconnectionBaloon = null;
		}
	}

	setReconnectingCameraId(id)
	{
		this.reconnectingCameraId = id;

		if (id) {
			this.updateCameraSettingsInCurrentCallAfterReconnecting(id)
		}
	}

	updateCameraSettingsInCurrentCallAfterReconnecting(cameraId)
	{
		if (this.currentCall.cameraId === cameraId) {
			return;
		}

		const devicesList = Call.Hardware.getCameraList();

		if (!devicesList.find(device => device.deviceId === cameraId))
		{
			return;
		}

		this.currentCall.setCameraId(cameraId);
		this.setReconnectingCameraId(null);
	}

	onUpdateLastUsedCameraId()
	{
		const cameraId = this.currentCall.cameraId;
		if (cameraId)
		{
			this.lastUsedCameraId = cameraId;
		}
	}

	onReconnecting(e)
	{
		if (!(this.currentCall.provider === Call.Provider.Bitrix || this.currentCall.provider === Call.Provider.Plain))
		{
			// todo: restore after fixing balloon resurrection issue
			// related to multiple simultaneous calls to the balloon manager
			// now it's enabled for Bitrix24 calls as a temp solution
			return false;
		}

		Analytics.getInstance().onReconnect({
			callId: this.currentCall.id,
			callType: Analytics.AnalyticsType.videoconf,
			reconnectionEventCount: e.reconnectionEventCount
		});

		// noinspection UnreachableCodeJS

		if (this.reconnectionBaloon)
		{
			return;
		}

		this.reconnectionBaloon = BX.UI.Notification.Center.notify({
			content: Text.encode(BX.message('IM_CALL_RECONNECTING')),
			autoHide: false,
			position: "top-right",
			closeButton: false,
		})
	}

	onReconnected()
	{
		this.setReconnectingCameraId(this.lastUsedCameraId)
		if (!(this.currentCall.provider === Call.Provider.Bitrix || this.currentCall.provider === Call.Provider.Plain))
		{
			// todo: restore after fixing balloon resurrection issue
			// related to multiple simultaneous calls to the balloon manager
			// now it's enabled for Bitrix24 calls as a temp solution
			return false;
		}

		// noinspection UnreachableCodeJS
		this.closeReconnectionBaloon();
	}

		setUserWasRenamed()
		{
			if (this.callView)
			{
				this.callView.localUser.userModel.wasRenamed = true;
			}
		}
		/* endregion 01. General actions */

		/* region 02. Store actions */
		setError(errorCode)
		{
			const currentError = this.getConference().common.error;
			// if user kicked from call - dont show him end of call form
			if (currentError && currentError === ConferenceErrorCode.kickedFromCall)
			{
				return;
			}

			this.controller.getStore().commit('conference/setError', {errorCode});
		}

		toggleSmiles()
		{
			this.controller.getStore().commit('conference/toggleSmiles');
		}

		setJoinType(joinWithVideo)
		{
			this.controller.getStore().commit('conference/setJoinType', {joinWithVideo});
		}

		setConferenceStatus(conferenceStarted)
		{
			this.controller.getStore().commit('conference/setConferenceStatus', {conferenceStarted});
		}

		setConferenceHasErrorInCall(hasErrorInCall)
		{
			this.controller.getStore().commit('conference/setConferenceHasErrorInCall', {hasErrorInCall});
		}

		setConferenceStartDate(conferenceStartDate)
		{
			this.controller.getStore().commit('conference/setConferenceStartDate', {conferenceStartDate});
		}

		setUserReadyToJoin()
		{
			this.controller.getStore().commit('conference/setUserReadyToJoin');
		}

		updateCallUser(userId, fields)
		{
			this.controller.getStore().dispatch('call/updateUser', {id: userId, fields});
		}
		/* endregion 02. Store actions */

		/* region 03. Rest actions */
		setUserName(name)
		{
			return new Promise((resolve, reject) => {
				this.restClient.callMethod('im.call.user.update', {
					name: name,
					chat_id: this.getChatId()
				}).then(() => {
					resolve();
				}).catch((error) => {
					reject(error)
				});
			});
		}

		checkPassword(password)
		{
			return new Promise((resolve, reject) => {
				this.restClient.callMethod('im.videoconf.password.check', { password, alias: this.params.alias })
					.then(result => {
						if (result.data() === true)
						{
							this.restClient.setPassword(password);
							this.controller.getStore().commit('conference/common', {
								passChecked: true
							});
							this.initUserComplete();
							resolve();
						}
						else
						{
							reject();
						}
					}).catch(result => {
						console.error('Password check error', result);
					});
			});
		}

		changeLink()
		{
			return new Promise((resolve, reject) => {
				this.restClient.callMethod('im.videoconf.share.change', {
					dialog_id: this.getDialogId()
				}).then(() => {
					resolve();
				}).catch((error) => {
					reject(error)
				});
			});
		}
		/* endregion 03. Rest actions */
	/* endregion 02. Component methods */

/* endregion 02. Methods */

/* region 03. Utils */
	ready()
	{
		if (this.inited)
		{
			let promise = new BX.Promise;
			promise.resolve(this);

			return promise;
		}

		return this.initPromise;
	}

	getConference()
	{
		return this.controller.getStore().state.conference;
	}

	isBroadcast()
	{
		return this.getConference().common.isBroadcast;
	}

	getBroadcastPresenters()
	{
		return this.getConference().common.presenters;
	}

	isExternalUser()
	{
		return !!this.getUserHash();
	}

	getChatId()
	{
		return parseInt(this.params.chatId);
	}

	getDialogId()
	{
		return this.params.dialogId;
	}

	getDialogData()
	{
		if (!this.dialogInited)
		{
			return false;
		}

		return this.controller.getStore().getters['dialogues/get'](this.getDialogId());
	}

	getHost()
	{
		return location.origin || '';
	}

	getStartupErrorCode()
	{
		return this.params.startupErrorCode? this.params.startupErrorCode : '';
	}

	isHttps()
	{
		return location.protocol === 'https:';
	}

	getUserHash()
	{
		return this.getConference().user.hash;
	}

	getUserHashCookie()
	{
		let userHash = '';

		let cookie = Cookie.get(null, 'BITRIX_CALL_HASH');
		if (typeof cookie === 'string' && cookie.match(/^[a-f0-9]{32}$/))
		{
			userHash = cookie;
		}

		return userHash;
	}

	switchToSessAuth()
	{
		this.restClient.restClient.queryParams = undefined;

		return true;
	}

/* endregion 03. Utils */
}

ConferenceApplication.FeatureState = {
	Enabled: 'enabled',
	Disabled: 'disabled',
	Limited: 'limited',
};

export {ConferenceApplication};
