/**
 * @module calendar/sync-page/wizard/sync-wizard
 */
jn.define('calendar/sync-page/wizard/sync-wizard', (require, exports, module) => {
	const AppTheme = require('apptheme');
	const { SyncWizardStage } = require('calendar/sync-page/wizard/stage');
	const { Loc } = require('loc');
	const { Color } = require('tokens');

	const MIN_UPDATE_STATE_DELAY = 2000;
	const FIRST_STAGE_CODE = 'connection_created';
	const SECOND_STAGE_CODE = 'import_finished';
	const THIRD_STAGE_CODE = 'export_finished';

	class SyncWizard extends LayoutComponent
	{
		constructor(props)
		{
			super(props);

			this.state = {
				completeState: false,
				errorState: false,
				title: props.title,
			};

			this.layoutWidget = props.layoutWidget;

			this.pullConfig = {
				commands: [
					'process_sync_connection',
				],
			};
			this.unsubscribe = null;
			this.lastUpdateStateTimestamp = Date.now();

			this.firstStageRef = null;
			this.secondStageRef = null;
			this.thirdStageRef = null;
			this.lottieRef = null;
		}

		get customEventEmitter()
		{
			return this.props.customEventEmitter;
		}

		componentDidMount()
		{
			this.pullSubscribe();

			this.setLoadingStage(this.firstStageRef);

			if (this.props.firstStageComplete)
			{
				this.setConnectionCreatedState();
			}
		}

		componentWillUnmount()
		{
			if (this.unsubscribe)
			{
				this.unsubscribe();
			}
		}

		componentDidUpdate(prevProps, prevState)
		{
			if (this.state.completeState && this.lottieRef)
			{
				setTimeout(() => {
					this.lottieRef.playAnimation();
				}, 1500);
			}
		}

		pullSubscribe()
		{
			this.unsubscribe = BX.PULL.subscribe({
				moduleId: 'calendar',
				callback: (data) => {
					const command = BX.prop.getString(data, 'command', '');
					const params = BX.prop.getObject(data, 'params', {});

					if (!this.pullConfig.commands.includes(command))
					{
						return;
					}

					if (command === 'process_sync_connection')
					{
						this.processSyncConnection(params);
					}
				},
			});
		}

		processSyncConnection(params)
		{
			if (!params)
			{
				return;
			}

			if (params.vendorName !== this.props.type)
			{
				return;
			}

			if (params.accountName)
			{
				this.setState({ title: params.accountName });
			}

			this.handleUpdateState(params);
		}

		handleUpdateState(params)
		{
			const currentTimestamp = Date.now();

			if (currentTimestamp - this.lastUpdateStateTimestamp > MIN_UPDATE_STATE_DELAY)
			{
				this.updateState(params);
			}
			else
			{
				setTimeout(() => {
					this.handleUpdateState(params);
				}, MIN_UPDATE_STATE_DELAY);
			}
		}

		updateState(params)
		{
			if (this.state.errorState)
			{
				return;
			}

			const updatingState = {};

			this.lastUpdateStateTimestamp = Date.now();

			switch (params.stage)
			{
				case FIRST_STAGE_CODE: {
					this.setCompleteStage(this.firstStageRef);
					this.setLoadingStage(this.secondStageRef);
					break;
				}

				case SECOND_STAGE_CODE: {
					this.setCompleteStage(this.firstStageRef);
					this.setCompleteStage(this.secondStageRef);
					this.setLoadingStage(this.thirdStageRef);

					break;
				}

				case THIRD_STAGE_CODE: {
					this.setCompleteStage(this.firstStageRef);
					this.setCompleteStage(this.secondStageRef);
					this.setCompleteStage(this.thirdStageRef);

					updatingState.completeState = true;

					this.customEventEmitter.emit('Calendar.Sync::onConnectionCreated', [{ type: this.props.type }]);

					setTimeout(() => {
						this.layoutWidget.close();
					}, 6000);

					break;
				}
				default:
					break;
			}

			this.setState(updatingState);
		}

		/**
		 *
		 * @param {SyncWizardStage} stageRef
		 */
		setCompleteStage(stageRef)
		{
			if (stageRef && !stageRef.isCompleted())
			{
				stageRef.setCompleteState();
			}
		}

		/**
		 *
		 * @param {SyncWizardStage} stageRef
		 */
		setLoadingStage(stageRef)
		{
			if (stageRef && stageRef.isDefault())
			{
				stageRef.setLoadingState();
			}
		}

		setErrorState()
		{
			this.setState({ errorState: true });
		}

		setConnectionCreatedState()
		{
			this.setCompleteStage(this.firstStageRef);
			this.setLoadingStage(this.secondStageRef);
		}

		render()
		{
			return View(
				{
					style: {
						backgroundColor: Color.bgContentPrimary.toHex(),
					},
					testId: 'sync_page_wizard_container',
				},
				this.renderTitle(),
				!this.state.errorState && this.renderStages(),
				!this.state.errorState && this.state.completeState && this.renderCompleteState(),
				this.state.errorState && this.renderErrorState(),
			);
		}

		renderTitle()
		{
			return View(
				{
					style: {
						flexDirection: 'row',
						padding: 18,
						paddingLeft: 41,
						paddingRight: 41,
						borderBottomWidth: 0.5,
						borderBottomColor: AppTheme.colors.base6,
					},
					testId: 'sync_page_wizard_title_container',
				},
				this.renderIcon(),
				this.renderConnectionInfo(),
			);
		}

		renderIcon()
		{
			return View(
				{
					style: {
						width: 66,
						height: 66,
						alignItems: 'center',
						justifyContent: 'center',
						borderRadius: 100,
						borderColor: AppTheme.colors.bgSeparatorSecondary,
						borderWidth: 1,
						backgroundColor: AppTheme.colors.bgNavigation,
						backgroundRepeat: 'no-repeat',
						backgroundPosition: 'center',
						backgroundResizeMode: 'cover',
					},
				},
				Image(
					{
						style: {
							width: this.props.icon.width,
							height: this.props.icon.height,
						},
						svg: {
							content: this.props.icon.svg,
						},
						tintColor: this.props.icon.tintColor,
					},
				),
			);
		}

		renderConnectionInfo()
		{
			return View(
				{
					style: {
						flexDirection: 'column',
						paddingRight: 45,
						marginLeft: 12,
					},
				},
				Text(
					{
						text: this.state.title,
						style: {
							fontSize: 18,
							color: AppTheme.colors.base1,
							marginBottom: 3,
						},
						ellipsize: 'end',
						numberOfLines: 1,
						testId: 'sync_page_wizard_title',
					},
				),
				Text(
					{
						text: this.state.completeState
							? Loc.getMessage('M_CALENDAR_SYNC_WIZARD_DONE')
							: Loc.getMessage('M_CALENDAR_SYNC_WIZARD_IN_PROGRESS'),
						style: {
							fontSize: 14,
							color: AppTheme.colors.base4,
						},
					},
				),
			);
		}

		renderStages()
		{
			return View(
				{
					style: {
						paddingLeft: 27,
						paddingTop: 24,
						paddingRight: 32,
						paddingBottom: 10,
					},
					testId: `sync_page_wizard_stages_container`,
				},
				new SyncWizardStage({
					title: this.props.firstStage,
					ref: (ref) => {
						this.firstStageRef = ref;
					},
					testId: `sync_page_wizard_first_stage`,
				}),
				new SyncWizardStage({
					title: this.props.secondStage,
					ref: (ref) => {
						this.secondStageRef = ref;
					},
					testId: `sync_page_wizard_second_stage`,
				}),
				new SyncWizardStage({
					title: this.props.thirdStage,
					ref: (ref) => {
						this.thirdStageRef = ref;
					},
					testId: `sync_page_wizard_third_stage`,
				}),
			);
		}

		renderCompleteState()
		{
			return View(
				{
					style: {
						height: 250,
						justifyContent: 'center',
					},
					testId: `sync_page_wizard_complete_state`,
				},
				View(
					{
						style: {
							height: 170,
							backgroundColor: AppTheme.colors.accentSoftBlue3,
							borderRadius: 10,
							marginHorizontal: 36,
							alignItems: 'center',
							justifyContent: 'center',
						},
					},
					Text(
						{
							text: Loc.getMessage('M_CALENDAR_SYNC_WIZARD_READY'),
							style: {
								fontSize: 24,
								color: AppTheme.colors.base0,
							},
						},
					),
					Text(
						{
							text: Loc.getMessage('M_CALENDAR_SYNC_WIZARD_CALENDAR_SYNCED'),
							style: {
								marginTop: 5,
								fontSize: 17,
								color: AppTheme.colors.base3,
							},
						},
					),
				),
				this.renderConfetti(),
			);
		}

		renderConfetti()
		{
			return View(
				{
					style: {
						position: 'absolute',
						left: 0,
						right: 0,
						top: 0,
						bottom: 0,
						justifyContent: 'center',
						alignItems: 'center',
					},
				},
				LottieView(
					{
						style: {
							width: 300,
							height: 300,
						},
						data: {
							content: lottie.confetti,
						},
						params: {
							loopMode: 'playOnce',
						},
						ref: (ref) => {
							this.lottieRef = ref;
						},
					},
				),
			);
		}

		renderErrorState()
		{
			return View(
				{
					style: {
						height: 170,
						borderRadius: 10,
						marginVertical: 20,
						marginHorizontal: 18,
						alignItems: 'center',
						justifyContent: 'center',
						padding: 40,
					},
					testId: `sync_page_wizard_error_state`,
				},
				Image(
					{
						svg: {
							content: icons.warning,
						},
						style: {
							width: 29,
							height: 26,
						},
					},
				),
				Text(
					{
						text: Loc.getMessage('M_CALENDAR_SYNC_WIZARD_NOT_SYNCED'),
						style: {
							fontWeight: '400',
							marginTop: 10,
							fontSize: 22,
							color: AppTheme.colors.base0,
							textAlign: 'center',
						},
					},
				),
				Text(
					{
						text: Loc.getMessage('M_CALENDAR_SYNC_WIZARD_TRY_AGAIN_LATER'),
						style: {
							fontWeight: '400',
							marginTop: 10,
							fontSize: 17,
							color: AppTheme.colors.base3,
							textAlign: 'center',
						},
					},
				),
			);
		}
	}

	const icons = {
		warning: '<svg width="29" height="26" viewBox="0 0 29 26" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M28.5272 21.8461L16.4888 1.79513C15.5612 0.255789 13.3509 0.255789 12.4431 1.79513L0.404629 21.8461C-0.542659 23.4249 0.601981 25.4181 2.43735 25.4181H26.5143C28.3299 25.4181 29.4745 23.4249 28.5272 21.8461ZM12.7391 8.78138C12.7391 7.8933 13.4496 7.18283 14.3377 7.18283H14.5547C15.4428 7.18283 16.1533 7.8933 16.1533 8.78138V14.7611C16.1533 15.6492 15.4428 16.3597 14.5547 16.3597H14.3377C13.4496 16.3597 12.7391 15.6492 12.7391 14.7611V8.78138ZM16.4691 20.3857C16.4691 21.4908 15.5612 22.3987 14.4561 22.3987C13.3509 22.3987 12.4431 21.4908 12.4431 20.3857C12.4431 19.2805 13.3509 18.3727 14.4561 18.3727C15.5612 18.3727 16.4691 19.2805 16.4691 20.3857Z" fill="#FF5752"/></svg>',
	};

	const lottie = {
		confetti: '{"v":"5.7.12","fr":30,"ip":0,"op":60,"w":512,"h":512,"nm":"Confetti 1","ddd":0,"assets":[],"layers":[{"ddd":0,"ind":1,"ty":3,"nm":"Null 3","sr":1,"ks":{"o":{"a":0,"k":0,"ix":11},"r":{"a":0,"k":260,"ix":10},"p":{"a":0,"k":[256,256,0],"ix":2,"l":2},"a":{"a":0,"k":[50,50,0],"ix":1,"l":2},"s":{"a":0,"k":[102,102,100],"ix":6,"l":2}},"ao":0,"ip":5,"op":60,"st":0,"bm":0},{"ddd":0,"ind":2,"ty":4,"nm":"Shape Layer 50","parent":1,"sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":14,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":19,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":41,"s":[100]},{"t":57,"s":[0]}],"ix":11},"r":{"a":0,"k":487,"ix":10},"p":{"a":0,"k":[50,50,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[41,41,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-19.362,34.498],[7.899,-25.433],[37.607,53.867],[11.905,108.592]],"o":[[0,0],[42.457,-75.648],[-9.025,29.061],[-47.221,-67.637],[-3.807,-34.72]],"v":[[1,2],[-48.536,-23.607],[-76.476,-100.021],[27.098,-130.112],[-120.809,-222.715]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[0.49411764705882355,0.8274509803921568,0.12941176470588237,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":17,"s":[0]},{"t":54,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":14,"s":[0]},{"t":49,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":14,"op":60,"st":14,"bm":0},{"ddd":0,"ind":3,"ty":4,"nm":"Shape Layer 49","parent":1,"sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":13,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":18,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":40,"s":[100]},{"t":56,"s":[0]}],"ix":11},"r":{"a":0,"k":353,"ix":10},"p":{"a":0,"k":[50,50,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[70,70,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-14,37],[8.506,-58.529],[28.328,67.272],[11.905,108.592]],"o":[[0,0],[9.966,-26.338],[-12.365,85.083],[-32.013,-76.024],[-3.807,-34.72]],"v":[[1,2],[-48.536,-23.607],[-76.476,-100.021],[27.098,-130.112],[-120.809,-222.715]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[0.728918397193,0.113771438599,1,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":16,"s":[0]},{"t":53,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":13,"s":[0]},{"t":48,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":13,"op":60,"st":13,"bm":0},{"ddd":0,"ind":4,"ty":4,"nm":"Shape Layer 48","parent":1,"sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":17,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":39,"s":[100]},{"t":55,"s":[0]}],"ix":11},"r":{"a":0,"k":187,"ix":10},"p":{"a":0,"k":[50,50,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[70,70,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-14,37],[8.254,-31.239],[20.685,70.001],[17.795,30.055]],"o":[[0,0],[9.966,-26.338],[-7.439,28.154],[-10.769,-36.442],[-17.795,-30.055]],"v":[[1,2],[2.692,-79.782],[-49.416,-90.653],[-14.589,-135.23],[-105.31,-168.421]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[1,0.89785563151,0.113771438599,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":15,"s":[0]},{"t":52,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":12,"s":[0]},{"t":47,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":12,"op":60,"st":12,"bm":0},{"ddd":0,"ind":5,"ty":4,"nm":"Shape Layer 47","parent":1,"sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":16,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":38,"s":[100]},{"t":54,"s":[0]}],"ix":11},"r":{"a":0,"k":203,"ix":10},"p":{"a":0,"k":[50,50,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[58,58,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-14,37],[-1,38],[35.009,-27.828],[-11,31]],"o":[[0,0],[9.966,-26.338],[1,-38],[-39,31],[11,-31]],"v":[[1,2],[48,-39],[29,-112],[136,-107],[103,-176]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[1,0.104197326361,0.38322987276,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":14,"s":[0]},{"t":51,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":11,"s":[0]},{"t":46,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":11,"op":60,"st":11,"bm":0},{"ddd":0,"ind":6,"ty":4,"nm":"Shape Layer 46","parent":1,"sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":15,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":37,"s":[100]},{"t":53,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":0,"k":[50,50,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[100,100,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-14,37],[-1,38],[35.009,-27.828],[-11,31]],"o":[[0,0],[9.966,-26.338],[1,-38],[-39,31],[11,-31]],"v":[[1,2],[48,-39],[29,-112],[136,-107],[103,-176]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":13,"s":[0]},{"t":50,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":10,"s":[0]},{"t":45,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":10,"op":60,"st":10,"bm":0},{"ddd":0,"ind":7,"ty":4,"nm":"Shape Layer 45","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":9,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":14,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":36,"s":[100]},{"t":52,"s":[0]}],"ix":11},"r":{"a":0,"k":487,"ix":10},"p":{"a":0,"k":[256,256,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[41,41,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-19.362,34.498],[7.899,-25.433],[37.607,53.867],[11.905,108.592]],"o":[[0,0],[42.457,-75.648],[-9.025,29.061],[-47.221,-67.637],[-3.807,-34.72]],"v":[[1,2],[-48.536,-23.607],[-76.476,-100.021],[27.098,-130.112],[-120.809,-222.715]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[0.49411764705882355,0.8274509803921568,0.12941176470588237,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":12,"s":[0]},{"t":49,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":9,"s":[0]},{"t":44,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":9,"op":60,"st":9,"bm":0},{"ddd":0,"ind":8,"ty":4,"nm":"Shape Layer 43","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":8,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":13,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":35,"s":[100]},{"t":51,"s":[0]}],"ix":11},"r":{"a":0,"k":353,"ix":10},"p":{"a":0,"k":[256,256,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[70,70,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-14,37],[8.506,-58.529],[28.328,67.272],[11.905,108.592]],"o":[[0,0],[9.966,-26.338],[-12.365,85.083],[-32.013,-76.024],[-3.807,-34.72]],"v":[[1,2],[-48.536,-23.607],[-76.476,-100.021],[27.098,-130.112],[-120.809,-222.715]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[0.728918397193,0.113771438599,1,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":11,"s":[0]},{"t":48,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":8,"s":[0]},{"t":43,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":8,"op":60,"st":8,"bm":0},{"ddd":0,"ind":9,"ty":4,"nm":"Shape Layer 42","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":7,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":34,"s":[100]},{"t":50,"s":[0]}],"ix":11},"r":{"a":0,"k":187,"ix":10},"p":{"a":0,"k":[256,256,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[70,70,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-14,37],[8.254,-31.239],[20.685,70.001],[17.795,30.055]],"o":[[0,0],[9.966,-26.338],[-7.439,28.154],[-10.769,-36.442],[-17.795,-30.055]],"v":[[1,2],[2.692,-79.782],[-49.416,-90.653],[-14.589,-135.23],[-105.31,-168.421]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[1,0.89785563151,0.113771438599,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":10,"s":[0]},{"t":47,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":7,"s":[0]},{"t":42,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":7,"op":60,"st":7,"bm":0},{"ddd":0,"ind":10,"ty":4,"nm":"Shape Layer 44","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":6,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":33,"s":[100]},{"t":49,"s":[0]}],"ix":11},"r":{"a":0,"k":203,"ix":10},"p":{"a":0,"k":[256,256,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[58,58,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-14,37],[-1,38],[35.009,-27.828],[-11,31]],"o":[[0,0],[9.966,-26.338],[1,-38],[-39,31],[11,-31]],"v":[[1,2],[48,-39],[29,-112],[136,-107],[103,-176]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[1,0.104197326361,0.38322987276,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":9,"s":[0]},{"t":46,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":6,"s":[0]},{"t":41,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":6,"op":60,"st":6,"bm":0},{"ddd":0,"ind":11,"ty":4,"nm":"Shape Layer 41","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":5,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":32,"s":[100]},{"t":48,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":0,"k":[256,256,0],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":0,"k":[100,100,100],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[-14,37],[-1,38],[35.009,-27.828],[-11,31]],"o":[[0,0],[9.966,-26.338],[1,-38],[-39,31],[11,-31]],"v":[[1,2],[48,-39],[29,-112],[136,-107],[103,-176]],"c":false},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"st","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":3},"o":{"a":0,"k":100,"ix":4},"w":{"a":0,"k":5,"ix":5,"x":"var $bm_rt;\\n$bm_rt = wiggle(5, 3);"},"lc":1,"lj":1,"ml":4,"bm":0,"nm":"Stroke 1","mn":"ADBE Vector Graphic - Stroke","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Shape 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false},{"ty":"tm","s":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":8,"s":[0]},{"t":45,"s":[100]}],"ix":1},"e":{"a":1,"k":[{"i":{"x":[0],"y":[1]},"o":{"x":[0.139],"y":[0]},"t":5,"s":[0]},{"t":40,"s":[100]}],"ix":2},"o":{"a":0,"k":0,"ix":3},"m":1,"ix":2,"nm":"Trim Paths 1","mn":"ADBE Vector Filter - Trim","hd":false}],"ip":5,"op":60,"st":5,"bm":0},{"ddd":0,"ind":12,"ty":4,"nm":"Shape Layer 40","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":6,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":33,"s":[100]},{"t":49,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.647},"o":{"x":0.167,"y":0.167},"t":6,"s":[269.95,307.113,0],"to":[-0.323,-0.36,0],"ti":[1.346,1.541,0]},{"i":{"x":0.833,"y":0.769},"o":{"x":0.167,"y":0.109},"t":7,"s":[268.011,304.951,0],"to":[-1.346,-1.541,0],"ti":[2.46,3.542,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.13},"t":8,"s":[261.875,297.864,0],"to":[-2.46,-3.542,0],"ti":[2.232,5.312,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.159},"t":9,"s":[253.252,283.701,0],"to":[-2.232,-5.312,0],"ti":[0.637,5.627,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.178},"t":10,"s":[248.481,265.992,0],"to":[-0.637,-5.627,0],"ti":[-0.921,4.788,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":11,"s":[249.431,249.939,0],"to":[0.921,-4.788,0],"ti":[-1.796,3.705,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":12,"s":[254.007,237.262,0],"to":[1.796,-3.705,0],"ti":[-2.147,2.784,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":13,"s":[260.206,227.707,0],"to":[2.147,-2.784,0],"ti":[-2.218,2.088,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":14,"s":[266.89,220.559,0],"to":[2.218,-2.088,0],"ti":[-2.161,1.578,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":15,"s":[273.513,215.179,0],"to":[2.161,-1.578,0],"ti":[-2.053,1.203,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.177},"t":16,"s":[279.855,211.089,0],"to":[2.053,-1.203,0],"ti":[-1.93,0.918,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[285.832,207.964,0],"to":[1.93,-0.918,0],"ti":[-1.801,0.7,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[291.433,205.582,0],"to":[1.801,-0.7,0],"ti":[-1.666,0.538,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":19,"s":[296.641,203.764,0],"to":[1.666,-0.538,0],"ti":[-1.53,0.417,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":20,"s":[301.428,202.354,0],"to":[1.53,-0.417,0],"ti":[-1.402,0.32,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":21,"s":[305.818,201.265,0],"to":[1.402,-0.32,0],"ti":[-1.283,0.243,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":22,"s":[309.838,200.433,0],"to":[1.283,-0.243,0],"ti":[-1.172,0.18,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[313.514,199.809,0],"to":[1.172,-0.18,0],"ti":[-1.068,0.128,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":24,"s":[316.868,199.355,0],"to":[1.068,-0.128,0],"ti":[-0.97,0.084,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":25,"s":[319.92,199.043,0],"to":[0.97,-0.084,0],"ti":[-0.879,0.048,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":26,"s":[322.69,198.85,0],"to":[0.879,-0.048,0],"ti":[-0.793,0.017,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":27,"s":[325.194,198.756,0],"to":[0.793,-0.017,0],"ti":[-0.712,-0.009,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":28,"s":[327.449,198.747,0],"to":[0.712,0.009,0],"ti":[-0.636,-0.03,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":29,"s":[329.469,198.808,0],"to":[0.636,0.03,0],"ti":[-0.564,-0.048,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":30,"s":[331.266,198.928,0],"to":[0.564,0.048,0],"ti":[-0.497,-0.063,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":31,"s":[332.854,199.098,0],"to":[0.497,0.063,0],"ti":[-0.434,-0.076,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":32,"s":[334.247,199.309,0],"to":[0.434,0.076,0],"ti":[-0.375,-0.085,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":33,"s":[335.457,199.552,0],"to":[0.375,0.085,0],"ti":[-0.32,-0.092,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":34,"s":[336.496,199.819,0],"to":[0.32,0.092,0],"ti":[-0.269,-0.097,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.181},"t":35,"s":[337.376,200.105,0],"to":[0.269,0.097,0],"ti":[-0.221,-0.099,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.182},"t":36,"s":[338.108,200.401,0],"to":[0.221,0.099,0],"ti":[-0.178,-0.1,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.183},"t":37,"s":[338.704,200.702,0],"to":[0.178,0.1,0],"ti":[-0.138,-0.098,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.184},"t":38,"s":[339.176,200.999,0],"to":[0.138,0.098,0],"ti":[-0.103,-0.094,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.186},"t":39,"s":[339.534,201.288,0],"to":[0.103,0.094,0],"ti":[-0.071,-0.087,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.187},"t":40,"s":[339.791,201.562,0],"to":[0.071,0.087,0],"ti":[-0.043,-0.079,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.188},"t":41,"s":[339.958,201.813,0],"to":[0.043,0.079,0],"ti":[-0.02,-0.068,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.189},"t":42,"s":[340.049,202.035,0],"to":[0.02,0.068,0],"ti":[0,-0.055,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.191},"t":43,"s":[340.075,202.222,0],"to":[0,0.055,0],"ti":[0.014,-0.04,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.191},"t":44,"s":[340.051,202.365,0],"to":[-0.014,0.04,0],"ti":[0.025,-0.022,0]},{"i":{"x":0.833,"y":0.855},"o":{"x":0.167,"y":0.184},"t":45,"s":[339.989,202.459,0],"to":[-0.025,0.022,0],"ti":[0.024,-0.002,0]},{"i":{"x":0.833,"y":0.799},"o":{"x":0.167,"y":0.197},"t":46,"s":[339.903,202.496,0],"to":[-0.024,0.002,0],"ti":[0.008,0.02,0]},{"i":{"x":0.833,"y":0.778},"o":{"x":0.167,"y":0.143},"t":47,"s":[339.844,202.469,0],"to":[-0.008,-0.02,0],"ti":[-0.011,0.042,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.133},"t":48,"s":[339.858,202.373,0],"to":[0.011,-0.042,0],"ti":[-0.012,0.057,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.156},"t":49,"s":[339.908,202.219,0],"to":[0.012,-0.057,0],"ti":[-0.001,0.066,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.158},"t":50,"s":[339.927,202.033,0],"to":[0.001,-0.066,0],"ti":[0.008,0.072,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":51,"s":[339.916,201.823,0],"to":[-0.008,-0.072,0],"ti":[0.017,0.076,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":52,"s":[339.877,201.599,0],"to":[-0.017,-0.076,0],"ti":[0.026,0.078,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.163},"t":53,"s":[339.812,201.365,0],"to":[-0.026,-0.078,0],"ti":[0.033,0.078,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":54,"s":[339.723,201.128,0],"to":[-0.033,-0.078,0],"ti":[0.04,0.076,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":55,"s":[339.613,200.895,0],"to":[-0.04,-0.076,0],"ti":[0.046,0.073,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.167},"t":56,"s":[339.484,200.67,0],"to":[-0.046,-0.073,0],"ti":[0.051,0.068,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":57,"s":[339.339,200.457,0],"to":[-0.051,-0.068,0],"ti":[0.055,0.062,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.169},"t":58,"s":[339.179,200.262,0],"to":[-0.055,-0.062,0],"ti":[0.029,0.029,0]},{"t":59,"s":[339.007,200.087,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.27,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":6,"s":[140.028,87.797,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.945,4.379,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,0.567,0]},"t":7,"s":[149.724,81.426,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.242,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.158,0.081,0]},"t":8,"s":[108.654,80.328,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,1.096,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,-0.235,0]},"t":9,"s":[122.85,125.952,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.361,0.045,0]},"t":10,"s":[67.435,113.999,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.46,0.786,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.186,-0.016,0]},"t":11,"s":[77.828,139.764,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.921,0.918,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,0.136,0]},"t":12,"s":[86.275,118.232,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.039,-1.479,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.604,-4.077,0]},"t":13,"s":[132.517,84.447,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.895,1.074,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,0.086,0]},"t":14,"s":[130.233,85.124,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.337,1.045,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.399,0.039,0]},"t":15,"s":[104.061,104.576,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,0.992,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,0.029,0]},"t":16,"s":[97.155,67.886,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.652,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.607,-0.009,0]},"t":17,"s":[131.988,124.186,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.86,0.858,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,-0.045,0]},"t":18,"s":[137.528,73.357,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.036,1.045,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.207,0.201,0]},"t":19,"s":[88.621,106.456,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,0.915,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,0.029,0]},"t":20,"s":[55.617,129.849,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.262,4.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.634,4.996,0]},"t":21,"s":[102.914,93.756,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,0.946,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,0.082,0]},"t":22,"s":[97.417,93.144,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.703,0.499,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.154,-0.157,0]},"t":23,"s":[54.264,122.898,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.998,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.116,0.1,0]},"t":24,"s":[69.385,112.598,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.949,1.263,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.002,0.458,0]},"t":25,"s":[108.191,60.937,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.891,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.135,0.063,0]},"t":26,"s":[70.4,49.44,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.585,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.486,0.354,0]},"t":27,"s":[84.854,97.16,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.234,0.72,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.104,-0.001,0]},"t":28,"s":[85.713,111.857,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,0.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,0.119,0]},"t":29,"s":[89.136,97.385,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.375,0.152,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,-3.19,0]},"t":30,"s":[76.083,63.267,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,0.8,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.068,0.092,0]},"t":31,"s":[85.219,64.136,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.877,1.035,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,0.143,0]},"t":32,"s":[34.985,72.111,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.257,0.025,0]},"t":33,"s":[97.846,83.275,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.901,2.92,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.02,2.259,0]},"t":34,"s":[127.941,67.428,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.553,0.981,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.536,0.08,0]},"t":35,"s":[103.695,66.821,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.782,0.701,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.102,-0.025,0]},"t":36,"s":[99.227,81.408,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.859,0.804,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,0.116,0]},"t":37,"s":[79.727,70.137,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.953,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.203,0.145,0]},"t":38,"s":[48.153,40.939,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.831,0.706,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.106,-0.113,0]},"t":39,"s":[26.225,1.63,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.052,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.164,0.116,0]},"t":40,"s":[35.878,18.283,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.639,0.832,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.032,-0.069,0]},"t":41,"s":[45.795,60.473,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.99,0.924,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.108,0.165,0]},"t":42,"s":[29.734,37.417,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,-0.589,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.011,-0.907,0]},"t":43,"s":[-23.828,13.866,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.893,0.871,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.06,0.088,0]},"t":44,"s":[23.282,15.847,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.57,1.036,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.379,0.235,0]},"t":45,"s":[-4.081,51.66,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,0.796,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.025,0]},"t":46,"s":[-11.795,71.385,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,0.979,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.061,0.141,0]},"t":47,"s":[48.641,43.137,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.129,0.825,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,-0.028,0]},"t":48,"s":[13.642,2.267,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,1.021,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,0.159,0]},"t":49,"s":[-18.389,32.951,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.864,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.028,0.017,0]},"t":50,"s":[63.189,66.722,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.844,1.094,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.146,0.216,0]},"t":51,"s":[2.261,24.31,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.977,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.179,0.044,0]},"t":52,"s":[24.372,-2.358,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.521,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.314,-0.032,0]},"t":53,"s":[43.66,54.3,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,0.99,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.101,-0.001,0]},"t":54,"s":[39.61,13.352,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.265,0.893,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.359,-0.012,0]},"t":55,"s":[20.359,53.589,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.823,0.847,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,0.371,0]},"t":56,"s":[23.988,18.274,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.068,1.058,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.157,0.183,0]},"t":57,"s":[52.384,8.05,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.034,0]},"t":58,"s":[84.408,-0.484,100]},{"t":59,"s":[26.091,14.017,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":6,"op":60,"st":6,"bm":0},{"ddd":0,"ind":13,"ty":4,"nm":"Shape Layer 39","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":16,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":38,"s":[100]},{"t":54,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.587},"o":{"x":0.167,"y":0.167},"t":11,"s":[266.709,299.63,0],"to":[-0.015,0.376,0],"ti":[0.204,-1.854,0]},{"i":{"x":0.833,"y":0.764},"o":{"x":0.167,"y":0.104},"t":12,"s":[266.621,301.887,0],"to":[-0.204,1.854,0],"ti":[0.96,-4.091,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":13,"s":[265.488,310.754,0],"to":[-0.96,4.091,0],"ti":[2.202,-5.284,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":14,"s":[260.859,326.432,0],"to":[-2.202,5.284,0],"ti":[3.145,-4.688,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":15,"s":[252.276,342.456,0],"to":[-3.145,4.688,0],"ti":[3.441,-3.397,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":16,"s":[241.99,354.561,0],"to":[-3.441,3.397,0],"ti":[3.35,-2.293,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":17,"s":[231.629,362.84,0],"to":[-3.35,2.293,0],"ti":[3.107,-1.507,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":18,"s":[221.889,368.32,0],"to":[-3.107,1.507,0],"ti":[2.823,-0.973,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":19,"s":[212.988,371.885,0],"to":[-2.823,0.973,0],"ti":[2.545,-0.61,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":20,"s":[204.948,374.155,0],"to":[-2.545,0.61,0],"ti":[2.287,-0.361,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":21,"s":[197.719,375.542,0],"to":[-2.287,0.361,0],"ti":[2.057,-0.189,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":22,"s":[191.223,376.321,0],"to":[-2.057,0.189,0],"ti":[1.855,-0.069,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":23,"s":[185.374,376.678,0],"to":[-1.855,0.069,0],"ti":[1.678,0.016,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":24,"s":[180.091,376.736,0],"to":[-1.678,-0.016,0],"ti":[1.52,0.078,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":25,"s":[175.308,376.58,0],"to":[-1.52,-0.078,0],"ti":[1.379,0.122,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[170.97,376.27,0],"to":[-1.379,-0.122,0],"ti":[1.252,0.155,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[167.033,375.845,0],"to":[-1.252,-0.155,0],"ti":[1.137,0.18,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":28,"s":[163.458,375.337,0],"to":[-1.137,-0.18,0],"ti":[1.031,0.198,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":29,"s":[160.214,374.766,0],"to":[-1.031,-0.198,0],"ti":[0.933,0.212,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[157.274,374.147,0],"to":[-0.933,-0.212,0],"ti":[0.843,0.223,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":31,"s":[154.615,373.492,0],"to":[-0.843,-0.223,0],"ti":[0.758,0.232,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":32,"s":[152.218,372.807,0],"to":[-0.758,-0.232,0],"ti":[0.683,0.239,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":33,"s":[150.067,372.098,0],"to":[-0.683,-0.239,0],"ti":[0.619,0.242,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":34,"s":[148.12,371.372,0],"to":[-0.619,-0.242,0],"ti":[0.56,0.239,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":35,"s":[146.355,370.647,0],"to":[-0.56,-0.239,0],"ti":[0.506,0.232,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":36,"s":[144.758,369.937,0],"to":[-0.506,-0.232,0],"ti":[0.454,0.219,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":37,"s":[143.32,369.258,0],"to":[-0.454,-0.219,0],"ti":[0.406,0.203,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":38,"s":[142.032,368.621,0],"to":[-0.406,-0.203,0],"ti":[0.36,0.184,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":39,"s":[140.885,368.038,0],"to":[-0.36,-0.184,0],"ti":[0.317,0.162,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":40,"s":[139.871,367.517,0],"to":[-0.317,-0.162,0],"ti":[0.277,0.138,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.18},"t":41,"s":[138.982,367.066,0],"to":[-0.277,-0.138,0],"ti":[0.238,0.111,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":42,"s":[138.211,366.692,0],"to":[-0.238,-0.111,0],"ti":[0.201,0.083,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.185},"t":43,"s":[137.554,366.399,0],"to":[-0.201,-0.083,0],"ti":[0.166,0.055,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.189},"t":44,"s":[137.005,366.191,0],"to":[-0.166,-0.055,0],"ti":[0.132,0.025,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.194},"t":45,"s":[136.559,366.071,0],"to":[-0.132,-0.025,0],"ti":[0.1,-0.005,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.195},"t":46,"s":[136.211,366.04,0],"to":[-0.1,0.005,0],"ti":[0.069,-0.034,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.182},"t":47,"s":[135.959,366.098,0],"to":[-0.069,0.034,0],"ti":[0.039,-0.063,0]},{"i":{"x":0.833,"y":0.81},"o":{"x":0.167,"y":0.158},"t":48,"s":[135.797,366.244,0],"to":[-0.039,0.063,0],"ti":[0.011,-0.091,0]},{"i":{"x":0.833,"y":0.81},"o":{"x":0.167,"y":0.148},"t":49,"s":[135.723,366.476,0],"to":[-0.011,0.091,0],"ti":[-0.016,-0.118,0]},{"i":{"x":0.833,"y":0.819},"o":{"x":0.167,"y":0.149},"t":50,"s":[135.732,366.79,0],"to":[0.016,0.118,0],"ti":[-0.036,-0.141,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.154},"t":51,"s":[135.82,367.181,0],"to":[0.036,0.141,0],"ti":[-0.044,-0.159,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.159},"t":52,"s":[135.95,367.634,0],"to":[0.044,0.159,0],"ti":[-0.046,-0.173,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.161},"t":53,"s":[136.085,368.134,0],"to":[0.046,0.173,0],"ti":[-0.047,-0.184,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":54,"s":[136.223,368.671,0],"to":[0.047,0.184,0],"ti":[-0.048,-0.193,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.164},"t":55,"s":[136.365,369.239,0],"to":[0.048,0.193,0],"ti":[-0.048,-0.197,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.165},"t":56,"s":[136.509,369.827,0],"to":[0.048,0.197,0],"ti":[-0.048,-0.198,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.167},"t":57,"s":[136.653,370.423,0],"to":[0.048,0.198,0],"ti":[-0.047,-0.195,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.169},"t":58,"s":[136.796,371.017,0],"to":[0.047,0.195,0],"ti":[-0.023,-0.096,0]},{"t":59,"s":[136.937,371.594,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.877,0.953,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":11,"s":[59.343,96.124,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.969,0.832,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.259,-0.107,0]},"t":12,"s":[89.744,43.042,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.861,1.087,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.05,0.166,0]},"t":13,"s":[104.203,66.307,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.424,0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.208,0.043,0]},"t":14,"s":[95.197,89.821,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.819,0.904,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,-0.039,0]},"t":15,"s":[89.153,41.816,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.993,0.548,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.154,0.617,0]},"t":16,"s":[125.946,74.556,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.815,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.008,0.102,0]},"t":17,"s":[169.121,79.673,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,1.513,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.152,-0.669,0]},"t":18,"s":[129.751,102.297,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.747,1.071,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.266,0.072,0]},"t":19,"s":[81.726,99.791,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.055,0.899,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.124,0.038,0]},"t":20,"s":[93.197,117.733,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,1.324,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.033,0.474,0]},"t":21,"s":[116.588,84.536,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.473,1.053,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.179,0.066,0]},"t":22,"s":[77.774,77.458,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,0.033,0]},"t":23,"s":[80.336,112.065,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.026,1.108,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.364,0.57,0]},"t":24,"s":[93.979,55.33,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.832,0.728,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.047,0]},"t":25,"s":[91.437,45.624,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.069,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.076,0.12,0]},"t":26,"s":[94.777,67.955,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,-0.116,0]},"t":27,"s":[58.109,118.474,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,0.352,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.102,0.493,0]},"t":28,"s":[125.074,97.355,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.408,0.998,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.104,0.096,0]},"t":29,"s":[95.015,93.056,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.982,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,-0.002,0]},"t":30,"s":[108.364,63.932,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.838,-0.683,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.022,-1.218,0]},"t":31,"s":[29.719,92.263,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.975,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.172,0.088,0]},"t":32,"s":[91.786,90.449,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,0.834,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.177,-0.036,0]},"t":33,"s":[150.239,55.635,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.842,0.888,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,0.167,0]},"t":34,"s":[131.528,79.867,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,1.053,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.177,0.326,0]},"t":35,"s":[114.424,104,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.464,0.726,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.141,0.032,0]},"t":36,"s":[99.134,112.3,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.762,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.12,0]},"t":37,"s":[104.825,98.77,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.398,1.006,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.54,0.128,0]},"t":38,"s":[67.48,67.824,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,0.877,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,0.006,0]},"t":39,"s":[65.344,10.519,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-3.208,1.093,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.911,0.258,0]},"t":40,"s":[77.671,71.977,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.93,0.905,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.085,0.044,0]},"t":41,"s":[76.639,101.368,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.96,1.238,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.443,0.68,0]},"t":42,"s":[25.532,39.221,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.134,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.077,0.062,0]},"t":43,"s":[33.627,30.543,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.683,0.714,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,-0.194,0]},"t":44,"s":[29.417,64.046,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.008,0.948,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,0.118,0]},"t":45,"s":[40.374,53.993,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.787,1.36,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,-0.137,0]},"t":46,"s":[71.134,29.6,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.078,1.002,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.137,0.068,0]},"t":47,"s":[37.38,38.828,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.937,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,0.002,0]},"t":48,"s":[-15.322,-10.291,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.695,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.139,-0.254,0]},"t":49,"s":[86.713,39.937,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.314,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.669,0.115,0]},"t":50,"s":[48.493,27.546,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.73,0.854,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.066,0.012,0]},"t":51,"s":[46.484,-5.44,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.035,0.884,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.12,0.194,0]},"t":52,"s":[56.051,33.097,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,1.27,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,0.299,0]},"t":53,"s":[77.513,62.169,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.866,0.851,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.324,0.064,0]},"t":54,"s":[46.909,73.396,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.037,0.912,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.222,0.188,0]},"t":55,"s":[53.173,25.847,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.024,1.423,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,1.71,0]},"t":56,"s":[56.946,-11.873,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.891,0.665,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.019,0.07,0]},"t":57,"s":[96.787,-13.805,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.356,0.111,0]},"t":58,"s":[45.548,-2.059,100]},{"t":59,"s":[29.917,33.378,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.167325382607,0.716982433843,1,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":11,"op":60,"st":11,"bm":0},{"ddd":0,"ind":14,"ty":4,"nm":"Shape Layer 38","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":7,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":34,"s":[100]},{"t":50,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.591},"o":{"x":0.167,"y":0.167},"t":7,"s":[262.436,299.331,0],"to":[0.489,-0.187,0],"ti":[-2.348,1.046,0]},{"i":{"x":0.833,"y":0.764},"o":{"x":0.167,"y":0.105},"t":8,"s":[265.373,298.21,0],"to":[2.348,-1.046,0],"ti":[-5.031,2.854,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":9,"s":[276.523,293.055,0],"to":[5.031,-2.854,0],"ti":[-6.175,4.882,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":10,"s":[295.56,281.084,0],"to":[6.175,-4.882,0],"ti":[-4.919,5.974,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":11,"s":[313.573,263.763,0],"to":[4.919,-5.974,0],"ti":[-2.793,5.995,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":12,"s":[325.076,245.24,0],"to":[2.793,-5.995,0],"ti":[-0.958,5.467,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":13,"s":[330.333,227.794,0],"to":[0.958,-5.467,0],"ti":[0.382,4.708,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":14,"s":[330.823,212.44,0],"to":[-0.382,-4.708,0],"ti":[1.261,3.892,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":15,"s":[328.039,199.545,0],"to":[-1.261,-3.892,0],"ti":[1.765,3.129,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":16,"s":[323.257,189.086,0],"to":[-1.765,-3.129,0],"ti":[2.001,2.482,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":17,"s":[317.447,180.769,0],"to":[-2.001,-2.482,0],"ti":[2.064,1.961,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":18,"s":[311.25,174.195,0],"to":[-2.064,-1.961,0],"ti":[2.024,1.552,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":19,"s":[305.064,169.001,0],"to":[-2.024,-1.552,0],"ti":[1.932,1.236,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":20,"s":[299.106,164.881,0],"to":[-1.932,-1.236,0],"ti":[1.816,0.99,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[293.474,161.588,0],"to":[-1.816,-0.99,0],"ti":[1.69,0.797,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[288.212,158.939,0],"to":[-1.69,-0.797,0],"ti":[1.563,0.644,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":23,"s":[283.333,156.803,0],"to":[-1.563,-0.644,0],"ti":[1.437,0.519,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":24,"s":[278.836,155.078,0],"to":[-1.437,-0.519,0],"ti":[1.316,0.416,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":25,"s":[274.712,153.691,0],"to":[-1.316,-0.416,0],"ti":[1.201,0.33,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":26,"s":[270.942,152.585,0],"to":[-1.201,-0.33,0],"ti":[1.094,0.257,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":27,"s":[267.505,151.714,0],"to":[-1.094,-0.257,0],"ti":[0.992,0.194,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":28,"s":[264.381,151.045,0],"to":[-0.992,-0.194,0],"ti":[0.897,0.139,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":29,"s":[261.551,150.551,0],"to":[-0.897,-0.139,0],"ti":[0.808,0.091,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":30,"s":[258.997,150.211,0],"to":[-0.808,-0.091,0],"ti":[0.724,0.048,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":31,"s":[256.702,150.007,0],"to":[-0.724,-0.048,0],"ti":[0.646,0.009,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":32,"s":[254.651,149.925,0],"to":[-0.646,-0.009,0],"ti":[0.572,-0.025,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":33,"s":[252.828,149.951,0],"to":[-0.572,0.025,0],"ti":[0.502,-0.055,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":34,"s":[251.221,150.073,0],"to":[-0.502,0.055,0],"ti":[0.437,-0.082,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":35,"s":[249.815,150.281,0],"to":[-0.437,0.082,0],"ti":[0.377,-0.105,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":36,"s":[248.597,150.563,0],"to":[-0.377,0.105,0],"ti":[0.321,-0.125,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":37,"s":[247.552,150.91,0],"to":[-0.321,0.125,0],"ti":[0.269,-0.141,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":38,"s":[246.671,151.311,0],"to":[-0.269,0.141,0],"ti":[0.22,-0.154,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":39,"s":[245.941,151.756,0],"to":[-0.22,0.154,0],"ti":[0.174,-0.163,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":40,"s":[245.353,152.233,0],"to":[-0.174,0.163,0],"ti":[0.132,-0.168,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":41,"s":[244.897,152.732,0],"to":[-0.132,0.168,0],"ti":[0.093,-0.169,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":42,"s":[244.562,153.241,0],"to":[-0.093,0.169,0],"ti":[0.057,-0.166,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":43,"s":[244.34,153.746,0],"to":[-0.057,0.166,0],"ti":[0.024,-0.158,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":44,"s":[244.221,154.236,0],"to":[-0.024,0.158,0],"ti":[-0.006,-0.145,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":45,"s":[244.197,154.694,0],"to":[0.006,0.145,0],"ti":[-0.033,-0.128,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":46,"s":[244.257,155.108,0],"to":[0.033,0.128,0],"ti":[-0.049,-0.109,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.175},"t":47,"s":[244.394,155.461,0],"to":[0.049,0.109,0],"ti":[-0.051,-0.108,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.16},"t":48,"s":[244.552,155.765,0],"to":[0.051,0.108,0],"ti":[-0.047,-0.12,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.16},"t":49,"s":[244.699,156.107,0],"to":[0.047,0.12,0],"ti":[-0.044,-0.131,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.163},"t":50,"s":[244.836,156.487,0],"to":[0.044,0.131,0],"ti":[-0.041,-0.137,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.164},"t":51,"s":[244.964,156.891,0],"to":[0.041,0.137,0],"ti":[-0.039,-0.14,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":52,"s":[245.085,157.309,0],"to":[0.039,0.14,0],"ti":[-0.037,-0.14,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":53,"s":[245.198,157.733,0],"to":[0.037,0.14,0],"ti":[-0.035,-0.138,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":54,"s":[245.305,158.152,0],"to":[0.035,0.138,0],"ti":[-0.033,-0.133,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.171},"t":55,"s":[245.406,158.559,0],"to":[0.033,0.133,0],"ti":[-0.031,-0.125,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":56,"s":[245.502,158.947,0],"to":[0.031,0.125,0],"ti":[-0.03,-0.116,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.174},"t":57,"s":[245.594,159.31,0],"to":[0.03,0.116,0],"ti":[-0.029,-0.104,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.176},"t":58,"s":[245.683,159.641,0],"to":[0.029,0.104,0],"ti":[-0.014,-0.049,0]},{"t":59,"s":[245.769,159.936,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.684,1.023,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":7,"s":[100.63,69.657,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.351,-1.07,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,0.018,0]},"t":8,"s":[104.514,69.052,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.905,1.153,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,0.087,0]},"t":9,"s":[115.346,69.821,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.398,0.948,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.695,0.054,0]},"t":10,"s":[58.871,88.161,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,0.738,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,-0.137,0]},"t":11,"s":[51.174,36.165,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.972,0.948,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.06,0.122,0]},"t":12,"s":[95.636,55.822,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.882,0.78,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.042,-0.141,0]},"t":13,"s":[69.725,97.853,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.246,1.072,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.285,0.134,0]},"t":14,"s":[86.959,82.233,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.96,0.883,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,0.039,0]},"t":15,"s":[94.096,56.625,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,1.134,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.077,0.29,0]},"t":16,"s":[151.503,104.404,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.845,0.96,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.166,0.051,0]},"t":17,"s":[121.653,123.646,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.987,1.023,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.181,-0.079,0]},"t":18,"s":[91.668,73.435,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,0.872,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.015,0.018,0]},"t":19,"s":[66.045,99.285,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,1.2,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.223,0.238,0]},"t":20,"s":[87.746,66.291,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.113,0.059,0]},"t":21,"s":[100.66,48.526,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.102,0.778,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.477,-0.182,0]},"t":22,"s":[95.172,108.893,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.575,0.936,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.046,0.133,0]},"t":23,"s":[95.987,89.94,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.108,0.256,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,-0.27,0]},"t":24,"s":[94.176,58.401,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.047,0.094,0]},"t":25,"s":[130.21,65.835,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,-0.034,0]},"t":26,"s":[47.499,124.757,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.822,1.024,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.43,-0.057,0]},"t":27,"s":[105.068,83.037,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.17,0.979,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.157,0.019,0]},"t":28,"s":[118.917,107.79,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.86,0.93,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.056,-0.028,0]},"t":29,"s":[134.616,75.912,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.094,0.206,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.205,-0.452,0]},"t":30,"s":[86.877,99.716,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.852,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,0.093,0]},"t":31,"s":[54.265,96.009,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.9,1.036,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.139,0.19,0]},"t":32,"s":[123.839,64.381,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.39,0.839,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.496,0.025,0]},"t":33,"s":[97.726,39.711,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,0.871,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.097,0.173,0]},"t":34,"s":[92.458,75.119,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.931,1.058,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.019,0.235,0]},"t":35,"s":[59.151,108.162,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.407,0.742,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.405,0.034,0]},"t":36,"s":[86.256,126.307,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.957,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.097,0.123,0]},"t":37,"s":[81.627,95.551,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.83,0.652,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.089,-0.391,0]},"t":38,"s":[53.31,31.043,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.098,1.066,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.164,0.11,0]},"t":39,"s":[67.002,42.37,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,0.972,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,0.037,0]},"t":40,"s":[81.207,78.404,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.908,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.224,-0.041,0]},"t":41,"s":[50.269,13.881,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.43,0.675,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.933,-1.114,0]},"t":42,"s":[31.981,57.039,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.97,0.375,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,0.112,0]},"t":43,"s":[30.188,54.035,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.919,0.947,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.046,0.096,0]},"t":44,"s":[41.246,45.317,100]},{"i":{"x":[0.833,0.833,0.833],"y":[17.435,1.051,1]},"o":{"x":[0.167,0.167,0.167],"y":[-2.661,-0.148,0]},"t":45,"s":[34.148,-11.396,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.997,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.083,0.032,0]},"t":46,"s":[34.363,9.033,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.893,0.829,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.004,-0.247,0]},"t":47,"s":[-8.365,-24.009,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.892,0.989,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.375,0.163,0]},"t":48,"s":[32.621,-15.669,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.847,1.17,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.364,-0.012,0]},"t":49,"s":[44.332,-6.94,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.476,0.712,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.183,0.056,0]},"t":50,"s":[47.804,-14.568,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.098,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,0.117,0]},"t":51,"s":[50.704,8.589,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,1.348,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,0.46,0]},"t":52,"s":[66.039,65.528,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.638,0.848,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.665,0.067,0]},"t":53,"s":[32.735,78.112,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.718,0.935,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,0.185,0]},"t":54,"s":[30.98,12.995,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.692,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.118,-0.305,0]},"t":55,"s":[46.164,-40.385,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.25,0.862,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.313,0.114,0]},"t":56,"s":[82.432,-28.944,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,0.97,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,0.211,0]},"t":57,"s":[74.813,1.96,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.112,-0.047,0]},"t":58,"s":[13.817,22.06,100]},{"t":59,"s":[39.779,9.173,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":7,"op":60,"st":7,"bm":0},{"ddd":0,"ind":15,"ty":4,"nm":"Shape Layer 37","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":9,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":14,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":36,"s":[100]},{"t":52,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.583},"o":{"x":0.167,"y":0.167},"t":9,"s":[264.075,300.144,0],"to":[-0.069,0.468,0],"ti":[0.355,-2.342,0]},{"i":{"x":0.833,"y":0.763},"o":{"x":0.167,"y":0.104},"t":10,"s":[263.663,302.955,0],"to":[-0.355,2.342,0],"ti":[0.559,-5.353,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":11,"s":[261.945,314.199,0],"to":[-0.559,5.353,0],"ti":[0.004,-7.357,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":12,"s":[260.312,335.072,0],"to":[-0.004,7.357,0],"ti":[-1.215,-7.142,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":13,"s":[261.922,358.34,0],"to":[1.215,7.142,0],"ti":[-2.396,-5.718,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":14,"s":[267.601,377.926,0],"to":[2.396,5.718,0],"ti":[-3.152,-4.167,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":15,"s":[276.299,392.647,0],"to":[3.152,4.167,0],"ti":[-3.444,-2.85,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":16,"s":[286.512,402.925,0],"to":[3.444,2.85,0],"ti":[-3.405,-1.874,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":17,"s":[296.962,409.748,0],"to":[3.405,1.874,0],"ti":[-3.2,-1.208,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":18,"s":[306.94,414.169,0],"to":[3.2,1.208,0],"ti":[-2.937,-0.769,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":19,"s":[316.159,416.998,0],"to":[2.937,0.769,0],"ti":[-2.672,-0.481,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":20,"s":[324.559,418.783,0],"to":[2.672,0.481,0],"ti":[-2.428,-0.291,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[332.191,419.883,0],"to":[2.428,0.291,0],"ti":[-2.206,-0.163,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":22,"s":[339.125,420.527,0],"to":[2.206,0.163,0],"ti":[-2.007,-0.077,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[345.429,420.863,0],"to":[2.007,0.077,0],"ti":[-1.827,-0.019,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":24,"s":[351.165,420.991,0],"to":[1.827,0.019,0],"ti":[-1.665,0.021,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":25,"s":[356.391,420.977,0],"to":[1.665,-0.021,0],"ti":[-1.519,0.049,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[361.156,420.865,0],"to":[1.519,-0.049,0],"ti":[-1.386,0.069,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[365.504,420.683,0],"to":[1.386,-0.069,0],"ti":[-1.264,0.084,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":28,"s":[369.472,420.452,0],"to":[1.264,-0.084,0],"ti":[-1.151,0.096,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":29,"s":[373.088,420.182,0],"to":[1.151,-0.096,0],"ti":[-1.046,0.107,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[376.379,419.878,0],"to":[1.046,-0.107,0],"ti":[-0.947,0.118,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":31,"s":[379.365,419.542,0],"to":[0.947,-0.118,0],"ti":[-0.856,0.124,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":32,"s":[382.063,419.172,0],"to":[0.856,-0.124,0],"ti":[-0.773,0.118,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":33,"s":[384.499,418.801,0],"to":[0.773,-0.118,0],"ti":[-0.697,0.107,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":34,"s":[386.7,418.464,0],"to":[0.697,-0.107,0],"ti":[-0.626,0.097,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":35,"s":[388.681,418.159,0],"to":[0.626,-0.097,0],"ti":[-0.561,0.088,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":36,"s":[390.459,417.883,0],"to":[0.561,-0.088,0],"ti":[-0.5,0.079,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":37,"s":[392.047,417.633,0],"to":[0.5,-0.079,0],"ti":[-0.443,0.072,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":38,"s":[393.458,417.406,0],"to":[0.443,-0.072,0],"ti":[-0.39,0.066,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":39,"s":[394.704,417.199,0],"to":[0.39,-0.066,0],"ti":[-0.34,0.061,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":40,"s":[395.795,417.009,0],"to":[0.34,-0.061,0],"ti":[-0.293,0.057,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":41,"s":[396.741,416.833,0],"to":[0.293,-0.057,0],"ti":[-0.249,0.053,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.182},"t":42,"s":[397.551,416.669,0],"to":[0.249,-0.053,0],"ti":[-0.207,0.05,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.184},"t":43,"s":[398.233,416.514,0],"to":[0.207,-0.05,0],"ti":[-0.168,0.048,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.186},"t":44,"s":[398.794,416.367,0],"to":[0.168,-0.048,0],"ti":[-0.132,0.046,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.19},"t":45,"s":[399.242,416.225,0],"to":[0.132,-0.046,0],"ti":[-0.097,0.045,0]},{"i":{"x":0.833,"y":0.857},"o":{"x":0.167,"y":0.195},"t":46,"s":[399.584,416.088,0],"to":[0.097,-0.045,0],"ti":[-0.064,0.044,0]},{"i":{"x":0.833,"y":0.857},"o":{"x":0.167,"y":0.2},"t":47,"s":[399.824,415.954,0],"to":[0.064,-0.044,0],"ti":[-0.034,0.043,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.2},"t":48,"s":[399.97,415.824,0],"to":[0.034,-0.043,0],"ti":[-0.012,0.044,0]},{"i":{"x":0.833,"y":0.82},"o":{"x":0.167,"y":0.168},"t":49,"s":[400.026,415.696,0],"to":[0.012,-0.044,0],"ti":[-0.006,0.049,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.156},"t":50,"s":[400.041,415.559,0],"to":[0.006,-0.049,0],"ti":[-0.008,0.056,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.158},"t":51,"s":[400.063,415.401,0],"to":[0.008,-0.056,0],"ti":[-0.011,0.062,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.159},"t":52,"s":[400.091,415.224,0],"to":[0.011,-0.062,0],"ti":[-0.014,0.066,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.161},"t":53,"s":[400.129,415.031,0],"to":[0.014,-0.066,0],"ti":[-0.017,0.07,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":54,"s":[400.175,414.825,0],"to":[0.017,-0.07,0],"ti":[-0.021,0.073,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.164},"t":55,"s":[400.232,414.61,0],"to":[0.021,-0.073,0],"ti":[-0.025,0.074,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":56,"s":[400.301,414.389,0],"to":[0.025,-0.074,0],"ti":[-0.029,0.073,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":57,"s":[400.382,414.168,0],"to":[0.029,-0.073,0],"ti":[-0.034,0.071,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.167},"t":58,"s":[400.476,413.95,0],"to":[0.034,-0.071,0],"ti":[-0.018,0.035,0]},{"t":59,"s":[400.585,413.741,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[2.237,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":9,"s":[60.433,69.807,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.354,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.078,0.012,0]},"t":10,"s":[57.035,63.909,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.869,0.914,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.136,0.096,0]},"t":11,"s":[110.865,70.792,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.993,2.539,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.23,2.807,0]},"t":12,"s":[90.384,117.236,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.615,1.021,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.008,0.079,0]},"t":13,"s":[78.755,118.657,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.957,0.878,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.106,0.017,0]},"t":14,"s":[89.349,90.995,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.759,1.314,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.09,0.263,0]},"t":15,"s":[127.76,125.63,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.85,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.128,0.066,0]},"t":16,"s":[109.298,141.669,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.93,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.187,-0.134,0]},"t":17,"s":[74.466,65.123,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.468,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.45,0.009,0]},"t":18,"s":[46.407,94.474,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.94,2.673,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,2.042,0]},"t":19,"s":[50.79,61.594,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.179,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.209,0.079,0]},"t":20,"s":[74.379,60.195,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,1.053,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.057,0.166,0]},"t":21,"s":[67.656,89.669,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.318,0.963,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.577,0.032,0]},"t":22,"s":[88.817,119.34,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.032,0.863,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,-0.068,0]},"t":23,"s":[92.391,70.889,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,1.002,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.023,0.212,0]},"t":24,"s":[145.365,97.56,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.702,0.895,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.509,0.002,0]},"t":25,"s":[71.948,114.853,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.886,1.201,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,0.407,0]},"t":26,"s":[67.657,97.077,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.837,0.928,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.312,0.059,0]},"t":27,"s":[108.077,92.496,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.302,1.255,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.171,-0.517,0]},"t":28,"s":[122.812,108.142,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.944,1.236,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.065,0.063,0]},"t":29,"s":[136.892,105.97,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.917,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.168,0.062,0]},"t":30,"s":[71.752,114.791,100]},{"i":{"x":[0.833,0.833,0.833],"y":[4.698,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[-10.728,0.167,0]},"t":31,"s":[93.325,80.942,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.281,2.111,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,0.997,0]},"t":32,"s":[93.159,47.153,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.918,0.796,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,0.078,0]},"t":33,"s":[100.704,44.07,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-6.847,0.936,1]},"o":{"x":[0.167,0.167,0.167],"y":[-7.898,0.141,0]},"t":34,"s":[67.68,88.243,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.94,0.681,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.084,-0.275,0]},"t":35,"s":[68.025,152.101,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.562,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.209,0.113,0]},"t":36,"s":[100.152,137.254,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.103,0.504,0]},"t":37,"s":[91.001,95.342,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.918,-2.094,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.001,0.554,0]},"t":38,"s":[52.026,87.036,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-2.601,0.998,1]},"o":{"x":[0.167,0.167,0.167],"y":[-3.839,0.086,0]},"t":39,"s":[90.362,85.565,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,0.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.085,-0.003,0]},"t":40,"s":[89.547,32.423,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.446,1.058,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.669,1.085,0]},"t":41,"s":[55.168,83.983,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.021,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,0.034,0]},"t":42,"s":[58.978,88.274,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.109,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.325,0.091,0]},"t":43,"s":[34.79,80.994,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,1.213,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.047,1.844,0]},"t":44,"s":[39.732,2.714,100]},{"i":{"x":[0.833,0.833,0.833],"y":[4.228,1.121,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.292,0.06,0]},"t":45,"s":[28.352,-0.992,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,0.049,0]},"t":46,"s":[29.041,12.182,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.923,0.921,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.02,-0.04,0]},"t":47,"s":[1.635,-20.19,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.207,3.708,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.991,-1.388,0]},"t":48,"s":[23.803,1.627,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.933,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.078,0.081,0]},"t":49,"s":[22.083,0.392,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.444,0.353,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.338,0.943,0]},"t":50,"s":[48.704,41.779,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.843,1.011,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.098,0.096,0]},"t":51,"s":[43.437,45.791,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.042,0.741,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.178,0.01,0]},"t":52,"s":[13.575,72.931,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.846,0.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.028,0.123,0]},"t":53,"s":[-12.851,42.13,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.907,1.471,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.181,1.058,0]},"t":54,"s":[26.868,-22.827,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.014,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.832,0.071,0]},"t":55,"s":[60.696,-28.382,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,6.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.077,11.039,0]},"t":56,"s":[64.461,8.569,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.945,1.005,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.093,0.082,0]},"t":57,"s":[14.897,8.85,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.162,0.005,0]},"t":58,"s":[38.326,-8.337,100]},{"t":59,"s":[30.376,9.931,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.167325382607,0.716982433843,1,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":9,"op":60,"st":9,"bm":0},{"ddd":0,"ind":16,"ty":4,"nm":"Shape Layer 36","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":5,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":32,"s":[100]},{"t":48,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.604},"o":{"x":0.167,"y":0.167},"t":5,"s":[268.501,303.894,0],"to":[0.373,-0.267,0],"ti":[-1.332,1.601,0]},{"i":{"x":0.833,"y":0.775},"o":{"x":0.167,"y":0.106},"t":6,"s":[270.738,302.29,0],"to":[1.332,-1.601,0],"ti":[1.952,1.768,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.133},"t":7,"s":[276.491,294.286,0],"to":[-1.952,-1.768,0],"ti":[6.31,0.166,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.156},"t":8,"s":[259.026,291.685,0],"to":[-6.31,-0.166,0],"ti":[6.364,-0.502,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":9,"s":[238.628,293.288,0],"to":[-6.364,0.502,0],"ti":[5.435,-0.368,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":10,"s":[220.844,294.694,0],"to":[-5.435,0.368,0],"ti":[4.549,-0.177,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.183},"t":11,"s":[206.017,295.497,0],"to":[-4.549,0.177,0],"ti":[3.853,-0.014,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":12,"s":[193.549,295.758,0],"to":[-3.853,0.014,0],"ti":[3.314,0.117,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":13,"s":[182.897,295.583,0],"to":[-3.314,-0.117,0],"ti":[2.886,0.223,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":14,"s":[173.668,295.056,0],"to":[-2.886,-0.223,0],"ti":[2.537,0.31,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":15,"s":[165.584,294.243,0],"to":[-2.537,-0.31,0],"ti":[2.246,0.383,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":16,"s":[158.446,293.194,0],"to":[-2.246,-0.383,0],"ti":[2.001,0.441,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[152.105,291.947,0],"to":[-2.001,-0.441,0],"ti":[1.789,0.485,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[146.441,290.551,0],"to":[-1.789,-0.485,0],"ti":[1.601,0.519,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":19,"s":[141.373,289.038,0],"to":[-1.601,-0.519,0],"ti":[1.435,0.544,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":20,"s":[136.833,287.437,0],"to":[-1.435,-0.544,0],"ti":[1.285,0.562,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":21,"s":[132.765,285.771,0],"to":[-1.285,-0.562,0],"ti":[1.15,0.573,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":22,"s":[129.121,284.063,0],"to":[-1.15,-0.573,0],"ti":[1.027,0.578,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[125.863,282.331,0],"to":[-1.027,-0.578,0],"ti":[0.915,0.576,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[122.957,280.597,0],"to":[-0.915,-0.576,0],"ti":[0.814,0.569,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[120.371,278.877,0],"to":[-0.814,-0.569,0],"ti":[0.721,0.558,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[118.075,277.184,0],"to":[-0.721,-0.558,0],"ti":[0.637,0.542,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":27,"s":[116.044,275.532,0],"to":[-0.637,-0.542,0],"ti":[0.56,0.523,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":28,"s":[114.254,273.931,0],"to":[-0.56,-0.523,0],"ti":[0.49,0.502,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":29,"s":[112.684,272.392,0],"to":[-0.49,-0.502,0],"ti":[0.427,0.478,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[111.312,270.921,0],"to":[-0.427,-0.478,0],"ti":[0.369,0.453,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":31,"s":[110.122,269.523,0],"to":[-0.369,-0.453,0],"ti":[0.317,0.426,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":32,"s":[109.095,268.204,0],"to":[-0.317,-0.426,0],"ti":[0.27,0.399,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":33,"s":[108.219,266.966,0],"to":[-0.27,-0.399,0],"ti":[0.226,0.37,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":34,"s":[107.478,265.812,0],"to":[-0.226,-0.37,0],"ti":[0.186,0.342,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":35,"s":[106.862,264.743,0],"to":[-0.186,-0.342,0],"ti":[0.15,0.313,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":36,"s":[106.36,263.762,0],"to":[-0.15,-0.313,0],"ti":[0.117,0.283,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":37,"s":[105.962,262.868,0],"to":[-0.117,-0.283,0],"ti":[0.086,0.254,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":38,"s":[105.66,262.062,0],"to":[-0.086,-0.254,0],"ti":[0.058,0.224,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":39,"s":[105.445,261.346,0],"to":[-0.058,-0.224,0],"ti":[0.032,0.194,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.182},"t":40,"s":[105.312,260.718,0],"to":[-0.032,-0.194,0],"ti":[0.008,0.164,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.184},"t":41,"s":[105.253,260.18,0],"to":[-0.008,-0.164,0],"ti":[-0.014,0.135,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.185},"t":42,"s":[105.263,259.732,0],"to":[0.014,-0.135,0],"ti":[-0.034,0.104,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.185},"t":43,"s":[105.336,259.373,0],"to":[0.034,-0.104,0],"ti":[-0.052,0.074,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.181},"t":44,"s":[105.467,259.105,0],"to":[0.052,-0.074,0],"ti":[-0.065,0.05,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":45,"s":[105.65,258.927,0],"to":[0.065,-0.05,0],"ti":[-0.069,0.036,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.172},"t":46,"s":[105.86,258.807,0],"to":[0.069,-0.036,0],"ti":[-0.062,0.029,0]},{"i":{"x":0.833,"y":0.856},"o":{"x":0.167,"y":0.186},"t":47,"s":[106.065,258.712,0],"to":[0.062,-0.029,0],"ti":[-0.047,0.024,0]},{"i":{"x":0.833,"y":0.863},"o":{"x":0.167,"y":0.197},"t":48,"s":[106.232,258.632,0],"to":[0.047,-0.024,0],"ti":[-0.031,0.02,0]},{"i":{"x":0.833,"y":0.876},"o":{"x":0.167,"y":0.214},"t":49,"s":[106.349,258.566,0],"to":[0.031,-0.02,0],"ti":[-0.015,0.014,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.252},"t":50,"s":[106.418,258.515,0],"to":[0.015,-0.014,0],"ti":[0.001,0.009,0]},{"i":{"x":0.833,"y":0.742},"o":{"x":0.167,"y":0.188},"t":51,"s":[106.44,258.479,0],"to":[-0.001,-0.009,0],"ti":[0.016,0.003,0]},{"i":{"x":0.833,"y":0.781},"o":{"x":0.167,"y":0.123},"t":52,"s":[106.414,258.46,0],"to":[-0.016,-0.003,0],"ti":[0.031,-0.003,0]},{"i":{"x":0.833,"y":0.801},"o":{"x":0.167,"y":0.134},"t":53,"s":[106.343,258.459,0],"to":[-0.031,0.003,0],"ti":[0.045,-0.008,0]},{"i":{"x":0.833,"y":0.811},"o":{"x":0.167,"y":0.144},"t":54,"s":[106.228,258.475,0],"to":[-0.045,0.008,0],"ti":[0.059,-0.014,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.149},"t":55,"s":[106.07,258.51,0],"to":[-0.059,0.014,0],"ti":[0.072,-0.02,0]},{"i":{"x":0.833,"y":0.82},"o":{"x":0.167,"y":0.153},"t":56,"s":[105.873,258.562,0],"to":[-0.072,0.02,0],"ti":[0.084,-0.026,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.155},"t":57,"s":[105.639,258.632,0],"to":[-0.084,0.026,0],"ti":[0.095,-0.032,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.157},"t":58,"s":[105.369,258.719,0],"to":[-0.095,0.032,0],"ti":[0.05,-0.017,0]},{"t":59,"s":[105.069,258.823,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.896,-0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":5,"s":[111.822,109.374,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.087,0.955,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.414,0.087,0]},"t":6,"s":[100.055,106.422,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.812,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,-0.097,0]},"t":7,"s":[97.093,40.951,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.773,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.117,0.15,0]},"t":8,"s":[67.613,71.247,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.893,1.157,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.132,-0.129,0]},"t":9,"s":[79.872,109.348,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.215,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.381,0.054,0]},"t":10,"s":[101.054,94.428,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,0.945,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.06,-0.015,0]},"t":11,"s":[106.979,137.485,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.974,0.793,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.477,-0.164,0]},"t":12,"s":[85.798,101.174,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.674,1.013,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.039,0.139,0]},"t":13,"s":[88.947,113.411,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,0.677,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,0.011,0]},"t":14,"s":[86.793,131.596,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.583,0.977,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.914,0.112,0]},"t":15,"s":[106.377,110.603,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.868,0.899,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.088,-0.033,0]},"t":16,"s":[104.74,50.192,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.873,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.225,0.471,0]},"t":17,"s":[75.287,93.627,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,0.963,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.244,0.557,0]},"t":18,"s":[57.971,102.957,100]},{"i":{"x":[0.833,0.833,0.833],"y":[6.07,1.272,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.53,-0.068,0]},"t":19,"s":[48.983,104.599,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.801,1.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.082,0.064,0]},"t":20,"s":[48.677,103.694,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.786,1.017,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.143,0.076,0]},"t":21,"s":[67.607,107.546,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.136,0.014,0]},"t":22,"s":[93.956,61.633,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.898,0.924,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.001,-0.051,0]},"t":23,"s":[135.372,117.12,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.936,1.36,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.448,-0.923,0]},"t":24,"s":[94.654,82.761,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.146,1.103,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.276,0.068,0]},"t":25,"s":[85.351,85.606,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.778,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,0.046,0]},"t":26,"s":[87.506,70.455,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.973,0.043,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.134,1.425,0]},"t":27,"s":[107.432,104.297,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.792,1.155,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.04,0.091,0]},"t":28,"s":[140.473,106.399,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.932,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.139,0.054,0]},"t":29,"s":[118.147,128.441,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.974,1.121,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.054,-0.359,0]},"t":30,"s":[84.776,65.43,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.673,0.886,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.038,0.049,0]},"t":31,"s":[105.029,77.29,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.925,1.408,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.112,0.311,0]},"t":32,"s":[91.068,48.266,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.647,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.747,0.069,0]},"t":33,"s":[50.228,37.635,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.827,0.939,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.109,-0.474,0]},"t":34,"s":[54.329,100.336,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.244,1.036,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.161,-0.223,0]},"t":35,"s":[67.613,90.964,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,0.595,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,0.025,0]},"t":36,"s":[81.902,93.51,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.785,0.829,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.101,0.105,0]},"t":37,"s":[25.699,89.858,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.976,0.897,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.136,0.163,0]},"t":38,"s":[51.156,75.74,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,1.149,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.034,0.43,0]},"t":39,"s":[91.35,60.958,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.534,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.231,0.053,0]},"t":40,"s":[62.734,57.409,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,0.526,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.072,-0.014,0]},"t":41,"s":[70.32,67.319,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.896,0.923,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.24,0.101,0]},"t":42,"s":[14.17,58.838,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.32,0.689,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.419,-0.944,0]},"t":43,"s":[28.636,19.113,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.869,1.114,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.066,0.114,0]},"t":44,"s":[32.23,22.336,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.092,1.006,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.228,0.048,0]},"t":45,"s":[14.851,31.15,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.899,1.001,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,0.005,0]},"t":46,"s":[4.812,10.283,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.075,1.195,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.472,0.001,0]},"t":47,"s":[25.925,32.587,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.799,0.984,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,0.058,0]},"t":48,"s":[30.453,9.9,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.842,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.142,-0.02,0]},"t":49,"s":[21.826,85.625,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.674,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.177,-0.039,0]},"t":50,"s":[9.608,24.619,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.022,0.324,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.112,-0.462,0]},"t":51,"s":[-1.289,66.096,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.018,0.095,0]},"t":52,"s":[-33.054,59.758,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,1.101,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.173,1.876,0]},"t":53,"s":[7.239,14.644,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.273,1.364,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.683,0.046,0]},"t":54,"s":[44.549,12.547,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.913,0.998,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,0.068,0]},"t":55,"s":[40.492,17.184,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.637,1.065,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.953,-0.003,0]},"t":56,"s":[9.161,-7.703,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.935,0.991,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,0.037,0]},"t":57,"s":[7.764,16.458,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.3,-0.01,0]},"t":58,"s":[36.594,-26.534,100]},{"t":59,"s":[30.332,11.798,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":5,"op":60,"st":5,"bm":0},{"ddd":0,"ind":17,"ty":4,"nm":"Shape Layer 35","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":4,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":9,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":31,"s":[100]},{"t":47,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.58},"o":{"x":0.167,"y":0.167},"t":4,"s":[269.847,301.952,0],"to":[0.17,-0.147,0],"ti":[-0.855,0.741,0]},{"i":{"x":0.833,"y":0.763},"o":{"x":0.167,"y":0.104},"t":5,"s":[270.865,301.071,0],"to":[0.855,-0.741,0],"ti":[-1.897,1.74,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":6,"s":[274.976,297.505,0],"to":[1.897,-1.74,0],"ti":[-2.487,2.492,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":7,"s":[282.245,290.631,0],"to":[2.487,-2.492,0],"ti":[-2.333,2.568,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":8,"s":[289.898,282.555,0],"to":[2.333,-2.568,0],"ti":[-1.909,2.26,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":9,"s":[296.244,275.226,0],"to":[1.909,-2.26,0],"ti":[-1.56,1.916,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.183},"t":10,"s":[301.355,268.994,0],"to":[1.56,-1.916,0],"ti":[-1.325,1.619,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":11,"s":[305.606,263.727,0],"to":[1.325,-1.619,0],"ti":[-1.185,1.356,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":12,"s":[309.304,259.281,0],"to":[1.185,-1.356,0],"ti":[-1.147,1.074,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":13,"s":[312.714,255.59,0],"to":[1.147,-1.074,0],"ti":[-1.205,0.459,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.179},"t":14,"s":[316.183,252.839,0],"to":[1.205,-0.459,0],"ti":[-0.955,-0.508,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.171},"t":15,"s":[319.947,252.839,0],"to":[0.955,0.508,0],"ti":[-0.499,-1.03,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":16,"s":[321.913,255.888,0],"to":[0.499,1.03,0],"ti":[-0.292,-1.003,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[322.944,259.018,0],"to":[0.292,1.003,0],"ti":[-0.216,-0.921,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[323.663,261.908,0],"to":[0.216,0.921,0],"ti":[-0.18,-0.838,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":19,"s":[324.24,264.543,0],"to":[0.18,0.838,0],"ti":[-0.16,-0.762,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":20,"s":[324.742,266.939,0],"to":[0.16,0.762,0],"ti":[-0.149,-0.692,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":21,"s":[325.202,269.115,0],"to":[0.149,0.692,0],"ti":[-0.143,-0.628,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":22,"s":[325.638,271.093,0],"to":[0.143,0.628,0],"ti":[-0.139,-0.568,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[326.06,272.884,0],"to":[0.139,0.568,0],"ti":[-0.136,-0.513,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":24,"s":[326.472,274.503,0],"to":[0.136,0.513,0],"ti":[-0.134,-0.461,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":25,"s":[326.877,275.962,0],"to":[0.134,0.461,0],"ti":[-0.131,-0.413,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":26,"s":[327.274,277.271,0],"to":[0.131,0.413,0],"ti":[-0.128,-0.369,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":27,"s":[327.663,278.442,0],"to":[0.128,0.369,0],"ti":[-0.123,-0.327,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":28,"s":[328.039,279.483,0],"to":[0.123,0.327,0],"ti":[-0.117,-0.287,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":29,"s":[328.401,280.402,0],"to":[0.117,0.287,0],"ti":[-0.109,-0.251,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.178},"t":30,"s":[328.742,281.208,0],"to":[0.109,0.251,0],"ti":[-0.095,-0.217,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":31,"s":[329.057,281.906,0],"to":[0.095,0.217,0],"ti":[-0.065,-0.188,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":32,"s":[329.315,282.507,0],"to":[0.065,0.188,0],"ti":[-0.024,-0.165,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.181},"t":33,"s":[329.447,283.036,0],"to":[0.024,0.165,0],"ti":[0.014,-0.145,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.176},"t":34,"s":[329.46,283.5,0],"to":[-0.014,0.145,0],"ti":[0.05,-0.126,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.169},"t":35,"s":[329.361,283.904,0],"to":[-0.05,0.126,0],"ti":[0.083,-0.109,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.163},"t":36,"s":[329.16,284.255,0],"to":[-0.083,0.109,0],"ti":[0.113,-0.092,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.16},"t":37,"s":[328.864,284.555,0],"to":[-0.113,0.092,0],"ti":[0.14,-0.078,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.159},"t":38,"s":[328.482,284.81,0],"to":[-0.14,0.078,0],"ti":[0.165,-0.064,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.159},"t":39,"s":[328.022,285.021,0],"to":[-0.165,0.064,0],"ti":[0.187,-0.05,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.16},"t":40,"s":[327.492,285.191,0],"to":[-0.187,0.05,0],"ti":[0.206,-0.038,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.161},"t":41,"s":[326.902,285.323,0],"to":[-0.206,0.038,0],"ti":[0.222,-0.026,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.162},"t":42,"s":[326.259,285.418,0],"to":[-0.222,0.026,0],"ti":[0.235,-0.015,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.163},"t":43,"s":[325.573,285.479,0],"to":[-0.235,0.015,0],"ti":[0.245,-0.007,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.164},"t":44,"s":[324.851,285.506,0],"to":[-0.245,0.007,0],"ti":[0.251,-0.008,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":45,"s":[324.104,285.522,0],"to":[-0.251,0.008,0],"ti":[0.255,-0.011,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":46,"s":[323.342,285.551,0],"to":[-0.255,0.011,0],"ti":[0.256,-0.015,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.167},"t":47,"s":[322.574,285.591,0],"to":[-0.256,0.015,0],"ti":[0.254,-0.018,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.168},"t":48,"s":[321.808,285.64,0],"to":[-0.254,0.018,0],"ti":[0.248,-0.02,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":49,"s":[321.053,285.698,0],"to":[-0.248,0.02,0],"ti":[0.24,-0.022,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.17},"t":50,"s":[320.318,285.762,0],"to":[-0.24,0.022,0],"ti":[0.229,-0.023,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.171},"t":51,"s":[319.612,285.831,0],"to":[-0.229,0.023,0],"ti":[0.214,-0.024,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.173},"t":52,"s":[318.945,285.902,0],"to":[-0.214,0.024,0],"ti":[0.197,-0.024,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.175},"t":53,"s":[318.326,285.973,0],"to":[-0.197,0.024,0],"ti":[0.176,-0.022,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.178},"t":54,"s":[317.764,286.043,0],"to":[-0.176,0.022,0],"ti":[0.153,-0.02,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.182},"t":55,"s":[317.268,286.108,0],"to":[-0.153,0.02,0],"ti":[0.126,-0.017,0]},{"i":{"x":0.833,"y":0.857},"o":{"x":0.167,"y":0.188},"t":56,"s":[316.848,286.165,0],"to":[-0.126,0.017,0],"ti":[0.096,-0.013,0]},{"i":{"x":0.833,"y":0.87},"o":{"x":0.167,"y":0.2},"t":57,"s":[316.513,286.212,0],"to":[-0.096,0.013,0],"ti":[0.062,-0.008,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.232},"t":58,"s":[316.273,286.245,0],"to":[-0.062,0.008,0],"ti":[0.022,-0.003,0]},{"t":59,"s":[316.139,286.261,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.112,0.057,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":4,"s":[102.583,113.906,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.982,0.964,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.048,0.091,0]},"t":5,"s":[89.34,107.749,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.826,0.789,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.023,-0.063,0]},"t":6,"s":[120.4,44.201,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.974,0.965,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.16,0.138,0]},"t":7,"s":[96.055,80.415,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.007,0.953,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.038,-0.061,0]},"t":8,"s":[69.666,135.77,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.054,1.124,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,-0.107,0]},"t":9,"s":[87.726,103.756,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.958,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.033,0.05,0]},"t":10,"s":[68.046,117.794,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.847,0.448,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.085,-0.418,0]},"t":11,"s":[100.548,82.836,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.184,0.098,0]},"t":12,"s":[84.441,88.646,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.653,0.719,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.581,-0.033,0]},"t":13,"s":[71.053,121.348,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,0.925,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,0.118,0]},"t":14,"s":[68.813,98.016,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.876,-0.382,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.52,-0.749,0]},"t":15,"s":[115.484,42.548,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.015,0.928,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.255,0.089,0]},"t":16,"s":[109.038,48.103,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.942,0.908,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,-0.516,0]},"t":17,"s":[105.912,134.682,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,-0.015,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.189,0.903,0]},"t":18,"s":[70.98,122.635,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.191,0.613,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.021,0.091,0]},"t":19,"s":[81.664,121.409,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.88,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,0.106,0]},"t":20,"s":[73.094,107.712,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.105,1.007,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.271,-0.001,0]},"t":21,"s":[101.314,57.875,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.034,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.046,0.006,0]},"t":22,"s":[113.878,107.014,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.024,1.786,0]},"t":23,"s":[85.547,53.745,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.812,-0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.094,-0.086,0]},"t":24,"s":[125.423,51.139,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,0.814,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.149,0.087,0]},"t":25,"s":[106.714,52.421,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,0.151,0]},"t":26,"s":[83.118,81.524,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.571,1.517,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.689,0.542,0]},"t":27,"s":[112.603,117.515,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.176,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.103,0.072,0]},"t":28,"s":[109.423,124.06,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.971,0.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.057,-0.391,0]},"t":29,"s":[96.227,76.928,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.758,5.317,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.044,1.133,0]},"t":30,"s":[137.222,85.21,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.901,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.127,0.082,0]},"t":31,"s":[110.401,85.868,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.016,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.527,-0.056,0]},"t":32,"s":[59.435,51.153,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.802,1.05,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.013,-0.013,0]},"t":33,"s":[49.857,71.912,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.11,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.144,0.031,0]},"t":34,"s":[61.259,54.038,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,0.88,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.047,-0.059,0]},"t":35,"s":[76.982,82.665,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,1.149,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.103,0.274,0]},"t":36,"s":[40.562,65.859,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.016,1.039,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.187,0.053,0]},"t":37,"s":[56.885,58.517,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.025,0.918,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.013,0.027,0]},"t":38,"s":[70.044,78.995,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.03,2.634,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.019,-7.024,0]},"t":39,"s":[54.364,48.853,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.85,0.699,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.022,0.079,0]},"t":40,"s":[74.752,49.207,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.102,1.073,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.188,0.115,0]},"t":41,"s":[46.926,41.924,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.008,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.046,0.039,0]},"t":42,"s":[24.667,22.936,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.923,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.008,-0.041,0]},"t":43,"s":[74.072,58.487,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.782,0.823,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.08,-0.204,0]},"t":44,"s":[19.734,34.651,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.772,1.167,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,0.158,0]},"t":45,"s":[23.626,41.571,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.128,1.109,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.131,0.056,0]},"t":46,"s":[29.896,49.314,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.837,1.026,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.05,0.047,0]},"t":47,"s":[40.776,26.063,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,0.975,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.171,0.02,0]},"t":48,"s":[13.174,79.764,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.928,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.153,-0.036,0]},"t":49,"s":[-13.177,9.252,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.831,0.665,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.32,-0.508,0]},"t":50,"s":[-3.882,58.452,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.879,0.883,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.076,0.111,0]},"t":51,"s":[-5.803,51.522,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.861,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.266,0.29,0]},"t":52,"s":[15.263,30.594,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.125,2.138,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.208,-0.398,0]},"t":53,"s":[24.861,22.138,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.882,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.05,0.078,0]},"t":54,"s":[31.285,23.603,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.155,1.18,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.282,-0.051,0]},"t":55,"s":[15.205,2.136,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.018,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,0.057,0]},"t":56,"s":[8.46,15.435,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.07,2.225,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.015,0.737,0]},"t":57,"s":[27.717,-26.636,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.078,0]},"t":58,"s":[4.398,-31.997,100]},{"t":59,"s":[47.278,52.184,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.167325382607,0.716982433843,1,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":4,"op":60,"st":4,"bm":0},{"ddd":0,"ind":18,"ty":4,"nm":"Shape Layer 34","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":15,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":37,"s":[100]},{"t":53,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.622},"o":{"x":0.167,"y":0.167},"t":10,"s":[271.252,303.813,0],"to":[0.284,-0.193,0],"ti":[-1.149,1.055,0]},{"i":{"x":0.833,"y":0.767},"o":{"x":0.167,"y":0.107},"t":11,"s":[272.957,302.654,0],"to":[1.149,-1.055,0],"ti":[-2.022,2.73,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.13},"t":12,"s":[278.144,297.481,0],"to":[2.022,-2.73,0],"ti":[-1.91,4.187,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":13,"s":[285.092,286.273,0],"to":[1.91,-4.187,0],"ti":[-1.055,4.438,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.178},"t":14,"s":[289.604,272.358,0],"to":[1.055,-4.438,0],"ti":[-0.359,3.914,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":15,"s":[291.42,259.644,0],"to":[0.359,-3.914,0],"ti":[0.022,3.308,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.182},"t":16,"s":[291.76,248.874,0],"to":[-0.022,-3.308,0],"ti":[0.239,2.795,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":17,"s":[291.291,239.797,0],"to":[-0.239,-2.795,0],"ti":[0.374,2.38,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":18,"s":[290.325,232.102,0],"to":[-0.374,-2.38,0],"ti":[0.463,2.046,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":19,"s":[289.046,225.517,0],"to":[-0.463,-2.046,0],"ti":[0.527,1.772,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":20,"s":[287.548,219.827,0],"to":[-0.527,-1.772,0],"ti":[0.577,1.542,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[285.886,214.884,0],"to":[-0.577,-1.542,0],"ti":[0.62,1.344,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[284.087,210.573,0],"to":[-0.62,-1.344,0],"ti":[0.662,1.168,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":23,"s":[282.165,206.818,0],"to":[-0.662,-1.168,0],"ti":[0.709,1.006,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":24,"s":[280.115,203.564,0],"to":[-0.709,-1.006,0],"ti":[0.773,0.841,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.172},"t":25,"s":[277.909,200.78,0],"to":[-0.773,-0.841,0],"ti":[0.882,0.613,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.169},"t":26,"s":[275.475,198.518,0],"to":[-0.882,-0.613,0],"ti":[0.968,0.094,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.169},"t":27,"s":[272.616,197.104,0],"to":[-0.968,-0.094,0],"ti":[0.794,-0.512,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.173},"t":28,"s":[269.666,197.956,0],"to":[-0.794,0.512,0],"ti":[0.517,-0.748,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.176},"t":29,"s":[267.851,200.177,0],"to":[-0.517,0.748,0],"ti":[0.389,-0.735,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":30,"s":[266.566,202.443,0],"to":[-0.389,0.735,0],"ti":[0.329,-0.691,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":31,"s":[265.514,204.588,0],"to":[-0.329,0.691,0],"ti":[0.293,-0.642,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":32,"s":[264.591,206.588,0],"to":[-0.293,0.642,0],"ti":[0.267,-0.593,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":33,"s":[263.756,208.44,0],"to":[-0.267,0.593,0],"ti":[0.245,-0.545,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":34,"s":[262.99,210.145,0],"to":[-0.245,0.545,0],"ti":[0.226,-0.497,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":35,"s":[262.283,211.707,0],"to":[-0.226,0.497,0],"ti":[0.206,-0.451,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":36,"s":[261.635,213.129,0],"to":[-0.206,0.451,0],"ti":[0.186,-0.406,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":37,"s":[261.046,214.413,0],"to":[-0.186,0.406,0],"ti":[0.164,-0.36,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.178},"t":38,"s":[260.52,215.562,0],"to":[-0.164,0.36,0],"ti":[0.14,-0.315,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.18},"t":39,"s":[260.062,216.575,0],"to":[-0.14,0.315,0],"ti":[0.114,-0.269,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.183},"t":40,"s":[259.679,217.45,0],"to":[-0.114,0.269,0],"ti":[0.086,-0.224,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.187},"t":41,"s":[259.376,218.188,0],"to":[-0.086,0.224,0],"ti":[0.055,-0.178,0]},{"i":{"x":0.833,"y":0.86},"o":{"x":0.167,"y":0.194},"t":42,"s":[259.162,218.791,0],"to":[-0.055,0.178,0],"ti":[0.022,-0.132,0]},{"i":{"x":0.833,"y":0.862},"o":{"x":0.167,"y":0.205},"t":43,"s":[259.045,219.258,0],"to":[-0.022,0.132,0],"ti":[-0.015,-0.086,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.21},"t":44,"s":[259.032,219.585,0],"to":[0.015,0.086,0],"ti":[-0.054,-0.038,0]},{"i":{"x":0.833,"y":0.802},"o":{"x":0.167,"y":0.162},"t":45,"s":[259.133,219.772,0],"to":[0.054,0.038,0],"ti":[-0.089,-0.002,0]},{"i":{"x":0.833,"y":0.818},"o":{"x":0.167,"y":0.144},"t":46,"s":[259.355,219.813,0],"to":[0.089,0.002,0],"ti":[-0.114,0.012,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.153},"t":47,"s":[259.669,219.787,0],"to":[0.114,-0.012,0],"ti":[-0.131,0.018,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.157},"t":48,"s":[260.041,219.743,0],"to":[0.131,-0.018,0],"ti":[-0.144,0.024,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.16},"t":49,"s":[260.458,219.681,0],"to":[0.144,-0.024,0],"ti":[-0.153,0.025,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.163},"t":50,"s":[260.907,219.601,0],"to":[0.153,-0.025,0],"ti":[-0.159,0.017,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":51,"s":[261.377,219.531,0],"to":[0.159,-0.017,0],"ti":[-0.161,0.006,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":52,"s":[261.86,219.498,0],"to":[0.161,-0.006,0],"ti":[-0.161,-0.005,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":53,"s":[262.346,219.497,0],"to":[0.161,0.005,0],"ti":[-0.157,-0.014,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":54,"s":[262.825,219.527,0],"to":[0.157,0.014,0],"ti":[-0.151,-0.023,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.17},"t":55,"s":[263.289,219.583,0],"to":[0.151,0.023,0],"ti":[-0.142,-0.03,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.171},"t":56,"s":[263.73,219.663,0],"to":[0.142,0.03,0],"ti":[-0.132,-0.036,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":57,"s":[264.142,219.763,0],"to":[0.132,0.036,0],"ti":[-0.119,-0.041,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.174},"t":58,"s":[264.519,219.88,0],"to":[0.119,0.041,0],"ti":[-0.056,-0.022,0]},{"t":59,"s":[264.857,220.011,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.023,-0.027,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":10,"s":[92.348,80.588,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.063,0.897,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.018,0.091,0]},"t":11,"s":[63.239,83.148,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.953,1.468,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.036,0.435,0]},"t":12,"s":[100.528,112.151,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.799,0.83,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.11,0.071,0]},"t":13,"s":[34.84,119.029,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.908,1.046,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.142,0.163,0]},"t":14,"s":[63.172,73.511,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.12,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.917,0.03,0]},"t":15,"s":[103.227,26.126,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.04,-2.324,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.049,10.207,0]},"t":16,"s":[107.23,99.638,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.987,0.97,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,0.085,0]},"t":17,"s":[97.456,100.244,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.007,0.792,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.016,-0.047,0]},"t":18,"s":[111.909,123.776,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.079,0.895,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,0.139,0]},"t":19,"s":[99.739,108.689,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,0.438,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,0.401,0]},"t":20,"s":[112.92,86.203,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.27,0.904,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.304,0.098,0]},"t":21,"s":[87.298,80.298,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.89,1.161,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,0.638,0]},"t":22,"s":[77.606,46.38,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.303,0.905,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.348,0.055,0]},"t":23,"s":[118.751,41.284,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,-2.246,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.065,0.688,0]},"t":24,"s":[131.696,56.225,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.997,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.244,0.086,0]},"t":25,"s":[71.627,58.284,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,0.56,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.003,-1.183,0]},"t":26,"s":[86.93,136.413,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.554,0.858,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.032,0.103,0]},"t":27,"s":[72.183,131.272,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.904,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.102,0.202,0]},"t":28,"s":[82.81,109.279,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.416,0.998,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.64,-0.034,0]},"t":29,"s":[129.066,93.807,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.845,0.765,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,-0.002,0]},"t":30,"s":[135.997,104.839,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.974,0.903,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.18,0.129,0]},"t":31,"s":[94.456,94.071,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.873,1.617,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.038,0.599,0]},"t":32,"s":[58.778,74.421,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.889,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.243,0.073,0]},"t":33,"s":[83.328,71.244,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.695,0.802,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.338,-0.134,0]},"t":34,"s":[96.122,97.929,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.062,1.055,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.115,0.144,0]},"t":35,"s":[100.314,87.693,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.795,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.036,0.033,0]},"t":36,"s":[111.459,73.549,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.993,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,-0.002,0]},"t":37,"s":[91.979,97.003,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.87,1.251,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.007,0.447,0]},"t":38,"s":[63.62,73.964,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.09,1.06,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.232,0.063,0]},"t":39,"s":[89.728,68.683,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,1,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.043,0.035,0]},"t":40,"s":[104.347,89.888,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,1.034,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.032,0,0]},"t":41,"s":[74.014,53.405,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.81,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.02,0.024,0]},"t":42,"s":[95.84,90.08,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.816,0.763,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.148,-0.184,0]},"t":43,"s":[78.32,38.425,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.899,0.993,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.152,0.129,0]},"t":44,"s":[55.867,54.538,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.701,0.99,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.487,-0.007,0]},"t":45,"s":[28.81,84.258,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.904,1.154,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.115,-0.011,0]},"t":46,"s":[23.23,56.921,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.701,0.959,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.611,0.054,0]},"t":47,"s":[8.759,81.032,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.089,0.935,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.116,-0.083,0]},"t":48,"s":[6.473,12.233,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.758,1.033,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.043,-0.293,0]},"t":49,"s":[0.558,46.8,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,1.273,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.127,0.023,0]},"t":50,"s":[12.808,39.141,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,0.953,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.02,0.064,0]},"t":51,"s":[36.176,49.79,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,0.954,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.242,-0.11,0]},"t":52,"s":[17.41,4.217,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.67,0.939,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.224,-0.102,0]},"t":53,"s":[22.21,23.867,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.825,1.394,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.112,-0.223,0]},"t":54,"s":[25.048,15.014,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.216,0.678,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.159,0.069,0]},"t":55,"s":[33.433,17.425,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.985,1.08,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.06,0.112,0]},"t":56,"s":[42.612,3.611,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.012,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.019,0.041,0]},"t":57,"s":[9.673,-36.008,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.01,-0.68,0]},"t":58,"s":[36.625,41.5,100]},{"t":59,"s":[5.886,33.044,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.8941176470588236,0.11372549019607843,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":10,"op":60,"st":10,"bm":0},{"ddd":0,"ind":19,"ty":4,"nm":"Shape Layer 33","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":1,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":6,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":28,"s":[100]},{"t":44,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.635},"o":{"x":0.167,"y":0.167},"t":1,"s":[270.415,303.961,0],"to":[0.303,-0.268,0],"ti":[-1.389,1.063,0]},{"i":{"x":0.833,"y":0.798},"o":{"x":0.167,"y":0.108},"t":2,"s":[272.233,302.354,0],"to":[1.389,-1.063,0],"ti":[-2.915,0.504,0]},{"i":{"x":0.833,"y":0.803},"o":{"x":0.167,"y":0.142},"t":3,"s":[278.751,297.585,0],"to":[2.915,-0.504,0],"ti":[-1.552,-2.903,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.144},"t":4,"s":[289.724,299.332,0],"to":[1.552,2.903,0],"ti":[0.807,-4.867,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":5,"s":[288.065,315.004,0],"to":[-0.807,4.867,0],"ti":[1.017,-4.14,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":6,"s":[284.884,328.537,0],"to":[-1.017,4.14,0],"ti":[0.905,-3.485,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":7,"s":[281.965,339.842,0],"to":[-0.905,3.485,0],"ti":[0.776,-2.987,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.18},"t":8,"s":[279.455,349.449,0],"to":[-0.776,2.987,0],"ti":[0.667,-2.607,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":9,"s":[277.308,357.764,0],"to":[-0.667,2.607,0],"ti":[0.578,-2.312,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":10,"s":[275.455,365.094,0],"to":[-0.578,2.312,0],"ti":[0.505,-2.074,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":11,"s":[273.842,371.638,0],"to":[-0.505,2.074,0],"ti":[0.444,-1.876,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":12,"s":[272.427,377.536,0],"to":[-0.444,1.876,0],"ti":[0.393,-1.708,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":13,"s":[271.178,382.891,0],"to":[-0.393,1.708,0],"ti":[0.349,-1.562,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":14,"s":[270.071,387.781,0],"to":[-0.349,1.562,0],"ti":[0.31,-1.434,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":15,"s":[269.086,392.264,0],"to":[-0.31,1.434,0],"ti":[0.275,-1.32,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":16,"s":[268.211,396.386,0],"to":[-0.275,1.32,0],"ti":[0.243,-1.215,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":17,"s":[267.436,400.181,0],"to":[-0.243,1.215,0],"ti":[0.213,-1.118,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":18,"s":[266.752,403.676,0],"to":[-0.213,1.118,0],"ti":[0.185,-1.028,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":19,"s":[266.156,406.891,0],"to":[-0.185,1.028,0],"ti":[0.158,-0.941,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":20,"s":[265.642,409.842,0],"to":[-0.158,0.941,0],"ti":[0.131,-0.859,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":21,"s":[265.209,412.54,0],"to":[-0.131,0.859,0],"ti":[0.105,-0.779,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":22,"s":[264.855,414.996,0],"to":[-0.105,0.779,0],"ti":[0.078,-0.702,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":23,"s":[264.581,417.216,0],"to":[-0.078,0.702,0],"ti":[0.051,-0.625,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":24,"s":[264.386,419.205,0],"to":[-0.051,0.625,0],"ti":[0.024,-0.55,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":25,"s":[264.272,420.968,0],"to":[-0.024,0.55,0],"ti":[-0.004,-0.475,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":26,"s":[264.241,422.506,0],"to":[0.004,0.475,0],"ti":[-0.032,-0.401,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":27,"s":[264.295,423.82,0],"to":[0.032,0.401,0],"ti":[-0.062,-0.326,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.185},"t":28,"s":[264.435,424.912,0],"to":[0.062,0.326,0],"ti":[-0.092,-0.251,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.187},"t":29,"s":[264.666,425.779,0],"to":[0.092,0.251,0],"ti":[-0.123,-0.178,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.184},"t":30,"s":[264.99,426.419,0],"to":[0.123,0.178,0],"ti":[-0.144,-0.125,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.173},"t":31,"s":[265.403,426.848,0],"to":[0.144,0.125,0],"ti":[-0.154,-0.094,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":32,"s":[265.856,427.168,0],"to":[0.154,0.094,0],"ti":[-0.158,-0.068,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.17},"t":33,"s":[266.328,427.409,0],"to":[0.158,0.068,0],"ti":[-0.158,-0.046,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":34,"s":[266.806,427.579,0],"to":[0.158,0.046,0],"ti":[-0.155,-0.026,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":35,"s":[267.278,427.686,0],"to":[0.155,0.026,0],"ti":[-0.148,-0.008,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":36,"s":[267.736,427.736,0],"to":[0.148,0.008,0],"ti":[-0.139,0.007,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":37,"s":[268.169,427.736,0],"to":[0.139,-0.007,0],"ti":[-0.127,0.022,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":38,"s":[268.571,427.692,0],"to":[0.127,-0.022,0],"ti":[-0.114,0.034,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":39,"s":[268.934,427.607,0],"to":[0.114,-0.034,0],"ti":[-0.098,0.046,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.175},"t":40,"s":[269.252,427.486,0],"to":[0.098,-0.046,0],"ti":[-0.081,0.051,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.18},"t":41,"s":[269.522,427.333,0],"to":[0.081,-0.051,0],"ti":[-0.064,0.046,0]},{"i":{"x":0.833,"y":0.858},"o":{"x":0.167,"y":0.191},"t":42,"s":[269.74,427.18,0],"to":[0.064,-0.046,0],"ti":[-0.047,0.035,0]},{"i":{"x":0.833,"y":0.868},"o":{"x":0.167,"y":0.202},"t":43,"s":[269.907,427.059,0],"to":[0.047,-0.035,0],"ti":[-0.029,0.025,0]},{"i":{"x":0.833,"y":0.882},"o":{"x":0.167,"y":0.227},"t":44,"s":[270.02,426.97,0],"to":[0.029,-0.025,0],"ti":[-0.011,0.015,0]},{"i":{"x":0.833,"y":0.799},"o":{"x":0.167,"y":0.281},"t":45,"s":[270.079,426.911,0],"to":[0.011,-0.015,0],"ti":[0.007,0.006,0]},{"i":{"x":0.833,"y":0.748},"o":{"x":0.167,"y":0.142},"t":46,"s":[270.083,426.879,0],"to":[-0.007,-0.006,0],"ti":[0.025,-0.002,0]},{"i":{"x":0.833,"y":0.791},"o":{"x":0.167,"y":0.125},"t":47,"s":[270.035,426.873,0],"to":[-0.025,0.002,0],"ti":[0.041,-0.009,0]},{"i":{"x":0.833,"y":0.808},"o":{"x":0.167,"y":0.139},"t":48,"s":[269.935,426.889,0],"to":[-0.041,0.009,0],"ti":[0.056,-0.014,0]},{"i":{"x":0.833,"y":0.816},"o":{"x":0.167,"y":0.147},"t":49,"s":[269.788,426.925,0],"to":[-0.056,0.014,0],"ti":[0.07,-0.019,0]},{"i":{"x":0.833,"y":0.822},"o":{"x":0.167,"y":0.153},"t":50,"s":[269.597,426.976,0],"to":[-0.07,0.019,0],"ti":[0.082,-0.022,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.156},"t":51,"s":[269.367,427.038,0],"to":[-0.082,0.022,0],"ti":[0.092,-0.023,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.16},"t":52,"s":[269.104,427.106,0],"to":[-0.092,0.023,0],"ti":[0.099,-0.022,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.162},"t":53,"s":[268.815,427.175,0],"to":[-0.099,0.022,0],"ti":[0.104,-0.02,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.165},"t":54,"s":[268.508,427.24,0],"to":[-0.104,0.02,0],"ti":[0.105,-0.015,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.167},"t":55,"s":[268.192,427.293,0],"to":[-0.105,0.015,0],"ti":[0.103,-0.008,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.17},"t":56,"s":[267.876,427.329,0],"to":[-0.103,0.008,0],"ti":[0.098,0.002,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.173},"t":57,"s":[267.571,427.341,0],"to":[-0.098,-0.002,0],"ti":[0.088,0.014,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.176},"t":58,"s":[267.289,427.32,0],"to":[-0.088,-0.014,0],"ti":[0.041,0.01,0]},{"t":59,"s":[267.042,427.258,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.161,0.992,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":1,"s":[88.706,125.309,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.987,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,-0.009,0]},"t":2,"s":[101.05,65.647,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.822,0.862,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.015,-0.05,0]},"t":3,"s":[64.822,119.625,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.03,1.194,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.156,0.21,0]},"t":4,"s":[95.37,85.974,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.022,0.058,0]},"t":5,"s":[130.225,63.782,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.866,0.793,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.229,-0.185,0]},"t":6,"s":[82.767,137.754,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.05,0.961,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.219,0.139,0]},"t":7,"s":[95.438,114.769,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.544,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.031,-0.072,0]},"t":8,"s":[103.2,80.591,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.032,0.895,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.102,0.224,0]},"t":9,"s":[90.818,98.938,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.336,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.023,0.402,0]},"t":10,"s":[35.459,109.792,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.842,1.105,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.056,0.095,0]},"t":11,"s":[111.828,112.627,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.085,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.176,0.046,0]},"t":12,"s":[66.113,132.396,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.924,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.042,0.248,0]},"t":13,"s":[25.062,87.733,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.884,1.494,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.328,-0.831,0]},"t":14,"s":[107.997,65.085,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.34,1.282,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.298,0.071,0]},"t":15,"s":[91.212,67.15,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,0.947,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,0.064,0]},"t":16,"s":[84.699,52.844,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.04,0.928,1]},"o":{"x":[0.167,0.167,0.167],"y":[-2.065,-0.147,0]},"t":17,"s":[117.811,115.589,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.695,1.984,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,-0.553,0]},"t":18,"s":[116.527,92.87,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.902,0.923,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,0.077,0]},"t":19,"s":[118.433,95.846,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.407,0.784,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.556,-0.959,0]},"t":20,"s":[77.764,57.746,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.863,0.619,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,0.136,0]},"t":21,"s":[70.59,60.792,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.073,1.124,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.213,0.107,0]},"t":22,"s":[112.827,65.643,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.871,1.062,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.039,0.05,0]},"t":23,"s":[139.959,82.972,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.858,0.887,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.235,0.036,0]},"t":24,"s":[89.044,39.953,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.03,1.025,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.202,0.321,0]},"t":25,"s":[61.106,115.024,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,0.861,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.022,0.019,0]},"t":26,"s":[41.52,141.346,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.086,0.818,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.674,0.209,0]},"t":27,"s":[68.262,107.246,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,0.94,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.077,0.154,0]},"t":28,"s":[65.317,84.613,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.919,1.127,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.001,-0.216,0]},"t":29,"s":[106.651,57.805,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[-3.288,0.05,0]},"t":30,"s":[66.005,65.265,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-1.723,0.574,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.143,-0.241,0]},"t":31,"s":[67.009,46.429,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,1.067,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.086,0.104,0]},"t":32,"s":[66.639,51.274,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.778,0.921,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,0.037,0]},"t":33,"s":[54.923,71.216,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.071,0.741,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.133,-1.444,0]},"t":34,"s":[65.427,35.22,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.878,1.091,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.123,0]},"t":35,"s":[82.893,37.183,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.906,0.828,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.264,0.043,0]},"t":36,"s":[50.525,41.333,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.063,1.344,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.747,0.162,0]},"t":37,"s":[35.563,32.66,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.889,1.002,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.077,0.067,0]},"t":38,"s":[33.685,23.422,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.055,0.924,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.331,0.002,0]},"t":39,"s":[59.517,70.825,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,1.608,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.033,-0.9,0]},"t":40,"s":[68.226,22.181,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.215,0.945,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.055,0.073,0]},"t":41,"s":[53.746,26.302,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.832,0.514,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.06,-0.163,0]},"t":42,"s":[62.499,-7.882,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.925,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.166,0.101,0]},"t":43,"s":[31.128,3.701,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.261,0.975,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.72,-0.052,0]},"t":44,"s":[-0.586,59.702,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.131,1.044,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.063,-0.035,0]},"t":45,"s":[2.703,25.328,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,0.936,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,0.029,0]},"t":46,"s":[-10.892,49.534,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.851,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.242,-0.278,0]},"t":47,"s":[24.109,12.637,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.541,1.12,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.189,-0.068,0]},"t":48,"s":[15.151,21.155,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.973,1.225,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.072,0.049,0]},"t":49,"s":[8.06,16.479,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.881,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.04,0.061,0]},"t":50,"s":[61.168,27.875,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.037,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.278,-0.07,0]},"t":51,"s":[25.413,-14.304,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.927,1.673,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.026,-0.462,0]},"t":52,"s":[10.106,8.552,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.809,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.597,0.074,0]},"t":53,"s":[32.24,5.058,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.896,0.766,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.148,-0.321,0]},"t":54,"s":[29.528,36.767,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.876,1.171,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.422,0.129,0]},"t":55,"s":[26.015,30.226,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.716,0.961,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.252,0.056,0]},"t":56,"s":[25.151,18.362,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.719,0.812,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.118,-0.074,0]},"t":57,"s":[24.725,54.585,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.075,0.15,0]},"t":58,"s":[23.696,35.402,100]},{"t":59,"s":[33.602,11.392,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":1,"op":60,"st":1,"bm":0},{"ddd":0,"ind":20,"ty":4,"nm":"Shape Layer 32","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":4,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":9,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":31,"s":[100]},{"t":47,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.569},"o":{"x":0.167,"y":0.167},"t":4,"s":[267.904,308.221,0],"to":[-0.062,-0.384,0],"ti":[-0.406,2.013,0]},{"i":{"x":0.833,"y":0.761},"o":{"x":0.167,"y":0.103},"t":5,"s":[267.529,305.917,0],"to":[0.406,-2.013,0],"ti":[-2.229,4.277,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.128},"t":6,"s":[270.337,296.143,0],"to":[2.229,-4.277,0],"ti":[-4.056,5.365,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":7,"s":[280.902,280.252,0],"to":[4.056,-5.365,0],"ti":[-4.394,5.019,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.178},"t":8,"s":[294.676,263.955,0],"to":[4.394,-5.019,0],"ti":[-3.868,4.214,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":9,"s":[307.264,250.141,0],"to":[3.868,-4.214,0],"ti":[-3.257,3.538,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":10,"s":[317.883,238.67,0],"to":[3.257,-3.538,0],"ti":[-2.752,3.047,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.18},"t":11,"s":[326.809,228.915,0],"to":[2.752,-3.047,0],"ti":[-2.349,2.69,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":12,"s":[334.394,220.39,0],"to":[2.349,-2.69,0],"ti":[-2.023,2.427,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.177},"t":13,"s":[340.902,212.773,0],"to":[2.023,-2.427,0],"ti":[-1.753,2.23,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":14,"s":[346.531,205.826,0],"to":[1.753,-2.23,0],"ti":[-1.519,2.072,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":15,"s":[351.417,199.393,0],"to":[1.519,-2.072,0],"ti":[-1.306,1.928,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":16,"s":[355.644,193.397,0],"to":[1.306,-1.928,0],"ti":[-1.106,1.797,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":17,"s":[359.255,187.824,0],"to":[1.106,-1.797,0],"ti":[-0.913,1.682,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[362.28,182.616,0],"to":[0.913,-1.682,0],"ti":[-0.72,1.579,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":19,"s":[364.731,177.73,0],"to":[0.72,-1.579,0],"ti":[-0.524,1.479,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":20,"s":[366.601,173.143,0],"to":[0.524,-1.479,0],"ti":[-0.321,1.378,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":21,"s":[367.872,168.855,0],"to":[0.321,-1.378,0],"ti":[-0.115,1.264,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":22,"s":[368.526,164.876,0],"to":[0.115,-1.264,0],"ti":[0.083,1.13,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":23,"s":[368.561,161.269,0],"to":[-0.083,-1.13,0],"ti":[0.254,0.978,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":24,"s":[368.028,158.096,0],"to":[-0.254,-0.978,0],"ti":[0.382,0.82,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":25,"s":[367.038,155.4,0],"to":[-0.382,-0.82,0],"ti":[0.464,0.67,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":26,"s":[365.735,153.176,0],"to":[-0.464,-0.67,0],"ti":[0.506,0.539,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":27,"s":[364.254,151.378,0],"to":[-0.506,-0.539,0],"ti":[0.52,0.43,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":28,"s":[362.7,149.94,0],"to":[-0.52,-0.43,0],"ti":[0.517,0.343,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":29,"s":[361.136,148.795,0],"to":[-0.517,-0.343,0],"ti":[0.505,0.273,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":30,"s":[359.597,147.883,0],"to":[-0.505,-0.273,0],"ti":[0.487,0.216,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":31,"s":[358.105,147.16,0],"to":[-0.487,-0.216,0],"ti":[0.465,0.169,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":32,"s":[356.676,146.589,0],"to":[-0.465,-0.169,0],"ti":[0.441,0.132,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":33,"s":[355.317,146.143,0],"to":[-0.441,-0.132,0],"ti":[0.415,0.101,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":34,"s":[354.033,145.799,0],"to":[-0.415,-0.101,0],"ti":[0.388,0.076,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":35,"s":[352.827,145.538,0],"to":[-0.388,-0.076,0],"ti":[0.36,0.056,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":36,"s":[351.704,145.343,0],"to":[-0.36,-0.056,0],"ti":[0.33,0.04,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":37,"s":[350.668,145.203,0],"to":[-0.33,-0.04,0],"ti":[0.298,0.027,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":38,"s":[349.724,145.104,0],"to":[-0.298,-0.027,0],"ti":[0.263,0.019,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.179},"t":39,"s":[348.882,145.038,0],"to":[-0.263,-0.019,0],"ti":[0.227,0.013,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.181},"t":40,"s":[348.144,144.993,0],"to":[-0.227,-0.013,0],"ti":[0.189,0.011,0]},{"i":{"x":0.833,"y":0.855},"o":{"x":0.167,"y":0.187},"t":41,"s":[347.517,144.959,0],"to":[-0.189,-0.011,0],"ti":[0.146,0.011,0]},{"i":{"x":0.833,"y":0.865},"o":{"x":0.167,"y":0.196},"t":42,"s":[347.011,144.929,0],"to":[-0.146,-0.011,0],"ti":[0.1,0.014,0]},{"i":{"x":0.833,"y":0.867},"o":{"x":0.167,"y":0.218},"t":43,"s":[346.639,144.893,0],"to":[-0.1,-0.014,0],"ti":[0.056,0.022,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.224},"t":44,"s":[346.413,144.843,0],"to":[-0.056,-0.022,0],"ti":[0.03,0.033,0]},{"i":{"x":0.833,"y":0.816},"o":{"x":0.167,"y":0.169},"t":45,"s":[346.304,144.761,0],"to":[-0.03,-0.033,0],"ti":[0.031,0.038,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.152},"t":46,"s":[346.234,144.648,0],"to":[-0.031,-0.038,0],"ti":[0.045,0.038,0]},{"i":{"x":0.833,"y":0.82},"o":{"x":0.167,"y":0.153},"t":47,"s":[346.12,144.534,0],"to":[-0.045,-0.038,0],"ti":[0.059,0.037,0]},{"i":{"x":0.833,"y":0.822},"o":{"x":0.167,"y":0.155},"t":48,"s":[345.962,144.422,0],"to":[-0.059,-0.037,0],"ti":[0.072,0.036,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.157},"t":49,"s":[345.764,144.313,0],"to":[-0.072,-0.036,0],"ti":[0.083,0.034,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.158},"t":50,"s":[345.53,144.207,0],"to":[-0.083,-0.034,0],"ti":[0.093,0.033,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.16},"t":51,"s":[345.264,144.107,0],"to":[-0.093,-0.033,0],"ti":[0.102,0.031,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.161},"t":52,"s":[344.97,144.011,0],"to":[-0.102,-0.031,0],"ti":[0.11,0.029,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.162},"t":53,"s":[344.65,143.922,0],"to":[-0.11,-0.029,0],"ti":[0.117,0.026,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.163},"t":54,"s":[344.308,143.839,0],"to":[-0.117,-0.026,0],"ti":[0.123,0.024,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.164},"t":55,"s":[343.947,143.764,0],"to":[-0.123,-0.024,0],"ti":[0.127,0.022,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.164},"t":56,"s":[343.572,143.695,0],"to":[-0.127,-0.022,0],"ti":[0.13,0.019,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":57,"s":[343.185,143.634,0],"to":[-0.13,-0.019,0],"ti":[0.133,0.017,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":58,"s":[342.79,143.581,0],"to":[-0.133,-0.017,0],"ti":[0.067,0.008,0]},{"t":59,"s":[342.389,143.535,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.916,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":4,"s":[146.545,56.981,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.984,6.136,1]},"o":{"x":[0.167,0.167,0.167],"y":[7.109,6.416,0]},"t":5,"s":[102.654,113.791,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.082,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.08,0.082,0]},"t":6,"s":[102.133,114.538,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.769,0.88,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.041,-0.024,0]},"t":7,"s":[115.051,67.711,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,0.924,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.13,0.272,0]},"t":8,"s":[89.461,104.174,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.843,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.029,-0.825,0]},"t":9,"s":[44.049,120.312,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.068,5.668,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.177,-0.052,0]},"t":10,"s":[77.823,118.832,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.009,0.967,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.037,0.082,0]},"t":11,"s":[107.747,119.743,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,0.945,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.009,-0.054,0]},"t":12,"s":[53.479,67.782,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-1.061,0.515,1]},"o":{"x":[0.167,0.167,0.167],"y":[-2.064,-0.16,0]},"t":13,"s":[113.921,99.393,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.981,1.019,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.087,0.101,0]},"t":14,"s":[111.576,88.565,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.95,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.025,0.015,0]},"t":15,"s":[55.899,36.375,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.094,1.148,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.128,0.737,0]},"t":16,"s":[98.768,100.4,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.97,0.816,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,0.053,0]},"t":17,"s":[81.873,108.568,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.853,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.047,0.153,0]},"t":18,"s":[117.879,85.877,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.034,0.862,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.192,-0.017,0]},"t":19,"s":[94.849,58.578,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.781,0.836,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.024,0.211,0]},"t":20,"s":[77.175,81.139,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.98,1.271,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,0.169,0]},"t":21,"s":[102.134,95.906,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.835,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.026,0.064,0]},"t":22,"s":[142.654,110.218,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.797,0.841,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.168,-0.034,0]},"t":23,"s":[111.887,49.352,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.009,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,0.175,0]},"t":24,"s":[81.638,92.763,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.008,0.009,0]},"t":25,"s":[38.028,132.048,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.012,0.82,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.143,0.392,0]},"t":26,"s":[86.128,88.194,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.088,0.901,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.011,0.155,0]},"t":27,"s":[68.431,76.37,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.042,1.263,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.043,0.518,0]},"t":28,"s":[88.699,62.632,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.935,1.037,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.028,0.063,0]},"t":29,"s":[46.98,59.998,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.851,1.103,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.298,0.026,0]},"t":30,"s":[109.874,70.958,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.881,1.047,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.189,0.046,0]},"t":31,"s":[96.114,55.137,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.231,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.278,0.03,0]},"t":32,"s":[85.214,90.54,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.189,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,-0.071,0]},"t":33,"s":[80.545,35.132,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.94,0.796,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,-0.412,0]},"t":34,"s":[98.178,65.031,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.966,0.606,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.209,0.141,0]},"t":35,"s":[40.533,60.003,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,1.021,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.059,0.106,0]},"t":36,"s":[56.966,52.732,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.747,1.016,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.244,0.016,0]},"t":37,"s":[47.335,25.638,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.818,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.124,0.014,0]},"t":38,"s":[49.783,59.415,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.061,0.813,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.154,0.718,0]},"t":39,"s":[54.761,18.968,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.894,1.058,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.035,0.15,0]},"t":40,"s":[60.644,13.657,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.945,0.618,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.384,0.034,0]},"t":41,"s":[50.471,7.064,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.973,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.087,0.107,0]},"t":42,"s":[47.647,18.253,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.927,0.88,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.04,-0.023,0]},"t":43,"s":[-15.426,58.31,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.52,0.892,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.604,0.275,0]},"t":44,"s":[27.159,27.049,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,0.901,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.072,0.364,0]},"t":45,"s":[21.997,13.425,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.588,1.135,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.54,0.54,0]},"t":46,"s":[59.347,9.373,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.055,1.741,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.104,0.051,0]},"t":47,"s":[54.352,8.634,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,0.829,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.033,0.075,0]},"t":48,"s":[34.643,10.569,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.881,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.001,0.163,0]},"t":49,"s":[67.47,-8.566,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,0.842,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.276,-0.04,0]},"t":50,"s":[34.991,-28.664,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.321,0.639,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.801,0.176,0]},"t":51,"s":[20.908,-15.133,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.876,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.08,0.108,0]},"t":52,"s":[21.531,-2.998,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.331,0.787,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.256,0.39,0]},"t":53,"s":[3.556,37.39,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.945,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,0.137,0]},"t":54,"s":[-5.112,48.357,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.837,0.83,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.482,-0.165,0]},"t":55,"s":[37.958,65.386,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.163,0.146,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.171,0.163,0]},"t":56,"s":[40.525,59.682,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.082,0.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.093,0.092,0]},"t":57,"s":[42.972,53.735,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.041,-0.57,0]},"t":58,"s":[65.106,-1.291,100]},{"t":59,"s":[21.097,5.723,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.8941176470588236,0.11372549019607843,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":4,"op":60,"st":4,"bm":0},{"ddd":0,"ind":21,"ty":4,"nm":"Shape Layer 31","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":5,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":32,"s":[100]},{"t":48,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.6},"o":{"x":0.167,"y":0.167},"t":5,"s":[274.616,306.243,0],"to":[-0.115,-0.59,0],"ti":[0.824,2.77,0]},{"i":{"x":0.833,"y":0.765},"o":{"x":0.167,"y":0.105},"t":6,"s":[273.925,302.703,0],"to":[-0.824,-2.77,0],"ti":[2.181,6.093,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":7,"s":[269.674,289.624,0],"to":[-2.181,-6.093,0],"ti":[3.267,8.196,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":8,"s":[260.838,266.142,0],"to":[-3.267,-8.196,0],"ti":[3.485,7.964,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":9,"s":[250.074,240.449,0],"to":[-3.485,-7.964,0],"ti":[3.198,6.707,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":10,"s":[239.929,218.357,0],"to":[-3.198,-6.707,0],"ti":[2.853,5.529,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":11,"s":[230.887,200.209,0],"to":[-2.853,-5.529,0],"ti":[2.563,4.612,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":12,"s":[222.813,185.182,0],"to":[-2.563,-4.612,0],"ti":[2.332,3.906,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":13,"s":[215.51,172.535,0],"to":[-2.332,-3.906,0],"ti":[2.146,3.349,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":14,"s":[208.822,161.746,0],"to":[-2.146,-3.349,0],"ti":[1.992,2.898,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":15,"s":[202.636,152.442,0],"to":[-1.992,-2.898,0],"ti":[1.863,2.525,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":16,"s":[196.868,144.357,0],"to":[-1.863,-2.525,0],"ti":[1.751,2.21,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[191.458,137.291,0],"to":[-1.751,-2.21,0],"ti":[1.653,1.941,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[186.36,131.094,0],"to":[-1.653,-1.941,0],"ti":[1.563,1.706,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":19,"s":[181.542,125.648,0],"to":[-1.563,-1.706,0],"ti":[1.481,1.501,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":20,"s":[176.98,120.856,0],"to":[-1.481,-1.501,0],"ti":[1.403,1.32,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":21,"s":[172.656,116.64,0],"to":[-1.403,-1.32,0],"ti":[1.328,1.158,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":22,"s":[168.561,112.936,0],"to":[-1.328,-1.158,0],"ti":[1.255,1.014,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[164.687,109.689,0],"to":[-1.255,-1.014,0],"ti":[1.183,0.886,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[161.03,106.849,0],"to":[-1.183,-0.886,0],"ti":[1.111,0.771,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":25,"s":[157.59,104.374,0],"to":[-1.111,-0.771,0],"ti":[1.038,0.668,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[154.367,102.223,0],"to":[-1.038,-0.668,0],"ti":[0.964,0.577,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":27,"s":[151.363,100.364,0],"to":[-0.964,-0.577,0],"ti":[0.888,0.495,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":28,"s":[148.583,98.763,0],"to":[-0.888,-0.495,0],"ti":[0.81,0.422,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":29,"s":[146.035,97.394,0],"to":[-0.81,-0.422,0],"ti":[0.741,0.358,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":30,"s":[143.723,96.23,0],"to":[-0.741,-0.358,0],"ti":[0.687,0.303,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":31,"s":[141.59,95.244,0],"to":[-0.687,-0.303,0],"ti":[0.639,0.255,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":32,"s":[139.604,94.413,0],"to":[-0.639,-0.255,0],"ti":[0.592,0.214,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":33,"s":[137.759,93.715,0],"to":[-0.592,-0.214,0],"ti":[0.547,0.179,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":34,"s":[136.052,93.13,0],"to":[-0.547,-0.179,0],"ti":[0.502,0.149,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":35,"s":[134.48,92.643,0],"to":[-0.502,-0.149,0],"ti":[0.46,0.124,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":36,"s":[133.038,92.238,0],"to":[-0.46,-0.124,0],"ti":[0.418,0.103,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":37,"s":[131.723,91.901,0],"to":[-0.418,-0.103,0],"ti":[0.378,0.085,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":38,"s":[130.53,91.622,0],"to":[-0.378,-0.085,0],"ti":[0.339,0.071,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":39,"s":[129.457,91.39,0],"to":[-0.339,-0.071,0],"ti":[0.301,0.059,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":40,"s":[128.499,91.197,0],"to":[-0.301,-0.059,0],"ti":[0.264,0.049,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":41,"s":[127.654,91.037,0],"to":[-0.264,-0.049,0],"ti":[0.228,0.041,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":42,"s":[126.917,90.903,0],"to":[-0.228,-0.041,0],"ti":[0.193,0.034,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":43,"s":[126.287,90.791,0],"to":[-0.193,-0.034,0],"ti":[0.158,0.028,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.187},"t":44,"s":[125.761,90.698,0],"to":[-0.158,-0.028,0],"ti":[0.134,0.025,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.177},"t":45,"s":[125.337,90.622,0],"to":[-0.134,-0.025,0],"ti":[0.126,0.027,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":46,"s":[124.96,90.547,0],"to":[-0.126,-0.027,0],"ti":[0.127,0.03,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":47,"s":[124.579,90.461,0],"to":[-0.127,-0.03,0],"ti":[0.127,0.034,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":48,"s":[124.196,90.364,0],"to":[-0.127,-0.034,0],"ti":[0.126,0.037,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.167},"t":49,"s":[123.816,90.257,0],"to":[-0.126,-0.037,0],"ti":[0.124,0.04,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.168},"t":50,"s":[123.44,90.141,0],"to":[-0.124,-0.04,0],"ti":[0.121,0.042,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.169},"t":51,"s":[123.073,90.017,0],"to":[-0.121,-0.042,0],"ti":[0.116,0.044,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.17},"t":52,"s":[122.717,89.887,0],"to":[-0.116,-0.044,0],"ti":[0.11,0.046,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":53,"s":[122.376,89.751,0],"to":[-0.11,-0.046,0],"ti":[0.103,0.046,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":54,"s":[122.055,89.613,0],"to":[-0.103,-0.046,0],"ti":[0.095,0.046,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.173},"t":55,"s":[121.755,89.474,0],"to":[-0.095,-0.046,0],"ti":[0.086,0.045,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.175},"t":56,"s":[121.483,89.337,0],"to":[-0.086,-0.045,0],"ti":[0.075,0.043,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.178},"t":57,"s":[121.241,89.204,0],"to":[-0.075,-0.043,0],"ti":[0.062,0.041,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.182},"t":58,"s":[121.035,89.077,0],"to":[-0.062,-0.041,0],"ti":[0.028,0.019,0]},{"t":59,"s":[120.867,88.96,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.272,0.954,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":5,"s":[88.213,98.042,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.933,0.787,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,-0.101,0]},"t":6,"s":[98.671,49.922,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,0.883,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.333,0.137,0]},"t":7,"s":[54.098,71.655,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.856,1.001,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.111,0.287,0]},"t":8,"s":[63.017,105.433,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.719,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.076,0.001,0]},"t":9,"s":[59.185,119.254,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,1.027,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.071,0.118,0]},"t":10,"s":[102.36,105.209,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,0.921,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.001,0.02,0]},"t":11,"s":[79.066,71.811,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.893,-1.794,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.579,-1.652,0]},"t":12,"s":[101.985,115.908,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.972,0.937,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.076,0.086,0]},"t":13,"s":[105.84,113.791,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.033,0.698,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.042,-0.255,0]},"t":14,"s":[60.688,44.915,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.04,0.92,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.024,0.115,0]},"t":15,"s":[90.735,61.894,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,0.271,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,-2.022,0]},"t":16,"s":[48.636,106.533,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.029,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.867,0.094,0]},"t":17,"s":[110.804,104.766,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.309,0.797,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.022,-0.018,0]},"t":18,"s":[105.35,91.067,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.026,0.865,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.066,0.141,0]},"t":19,"s":[112.726,102.388,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,1.297,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.217,0]},"t":20,"s":[78.021,118.647,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.73,0.94,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.493,0.065,0]},"t":21,"s":[123.499,128.748,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.684,0.681,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.12,-0.217,0]},"t":22,"s":[116.924,82.593,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,1.037,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,0.113,0]},"t":23,"s":[102.179,95.418,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.886,0.872,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,0.026,0]},"t":24,"s":[61.093,131.691,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.995,1.018,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.309,0.239,0]},"t":25,"s":[97.755,79.275,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.561,0.959,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.006,0.015,0]},"t":26,"s":[111.269,51.174,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.103,-0.081,0]},"t":27,"s":[98.597,85.374,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.896,0.259,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.031,0.937,0]},"t":28,"s":[44.555,68.006,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.03,1.073,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.425,0.094,0]},"t":29,"s":[83.792,66.311,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.079,1.021,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.022,0.039,0]},"t":30,"s":[93.366,52.935,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.872,0.97,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.041,0.017,0]},"t":31,"s":[80.351,77.978,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.459,0.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.24,-0.047,0]},"t":32,"s":[105.736,46.617,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.918,1.164,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,1.147,0]},"t":33,"s":[119.231,66.597,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.338,0.408,1]},"o":{"x":[0.167,0.167,0.167],"y":[-4.839,0.055,0]},"t":34,"s":[31.333,68.163,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.923,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,0.097,0]},"t":35,"s":[32.821,63.516,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.316,1.245,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.049,-0.192,0]},"t":36,"s":[55.229,35.157,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.082,0.955,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.095,0.062,0]},"t":37,"s":[53.579,43.75,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.008,0.863,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.041,-0.1,0]},"t":38,"s":[41.681,9.938,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,1.024,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.008,0.213,0]},"t":39,"s":[65.338,25.314,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.135,1.097,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,0.019,0]},"t":40,"s":[39.293,35.228,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.942,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.052,0.045,0]},"t":41,"s":[71.909,22.453,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,1.746,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.192,0.562,0]},"t":42,"s":[-13.722,50.113,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.78,0.885,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,0.075,0]},"t":43,"s":[12.165,54.928,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,1.049,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.134,0.304,0]},"t":44,"s":[36.047,7.012,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.981,0.875,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.369,0.031,0]},"t":45,"s":[75.273,-11.082,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.976,1.192,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.024,0.248,0]},"t":46,"s":[68.05,17.747,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.142,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.033,0.058,0]},"t":47,"s":[73.648,32.314,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.915,0.932,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,0.553,0]},"t":48,"s":[69.641,-15.771,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.919,-0.285,1]},"o":{"x":[0.167,0.167,0.167],"y":[4.378,-0.361,0]},"t":49,"s":[32.387,-24.312,100]},{"i":{"x":[0.833,0.833,0.833],"y":[216.656,0.822,1]},"o":{"x":[0.167,0.167,0.167],"y":[-3.24,0.089,0]},"t":50,"s":[31.664,-22.708,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,0.876,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.083,0.156,0]},"t":51,"s":[31.682,0.419,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.616,0.585,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.235,0.252,0]},"t":52,"s":[-15.245,26.807,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.106,0.104,0]},"t":53,"s":[-2.957,39.841,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,0.871,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.581,-0.016,0]},"t":54,"s":[41.396,91.664,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.754,0.858,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.15,0.236,0]},"t":55,"s":[48.828,48.327,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.071,0.928,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.075,0.203,0]},"t":56,"s":[46.178,24.721,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.871,0.408,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,-0.524,0]},"t":57,"s":[72.828,8.231,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.235,0.097,0]},"t":58,"s":[23.424,10.493,100]},{"t":59,"s":[-3.769,24.296,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":5,"op":60,"st":5,"bm":0},{"ddd":0,"ind":22,"ty":4,"nm":"Shape Layer 30","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":8,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":13,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":35,"s":[100]},{"t":51,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.613},"o":{"x":0.167,"y":0.167},"t":8,"s":[273.484,300.616,0],"to":[-0.073,-0.268,0],"ti":[0.394,1.23,0]},{"i":{"x":0.833,"y":0.766},"o":{"x":0.167,"y":0.106},"t":9,"s":[273.044,299.005,0],"to":[-0.394,-1.23,0],"ti":[0.97,2.68,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":10,"s":[271.118,293.234,0],"to":[-0.97,-2.68,0],"ti":[1.519,3.557,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.159},"t":11,"s":[267.222,282.925,0],"to":[-1.519,-3.557,0],"ti":[1.983,3.196,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.179},"t":12,"s":[262.002,271.894,0],"to":[-1.983,-3.196,0],"ti":[2.392,1.007,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.191},"t":13,"s":[255.326,263.748,0],"to":[-2.392,-1.007,0],"ti":[1.906,-1.439,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.174},"t":14,"s":[247.647,265.854,0],"to":[-1.906,1.439,0],"ti":[1.029,-2.094,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":15,"s":[243.888,272.381,0],"to":[-1.029,2.094,0],"ti":[0.709,-1.9,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":16,"s":[241.473,278.416,0],"to":[-0.709,1.9,0],"ti":[0.56,-1.691,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":17,"s":[239.636,283.781,0],"to":[-0.56,1.691,0],"ti":[0.476,-1.51,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":18,"s":[238.111,288.564,0],"to":[-0.476,1.51,0],"ti":[0.423,-1.353,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":19,"s":[236.781,292.841,0],"to":[-0.423,1.353,0],"ti":[0.389,-1.218,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":20,"s":[235.575,296.684,0],"to":[-0.389,1.218,0],"ti":[0.367,-1.099,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":21,"s":[234.45,300.149,0],"to":[-0.367,1.099,0],"ti":[0.353,-0.992,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":22,"s":[233.375,303.275,0],"to":[-0.353,0.992,0],"ti":[0.346,-0.895,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":23,"s":[232.33,306.098,0],"to":[-0.346,0.895,0],"ti":[0.345,-0.807,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[231.297,308.646,0],"to":[-0.345,0.807,0],"ti":[0.347,-0.725,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[230.261,310.941,0],"to":[-0.347,0.725,0],"ti":[0.353,-0.648,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[229.214,312.998,0],"to":[-0.353,0.648,0],"ti":[0.361,-0.575,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":27,"s":[228.146,314.83,0],"to":[-0.361,0.575,0],"ti":[0.371,-0.504,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":28,"s":[227.049,316.449,0],"to":[-0.371,0.504,0],"ti":[0.381,-0.434,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":29,"s":[225.922,317.856,0],"to":[-0.381,0.434,0],"ti":[0.391,-0.366,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":30,"s":[224.764,319.055,0],"to":[-0.391,0.366,0],"ti":[0.401,-0.298,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":31,"s":[223.576,320.049,0],"to":[-0.401,0.298,0],"ti":[0.41,-0.232,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":32,"s":[222.359,320.843,0],"to":[-0.41,0.232,0],"ti":[0.415,-0.168,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":33,"s":[221.118,321.443,0],"to":[-0.415,0.168,0],"ti":[0.412,-0.106,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":34,"s":[219.872,321.85,0],"to":[-0.412,0.106,0],"ti":[0.401,-0.05,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":35,"s":[218.647,322.076,0],"to":[-0.401,0.05,0],"ti":[0.385,-0.003,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":36,"s":[217.463,322.147,0],"to":[-0.385,0.003,0],"ti":[0.364,0.034,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":37,"s":[216.337,322.093,0],"to":[-0.364,-0.034,0],"ti":[0.341,0.06,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":38,"s":[215.276,321.946,0],"to":[-0.341,-0.06,0],"ti":[0.316,0.077,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":39,"s":[214.288,321.735,0],"to":[-0.316,-0.077,0],"ti":[0.29,0.086,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":40,"s":[213.378,321.486,0],"to":[-0.29,-0.086,0],"ti":[0.265,0.09,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":41,"s":[212.546,321.219,0],"to":[-0.265,-0.09,0],"ti":[0.239,0.088,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":42,"s":[211.79,320.949,0],"to":[-0.239,-0.088,0],"ti":[0.212,0.083,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":43,"s":[211.114,320.689,0],"to":[-0.212,-0.083,0],"ti":[0.186,0.075,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.178},"t":44,"s":[210.521,320.45,0],"to":[-0.186,-0.075,0],"ti":[0.176,0.067,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.167},"t":45,"s":[209.997,320.238,0],"to":[-0.176,-0.067,0],"ti":[0.178,0.059,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.167},"t":46,"s":[209.465,320.047,0],"to":[-0.178,-0.059,0],"ti":[0.179,0.05,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.168},"t":47,"s":[208.927,319.881,0],"to":[-0.179,-0.05,0],"ti":[0.179,0.042,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.166},"t":48,"s":[208.392,319.745,0],"to":[-0.179,-0.042,0],"ti":[0.184,0.036,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.164},"t":49,"s":[207.85,319.63,0],"to":[-0.184,-0.036,0],"ti":[0.189,0.031,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":50,"s":[207.287,319.528,0],"to":[-0.189,-0.031,0],"ti":[0.191,0.026,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.167},"t":51,"s":[206.714,319.442,0],"to":[-0.191,-0.026,0],"ti":[0.19,0.019,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":52,"s":[206.14,319.374,0],"to":[-0.19,-0.019,0],"ti":[0.186,0.013,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":53,"s":[205.574,319.326,0],"to":[-0.186,-0.013,0],"ti":[0.18,0.006,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.17},"t":54,"s":[205.024,319.298,0],"to":[-0.18,-0.006,0],"ti":[0.171,-0.002,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.171},"t":55,"s":[204.495,319.292,0],"to":[-0.171,0.002,0],"ti":[0.161,-0.009,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":56,"s":[203.995,319.308,0],"to":[-0.161,0.009,0],"ti":[0.149,-0.017,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":57,"s":[203.528,319.347,0],"to":[-0.149,0.017,0],"ti":[0.136,-0.024,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.174},"t":58,"s":[203.099,319.409,0],"to":[-0.136,0.024,0],"ti":[0.065,-0.014,0]},{"t":59,"s":[202.712,319.493,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.949,1.131,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":8,"s":[62.161,96.543,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.907,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.13,0.051,0]},"t":9,"s":[100.156,112.306,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.255,0.94,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.78,-0.001,0]},"t":10,"s":[85.337,71.714,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,0.737,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,-0.212,0]},"t":11,"s":[83.564,111.805,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.317,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.881,0.122,0]},"t":12,"s":[58.626,100.499,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.968,0.759,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.08,0.397,0]},"t":13,"s":[57.883,76.161,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.128,1.176,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.051,0.127,0]},"t":14,"s":[79.277,69.687,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,0.877,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,0.057,0]},"t":15,"s":[66.041,57.442,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.189,0.928,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.092,0.26,0]},"t":16,"s":[99.685,95.603,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.016,1.082,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,-0.51,0]},"t":17,"s":[83.661,113.563,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,1.121,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.013,0.041,0]},"t":18,"s":[136.015,111.04,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.746,0.893,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.873,0.049,0]},"t":19,"s":[73.781,116.06,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.529,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.124,0.375,0]},"t":20,"s":[79.202,103.731,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.027,1.092,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.055,0.101,0]},"t":21,"s":[90.283,100.207,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.852,0.985,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.044,0]},"t":22,"s":[83.612,83.82,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.603,0.729,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.191,-0.018,0]},"t":23,"s":[92.451,118.313,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.909,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.106,0.12,0]},"t":24,"s":[99.265,89.881,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.481,0.93,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.02,-0.001,0]},"t":25,"s":[124.872,26,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,0.628,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,-0.442,0]},"t":26,"s":[127.15,89.446,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.841,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[-2.332,0.107,0]},"t":27,"s":[57.069,79.381,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.572,0.791,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.175,-0.245,0]},"t":28,"s":[59.486,44.49,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.638,0.895,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.103,0.139,0]},"t":29,"s":[61.689,53.354,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.822,1.105,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.108,0.401,0]},"t":30,"s":[70.8,66.698,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.094,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.157,0.047,0]},"t":31,"s":[101.268,70.196,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.89,-1.504,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,-0.649,0]},"t":32,"s":[135.914,62.272,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.08,0.944,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.345,0.086,0]},"t":33,"s":[62.097,63.174,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.94,0.915,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.041,-0.167,0]},"t":34,"s":[38.578,89.373,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.898,-33.804,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.213,5.1,0]},"t":35,"s":[84.734,80.668,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.645,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.457,0.084,0]},"t":36,"s":[71.735,80.523,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,-0.034,0]},"t":37,"s":[68.832,20.271,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,1.005,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.053,0.563,0]},"t":38,"s":[94.213,63.191,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.499,0.724,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.065,0.004,0]},"t":39,"s":[78.745,70.653,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,1.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.119,0]},"t":40,"s":[87.422,62.786,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,0.984,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.658,0.011,0]},"t":41,"s":[26.733,44.597,100]},{"i":{"x":[0.833,0.833,0.833],"y":[8.954,0.747,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.609,-0.019,0]},"t":42,"s":[23.521,65.429,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.931,0.997,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.082,0.124,0]},"t":43,"s":[23.011,48.501,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.153,0.822,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.407,-0.003,0]},"t":44,"s":[72.162,14.125,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.994,1.075,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,0.156,0]},"t":45,"s":[63.801,47.128,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.742,0.92,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.006,0.039,0]},"t":46,"s":[87.498,84.73,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,1.295,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.123,-1.924,0]},"t":47,"s":[65.378,13.453,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.536,1.079,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.497,0.065,0]},"t":48,"s":[19.027,16.413,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.076,0.957,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.102,0.041,0]},"t":49,"s":[25.686,2.971,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,0.87,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,-0.09,0]},"t":50,"s":[56.116,29.236,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.444,1.634,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.906,0.231,0]},"t":51,"s":[-1.889,16.583,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.986,0.961,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.098,0.074,0]},"t":52,"s":[2.996,9.45,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,0.802,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.017,-0.074,0]},"t":53,"s":[30.717,70.822,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.484,0.967,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.111,0.144,0]},"t":54,"s":[7.584,38.216,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.944,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,-0.054,0]},"t":55,"s":[17.496,-6.813,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.773,1.079,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.143,-0.17,0]},"t":56,"s":[68.936,20.442,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.98,0.725,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.132,0.041,0]},"t":57,"s":[50.024,11.466,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.027,0.12,0]},"t":58,"s":[17.481,29.004,100]},{"t":59,"s":[42.14,69.391,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.8941176470588236,0.11372549019607843,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":8,"op":60,"st":8,"bm":0},{"ddd":0,"ind":23,"ty":4,"nm":"Shape Layer 29","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":17,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":39,"s":[100]},{"t":55,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.577},"o":{"x":0.167,"y":0.167},"t":12,"s":[270.833,305.959,0],"to":[-0.088,-0.333,0],"ti":[0.479,1.681,0]},{"i":{"x":0.833,"y":0.762},"o":{"x":0.167,"y":0.104},"t":13,"s":[270.305,303.959,0],"to":[-0.479,-1.681,0],"ti":[0.843,3.89,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.128},"t":14,"s":[267.957,295.871,0],"to":[-0.843,-3.89,0],"ti":[-1.367,3.831,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.177},"t":15,"s":[265.25,280.619,0],"to":[1.367,-3.831,0],"ti":[-4.031,0.131,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.16},"t":16,"s":[276.159,272.888,0],"to":[4.031,-0.131,0],"ti":[-3.947,-2.293,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.185},"t":17,"s":[289.438,279.833,0],"to":[3.947,2.293,0],"ti":[-3.163,-2.107,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":18,"s":[299.839,286.644,0],"to":[3.163,2.107,0],"ti":[-2.643,-1.789,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":19,"s":[308.416,292.475,0],"to":[2.643,1.789,0],"ti":[-2.265,-1.502,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":20,"s":[315.699,297.38,0],"to":[2.265,1.502,0],"ti":[-1.975,-1.258,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":21,"s":[322.005,301.489,0],"to":[1.975,1.258,0],"ti":[-1.746,-1.051,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":22,"s":[327.55,304.928,0],"to":[1.746,1.051,0],"ti":[-1.558,-0.872,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":23,"s":[332.48,307.794,0],"to":[1.558,0.872,0],"ti":[-1.4,-0.715,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":24,"s":[336.899,310.159,0],"to":[1.4,0.715,0],"ti":[-1.265,-0.574,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":25,"s":[340.882,312.081,0],"to":[1.265,0.574,0],"ti":[-1.148,-0.448,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":26,"s":[344.489,313.605,0],"to":[1.148,0.448,0],"ti":[-1.045,-0.333,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":27,"s":[347.771,314.769,0],"to":[1.045,0.333,0],"ti":[-0.954,-0.228,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":28,"s":[350.762,315.603,0],"to":[0.954,0.228,0],"ti":[-0.867,-0.129,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":29,"s":[353.494,316.134,0],"to":[0.867,0.129,0],"ti":[-0.779,-0.034,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":30,"s":[355.966,316.376,0],"to":[0.779,0.034,0],"ti":[-0.692,0.054,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":31,"s":[358.169,316.339,0],"to":[0.692,-0.054,0],"ti":[-0.607,0.132,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":32,"s":[360.118,316.051,0],"to":[0.607,-0.132,0],"ti":[-0.525,0.197,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":33,"s":[361.814,315.546,0],"to":[0.525,-0.197,0],"ti":[-0.446,0.247,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":34,"s":[363.265,314.868,0],"to":[0.446,-0.247,0],"ti":[-0.374,0.281,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":35,"s":[364.49,314.065,0],"to":[0.374,-0.281,0],"ti":[-0.309,0.299,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":36,"s":[365.508,313.183,0],"to":[0.309,-0.299,0],"ti":[-0.254,0.304,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":37,"s":[366.345,312.269,0],"to":[0.254,-0.304,0],"ti":[-0.208,0.299,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":38,"s":[367.029,311.356,0],"to":[0.208,-0.299,0],"ti":[-0.171,0.285,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":39,"s":[367.591,310.475,0],"to":[0.171,-0.285,0],"ti":[-0.143,0.266,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":40,"s":[368.057,309.644,0],"to":[0.143,-0.266,0],"ti":[-0.123,0.244,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":41,"s":[368.451,308.877,0],"to":[0.123,-0.244,0],"ti":[-0.108,0.219,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":42,"s":[368.793,308.182,0],"to":[0.108,-0.219,0],"ti":[-0.097,0.192,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":43,"s":[369.097,307.564,0],"to":[0.097,-0.192,0],"ti":[-0.091,0.166,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":44,"s":[369.377,307.027,0],"to":[0.091,-0.166,0],"ti":[-0.087,0.139,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":45,"s":[369.642,306.571,0],"to":[0.087,-0.139,0],"ti":[-0.085,0.112,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":46,"s":[369.899,306.194,0],"to":[0.085,-0.112,0],"ti":[-0.083,0.087,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":47,"s":[370.15,305.896,0],"to":[0.083,-0.087,0],"ti":[-0.082,0.062,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":48,"s":[370.398,305.674,0],"to":[0.082,-0.062,0],"ti":[-0.081,0.039,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.178},"t":49,"s":[370.644,305.522,0],"to":[0.081,-0.039,0],"ti":[-0.079,0.017,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.175},"t":50,"s":[370.885,305.438,0],"to":[0.079,-0.017,0],"ti":[-0.075,-0.004,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.171},"t":51,"s":[371.117,305.419,0],"to":[0.075,0.004,0],"ti":[-0.071,-0.019,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":52,"s":[371.334,305.462,0],"to":[0.071,0.019,0],"ti":[-0.071,-0.023,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.167},"t":53,"s":[371.546,305.533,0],"to":[0.071,0.023,0],"ti":[-0.069,-0.021,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.171},"t":54,"s":[371.758,305.598,0],"to":[0.069,0.021,0],"ti":[-0.064,-0.018,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.176},"t":55,"s":[371.962,305.656,0],"to":[0.064,0.018,0],"ti":[-0.056,-0.014,0]},{"i":{"x":0.833,"y":0.859},"o":{"x":0.167,"y":0.185},"t":56,"s":[372.145,305.704,0],"to":[0.056,0.014,0],"ti":[-0.042,-0.01,0]},{"i":{"x":0.833,"y":0.882},"o":{"x":0.167,"y":0.204},"t":57,"s":[372.295,305.741,0],"to":[0.042,0.01,0],"ti":[-0.025,-0.005,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.285},"t":58,"s":[372.399,305.764,0],"to":[0.025,0.005,0],"ti":[-0.007,-0.001,0]},{"t":59,"s":[372.443,305.77,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.332,1.186,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":12,"s":[64.57,85.432,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.875,0.947,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.095,0.058,0]},"t":13,"s":[69.942,108.953,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.206,0.842,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.249,-0.144,0]},"t":14,"s":[107.607,32.831,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.969,0.836,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.059,0.176,0]},"t":15,"s":[126.531,60.735,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.048,0.17,0]},"t":16,"s":[60.769,85.755,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.908,1.153,1]},"o":{"x":[0.167,0.167,0.167],"y":[0,0.222,0]},"t":17,"s":[102.39,109.836,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.333,0.945,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.877,0.054,0]},"t":18,"s":[60.887,124.258,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.949,1.054,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,-0.162,0]},"t":19,"s":[56.529,83.394,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.322,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.133,0.033,0]},"t":20,"s":[78.284,97.261,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.976,0.786,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.066,-0.05,0]},"t":21,"s":[69.892,74.45,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.923,1.181,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.033,0.137,0]},"t":22,"s":[110.724,88.677,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.066,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.984,0.057,0]},"t":23,"s":[81.454,110.927,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.003,0.937,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,-0.193,0]},"t":24,"s":[83.739,40.242,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.86,0.536,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.002,-0.266,0]},"t":25,"s":[107.061,61.553,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.865,1.004,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.206,0.102,0]},"t":26,"s":[83.024,56.466,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.851,0.868,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.217,0.004,0]},"t":27,"s":[66.63,33.237,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.158,0.859,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.189,0.226,0]},"t":28,"s":[56.437,57.54,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.865,1.267,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.203,0]},"t":29,"s":[48.398,71.747,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.754,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.218,0.064,0]},"t":30,"s":[71.71,81.652,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.096,0.683,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.126,-0.203,0]},"t":31,"s":[86.087,39.983,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.99,0.804,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,0.113,0]},"t":32,"s":[114.075,52.122,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.96,0.963,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.012,0.145,0]},"t":33,"s":[53.751,86.117,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,0.79,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.077,-0.066,0]},"t":34,"s":[106.75,131.89,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.204,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.532,0.138,0]},"t":35,"s":[79.216,106.312,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.915,0.889,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.093,-0.024,0]},"t":36,"s":[82.944,67.378,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.454,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[3.696,0.337,0]},"t":37,"s":[114.826,97.73,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.872,0.088,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.098,-0.207,0]},"t":38,"s":[115.562,107.689,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.944,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.076,0.092,0]},"t":39,"s":[119.647,104.826,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.157,0.579,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.17,0.758,0]},"t":40,"s":[72.833,76.34,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,0.104,0]},"t":41,"s":[88.225,72.82,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.853,0.888,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.07,-0.05,0]},"t":42,"s":[43.922,58.537,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,-0.216,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.193,0.325,0]},"t":43,"s":[67.927,67.475,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.471,1.047,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.3,0.089,0]},"t":44,"s":[86.125,70.552,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.904,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.03,0]},"t":45,"s":[93.113,112.383,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.526,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.633,-0.31,0]},"t":46,"s":[46.616,46.845,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.045,1.106,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.101,0.012,0]},"t":47,"s":[39.563,60.724,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.969,1.027,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.029,0.047,0]},"t":48,"s":[6.525,44.501,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,0.915,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.05,0.021,0]},"t":49,"s":[57.474,81.367,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.854,1.348,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.48,3.659,0]},"t":50,"s":[25.518,32.394,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.966,0.815,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.195,0.067,0]},"t":51,"s":[30.245,31.253,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.921,1.317,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.077,0.152,0]},"t":52,"s":[33.783,37.163,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.996,0.978,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.48,0.066,0]},"t":53,"s":[-10.763,44.348,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.91,0.989,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.087,-0.029,0]},"t":54,"s":[-8.389,9.855,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.194,0.978,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.177,-0.012,0]},"t":55,"s":[46.091,35.382,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.01,0.743,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,-0.03,0]},"t":56,"s":[50.24,13.115,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.876,0.981,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.009,0.123,0]},"t":57,"s":[36.428,29.519,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.252,-0.025,0]},"t":58,"s":[51.9,63.715,100]},{"t":59,"s":[59.535,37.52,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":12,"op":60,"st":12,"bm":0},{"ddd":0,"ind":24,"ty":4,"nm":"Shape Layer 28","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":3,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":8,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":30,"s":[100]},{"t":46,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.613},"o":{"x":0.167,"y":0.167},"t":3,"s":[270.565,305.387,0],"to":[-0.175,-0.467,0],"ti":[0.928,2.118,0]},{"i":{"x":0.833,"y":0.766},"o":{"x":0.167,"y":0.106},"t":4,"s":[269.518,302.585,0],"to":[-0.928,-2.118,0],"ti":[2.866,4.132,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.13},"t":5,"s":[264.999,292.679,0],"to":[-2.866,-4.132,0],"ti":[5.573,3.332,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.159},"t":6,"s":[252.323,277.795,0],"to":[-5.573,-3.332,0],"ti":[6.454,-0.145,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.178},"t":7,"s":[231.564,272.688,0],"to":[-6.454,0.145,0],"ti":[5.175,-2.485,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":8,"s":[213.599,278.665,0],"to":[-5.175,2.485,0],"ti":[3.799,-3.017,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":9,"s":[200.516,287.597,0],"to":[-3.799,3.017,0],"ti":[2.848,-2.986,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":10,"s":[190.807,296.767,0],"to":[-2.848,2.986,0],"ti":[2.177,-2.823,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":11,"s":[183.426,305.511,0],"to":[-2.177,2.823,0],"ti":[1.672,-2.64,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":12,"s":[177.747,313.704,0],"to":[-1.672,2.64,0],"ti":[1.269,-2.465,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":13,"s":[173.394,321.352,0],"to":[-1.269,2.465,0],"ti":[0.931,-2.298,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":14,"s":[170.134,328.494,0],"to":[-0.931,2.298,0],"ti":[0.638,-2.135,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":15,"s":[167.806,335.14,0],"to":[-0.638,2.135,0],"ti":[0.374,-1.972,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":16,"s":[166.306,341.307,0],"to":[-0.374,1.972,0],"ti":[0.132,-1.802,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[165.565,346.975,0],"to":[-0.132,1.802,0],"ti":[-0.087,-1.628,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[165.516,352.122,0],"to":[0.087,1.628,0],"ti":[-0.279,-1.446,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":19,"s":[166.088,356.74,0],"to":[0.279,1.446,0],"ti":[-0.435,-1.255,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":20,"s":[167.189,360.798,0],"to":[0.435,1.255,0],"ti":[-0.549,-1.065,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[168.697,364.273,0],"to":[0.549,1.065,0],"ti":[-0.622,-0.888,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":22,"s":[170.482,367.189,0],"to":[0.622,0.888,0],"ti":[-0.661,-0.731,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[172.43,369.6,0],"to":[0.661,0.731,0],"ti":[-0.672,-0.596,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":24,"s":[174.447,371.574,0],"to":[0.672,0.596,0],"ti":[-0.665,-0.483,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[176.464,373.178,0],"to":[0.665,0.483,0],"ti":[-0.646,-0.39,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[178.436,374.475,0],"to":[0.646,0.39,0],"ti":[-0.62,-0.314,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":27,"s":[180.337,375.52,0],"to":[0.62,0.314,0],"ti":[-0.589,-0.251,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":28,"s":[182.153,376.359,0],"to":[0.589,0.251,0],"ti":[-0.557,-0.198,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":29,"s":[183.874,377.025,0],"to":[0.557,0.198,0],"ti":[-0.524,-0.154,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":30,"s":[185.496,377.549,0],"to":[0.524,0.154,0],"ti":[-0.49,-0.117,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":31,"s":[187.016,377.951,0],"to":[0.49,0.117,0],"ti":[-0.456,-0.086,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":32,"s":[188.435,378.252,0],"to":[0.456,0.086,0],"ti":[-0.422,-0.059,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":33,"s":[189.752,378.465,0],"to":[0.422,0.059,0],"ti":[-0.387,-0.035,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":34,"s":[190.966,378.603,0],"to":[0.387,0.035,0],"ti":[-0.352,-0.015,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":35,"s":[192.076,378.677,0],"to":[0.352,0.015,0],"ti":[-0.316,0.002,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":36,"s":[193.079,378.696,0],"to":[0.316,-0.002,0],"ti":[-0.278,0.017,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":37,"s":[193.971,378.666,0],"to":[0.278,-0.017,0],"ti":[-0.239,0.029,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.181},"t":38,"s":[194.747,378.596,0],"to":[0.239,-0.029,0],"ti":[-0.199,0.039,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.184},"t":39,"s":[195.405,378.492,0],"to":[0.199,-0.039,0],"ti":[-0.156,0.047,0]},{"i":{"x":0.833,"y":0.856},"o":{"x":0.167,"y":0.19},"t":40,"s":[195.939,378.361,0],"to":[0.156,-0.047,0],"ti":[-0.11,0.054,0]},{"i":{"x":0.833,"y":0.86},"o":{"x":0.167,"y":0.199},"t":41,"s":[196.341,378.209,0],"to":[0.11,-0.054,0],"ti":[-0.061,0.058,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.206},"t":42,"s":[196.601,378.04,0],"to":[0.061,-0.058,0],"ti":[-0.017,0.06,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.18},"t":43,"s":[196.709,377.862,0],"to":[0.017,-0.06,0],"ti":[0.003,0.066,0]},{"i":{"x":0.833,"y":0.816},"o":{"x":0.167,"y":0.153},"t":44,"s":[196.702,377.682,0],"to":[-0.003,-0.066,0],"ti":[-0.001,0.08,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.153},"t":45,"s":[196.693,377.464,0],"to":[0.001,-0.08,0],"ti":[-0.01,0.093,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.156},"t":46,"s":[196.709,377.203,0],"to":[0.01,-0.093,0],"ti":[-0.019,0.104,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.158},"t":47,"s":[196.752,376.906,0],"to":[0.019,-0.104,0],"ti":[-0.028,0.112,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":48,"s":[196.823,376.58,0],"to":[0.028,-0.112,0],"ti":[-0.038,0.118,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.162},"t":49,"s":[196.923,376.233,0],"to":[0.038,-0.118,0],"ti":[-0.049,0.122,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.163},"t":50,"s":[197.053,375.871,0],"to":[0.049,-0.122,0],"ti":[-0.059,0.124,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.164},"t":51,"s":[197.215,375.501,0],"to":[0.059,-0.124,0],"ti":[-0.069,0.124,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.164},"t":52,"s":[197.407,375.127,0],"to":[0.069,-0.124,0],"ti":[-0.079,0.122,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":53,"s":[197.63,374.756,0],"to":[0.079,-0.122,0],"ti":[-0.089,0.119,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":54,"s":[197.883,374.392,0],"to":[0.089,-0.119,0],"ti":[-0.099,0.115,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":55,"s":[198.166,374.04,0],"to":[0.099,-0.115,0],"ti":[-0.108,0.109,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":56,"s":[198.477,373.703,0],"to":[0.108,-0.109,0],"ti":[-0.116,0.102,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":57,"s":[198.814,373.385,0],"to":[0.116,-0.102,0],"ti":[-0.124,0.095,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":58,"s":[199.175,373.088,0],"to":[0.124,-0.095,0],"ti":[-0.064,0.045,0]},{"t":59,"s":[199.559,372.816,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.903,1.422,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":3,"s":[94.702,122.655,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.272,0.85,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.6,0.07,0]},"t":4,"s":[127.289,129.977,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.732,1.117,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,0.187,0]},"t":5,"s":[132.545,85.581,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.121,0.049,0]},"t":6,"s":[110.128,50.029,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.861,2.842,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.066,6.415,0]},"t":7,"s":[60.526,135.359,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.066,0.745,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.209,0.08,0]},"t":8,"s":[88.167,136.482,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.966,0.932,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.037,0.124,0]},"t":9,"s":[106.49,110.542,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.936,0.426,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.056,-0.36,0]},"t":10,"s":[73.621,57.227,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.484,1.02,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.27,0.097,0]},"t":11,"s":[93.234,67.259,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.866,0.935,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.016,0]},"t":12,"s":[88.608,126.279,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.185,0.861,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.221,-0.293,0]},"t":13,"s":[120.076,52.9,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.916,0.706,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.057,0.207,0]},"t":14,"s":[139.099,69.169,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-1.441,0.819,1]},"o":{"x":[0.167,0.167,0.167],"y":[6.732,0.116,0]},"t":15,"s":[77.908,80.095,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.941,1.013,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.086,0.155,0]},"t":16,"s":[77.141,107.685,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.598,0.859,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.198,0.011,0]},"t":17,"s":[55.437,139.887,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.889,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.105,0.204,0]},"t":18,"s":[61.857,102.633,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.557,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.337,-0.045,0]},"t":19,"s":[86.378,76.933,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.048,1.193,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.103,-0.014,0]},"t":20,"s":[94.434,93.656,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.894,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.03,0.058,0]},"t":21,"s":[129.176,79.379,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.003,0.846,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.385,-0.049,0]},"t":22,"s":[74.62,126.637,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.783,0.843,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.003,0.182,0]},"t":23,"s":[59.528,96.902,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.987,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,0.178,0]},"t":24,"s":[75.163,71.778,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.98,0.157,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.016,-0.467,0]},"t":25,"s":[100.236,49.638,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.196,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.026,0.092,0]},"t":26,"s":[79.183,52.988,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,1.011,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,0.384,0]},"t":27,"s":[95.26,83.538,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.998,0.441,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.251,0.01,0]},"t":28,"s":[41.321,92.009,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.333,0.953,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.002,0.098,0]},"t":29,"s":[54.762,82.447,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.996,0.891,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,-0.107,0]},"t":30,"s":[41.586,27.852,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.941,0.229,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.004,0.35,0]},"t":31,"s":[107.415,51.724,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.942,0.994,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.201,0.093,0]},"t":32,"s":[44.413,59.172,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.637,0.897,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.19,-0.007,0]},"t":33,"s":[62.871,120.641,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,1.033,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.108,0.443,0]},"t":34,"s":[57.246,63.863,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.767,0.841,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.63,0.024,0]},"t":35,"s":[38.372,50.708,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.822,1.066,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,0.175,0]},"t":36,"s":[37.754,69.059,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.994,0.844,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.156,0.037,0]},"t":37,"s":[58.88,85.746,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,1.009,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.007,0.179,0]},"t":38,"s":[83,55.846,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.235,1.083,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.184,0.008,0]},"t":39,"s":[60.74,29.847,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.881,0.964,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,0.042,0]},"t":40,"s":[67.668,58.5,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.257,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.279,-0.062,0]},"t":41,"s":[41.227,1.342,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,0.99,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.063,0.452,0]},"t":42,"s":[29.972,34.049,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.558,0.889,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.356,-0.011,0]},"t":43,"s":[75.997,41.447,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.923,0.757,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,0.337,0]},"t":44,"s":[77.684,34.902,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.665,1.132,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.086,0.127,0]},"t":45,"s":[24.202,32.753,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.007,0.637,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,0.051,0]},"t":46,"s":[28.012,28.648,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.981,0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,0.108,0]},"t":47,"s":[-6.22,39.267,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1,0.981,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.024,-0.038,0]},"t":48,"s":[30.742,74.855,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,1.113,1]},"o":{"x":[0.167,0.167,0.167],"y":[0,-0.024,0]},"t":49,"s":[2.023,50.337,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.727,0.883,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.326,0.048,0]},"t":50,"s":[30.77,69.402,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.902,1.045,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.12,0.292,0]},"t":51,"s":[24.92,24.502,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.108,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.55,0.029,0]},"t":52,"s":[11.611,6.547,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.904,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.078,0.962,0]},"t":53,"s":[9.235,34.159,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.825,-0.409,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.138,0.611,0]},"t":54,"s":[43.201,36.778,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.973,0.815,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.159,0.089,0]},"t":55,"s":[30.429,37.192,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,1.034,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.039,0.152,0]},"t":56,"s":[16.318,43.768,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.408,0.765,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.6,0.024,0]},"t":57,"s":[25.931,51.757,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,0.129,0]},"t":58,"s":[27.481,40.555,100]},{"t":59,"s":[52.128,20.168,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":3,"op":60,"st":3,"bm":0},{"ddd":0,"ind":25,"ty":4,"nm":"Shape Layer 27","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":1,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":6,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":28,"s":[100]},{"t":44,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.551},"o":{"x":0.167,"y":0.167},"t":1,"s":[264.583,303.394,0],"to":[0.152,-0.309,0],"ti":[-0.732,1.703,0]},{"i":{"x":0.833,"y":0.762},"o":{"x":0.167,"y":0.102},"t":2,"s":[265.494,301.541,0],"to":[0.732,-1.703,0],"ti":[-1.362,4.094,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.128},"t":3,"s":[268.975,293.174,0],"to":[1.362,-4.094,0],"ti":[-1.445,5.76,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":4,"s":[273.666,276.977,0],"to":[1.445,-5.76,0],"ti":[-1.23,5.729,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":5,"s":[277.646,258.613,0],"to":[1.23,-5.729,0],"ti":[-1.123,4.869,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":6,"s":[281.044,242.601,0],"to":[1.123,-4.869,0],"ti":[-1.146,4.013,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.183},"t":7,"s":[284.385,229.397,0],"to":[1.146,-4.013,0],"ti":[-1.227,3.315,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":8,"s":[287.919,218.524,0],"to":[1.227,-3.315,0],"ti":[-1.326,2.748,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":9,"s":[291.745,209.508,0],"to":[1.326,-2.748,0],"ti":[-1.42,2.27,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":10,"s":[295.873,202.037,0],"to":[1.42,-2.27,0],"ti":[-1.493,1.856,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":11,"s":[300.264,195.89,0],"to":[1.493,-1.856,0],"ti":[-1.535,1.495,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":12,"s":[304.832,190.902,0],"to":[1.535,-1.495,0],"ti":[-1.541,1.186,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":13,"s":[309.473,186.917,0],"to":[1.541,-1.186,0],"ti":[-1.513,0.928,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":14,"s":[314.076,183.786,0],"to":[1.513,-0.928,0],"ti":[-1.459,0.719,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":15,"s":[318.551,181.352,0],"to":[1.459,-0.719,0],"ti":[-1.388,0.557,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":16,"s":[322.832,179.47,0],"to":[1.388,-0.557,0],"ti":[-1.306,0.432,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":17,"s":[326.879,178.013,0],"to":[1.306,-0.432,0],"ti":[-1.218,0.338,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":18,"s":[330.667,176.878,0],"to":[1.218,-0.338,0],"ti":[-1.129,0.269,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":19,"s":[334.187,175.983,0],"to":[1.129,-0.269,0],"ti":[-1.043,0.219,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":20,"s":[337.444,175.264,0],"to":[1.043,-0.219,0],"ti":[-0.959,0.182,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":21,"s":[340.444,174.672,0],"to":[0.959,-0.182,0],"ti":[-0.877,0.155,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":22,"s":[343.196,174.174,0],"to":[0.877,-0.155,0],"ti":[-0.798,0.134,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":23,"s":[345.706,173.744,0],"to":[0.798,-0.134,0],"ti":[-0.72,0.119,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":24,"s":[347.981,173.367,0],"to":[0.72,-0.119,0],"ti":[-0.644,0.106,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":25,"s":[350.026,173.032,0],"to":[0.644,-0.106,0],"ti":[-0.57,0.094,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.178},"t":26,"s":[351.847,172.732,0],"to":[0.57,-0.094,0],"ti":[-0.497,0.083,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.18},"t":27,"s":[353.446,172.467,0],"to":[0.497,-0.083,0],"ti":[-0.429,0.075,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.18},"t":28,"s":[354.827,172.237,0],"to":[0.429,-0.075,0],"ti":[-0.38,0.083,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":29,"s":[356.018,172.02,0],"to":[0.38,-0.083,0],"ti":[-0.347,0.103,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":30,"s":[357.106,171.74,0],"to":[0.347,-0.103,0],"ti":[-0.318,0.12,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":31,"s":[358.103,171.404,0],"to":[0.318,-0.12,0],"ti":[-0.291,0.134,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":32,"s":[359.015,171.022,0],"to":[0.291,-0.134,0],"ti":[-0.265,0.144,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":33,"s":[359.846,170.603,0],"to":[0.265,-0.144,0],"ti":[-0.24,0.151,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":34,"s":[360.602,170.159,0],"to":[0.24,-0.151,0],"ti":[-0.216,0.155,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":35,"s":[361.284,169.698,0],"to":[0.216,-0.155,0],"ti":[-0.193,0.156,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":36,"s":[361.896,169.23,0],"to":[0.193,-0.156,0],"ti":[-0.172,0.154,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":37,"s":[362.443,168.763,0],"to":[0.172,-0.154,0],"ti":[-0.151,0.15,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":38,"s":[362.927,168.306,0],"to":[0.151,-0.15,0],"ti":[-0.13,0.143,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":39,"s":[363.348,167.866,0],"to":[0.13,-0.143,0],"ti":[-0.109,0.134,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.177},"t":40,"s":[363.707,167.448,0],"to":[0.109,-0.134,0],"ti":[-0.095,0.123,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.175},"t":41,"s":[364.004,167.059,0],"to":[0.095,-0.123,0],"ti":[-0.093,0.109,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":42,"s":[364.278,166.709,0],"to":[0.093,-0.109,0],"ti":[-0.096,0.093,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":43,"s":[364.562,166.405,0],"to":[0.096,-0.093,0],"ti":[-0.099,0.076,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":44,"s":[364.856,166.152,0],"to":[0.099,-0.076,0],"ti":[-0.1,0.058,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":45,"s":[365.154,165.951,0],"to":[0.1,-0.058,0],"ti":[-0.1,0.041,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":46,"s":[365.455,165.803,0],"to":[0.1,-0.041,0],"ti":[-0.099,0.023,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":47,"s":[365.754,165.707,0],"to":[0.099,-0.023,0],"ti":[-0.096,0.006,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.17},"t":48,"s":[366.048,165.664,0],"to":[0.096,-0.006,0],"ti":[-0.093,-0.01,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.169},"t":49,"s":[366.333,165.672,0],"to":[0.093,0.01,0],"ti":[-0.088,-0.026,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.168},"t":50,"s":[366.605,165.727,0],"to":[0.088,0.026,0],"ti":[-0.082,-0.04,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.167},"t":51,"s":[366.86,165.827,0],"to":[0.082,0.04,0],"ti":[-0.074,-0.052,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.167},"t":52,"s":[367.094,165.966,0],"to":[0.074,0.052,0],"ti":[-0.065,-0.063,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.167},"t":53,"s":[367.302,166.14,0],"to":[0.065,0.063,0],"ti":[-0.054,-0.071,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":54,"s":[367.481,166.343,0],"to":[0.054,0.071,0],"ti":[-0.042,-0.077,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":55,"s":[367.625,166.567,0],"to":[0.042,0.077,0],"ti":[-0.028,-0.08,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.17},"t":56,"s":[367.73,166.804,0],"to":[0.028,0.08,0],"ti":[-0.012,-0.079,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.171},"t":57,"s":[367.791,167.045,0],"to":[0.012,0.079,0],"ti":[0.003,-0.077,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.169},"t":58,"s":[367.803,167.28,0],"to":[-0.003,0.077,0],"ti":[0.005,-0.038,0]},{"t":59,"s":[367.776,167.508,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[-0.282,0.584,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":1,"s":[72.171,102.828,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,1.039,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,0.104,0]},"t":2,"s":[75.523,112.995,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.019,0.877,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.304,0.027,0]},"t":3,"s":[123.744,153.636,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.693,1.033,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,0.256,0]},"t":4,"s":[141.925,93.916,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.936,0.883,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.114,0.023,0]},"t":5,"s":[119.587,65.139,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.769,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.276,0.289,0]},"t":6,"s":[59.668,105.203,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.003,2.032,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.13,0.39,0]},"t":7,"s":[73.555,121.412,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.032,0.905,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.003,0.077,0]},"t":8,"s":[98.171,125.813,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.887,1.503,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.023,0.67,0]},"t":9,"s":[72.643,66.925,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.904,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.315,0.071,0]},"t":10,"s":[108.084,58.564,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.479,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.612,-0.05,0]},"t":11,"s":[120.859,117.368,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.764,0.88,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,-0.052,0]},"t":12,"s":[122.874,80.731,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.916,1.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.129,0.273,0]},"t":13,"s":[109.267,103.32,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.633,1.171,1]},"o":{"x":[0.167,0.167,0.167],"y":[12.53,0.011,0]},"t":14,"s":[84.332,113.266,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.72,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,0.056,0]},"t":15,"s":[84.165,101.864,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.993,0.575,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.119,-0.115,0]},"t":16,"s":[89.607,136.724,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.905,0.946,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.007,0.104,0]},"t":17,"s":[102.429,122.041,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.566,0.936,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.667,-0.15,0]},"t":18,"s":[90.653,61.798,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.025,1.312,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,-0.274,0]},"t":19,"s":[88.97,83.343,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.879,0.882,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.019,0.066,0]},"t":20,"s":[122.282,78.321,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.099,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.27,0.282,0]},"t":21,"s":[78.796,102.131,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.927,5.118,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,0.945,0]},"t":22,"s":[59.361,112.137,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.136,0.975,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.587,0.082,0]},"t":23,"s":[102.002,113.104,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.921,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.052,-0.036,0]},"t":24,"s":[96.704,64.316,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.841,2.162,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.183,-1.67,0]},"t":25,"s":[110.65,98.408,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.252,1.02,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.176,0.078,0]},"t":26,"s":[106.279,96.788,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.859,0.808,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,0.016,0]},"t":27,"s":[102.331,121.007,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.964,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.205,0.147,0]},"t":28,"s":[70.842,91.027,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.045,0.592,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.063,-0.479,0]},"t":29,"s":[49.193,51.754,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.816,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.029,0.105,0]},"t":30,"s":[61.557,57.576,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.115,0.844,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.152,-0.022,0]},"t":31,"s":[42.519,80.235,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.93,0.834,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.048,0.179,0]},"t":32,"s":[19.507,62.344,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.463,1.029,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.45,0.167,0]},"t":33,"s":[74.401,46.72,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.852,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,0.021,0]},"t":34,"s":[65.831,31.185,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.744,1.068,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.12,0.191,0]},"t":35,"s":[19.228,52.105,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.91,0.811,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.124,0.037,0]},"t":36,"s":[38.347,68.326,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.615,1.069,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.166,0.149,0]},"t":37,"s":[77.863,38.956,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,0.995,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.038,0]},"t":38,"s":[80.904,1.852,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,0.924,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,-0.006,0]},"t":39,"s":[55.41,69.603,100]},{"i":{"x":[0.833,0.833,0.833],"y":[4.375,0.208,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.748,-0.921,0]},"t":40,"s":[31.879,6.237,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,1.05,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,0.093,0]},"t":41,"s":[31.143,11.494,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.257,0.972,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.59,0.031,0]},"t":42,"s":[61.679,56.19,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.875,0.989,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.063,-0.042,0]},"t":43,"s":[66.703,-15.414,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.777,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.252,-0.012,0]},"t":44,"s":[46.165,32.254,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.071,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.133,-0.048,0]},"t":45,"s":[36.01,-9.217,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.042,0.895,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.166,0]},"t":46,"s":[18.977,17.016,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.64,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.028,0.41,0]},"t":47,"s":[50.584,43.433,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,1.026,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.054,0.108,0]},"t":48,"s":[3.2,50.175,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.826,0.687,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.121,0.02,0]},"t":49,"s":[31.978,72.587,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.958,0.985,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.16,0.114,0]},"t":50,"s":[20.225,43.283,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.531,0.899,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.084,-0.019,0]},"t":51,"s":[7.424,-37.423,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,0.885,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.101,0.475,0]},"t":52,"s":[13.811,28.463,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.207,0.722,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.872,0.302,0]},"t":53,"s":[43.338,42.467,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.09,0.119,0]},"t":54,"s":[40.761,47.811,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.074,0.996,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.092,-0.177,0]},"t":55,"s":[6.006,60.269,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,2.021,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.039,-0.004,0]},"t":56,"s":[22.502,56.273,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.823,0.991,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.12,0.077,0]},"t":57,"s":[-8.669,60.085,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.158,-0.01,0]},"t":58,"s":[4.137,9.589,100]},{"t":59,"s":[18.516,54.805,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":1,"op":60,"st":1,"bm":0},{"ddd":0,"ind":26,"ty":4,"nm":"Shape Layer 26","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":4,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":9,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":31,"s":[100]},{"t":47,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.589},"o":{"x":0.167,"y":0.167},"t":4,"s":[269.491,306.01,0],"to":[0.367,0.261,0],"ti":[-1.762,-1.355,0]},{"i":{"x":0.833,"y":0.764},"o":{"x":0.167,"y":0.105},"t":5,"s":[271.692,307.579,0],"to":[1.762,1.355,0],"ti":[-4.055,-2.961,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":6,"s":[280.065,314.139,0],"to":[4.055,2.961,0],"ti":[-5.763,-3.712,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":7,"s":[296.024,325.342,0],"to":[5.763,3.712,0],"ti":[-5.909,-3.26,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":8,"s":[314.641,336.411,0],"to":[5.909,3.26,0],"ti":[-5.199,-2.461,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":9,"s":[331.476,344.903,0],"to":[5.199,2.461,0],"ti":[-4.435,-1.834,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":10,"s":[345.833,351.179,0],"to":[4.435,1.834,0],"ti":[-3.802,-1.398,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":11,"s":[358.083,355.908,0],"to":[3.802,1.398,0],"ti":[-3.292,-1.093,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":12,"s":[368.642,359.567,0],"to":[3.292,1.093,0],"ti":[-2.886,-0.867,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":13,"s":[377.836,362.467,0],"to":[2.886,0.867,0],"ti":[-2.566,-0.687,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":14,"s":[385.959,364.771,0],"to":[2.566,0.687,0],"ti":[-2.306,-0.543,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":15,"s":[393.233,366.589,0],"to":[2.306,0.543,0],"ti":[-2.085,-0.431,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":16,"s":[399.792,368.03,0],"to":[2.085,0.431,0],"ti":[-1.895,-0.342,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":17,"s":[405.741,369.174,0],"to":[1.895,0.342,0],"ti":[-1.729,-0.272,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":18,"s":[411.16,370.082,0],"to":[1.729,0.272,0],"ti":[-1.583,-0.217,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":19,"s":[416.115,370.805,0],"to":[1.583,0.217,0],"ti":[-1.452,-0.173,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":20,"s":[420.657,371.382,0],"to":[1.452,0.173,0],"ti":[-1.335,-0.139,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":21,"s":[424.829,371.844,0],"to":[1.335,0.139,0],"ti":[-1.229,-0.113,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":22,"s":[428.668,372.218,0],"to":[1.229,0.113,0],"ti":[-1.131,-0.094,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[432.201,372.525,0],"to":[1.131,0.094,0],"ti":[-1.041,-0.08,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[435.454,372.783,0],"to":[1.041,0.08,0],"ti":[-0.958,-0.071,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[438.448,373.008,0],"to":[0.958,0.071,0],"ti":[-0.88,-0.066,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[441.201,373.211,0],"to":[0.88,0.066,0],"ti":[-0.807,-0.063,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":27,"s":[443.727,373.403,0],"to":[0.807,0.063,0],"ti":[-0.738,-0.063,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":28,"s":[446.042,373.591,0],"to":[0.738,0.063,0],"ti":[-0.673,-0.065,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":29,"s":[448.156,373.783,0],"to":[0.673,0.065,0],"ti":[-0.611,-0.069,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[450.079,373.983,0],"to":[0.611,0.069,0],"ti":[-0.552,-0.073,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":31,"s":[451.822,374.195,0],"to":[0.552,0.073,0],"ti":[-0.495,-0.077,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":32,"s":[453.39,374.419,0],"to":[0.495,0.077,0],"ti":[-0.44,-0.082,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":33,"s":[454.792,374.659,0],"to":[0.44,0.082,0],"ti":[-0.387,-0.086,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":34,"s":[456.031,374.911,0],"to":[0.387,0.086,0],"ti":[-0.336,-0.09,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":35,"s":[457.115,375.176,0],"to":[0.336,0.09,0],"ti":[-0.286,-0.092,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":36,"s":[458.047,375.45,0],"to":[0.286,0.092,0],"ti":[-0.237,-0.093,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":37,"s":[458.831,375.73,0],"to":[0.237,0.093,0],"ti":[-0.19,-0.093,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.186},"t":38,"s":[459.47,376.01,0],"to":[0.19,0.093,0],"ti":[-0.143,-0.09,0]},{"i":{"x":0.833,"y":0.855},"o":{"x":0.167,"y":0.19},"t":39,"s":[459.968,376.286,0],"to":[0.143,0.09,0],"ti":[-0.097,-0.085,0]},{"i":{"x":0.833,"y":0.857},"o":{"x":0.167,"y":0.195},"t":40,"s":[460.328,376.55,0],"to":[0.097,0.085,0],"ti":[-0.052,-0.077,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.2},"t":41,"s":[460.55,376.795,0],"to":[0.052,0.077,0],"ti":[-0.007,-0.066,0]},{"i":{"x":0.833,"y":0.811},"o":{"x":0.167,"y":0.188},"t":42,"s":[460.639,377.012,0],"to":[0.007,0.066,0],"ti":[0.032,-0.061,0]},{"i":{"x":0.833,"y":0.813},"o":{"x":0.167,"y":0.149},"t":43,"s":[460.595,377.193,0],"to":[-0.032,0.061,0],"ti":[0.056,-0.069,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.151},"t":44,"s":[460.445,377.377,0],"to":[-0.056,0.069,0],"ti":[0.06,-0.083,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.159},"t":45,"s":[460.259,377.605,0],"to":[-0.06,0.083,0],"ti":[0.055,-0.097,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":46,"s":[460.084,377.877,0],"to":[-0.055,0.097,0],"ti":[0.049,-0.11,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.161},"t":47,"s":[459.926,378.189,0],"to":[-0.049,0.11,0],"ti":[0.042,-0.121,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.162},"t":48,"s":[459.788,378.537,0],"to":[-0.042,0.121,0],"ti":[0.034,-0.13,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.163},"t":49,"s":[459.673,378.915,0],"to":[-0.034,0.13,0],"ti":[0.025,-0.138,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.163},"t":50,"s":[459.585,379.319,0],"to":[-0.025,0.138,0],"ti":[0.015,-0.145,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.164},"t":51,"s":[459.525,379.745,0],"to":[-0.015,0.145,0],"ti":[0.004,-0.15,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.164},"t":52,"s":[459.496,380.189,0],"to":[-0.004,0.15,0],"ti":[-0.006,-0.154,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":53,"s":[459.499,380.647,0],"to":[0.006,0.154,0],"ti":[-0.017,-0.157,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":54,"s":[459.535,381.115,0],"to":[0.017,0.157,0],"ti":[-0.028,-0.158,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":55,"s":[459.603,381.589,0],"to":[0.028,0.158,0],"ti":[-0.039,-0.159,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.166},"t":56,"s":[459.705,382.066,0],"to":[0.039,0.159,0],"ti":[-0.05,-0.158,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":57,"s":[459.838,382.542,0],"to":[0.05,0.158,0],"ti":[-0.059,-0.156,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":58,"s":[460.002,383.013,0],"to":[0.059,0.156,0],"ti":[-0.032,-0.077,0]},{"t":59,"s":[460.195,383.477,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":4,"s":[97.409,66.055,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.753,1.579,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.148,-0.189,0]},"t":5,"s":[44.567,76.147,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.004,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.126,0.073,0]},"t":6,"s":[63.613,73.054,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.971,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.004,0.009,0]},"t":7,"s":[100.946,97.634,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.792,1.619,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.045,0.494,0]},"t":8,"s":[61.805,70.187,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.93,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.139,0.073,0]},"t":9,"s":[87.307,64.612,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.68,0.923,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.12,-0.436,0]},"t":10,"s":[125.487,111.62,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.883,-3.054,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,-0.938,0]},"t":11,"s":[109.816,104.073,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.123,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.289,0.085,0]},"t":12,"s":[65.323,104.689,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,1.021,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.05,-0.013,0]},"t":13,"s":[47.252,134.038,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.754,0.996,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.445,0.017,0]},"t":14,"s":[91.917,108.765,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,0.782,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.126,-0.004,0]},"t":15,"s":[102.196,140.322,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.598,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.059,0.135,0]},"t":16,"s":[122.314,110.374,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,0.983,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.105,-0.179,0]},"t":17,"s":[110.56,62.157,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.992,0.811,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.102,-0.022,0]},"t":18,"s":[65.661,77.47,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.964,1.174,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.009,0.149,0]},"t":19,"s":[85.871,65.321,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.389,0.889,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.062,0.056,0]},"t":20,"s":[67.714,49.917,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.983,1.088,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,0.335,0]},"t":21,"s":[78.115,97.527,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.976,1.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.021,0.043,0]},"t":22,"s":[143.974,113.298,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.034,0.01,0]},"t":23,"s":[91.205,80.878,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.004,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,0.012,0]},"t":24,"s":[128.534,117.961,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,0.878,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.004,-0.04,0]},"t":25,"s":[81.833,74.831,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.701,1.076,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.245,0.261,0]},"t":26,"s":[130.604,103.951,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.848,0.884,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.115,0.04,0]},"t":27,"s":[118.24,117.585,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,1.085,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.185,0.297,0]},"t":28,"s":[86.185,91.54,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.239,1.079,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.857,0.042,0]},"t":29,"s":[59.806,81.372,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.06,0.921,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.078,0.041,0]},"t":30,"s":[62.144,101.879,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.904,1.073,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.035,-1.454,0]},"t":31,"s":[25.053,61.948,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.113,0.614,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.64,0.039,0]},"t":32,"s":[88.839,64.113,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.878,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.048,0.106,0]},"t":33,"s":[98.389,60.046,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.973,1.366,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.265,-0.316,0]},"t":34,"s":[75.918,45.28,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.487,0.892,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.04,0.068,0]},"t":35,"s":[65.601,48.361,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.02,1.735,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.1,0.366,0]},"t":36,"s":[72.571,31.73,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.839,1.002,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,0.075,0]},"t":37,"s":[108.482,26.833,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,0.884,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.173,0.002,0]},"t":38,"s":[64.065,74.946,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.807,1.122,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.226,0.296,0]},"t":39,"s":[22.794,25.686,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.039,1.029,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.147,0.05,0]},"t":40,"s":[33.918,6.407,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,0.021,0]},"t":41,"s":[48.539,53.893,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.843,1.001,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.114,-0.046,0]},"t":42,"s":[27.022,-9.839,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.415,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.177,0.001,0]},"t":43,"s":[36.118,31.361,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.042,0.504,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,-0.476,0]},"t":44,"s":[44.181,-10.557,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.028,0.1,0]},"t":45,"s":[-4.037,-4.307,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.945,0.884,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.027,0.248,0]},"t":46,"s":[68.255,26.661,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.863,0.961,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.164,0.298,0]},"t":47,"s":[13.766,42.343,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.171,-1.3,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.213,-0.072,0]},"t":48,"s":[32.121,48.441,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.056,0.086,0]},"t":49,"s":[43.941,45.171,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.907,0.82,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.238,-0.116,0]},"t":50,"s":[7.866,-41.801,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.765,0.908,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.767,0.155,0]},"t":51,"s":[17.231,-5.411,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,0.772,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.087,0.843,0]},"t":52,"s":[18.373,36.907,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.047,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.066,0.131,0]},"t":53,"s":[41.417,41.547,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.142,1.256,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.03,-0.64,0]},"t":54,"s":[28.526,49.618,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.933,1.594,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.053,0.063,0]},"t":55,"s":[48.627,48.688,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.831,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.344,0.073,0]},"t":56,"s":[-5.742,52.477,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.959,1.011,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.164,-0.017,0]},"t":57,"s":[4.853,21.683,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.082,0.01,0]},"t":58,"s":[15.758,47.142,100]},{"t":59,"s":[10.26,18.203,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":4,"op":60,"st":4,"bm":0},{"ddd":0,"ind":27,"ty":4,"nm":"Shape Layer 25","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":15,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":37,"s":[100]},{"t":53,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.59},"o":{"x":0.167,"y":0.167},"t":10,"s":[269.753,298.518,0],"to":[0.371,-0.157,0],"ti":[-1.794,0.84,0]},{"i":{"x":0.833,"y":0.764},"o":{"x":0.167,"y":0.105},"t":11,"s":[271.976,297.574,0],"to":[1.794,-0.84,0],"ti":[-3.913,2.158,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":12,"s":[280.516,293.478,0],"to":[3.913,-2.158,0],"ti":[-5.148,3.28,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":13,"s":[295.452,284.625,0],"to":[5.148,-3.28,0],"ti":[-4.973,3.376,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":14,"s":[311.406,273.799,0],"to":[4.973,-3.376,0],"ti":[-4.286,2.798,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":15,"s":[325.289,264.369,0],"to":[4.286,-2.798,0],"ti":[-3.685,2.155,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":16,"s":[337.123,257.014,0],"to":[3.685,-2.155,0],"ti":[-3.222,1.625,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":17,"s":[347.398,251.439,0],"to":[3.222,-1.625,0],"ti":[-2.85,1.213,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":18,"s":[356.453,247.262,0],"to":[2.85,-1.213,0],"ti":[-2.541,0.896,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":19,"s":[364.5,244.163,0],"to":[2.541,-0.896,0],"ti":[-2.276,0.656,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":20,"s":[371.696,241.885,0],"to":[2.276,-0.656,0],"ti":[-2.047,0.474,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":21,"s":[378.157,240.228,0],"to":[2.047,-0.474,0],"ti":[-1.845,0.337,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":22,"s":[383.976,239.042,0],"to":[1.845,-0.337,0],"ti":[-1.666,0.235,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":23,"s":[389.227,238.206,0],"to":[1.666,-0.235,0],"ti":[-1.505,0.161,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":24,"s":[393.971,237.63,0],"to":[1.505,-0.161,0],"ti":[-1.361,0.108,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":25,"s":[398.259,237.241,0],"to":[1.361,-0.108,0],"ti":[-1.229,0.074,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":26,"s":[402.134,236.98,0],"to":[1.229,-0.074,0],"ti":[-1.111,0.051,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":27,"s":[405.636,236.799,0],"to":[1.111,-0.051,0],"ti":[-1.01,0.021,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":28,"s":[408.801,236.672,0],"to":[1.01,-0.021,0],"ti":[-0.926,-0.021,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":29,"s":[411.696,236.673,0],"to":[0.926,0.021,0],"ti":[-0.853,-0.06,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":30,"s":[414.357,236.801,0],"to":[0.853,0.06,0],"ti":[-0.787,-0.092,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":31,"s":[416.812,237.033,0],"to":[0.787,0.092,0],"ti":[-0.729,-0.118,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":32,"s":[419.082,237.352,0],"to":[0.729,0.118,0],"ti":[-0.677,-0.139,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":33,"s":[421.188,237.741,0],"to":[0.677,0.139,0],"ti":[-0.631,-0.156,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":34,"s":[423.146,238.187,0],"to":[0.631,0.156,0],"ti":[-0.589,-0.169,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":35,"s":[424.973,238.678,0],"to":[0.589,0.169,0],"ti":[-0.552,-0.179,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":36,"s":[426.682,239.204,0],"to":[0.552,0.179,0],"ti":[-0.518,-0.186,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":37,"s":[428.285,239.754,0],"to":[0.518,0.186,0],"ti":[-0.488,-0.19,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":38,"s":[429.793,240.32,0],"to":[0.488,0.19,0],"ti":[-0.46,-0.192,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":39,"s":[431.212,240.895,0],"to":[0.46,0.192,0],"ti":[-0.433,-0.191,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":40,"s":[432.55,241.471,0],"to":[0.433,0.191,0],"ti":[-0.408,-0.188,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":41,"s":[433.811,242.041,0],"to":[0.408,0.188,0],"ti":[-0.384,-0.184,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":42,"s":[434.998,242.6,0],"to":[0.384,0.184,0],"ti":[-0.36,-0.177,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":43,"s":[436.114,243.143,0],"to":[0.36,0.177,0],"ti":[-0.337,-0.169,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":44,"s":[437.16,243.663,0],"to":[0.337,0.169,0],"ti":[-0.314,-0.159,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":45,"s":[438.136,244.156,0],"to":[0.314,0.159,0],"ti":[-0.29,-0.148,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":46,"s":[439.041,244.617,0],"to":[0.29,0.148,0],"ti":[-0.265,-0.135,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":47,"s":[439.873,245.043,0],"to":[0.265,0.135,0],"ti":[-0.239,-0.121,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":48,"s":[440.629,245.429,0],"to":[0.239,0.121,0],"ti":[-0.211,-0.106,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.179},"t":49,"s":[441.306,245.771,0],"to":[0.211,0.106,0],"ti":[-0.188,-0.091,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.176},"t":50,"s":[441.897,246.067,0],"to":[0.188,0.091,0],"ti":[-0.176,-0.076,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":51,"s":[442.436,246.318,0],"to":[0.176,0.076,0],"ti":[-0.167,-0.062,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.174},"t":52,"s":[442.952,246.526,0],"to":[0.167,0.062,0],"ti":[-0.154,-0.046,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.177},"t":53,"s":[443.435,246.688,0],"to":[0.154,0.046,0],"ti":[-0.137,-0.029,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.181},"t":54,"s":[443.874,246.8,0],"to":[0.137,0.029,0],"ti":[-0.117,-0.011,0]},{"i":{"x":0.833,"y":0.853},"o":{"x":0.167,"y":0.186},"t":55,"s":[444.258,246.86,0],"to":[0.117,0.011,0],"ti":[-0.092,0.008,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.192},"t":56,"s":[444.574,246.866,0],"to":[0.092,-0.008,0],"ti":[-0.065,0.028,0]},{"i":{"x":0.833,"y":0.809},"o":{"x":0.167,"y":0.188},"t":57,"s":[444.81,246.814,0],"to":[0.065,-0.028,0],"ti":[-0.052,0.051,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.148},"t":58,"s":[444.965,246.7,0],"to":[0.052,-0.051,0],"ti":[-0.027,0.032,0]},{"t":59,"s":[445.124,246.509,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.872,1.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":10,"s":[110.669,116.358,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.04,0.96,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.239,0.011,0]},"t":11,"s":[59.779,98.507,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,1.168,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,-0.078,0]},"t":12,"s":[32.558,118.992,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.283,0.983,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.327,0.056,0]},"t":13,"s":[72.99,108.43,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.904,0.749,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,-0.022,0]},"t":14,"s":[64.776,140.246,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.471,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.643,0.125,0]},"t":15,"s":[100.849,115.061,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.917,0.951,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,-0.015,0]},"t":16,"s":[106.224,64.521,100]},{"i":{"x":[0.833,0.833,0.833],"y":[7.839,0.661,1]},"o":{"x":[0.167,0.167,0.167],"y":[-10.254,-0.119,0]},"t":17,"s":[70.46,107.342,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.07,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.082,0.111,0]},"t":18,"s":[70.749,89.722,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.822,0.875,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,-0.015,0]},"t":19,"s":[46.798,35.679,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.002,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.157,0.249,0]},"t":20,"s":[90.882,81.249,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.942,1.169,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.002,-0.177,0]},"t":21,"s":[140.894,104.127,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.056,1.071,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.196,0.056,0]},"t":22,"s":[89.564,96.8,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.972,0.871,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.034,0.038,0]},"t":23,"s":[104.901,119.028,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.839,1.124,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.041,0.236,0]},"t":24,"s":[79.173,77.868,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.075,0.964,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.172,0.05,0]},"t":25,"s":[96.359,55.376,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.864,0.984,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,-0.064,0]},"t":26,"s":[112.425,111.326,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.964,0.96,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.216,-0.02,0]},"t":27,"s":[81.872,79.606,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.136,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.063,-0.076,0]},"t":28,"s":[62.674,105.212,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.04,0.329,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.052,0.758,0]},"t":29,"s":[73.611,91.797,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,0.991,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,0.095,0]},"t":30,"s":[44.883,90.141,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.864,1.007,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.445,-0.009,0]},"t":31,"s":[87.448,78.456,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.687,0.817,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.216,0.007,0]},"t":32,"s":[97.261,88.946,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.172,0.78,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.114,0.153,0]},"t":33,"s":[103.41,77.55,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.025,0.995,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.056,0.134,0]},"t":34,"s":[120.354,63.853,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,0.776,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.019,-0.005,0]},"t":35,"s":[68.478,41.315,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.781,1.002,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.226,0.133,0]},"t":36,"s":[135.85,62.489,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.836,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,0.002,0]},"t":37,"s":[117.729,98.108,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.955,0.732,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.17,0.45,0]},"t":38,"s":[88.261,61.749,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.161,0.879,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.097,0.121,0]},"t":39,"s":[59.775,53.492,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.268,0]},"t":40,"s":[72.912,35.197,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.014,1.011,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.932,0.009,0]},"t":41,"s":[34.35,26.941,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.166,0.799,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,0.01,0]},"t":42,"s":[35.945,36.212,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.01,1.008,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.142,0]},"t":43,"s":[53.22,25.739,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.804,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.009,0.007,0]},"t":44,"s":[1.613,10.927,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.916,0.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.054,0.145,0]},"t":45,"s":[59.191,27.12,100]},{"i":{"x":[0.833,0.833,0.833],"y":[32.334,1.954,1]},"o":{"x":[0.167,0.167,0.167],"y":[28.082,-2.647,0]},"t":46,"s":[24.376,49.1,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,1.605,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.083,0.077,0]},"t":47,"s":[24.272,48.429,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.911,0.914,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.066,0.073,0]},"t":48,"s":[63.337,56.778,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.402,1.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.325,2.783,0]},"t":49,"s":[41.555,-12.234,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.891,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,0.076,0]},"t":50,"s":[40.094,-14.365,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.348,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.357,0.224,0]},"t":51,"s":[16.969,11.25,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.893,1.059,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,-0.033,0]},"t":52,"s":[9.934,26.464,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.223,1.041,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.372,0.034,0]},"t":53,"s":[46.338,15.586,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.949,0.977,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,0.028,0]},"t":54,"s":[56.833,34.129,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.872,0.807,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.133,-0.032,0]},"t":55,"s":[18.206,6.413,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.098,0.917,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.239,0.147,0]},"t":56,"s":[33.108,26.401,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.855,-124.032,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,-45.948,0]},"t":57,"s":[41.083,52.674,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.195,0.083,0]},"t":58,"s":[23.746,52.627,100]},{"t":59,"s":[10.82,-18.691,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":10,"op":60,"st":10,"bm":0},{"ddd":0,"ind":28,"ty":4,"nm":"Shape Layer 24","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":7,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":34,"s":[100]},{"t":50,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.604},"o":{"x":0.167,"y":0.167},"t":7,"s":[265.04,301.431,0],"to":[-0.341,-0.143,0],"ti":[1.527,0.877,0]},{"i":{"x":0.833,"y":0.765},"o":{"x":0.167,"y":0.106},"t":8,"s":[262.993,300.574,0],"to":[-1.527,-0.877,0],"ti":[3.095,2.404,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":9,"s":[255.881,296.17,0],"to":[-3.095,-2.404,0],"ti":[3.636,3.902,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":10,"s":[244.425,286.149,0],"to":[-3.636,-3.902,0],"ti":[2.978,4.369,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.178},"t":11,"s":[234.064,272.758,0],"to":[-2.978,-4.369,0],"ti":[2.214,3.975,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":12,"s":[226.554,259.936,0],"to":[-2.214,-3.975,0],"ti":[1.769,3.38,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":13,"s":[220.781,248.906,0],"to":[-1.769,-3.38,0],"ti":[1.52,2.839,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":14,"s":[215.943,239.657,0],"to":[-1.52,-2.839,0],"ti":[1.364,2.397,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":15,"s":[211.658,231.874,0],"to":[-1.364,-2.397,0],"ti":[1.247,2.04,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":16,"s":[207.761,225.273,0],"to":[-1.247,-2.04,0],"ti":[1.147,1.749,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":17,"s":[204.179,219.632,0],"to":[-1.147,-1.749,0],"ti":[1.054,1.508,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":18,"s":[200.881,214.781,0],"to":[-1.054,-1.508,0],"ti":[0.966,1.308,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":19,"s":[197.852,210.583,0],"to":[-0.966,-1.308,0],"ti":[0.881,1.139,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":20,"s":[195.083,206.935,0],"to":[-0.881,-1.139,0],"ti":[0.798,0.996,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":21,"s":[192.565,203.748,0],"to":[-0.798,-0.996,0],"ti":[0.718,0.875,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":22,"s":[190.293,200.956,0],"to":[-0.718,-0.875,0],"ti":[0.64,0.771,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":23,"s":[188.258,198.5,0],"to":[-0.64,-0.771,0],"ti":[0.565,0.681,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":24,"s":[186.453,196.332,0],"to":[-0.565,-0.681,0],"ti":[0.494,0.604,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":25,"s":[184.867,194.413,0],"to":[-0.494,-0.604,0],"ti":[0.425,0.537,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":26,"s":[183.491,192.71,0],"to":[-0.425,-0.537,0],"ti":[0.361,0.479,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":27,"s":[182.314,191.193,0],"to":[-0.361,-0.479,0],"ti":[0.301,0.428,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":28,"s":[181.325,189.838,0],"to":[-0.301,-0.428,0],"ti":[0.244,0.384,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":29,"s":[180.511,188.625,0],"to":[-0.244,-0.384,0],"ti":[0.192,0.346,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":30,"s":[179.859,187.534,0],"to":[-0.192,-0.346,0],"ti":[0.145,0.312,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":31,"s":[179.357,186.551,0],"to":[-0.145,-0.312,0],"ti":[0.103,0.284,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":32,"s":[178.99,185.66,0],"to":[-0.103,-0.284,0],"ti":[0.066,0.259,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.177},"t":33,"s":[178.741,184.85,0],"to":[-0.066,-0.259,0],"ti":[0.034,0.238,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":34,"s":[178.596,184.107,0],"to":[-0.034,-0.238,0],"ti":[0.008,0.219,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":35,"s":[178.536,183.424,0],"to":[-0.008,-0.219,0],"ti":[-0.011,0.204,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":36,"s":[178.545,182.791,0],"to":[0.011,-0.204,0],"ti":[-0.025,0.191,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":37,"s":[178.604,182.2,0],"to":[0.025,-0.191,0],"ti":[-0.033,0.18,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":38,"s":[178.696,181.645,0],"to":[0.033,-0.18,0],"ti":[-0.033,0.171,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":39,"s":[178.8,181.121,0],"to":[0.033,-0.171,0],"ti":[-0.027,0.163,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.171},"t":40,"s":[178.897,180.622,0],"to":[0.027,-0.163,0],"ti":[-0.014,0.157,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.17},"t":41,"s":[178.964,180.143,0],"to":[0.014,-0.157,0],"ti":[-0.002,0.154,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.167},"t":42,"s":[178.981,179.68,0],"to":[0.002,-0.154,0],"ti":[-0.011,0.158,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":43,"s":[178.978,179.217,0],"to":[0.011,-0.158,0],"ti":[-0.033,0.162,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.164},"t":44,"s":[179.048,178.732,0],"to":[0.033,-0.162,0],"ti":[-0.051,0.161,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.166},"t":45,"s":[179.179,178.243,0],"to":[0.051,-0.161,0],"ti":[-0.064,0.155,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":46,"s":[179.355,177.765,0],"to":[0.064,-0.155,0],"ti":[-0.069,0.148,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.17},"t":47,"s":[179.564,177.312,0],"to":[0.069,-0.148,0],"ti":[-0.061,0.142,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":48,"s":[179.767,176.879,0],"to":[0.061,-0.142,0],"ti":[-0.046,0.136,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.174},"t":49,"s":[179.929,176.457,0],"to":[0.046,-0.136,0],"ti":[-0.028,0.126,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.177},"t":50,"s":[180.041,176.061,0],"to":[0.028,-0.126,0],"ti":[-0.009,0.113,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.18},"t":51,"s":[180.098,175.7,0],"to":[0.009,-0.113,0],"ti":[0.012,0.096,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.181},"t":52,"s":[180.095,175.385,0],"to":[-0.012,-0.096,0],"ti":[0.034,0.077,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.176},"t":53,"s":[180.027,175.124,0],"to":[-0.034,-0.077,0],"ti":[0.057,0.055,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.166},"t":54,"s":[179.891,174.924,0],"to":[-0.057,-0.055,0],"ti":[0.08,0.032,0]},{"i":{"x":0.833,"y":0.815},"o":{"x":0.167,"y":0.156},"t":55,"s":[179.687,174.792,0],"to":[-0.08,-0.032,0],"ti":[0.103,0.008,0]},{"i":{"x":0.833,"y":0.815},"o":{"x":0.167,"y":0.152},"t":56,"s":[179.413,174.732,0],"to":[-0.103,-0.008,0],"ti":[0.125,-0.018,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.152},"t":57,"s":[179.07,174.747,0],"to":[-0.125,0.018,0],"ti":[0.147,-0.043,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.153},"t":58,"s":[178.66,174.838,0],"to":[-0.147,0.043,0],"ti":[0.079,-0.028,0]},{"t":59,"s":[178.187,175.006,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.856,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":7,"s":[67.557,101.258,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,0.856,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.076,-0.033,0]},"t":8,"s":[62.081,60.567,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.899,1,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.062,0.197,0]},"t":9,"s":[123.801,89.604,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.652,0.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.485,0,0]},"t":10,"s":[88.344,110.822,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.902,1.594,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.11,1.18,0]},"t":11,"s":[80.981,89.674,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.494,0.831,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.569,0.073,0]},"t":12,"s":[57.576,88.067,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.815,1.117,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.164,0]},"t":13,"s":[53.56,101.12,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.035,0.99,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.152,0.049,0]},"t":14,"s":[81.37,114.583,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,-0.011,0]},"t":15,"s":[115.142,82.186,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.154,1.445,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.235,0.491,0]},"t":16,"s":[67.083,110.768,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.037,0.85,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,0.07,0]},"t":17,"s":[79.677,116.611,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,1.062,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.026,0.187,0]},"t":18,"s":[43.885,79.546,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.972,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.301,0.035,0]},"t":19,"s":[95.707,49.721,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.783,0.41,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.043,0.938,0]},"t":20,"s":[115.579,101.629,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.957,0.995,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,0.097,0]},"t":21,"s":[102.439,106.687,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.875,0.793,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.087,-0.005,0]},"t":22,"s":[81.407,137.444,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.844,0.981,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.25,0.14,0]},"t":23,"s":[91.687,108.575,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.4,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.178,-0.024,0]},"t":24,"s":[96.813,65.889,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,1.015,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,0.009,0]},"t":25,"s":[101.302,98.9,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.186,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.902,0.013,0]},"t":26,"s":[75.267,62.121,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.894,0.887,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,0.724,0]},"t":27,"s":[77.47,105.409,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.167,2.294,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.393,0.315,0]},"t":28,"s":[70.35,111.04,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.394,0.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.056,0.078,0]},"t":29,"s":[68.432,113.069,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,1.578,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.097,1.095,0]},"t":30,"s":[74.202,79.548,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,0.978,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.001,0.073,0]},"t":31,"s":[110.376,76.788,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.071,0.748,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.029,-0.029,0]},"t":32,"s":[74.587,98.699,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.005,0.895,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.125,0]},"t":33,"s":[101.235,82.443,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,1.096,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.004,0.401,0]},"t":34,"s":[51.784,49.555,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.916,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.43,0.045,0]},"t":35,"s":[103.982,40.922,100]},{"i":{"x":[0.833,0.833,0.833],"y":[22.316,1.143,1]},"o":{"x":[0.167,0.167,0.167],"y":[6.077,-0.023,0]},"t":36,"s":[116.519,59.525,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.868,1.051,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.083,0.053,0]},"t":37,"s":[116.693,44.996,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.915,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.226,0.031,0]},"t":38,"s":[71.936,84.414,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.19,1.045,1]},"o":{"x":[0.167,0.167,0.167],"y":[4.525,-0.385,0]},"t":39,"s":[45.732,21.041,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.945,0.893,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.09,0.029,0]},"t":40,"s":[45.24,32.32,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.147,0.824,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.16,0.379,0]},"t":41,"s":[38.713,14.915,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.02,0.834,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.078,0.158,0]},"t":42,"s":[40.945,10.011,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.96,1.294,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,0.167,0]},"t":43,"s":[7.998,4.532,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.811,0.925,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.078,0.065,0]},"t":44,"s":[48.948,-0.907,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,2.089,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.149,-0.765,0]},"t":45,"s":[27.778,23.742,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.789,1.036,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.033,0.077,0]},"t":46,"s":[0.947,21.32,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.138,0.025,0]},"t":47,"s":[20.246,55.394,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.839,1.071,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.317,0.568,0]},"t":48,"s":[49.89,6.617,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.533,0.903,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.173,0.038,0]},"t":49,"s":[43.72,-1.776,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,1.723,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.101,0.578,0]},"t":50,"s":[37.975,13.785,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.517,1.043,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.234,0.075,0]},"t":51,"s":[11.499,16.406,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,0.978,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.101,0.028,0]},"t":52,"s":[18.462,-8.949,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.022,0.904,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.068,-0.031,0]},"t":53,"s":[51.821,29.36,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.996,1.698,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,0.617,0]},"t":54,"s":[33.392,1.359,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.888,0.865,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.004,0.074,0]},"t":55,"s":[56.645,-3.016,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.486,0.985,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.324,0.219,0]},"t":56,"s":[34.461,37.994,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.009,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,-0.018,0]},"t":57,"s":[26.772,63.226,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.008,0.247,0]},"t":58,"s":[-12.998,42.488,100]},{"t":59,"s":[31.14,31.954,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":7,"op":60,"st":7,"bm":0},{"ddd":0,"ind":29,"ty":4,"nm":"Shape Layer 23","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":5,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":32,"s":[100]},{"t":48,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.598},"o":{"x":0.167,"y":0.167},"t":5,"s":[261.789,302.009,0],"to":[-0.181,-0.464,0],"ti":[0.71,2.293,0]},{"i":{"x":0.833,"y":0.765},"o":{"x":0.167,"y":0.105},"t":6,"s":[260.702,299.226,0],"to":[-0.71,-2.293,0],"ti":[1.003,5.268,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":7,"s":[257.531,288.249,0],"to":[-1.003,-5.268,0],"ti":[0.104,7.272,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":8,"s":[254.686,267.621,0],"to":[-0.104,-7.272,0],"ti":[-1.513,7.003,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":9,"s":[256.909,244.618,0],"to":[1.513,-7.003,0],"ti":[-2.58,5.598,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":10,"s":[263.763,225.603,0],"to":[2.58,-5.598,0],"ti":[-2.89,4.322,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":11,"s":[272.387,211.032,0],"to":[2.89,-4.322,0],"ti":[-2.817,3.434,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":12,"s":[281.103,199.67,0],"to":[2.817,-3.434,0],"ti":[-2.604,2.843,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":13,"s":[289.287,190.426,0],"to":[2.604,-2.843,0],"ti":[-2.352,2.444,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":14,"s":[296.729,182.612,0],"to":[2.352,-2.444,0],"ti":[-2.098,2.172,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":15,"s":[303.402,175.762,0],"to":[2.098,-2.172,0],"ti":[-1.847,1.982,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":16,"s":[309.316,169.581,0],"to":[1.847,-1.982,0],"ti":[-1.599,1.847,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[314.483,163.871,0],"to":[1.599,-1.847,0],"ti":[-1.351,1.749,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[318.909,158.499,0],"to":[1.351,-1.749,0],"ti":[-1.1,1.67,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":19,"s":[322.589,153.379,0],"to":[1.1,-1.67,0],"ti":[-0.847,1.597,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":20,"s":[325.509,148.477,0],"to":[0.847,-1.597,0],"ti":[-0.598,1.514,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":21,"s":[327.669,143.798,0],"to":[0.598,-1.514,0],"ti":[-0.368,1.415,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":22,"s":[329.098,139.39,0],"to":[0.368,-1.415,0],"ti":[-0.17,1.301,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":23,"s":[329.877,135.307,0],"to":[0.17,-1.301,0],"ti":[-0.012,1.178,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":24,"s":[330.12,131.586,0],"to":[0.012,-1.178,0],"ti":[0.108,1.057,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":25,"s":[329.95,128.236,0],"to":[-0.108,-1.057,0],"ti":[0.197,0.942,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[329.473,125.245,0],"to":[-0.197,-0.942,0],"ti":[0.245,0.841,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[328.77,122.586,0],"to":[-0.245,-0.841,0],"ti":[0.252,0.756,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":28,"s":[328.003,120.201,0],"to":[-0.252,-0.756,0],"ti":[0.241,0.68,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":29,"s":[327.259,118.051,0],"to":[-0.241,-0.68,0],"ti":[0.227,0.611,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":30,"s":[326.555,116.118,0],"to":[-0.227,-0.611,0],"ti":[0.211,0.546,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":31,"s":[325.898,114.387,0],"to":[-0.211,-0.546,0],"ti":[0.196,0.484,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":32,"s":[325.288,112.845,0],"to":[-0.196,-0.484,0],"ti":[0.183,0.426,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":33,"s":[324.72,111.482,0],"to":[-0.183,-0.426,0],"ti":[0.173,0.371,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":34,"s":[324.187,110.287,0],"to":[-0.173,-0.371,0],"ti":[0.164,0.318,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":35,"s":[323.684,109.255,0],"to":[-0.164,-0.318,0],"ti":[0.159,0.267,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.18},"t":36,"s":[323.201,108.379,0],"to":[-0.159,-0.267,0],"ti":[0.156,0.217,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.18},"t":37,"s":[322.731,107.655,0],"to":[-0.156,-0.217,0],"ti":[0.154,0.168,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":38,"s":[322.267,107.078,0],"to":[-0.154,-0.168,0],"ti":[0.155,0.121,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.179},"t":39,"s":[321.804,106.645,0],"to":[-0.155,-0.121,0],"ti":[0.157,0.075,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.175},"t":40,"s":[321.338,106.352,0],"to":[-0.157,-0.075,0],"ti":[0.16,0.031,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.169},"t":41,"s":[320.863,106.193,0],"to":[-0.16,-0.031,0],"ti":[0.165,-0.011,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.163},"t":42,"s":[320.376,106.164,0],"to":[-0.165,0.011,0],"ti":[0.169,-0.052,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.16},"t":43,"s":[319.875,106.261,0],"to":[-0.169,0.052,0],"ti":[0.172,-0.092,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.158},"t":44,"s":[319.363,106.479,0],"to":[-0.172,0.092,0],"ti":[0.18,-0.125,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.158},"t":45,"s":[318.841,106.814,0],"to":[-0.18,0.125,0],"ti":[0.197,-0.144,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.159},"t":46,"s":[318.282,107.227,0],"to":[-0.197,0.144,0],"ti":[0.217,-0.157,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":47,"s":[317.66,107.68,0],"to":[-0.217,0.157,0],"ti":[0.233,-0.168,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":48,"s":[316.983,108.169,0],"to":[-0.233,0.168,0],"ti":[0.246,-0.177,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.163},"t":49,"s":[316.262,108.687,0],"to":[-0.246,0.177,0],"ti":[0.254,-0.183,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.164},"t":50,"s":[315.509,109.229,0],"to":[-0.254,0.183,0],"ti":[0.259,-0.188,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":51,"s":[314.735,109.788,0],"to":[-0.259,0.188,0],"ti":[0.258,-0.189,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.167},"t":52,"s":[313.957,110.355,0],"to":[-0.258,0.189,0],"ti":[0.251,-0.187,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.169},"t":53,"s":[313.19,110.921,0],"to":[-0.251,0.187,0],"ti":[0.238,-0.182,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":54,"s":[312.452,111.477,0],"to":[-0.238,0.182,0],"ti":[0.219,-0.173,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.174},"t":55,"s":[311.761,112.011,0],"to":[-0.219,0.173,0],"ti":[0.192,-0.16,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.178},"t":56,"s":[311.139,112.513,0],"to":[-0.192,0.16,0],"ti":[0.164,-0.143,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":57,"s":[310.606,112.97,0],"to":[-0.164,0.143,0],"ti":[0.139,-0.125,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.181},"t":58,"s":[310.154,113.373,0],"to":[-0.139,0.125,0],"ti":[0.064,-0.057,0]},{"t":59,"s":[309.77,113.717,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.657,0.977,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":5,"s":[122.178,94.449,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.995,0.673,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.11,-0.032,0]},"t":6,"s":[102.174,113.837,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.918,0.997,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.005,0.112,0]},"t":7,"s":[39.796,99.783,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.373,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[-5.85,-0.003,0]},"t":8,"s":[98.461,58.708,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.007,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,-0.182,0]},"t":9,"s":[97.637,98.226,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.861,-0.461,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,2.089,0]},"t":10,"s":[121.922,85.826,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.953,1.17,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.208,0.088,0]},"t":11,"s":[95.646,85.311,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.37,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.109,0.056,0]},"t":12,"s":[78.043,76.794,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.019,1.331,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,1.985,0]},"t":13,"s":[85.681,102.643,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,1.177,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,0.067,0]},"t":14,"s":[135.804,103.775,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,0.896,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.12,0.057,0]},"t":15,"s":[74.18,98.145,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.067,2.296,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.019,0.418,0]},"t":16,"s":[99.49,115.703,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.037,0.078,0]},"t":17,"s":[78.946,120.079,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.978,0.68,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,-0.197,0]},"t":18,"s":[115.943,47.625,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.04,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.029,0.113,0]},"t":19,"s":[90.116,69.173,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.011,1.066,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,0.387,0]},"t":20,"s":[109.209,130.363,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.972,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.01,0.037,0]},"t":21,"s":[80.945,147.149,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.048,0.791,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.043,0.733,0]},"t":22,"s":[112.856,117.083,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.002,0.619,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.03,0.139,0]},"t":23,"s":[91.758,113.226,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.913,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.002,0.107,0]},"t":24,"s":[124.953,107.402,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.03,0.669,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.871,-0.086,0]},"t":25,"s":[90.896,86.565,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.06,0.86,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,0.111,0]},"t":26,"s":[89.309,96.814,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.994,1.197,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.035,0.207,0]},"t":27,"s":[71.266,127.218,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.064,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.007,0.059,0]},"t":28,"s":[102.251,147.752,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.023,1.152,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.036,1.009,0]},"t":29,"s":[73.55,78.754,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.89,0.917,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.018,0.054,0]},"t":30,"s":[124.448,72.546,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,-191.097,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.347,-188.615,0]},"t":31,"s":[59.508,90.091,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.122,0.759,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.254,0.083,0]},"t":32,"s":[39,90.083,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,0.127,0]},"t":33,"s":[44.062,72.23,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.05,0.825,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.262,-1.141,0]},"t":34,"s":[92.349,38.4,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.058,-0.83,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.031,0.159,0]},"t":35,"s":[80.711,40.702,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.872,0.959,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.034,0.087,0]},"t":36,"s":[99.332,43.228,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.945,0.925,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.238,-0.08,0]},"t":37,"s":[67.744,96.178,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.837,1.576,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.16,-0.761,0]},"t":38,"s":[50.721,69.109,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.787,0.792,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.171,0.073,0]},"t":39,"s":[56.555,71.779,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.453,0.836,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.137,0.139,0]},"t":40,"s":[62.129,50.654,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,0.889,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,0.169,0]},"t":41,"s":[70.779,19.084,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.921,1.022,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.02,0.334,0]},"t":42,"s":[15.062,-11.663,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.296,0.99,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.554,0.018,0]},"t":43,"s":[59.871,-21.902,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,1.249,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,-0.012,0]},"t":44,"s":[57.59,-8.938,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.049,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,0.062,0]},"t":45,"s":[24.406,-20.323,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.841,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.031,-0.187,0]},"t":46,"s":[-6.116,25.019,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.989,1.089,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.626,0.175,0]},"t":47,"s":[42.39,11.059,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.274,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.013,0.043,0]},"t":48,"s":[45.011,-1.68,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.577,1.034,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,0.225,0]},"t":49,"s":[42.735,24.617,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.024,0]},"t":50,"s":[52.491,40.139,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.91,0.866,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,-0.326,0]},"t":51,"s":[-24.813,18.208,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.149,1.734,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.154,0.22,0]},"t":52,"s":[29.263,22.673,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.047,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,0.075,0]},"t":53,"s":[33.471,25.402,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.891,0.786,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.03,-0.07,0]},"t":54,"s":[72.245,-1.354,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,0.815,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.355,0.137,0]},"t":55,"s":[11.503,13.161,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.127,0.918,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.26,0.152,0]},"t":56,"s":[-7.15,35.893,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.908,-1.79,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,-4.527,0]},"t":57,"s":[-2.619,63.648,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.867,0.086,0]},"t":58,"s":[40.328,63.147,100]},{"t":59,"s":[44.897,46.852,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":5,"op":60,"st":5,"bm":0},{"ddd":0,"ind":30,"ty":4,"nm":"Shape Layer 22","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":2,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":7,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":29,"s":[100]},{"t":45,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.555},"o":{"x":0.167,"y":0.167},"t":2,"s":[263.182,299.465,0],"to":[-0.365,0.027,0],"ti":[1.949,-0.179,0]},{"i":{"x":0.833,"y":0.762},"o":{"x":0.167,"y":0.103},"t":3,"s":[260.99,299.63,0],"to":[-1.949,0.179,0],"ti":[4.498,-0.649,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.128},"t":4,"s":[251.489,300.54,0],"to":[-4.498,0.649,0],"ti":[6.073,-1.426,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":5,"s":[234.001,303.526,0],"to":[-6.073,1.426,0],"ti":[5.777,-2.109,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":6,"s":[215.049,309.096,0],"to":[-5.777,2.109,0],"ti":[4.639,-2.46,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":7,"s":[199.338,316.18,0],"to":[-4.639,2.46,0],"ti":[3.563,-2.581,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":8,"s":[187.213,323.857,0],"to":[-3.563,2.581,0],"ti":[2.716,-2.584,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":9,"s":[177.957,331.668,0],"to":[-2.716,2.584,0],"ti":[2.065,-2.517,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":10,"s":[170.915,339.358,0],"to":[-2.065,2.517,0],"ti":[1.564,-2.411,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":11,"s":[165.569,346.772,0],"to":[-1.564,2.411,0],"ti":[1.165,-2.278,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":12,"s":[161.531,353.821,0],"to":[-1.165,2.278,0],"ti":[0.843,-2.131,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":13,"s":[158.58,360.44,0],"to":[-0.843,2.131,0],"ti":[0.594,-1.982,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":14,"s":[156.475,366.609,0],"to":[-0.594,1.982,0],"ti":[0.404,-1.836,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":15,"s":[155.019,372.333,0],"to":[-0.404,1.836,0],"ti":[0.261,-1.696,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":16,"s":[154.052,377.626,0],"to":[-0.261,1.696,0],"ti":[0.155,-1.565,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":17,"s":[153.453,382.511,0],"to":[-0.155,1.565,0],"ti":[0.078,-1.441,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":18,"s":[153.124,387.013,0],"to":[-0.078,1.441,0],"ti":[0.024,-1.327,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":19,"s":[152.987,391.16,0],"to":[-0.024,1.327,0],"ti":[-0.012,-1.222,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":20,"s":[152.981,394.977,0],"to":[0.012,1.222,0],"ti":[-0.034,-1.124,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":21,"s":[153.059,398.49,0],"to":[0.034,1.124,0],"ti":[-0.044,-1.033,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":22,"s":[153.183,401.72,0],"to":[0.044,1.033,0],"ti":[-0.045,-0.949,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[153.322,404.688,0],"to":[0.045,0.949,0],"ti":[-0.039,-0.87,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[153.452,407.413,0],"to":[0.039,0.87,0],"ti":[-0.028,-0.797,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":25,"s":[153.556,409.91,0],"to":[0.028,0.797,0],"ti":[-0.013,-0.729,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[153.62,412.195,0],"to":[0.013,0.729,0],"ti":[0.005,-0.666,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[153.633,414.284,0],"to":[-0.005,0.666,0],"ti":[0.025,-0.607,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":28,"s":[153.589,416.19,0],"to":[-0.025,0.607,0],"ti":[0.045,-0.552,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":29,"s":[153.484,417.926,0],"to":[-0.045,0.552,0],"ti":[0.066,-0.501,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[153.316,419.502,0],"to":[-0.066,0.501,0],"ti":[0.086,-0.452,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":31,"s":[153.086,420.929,0],"to":[-0.086,0.452,0],"ti":[0.105,-0.407,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":32,"s":[152.797,422.216,0],"to":[-0.105,0.407,0],"ti":[0.122,-0.364,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":33,"s":[152.454,423.37,0],"to":[-0.122,0.364,0],"ti":[0.137,-0.324,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":34,"s":[152.063,424.4,0],"to":[-0.137,0.324,0],"ti":[0.149,-0.286,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":35,"s":[151.631,425.314,0],"to":[-0.149,0.286,0],"ti":[0.157,-0.251,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":36,"s":[151.17,426.118,0],"to":[-0.157,0.251,0],"ti":[0.161,-0.217,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":37,"s":[150.689,426.818,0],"to":[-0.161,0.217,0],"ti":[0.161,-0.186,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":38,"s":[150.202,427.422,0],"to":[-0.161,0.186,0],"ti":[0.156,-0.157,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":39,"s":[149.722,427.936,0],"to":[-0.156,0.157,0],"ti":[0.145,-0.13,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.179},"t":40,"s":[149.266,428.364,0],"to":[-0.145,0.13,0],"ti":[0.13,-0.103,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.184},"t":41,"s":[148.849,428.714,0],"to":[-0.13,0.103,0],"ti":[0.112,-0.075,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.187},"t":42,"s":[148.487,428.98,0],"to":[-0.112,0.075,0],"ti":[0.095,-0.052,0]},{"i":{"x":0.833,"y":0.853},"o":{"x":0.167,"y":0.188},"t":43,"s":[148.176,429.161,0],"to":[-0.095,0.052,0],"ti":[0.076,-0.036,0]},{"i":{"x":0.833,"y":0.858},"o":{"x":0.167,"y":0.193},"t":44,"s":[147.92,429.292,0],"to":[-0.076,0.036,0],"ti":[0.058,-0.02,0]},{"i":{"x":0.833,"y":0.864},"o":{"x":0.167,"y":0.202},"t":45,"s":[147.718,429.375,0],"to":[-0.058,0.02,0],"ti":[0.041,-0.005,0]},{"i":{"x":0.833,"y":0.858},"o":{"x":0.167,"y":0.214},"t":46,"s":[147.57,429.411,0],"to":[-0.041,0.005,0],"ti":[0.024,0.009,0]},{"i":{"x":0.833,"y":0.811},"o":{"x":0.167,"y":0.201},"t":47,"s":[147.473,429.404,0],"to":[-0.024,-0.009,0],"ti":[0.007,0.022,0]},{"i":{"x":0.833,"y":0.791},"o":{"x":0.167,"y":0.149},"t":48,"s":[147.427,429.356,0],"to":[-0.007,-0.022,0],"ti":[-0.008,0.035,0]},{"i":{"x":0.833,"y":0.803},"o":{"x":0.167,"y":0.139},"t":49,"s":[147.429,429.27,0],"to":[0.008,-0.035,0],"ti":[-0.023,0.046,0]},{"i":{"x":0.833,"y":0.812},"o":{"x":0.167,"y":0.144},"t":50,"s":[147.476,429.149,0],"to":[0.023,-0.046,0],"ti":[-0.037,0.056,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.15},"t":51,"s":[147.567,428.995,0],"to":[0.037,-0.056,0],"ti":[-0.05,0.065,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.153},"t":52,"s":[147.698,428.811,0],"to":[0.05,-0.065,0],"ti":[-0.061,0.073,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.156},"t":53,"s":[147.866,428.602,0],"to":[0.061,-0.073,0],"ti":[-0.072,0.08,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.158},"t":54,"s":[148.067,428.371,0],"to":[0.072,-0.08,0],"ti":[-0.081,0.086,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":55,"s":[148.297,428.121,0],"to":[0.081,-0.086,0],"ti":[-0.088,0.09,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":56,"s":[148.552,427.857,0],"to":[0.088,-0.09,0],"ti":[-0.094,0.093,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.163},"t":57,"s":[148.827,427.581,0],"to":[0.094,-0.093,0],"ti":[-0.099,0.094,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":58,"s":[149.118,427.299,0],"to":[0.099,-0.094,0],"ti":[-0.05,0.047,0]},{"t":59,"s":[149.419,427.014,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.986,0.936,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":2,"s":[78.01,99.254,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,0.531,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.017,-0.276,0]},"t":3,"s":[128.233,72.965,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.337,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.687,0.101,0]},"t":4,"s":[86.694,79.056,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.838,1.122,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,0.245,0]},"t":5,"s":[91.188,107.271,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.041,0.842,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.172,0.05,0]},"t":6,"s":[68.543,121.799,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.872,0.956,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,0.177,0]},"t":7,"s":[47.195,85.924,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.787,0.859,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.24,-0.094,0]},"t":8,"s":[79.057,53.992,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.016,0.785,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.137,0.203,0]},"t":9,"s":[95.971,69,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,0.823,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.013,0.136,0]},"t":10,"s":[122.24,79.447,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.768,0.977,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.111,0.157,0]},"t":11,"s":[91.043,95.99,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.026,0.806,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.13,-0.032,0]},"t":12,"s":[104.449,114.638,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,1.057,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.146,0]},"t":13,"s":[128.392,101.122,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.99,0.907,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.071,0.034,0]},"t":14,"s":[96.902,83.222,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.026,1.705,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.011,0.803,0]},"t":15,"s":[113.87,113.437,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.96,0.836,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.075,0]},"t":16,"s":[98.911,116.933,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.845,1.094,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.077,0.169,0]},"t":17,"s":[118.545,83.843,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.612,0.887,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.18,0.044,0]},"t":18,"s":[108.334,51.632,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.07,1.117,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.106,0.317,0]},"t":19,"s":[99.536,120.079,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.987,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.049,0]},"t":20,"s":[67.373,144.525,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.993,0.851,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.016,-0.19,0]},"t":21,"s":[126.372,85.891,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.911,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.008,0.19,0]},"t":22,"s":[76.632,103.785,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.779,0.97,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.349,0.009,0]},"t":23,"s":[122.04,117.809,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.843,1.086,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.075,-0.046,0]},"t":24,"s":[125.029,102.081,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.135,1.161,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.177,0.042,0]},"t":25,"s":[94.09,112.18,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.981,0.983,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.052,0.055,0]},"t":26,"s":[66.677,91.705,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,0.885,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.024,-0.021,0]},"t":27,"s":[138.63,151.835,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.1,0.917,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.153,0.304,0]},"t":28,"s":[82.794,103.616,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.825,23.963,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,-14.676,0]},"t":29,"s":[102.519,85.419,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.959,0.97,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.159,0.083,0]},"t":30,"s":[59.163,85.522,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.781,1.06,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.081,-0.046,0]},"t":31,"s":[11.254,57.108,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.97,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.134,0.035,0]},"t":32,"s":[35.57,75.366,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.91,0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.047,-0.198,0]},"t":33,"s":[75.263,44.021,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.16,1.441,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.081,-0.038,0]},"t":34,"s":[49.845,53.322,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.019,0.947,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.09,0.07,0]},"t":35,"s":[47.721,46.924,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,-0.142,0]},"t":36,"s":[20.292,87.156,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.992,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.252,-0.131,0]},"t":37,"s":[54.089,72.277,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.683,0.671,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.008,0.009,0]},"t":38,"s":[45.702,78.056,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.156,0.664,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,0.112,0]},"t":39,"s":[53.33,71.575,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.955,0.899,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,0.111,0]},"t":40,"s":[74.741,52.442,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.831,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.096,0.48,0]},"t":41,"s":[13.206,-5.553,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,1.051,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.164,-0.015,0]},"t":42,"s":[41.869,-17.736,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.774,1.016,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.027,0.032,0]},"t":43,"s":[71.33,-7.435,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.832,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.132,0.013,0]},"t":44,"s":[49.137,-24.086,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.882,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.056,0.165,0]},"t":45,"s":[11.219,-4.255,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.917,0.797,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.282,-0.239,0]},"t":46,"s":[33.923,16.004,100]},{"i":{"x":[0.833,0.833,0.833],"y":[26.628,1.515,1]},"o":{"x":[0.167,0.167,0.167],"y":[-41.355,0.141,0]},"t":47,"s":[43.431,10.768,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.023,0.936,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.083,0.072,0]},"t":48,"s":[43.412,3.234,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,0.76,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.077,-0.285,0]},"t":49,"s":[49.311,57.286,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.757,0.888,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.255,0.128,0]},"t":50,"s":[-29.011,45.06,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.807,1.116,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.127,0.328,0]},"t":51,"s":[-9.692,22.004,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.012,0.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.147,0.048,0]},"t":52,"s":[27.268,14.157,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.85,0.798,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.01,-0.607,0]},"t":53,"s":[75.93,32.89,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.972,0.796,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.187,0.142,0]},"t":54,"s":[20.351,30.629,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.765,1.805,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.041,0.141,0]},"t":55,"s":[-24.176,27.418,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.129,0.076,0]},"t":56,"s":[5.634,22.754,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.972,0.674,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.092,-0.394,0]},"t":57,"s":[59.79,72.494,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.043,0.112,0]},"t":58,"s":[34.087,63.804,100]},{"t":59,"s":[51.017,38.534,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":2,"op":60,"st":2,"bm":0},{"ddd":0,"ind":31,"ty":4,"nm":"Shape Layer 21","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":8,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":13,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":35,"s":[100]},{"t":51,"s":[0]}],"ix":11},"r":{"a":0,"k":92,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.611},"o":{"x":0.167,"y":0.167},"t":8,"s":[268.269,306.17,0],"to":[0.003,0.494,0],"ti":[-0.124,-2.305,0]},{"i":{"x":0.833,"y":0.766},"o":{"x":0.167,"y":0.106},"t":9,"s":[268.288,309.137,0],"to":[0.124,2.305,0],"ti":[-0.683,-5.046,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.129},"t":10,"s":[269.014,319.997,0],"to":[0.683,5.046,0],"ti":[-1.89,-6.606,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.159},"t":11,"s":[272.384,339.414,0],"to":[1.89,6.606,0],"ti":[-3.289,-5.823,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.179},"t":12,"s":[280.355,359.634,0],"to":[3.289,5.823,0],"ti":[-4.111,-3.921,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":13,"s":[292.117,374.353,0],"to":[4.111,3.921,0],"ti":[-4.169,-2.298,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":14,"s":[305.021,383.162,0],"to":[4.169,2.298,0],"ti":[-3.808,-1.326,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":15,"s":[317.131,388.142,0],"to":[3.808,1.326,0],"ti":[-3.36,-0.831,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":16,"s":[327.87,391.116,0],"to":[3.36,0.831,0],"ti":[-2.951,-0.601,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":17,"s":[337.294,393.129,0],"to":[2.951,0.601,0],"ti":[-2.599,-0.51,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":18,"s":[345.577,394.723,0],"to":[2.599,0.51,0],"ti":[-2.295,-0.495,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":19,"s":[352.885,396.19,0],"to":[2.295,0.495,0],"ti":[-2.032,-0.524,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":20,"s":[359.349,397.694,0],"to":[2.032,0.524,0],"ti":[-1.8,-0.577,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":21,"s":[365.075,399.331,0],"to":[1.8,0.577,0],"ti":[-1.589,-0.643,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":22,"s":[370.148,401.158,0],"to":[1.589,0.643,0],"ti":[-1.392,-0.71,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[374.61,403.19,0],"to":[1.392,0.71,0],"ti":[-1.21,-0.771,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":24,"s":[378.501,405.417,0],"to":[1.21,0.771,0],"ti":[-1.047,-0.823,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":25,"s":[381.873,407.817,0],"to":[1.047,0.823,0],"ti":[-0.897,-0.852,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.173},"t":26,"s":[384.78,410.354,0],"to":[0.897,0.852,0],"ti":[-0.758,-0.847,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[387.255,412.931,0],"to":[0.758,0.847,0],"ti":[-0.635,-0.817,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":28,"s":[389.33,415.438,0],"to":[0.635,0.817,0],"ti":[-0.53,-0.777,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":29,"s":[391.063,417.835,0],"to":[0.53,0.777,0],"ti":[-0.442,-0.729,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[392.509,420.098,0],"to":[0.442,0.729,0],"ti":[-0.37,-0.677,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":31,"s":[393.717,422.209,0],"to":[0.37,0.677,0],"ti":[-0.311,-0.623,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":32,"s":[394.73,424.158,0],"to":[0.311,0.623,0],"ti":[-0.263,-0.569,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":33,"s":[395.584,425.945,0],"to":[0.263,0.569,0],"ti":[-0.224,-0.517,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":34,"s":[396.309,427.572,0],"to":[0.224,0.517,0],"ti":[-0.192,-0.467,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":35,"s":[396.928,429.046,0],"to":[0.192,0.467,0],"ti":[-0.166,-0.419,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":36,"s":[397.461,430.372,0],"to":[0.166,0.419,0],"ti":[-0.144,-0.373,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":37,"s":[397.921,431.558,0],"to":[0.144,0.373,0],"ti":[-0.125,-0.33,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":38,"s":[398.322,432.612,0],"to":[0.125,0.33,0],"ti":[-0.11,-0.29,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":39,"s":[398.672,433.54,0],"to":[0.11,0.29,0],"ti":[-0.096,-0.253,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":40,"s":[398.98,434.351,0],"to":[0.096,0.253,0],"ti":[-0.085,-0.219,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":41,"s":[399.251,435.056,0],"to":[0.085,0.219,0],"ti":[-0.075,-0.187,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":42,"s":[399.491,435.663,0],"to":[0.075,0.187,0],"ti":[-0.066,-0.157,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.183},"t":43,"s":[399.701,436.178,0],"to":[0.066,0.157,0],"ti":[-0.058,-0.129,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.185},"t":44,"s":[399.886,436.606,0],"to":[0.058,0.129,0],"ti":[-0.05,-0.103,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.188},"t":45,"s":[400.047,436.953,0],"to":[0.05,0.103,0],"ti":[-0.042,-0.078,0]},{"i":{"x":0.833,"y":0.858},"o":{"x":0.167,"y":0.194},"t":46,"s":[400.184,437.223,0],"to":[0.042,0.078,0],"ti":[-0.035,-0.055,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.203},"t":47,"s":[400.3,437.422,0],"to":[0.035,0.055,0],"ti":[-0.029,-0.04,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.183},"t":48,"s":[400.393,437.552,0],"to":[0.029,0.04,0],"ti":[-0.025,-0.04,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.158},"t":49,"s":[400.471,437.661,0],"to":[0.025,0.04,0],"ti":[-0.022,-0.048,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.158},"t":50,"s":[400.541,437.792,0],"to":[0.022,0.048,0],"ti":[-0.019,-0.055,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.159},"t":51,"s":[400.602,437.946,0],"to":[0.019,0.055,0],"ti":[-0.015,-0.062,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.159},"t":52,"s":[400.653,438.122,0],"to":[0.015,0.062,0],"ti":[-0.012,-0.069,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.16},"t":53,"s":[400.694,438.32,0],"to":[0.012,0.069,0],"ti":[-0.008,-0.076,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.16},"t":54,"s":[400.724,438.538,0],"to":[0.008,0.076,0],"ti":[-0.004,-0.082,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.161},"t":55,"s":[400.743,438.775,0],"to":[0.004,0.082,0],"ti":[0.001,-0.09,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.157},"t":56,"s":[400.751,439.03,0],"to":[-0.001,0.09,0],"ti":[0.01,-0.102,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.156},"t":57,"s":[400.737,439.316,0],"to":[-0.01,0.102,0],"ti":[0.019,-0.113,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.16},"t":58,"s":[400.693,439.644,0],"to":[-0.019,0.113,0],"ti":[0.011,-0.059,0]},{"t":59,"s":[400.624,439.996,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[-5.266,0.476,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":8,"s":[101.238,48.218,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.994,0.881,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.084,0.099,0]},"t":9,"s":[100.863,56.49,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.87,1.004,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.006,0.279,0]},"t":10,"s":[73.006,100.241,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.716,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.233,0.004,0]},"t":11,"s":[98.954,118.86,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.692,0.993,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.185,0.118,0]},"t":12,"s":[113.422,99.365,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.95,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.114,-0.008,0]},"t":13,"s":[108.921,52.336,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.185,0.915,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.07,-0.128,0]},"t":14,"s":[96.758,95.263,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.076,1.566,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.057,3.934,0]},"t":15,"s":[103.361,78.328,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,0.67,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,0.073,0]},"t":16,"s":[82.062,77.962,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.435,0.477,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.254,0.111,0]},"t":17,"s":[122.901,80.818,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.986,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.098,0.099,0]},"t":18,"s":[112.827,89.288,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.908,0.891,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.016,0.009,0]},"t":19,"s":[54.614,133.963,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,1.123,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.871,0.351,0]},"t":20,"s":[103.222,83.744,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.079,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.032,0.05,0]},"t":21,"s":[108.364,68.125,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.812,-0.393,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,2.172,0]},"t":22,"s":[104.662,106.75,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.481,1.162,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.15,0.089,0]},"t":23,"s":[111.862,108.291,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.008,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.055,0]},"t":24,"s":[120.907,132.514,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.892,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,-0.052,0]},"t":25,"s":[59.613,61.164,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.83,1.094,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.141,0.365,0]},"t":26,"s":[126.782,105.209,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,1.033,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.163,0.044,0]},"t":27,"s":[101.882,118.255,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.119,1.091,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.059,0.024,0]},"t":28,"s":[75.979,90.498,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.983,0.951,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.049,0.043,0]},"t":29,"s":[91.099,129.32,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.873,0.878,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.022,-0.118,0]},"t":30,"s":[54.42,48.147,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.036,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.244,0.263,0]},"t":31,"s":[83.4,81.807,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,2.819,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,1.369,0]},"t":32,"s":[98.473,97.441,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-4.522,1.004,1]},"o":{"x":[0.167,0.167,0.167],"y":[3.172,0.08,0]},"t":33,"s":[76.975,98.454,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,1.005,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.085,0.004,0]},"t":34,"s":[76.395,75.312,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.837,0.918,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.032,0.005,0]},"t":35,"s":[38.536,99.649,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.978,2.101,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.171,-4.786,0]},"t":36,"s":[65.888,73.73,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.051,1.151,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.03,0.077,0]},"t":37,"s":[91.971,74.174,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.063,1.136,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.032,0.054,0]},"t":38,"s":[72.715,67.868,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.933,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.036,0.052,0]},"t":39,"s":[103.785,85.634,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.65,1.416,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.353,1.245,0]},"t":40,"s":[49.165,38.912,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,0.946,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.109,0.069,0]},"t":41,"s":[59.594,35.561,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.678,0.628,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.154,-0.156,0]},"t":42,"s":[92.977,55.629,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,0.983,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.112,0.107,0]},"t":43,"s":[81.252,48.654,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.985,0.899,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,-0.021,0]},"t":44,"s":[47.617,24.458,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,1.627,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.018,0.474,0]},"t":45,"s":[16.653,43.696,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.225,0.074,0]},"t":46,"s":[42.187,47.804,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.962,0.963,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.07,-0.023,0]},"t":47,"s":[35.289,12.804,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.877,0.815,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.077,-0.066,0]},"t":48,"s":[39.035,40.12,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.027,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.257,0.151,0]},"t":49,"s":[-7.944,24.911,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.741,1.341,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.223,0]},"t":50,"s":[-30.445,6.276,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.123,0.067,0]},"t":51,"s":[-0.773,-4.844,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.692,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.147,0.394,0]},"t":52,"s":[61.907,51.792,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,0.468,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.114,-0.024,0]},"t":53,"s":[39.231,67.006,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.829,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.035,0.099,0]},"t":54,"s":[-21.899,55.157,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.981,0.881,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.163,-0.053,0]},"t":55,"s":[21.199,-8.63,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.959,1.022,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.025,0.276,0]},"t":56,"s":[66.363,30.41,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.882,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.08,0.018,0]},"t":57,"s":[31.738,47.268,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.182,0.283,0]},"t":58,"s":[49.437,25.928,100]},{"t":59,"s":[43.876,17.029,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":8,"op":60,"st":8,"bm":0},{"ddd":0,"ind":32,"ty":4,"nm":"Shape Layer 20","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":6,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":33,"s":[100]},{"t":49,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.574},"o":{"x":0.167,"y":0.167},"t":6,"s":[273.987,258.212,0],"to":[-0.226,0.261,0],"ti":[1.235,-1.262,0]},{"i":{"x":0.833,"y":0.763},"o":{"x":0.167,"y":0.104},"t":7,"s":[272.634,259.781,0],"to":[-1.235,1.262,0],"ti":[3.172,-2.472,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":8,"s":[266.579,265.787,0],"to":[-3.172,2.472,0],"ti":[4.928,-2.338,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":9,"s":[253.605,274.613,0],"to":[-4.928,2.338,0],"ti":[5.281,-0.789,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":10,"s":[237.011,279.814,0],"to":[-5.281,0.789,0],"ti":[4.495,0.772,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":11,"s":[221.916,279.349,0],"to":[-4.495,-0.772,0],"ti":[3.459,1.674,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":12,"s":[210.041,275.183,0],"to":[-3.459,-1.674,0],"ti":[2.573,2.049,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":13,"s":[201.165,269.307,0],"to":[-2.573,-2.049,0],"ti":[1.908,2.132,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":14,"s":[194.6,262.891,0],"to":[-1.908,-2.132,0],"ti":[1.43,2.079,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":15,"s":[189.714,256.512,0],"to":[-1.43,-2.079,0],"ti":[1.088,1.968,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":16,"s":[186.021,250.417,0],"to":[-1.088,-1.968,0],"ti":[0.842,1.835,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[183.185,244.707,0],"to":[-0.842,-1.835,0],"ti":[0.663,1.699,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[180.968,239.407,0],"to":[-0.663,-1.699,0],"ti":[0.532,1.568,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":19,"s":[179.205,234.511,0],"to":[-0.532,-1.568,0],"ti":[0.434,1.443,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":20,"s":[177.777,230,0],"to":[-0.434,-1.443,0],"ti":[0.361,1.326,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":21,"s":[176.6,225.853,0],"to":[-0.361,-1.326,0],"ti":[0.305,1.218,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":22,"s":[175.612,222.043,0],"to":[-0.305,-1.218,0],"ti":[0.263,1.116,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[174.767,218.548,0],"to":[-0.263,-1.116,0],"ti":[0.231,1.022,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":24,"s":[174.033,215.345,0],"to":[-0.231,-1.022,0],"ti":[0.205,0.934,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":25,"s":[173.383,212.416,0],"to":[-0.205,-0.934,0],"ti":[0.185,0.851,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[172.801,209.743,0],"to":[-0.185,-0.851,0],"ti":[0.169,0.773,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[172.273,207.311,0],"to":[-0.169,-0.773,0],"ti":[0.155,0.7,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":28,"s":[171.79,205.104,0],"to":[-0.155,-0.7,0],"ti":[0.143,0.63,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":29,"s":[171.345,203.111,0],"to":[-0.143,-0.63,0],"ti":[0.131,0.565,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":30,"s":[170.934,201.322,0],"to":[-0.131,-0.565,0],"ti":[0.12,0.503,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":31,"s":[170.557,199.723,0],"to":[-0.12,-0.503,0],"ti":[0.109,0.444,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":32,"s":[170.213,198.306,0],"to":[-0.109,-0.444,0],"ti":[0.097,0.389,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":33,"s":[169.903,197.059,0],"to":[-0.097,-0.389,0],"ti":[0.085,0.336,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":34,"s":[169.629,195.974,0],"to":[-0.085,-0.336,0],"ti":[0.07,0.287,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.182},"t":35,"s":[169.396,195.042,0],"to":[-0.07,-0.287,0],"ti":[0.055,0.24,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.185},"t":36,"s":[169.206,194.254,0],"to":[-0.055,-0.24,0],"ti":[0.037,0.195,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.188},"t":37,"s":[169.067,193.604,0],"to":[-0.037,-0.195,0],"ti":[0.018,0.153,0]},{"i":{"x":0.833,"y":0.856},"o":{"x":0.167,"y":0.193},"t":38,"s":[168.983,193.083,0],"to":[-0.018,-0.153,0],"ti":[-0.004,0.114,0]},{"i":{"x":0.833,"y":0.855},"o":{"x":0.167,"y":0.199},"t":39,"s":[168.961,192.685,0],"to":[0.004,-0.114,0],"ti":[-0.028,0.076,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.195},"t":40,"s":[169.007,192.402,0],"to":[0.028,-0.076,0],"ti":[-0.046,0.036,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.19},"t":41,"s":[169.128,192.228,0],"to":[0.046,-0.036,0],"ti":[-0.055,-0.006,0]},{"i":{"x":0.833,"y":0.805},"o":{"x":0.167,"y":0.156},"t":42,"s":[169.284,192.183,0],"to":[0.055,0.006,0],"ti":[-0.06,-0.042,0]},{"i":{"x":0.833,"y":0.812},"o":{"x":0.167,"y":0.145},"t":43,"s":[169.457,192.261,0],"to":[0.06,0.042,0],"ti":[-0.063,-0.073,0]},{"i":{"x":0.833,"y":0.819},"o":{"x":0.167,"y":0.15},"t":44,"s":[169.642,192.438,0],"to":[0.063,0.073,0],"ti":[-0.065,-0.097,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.154},"t":45,"s":[169.834,192.698,0],"to":[0.065,0.097,0],"ti":[-0.066,-0.11,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.164},"t":46,"s":[170.032,193.022,0],"to":[0.066,0.11,0],"ti":[-0.066,-0.107,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.173},"t":47,"s":[170.231,193.359,0],"to":[0.066,0.107,0],"ti":[-0.066,-0.094,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.176},"t":48,"s":[170.43,193.661,0],"to":[0.066,0.094,0],"ti":[-0.064,-0.077,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.18},"t":49,"s":[170.626,193.92,0],"to":[0.064,0.077,0],"ti":[-0.062,-0.058,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":50,"s":[170.816,194.125,0],"to":[0.062,0.058,0],"ti":[-0.059,-0.037,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.185},"t":51,"s":[170.999,194.269,0],"to":[0.059,0.037,0],"ti":[-0.056,-0.014,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.18},"t":52,"s":[171.172,194.347,0],"to":[0.056,0.014,0],"ti":[-0.052,0.01,0]},{"i":{"x":0.833,"y":0.816},"o":{"x":0.167,"y":0.165},"t":53,"s":[171.335,194.353,0],"to":[0.052,-0.01,0],"ti":[-0.048,0.036,0]},{"i":{"x":0.833,"y":0.811},"o":{"x":0.167,"y":0.153},"t":54,"s":[171.485,194.284,0],"to":[0.048,-0.036,0],"ti":[-0.043,0.061,0]},{"i":{"x":0.833,"y":0.812},"o":{"x":0.167,"y":0.149},"t":55,"s":[171.621,194.138,0],"to":[0.043,-0.061,0],"ti":[-0.038,0.087,0]},{"i":{"x":0.833,"y":0.816},"o":{"x":0.167,"y":0.15},"t":56,"s":[171.743,193.915,0],"to":[0.038,-0.087,0],"ti":[-0.033,0.112,0]},{"i":{"x":0.833,"y":0.819},"o":{"x":0.167,"y":0.152},"t":57,"s":[171.851,193.615,0],"to":[0.033,-0.112,0],"ti":[-0.028,0.137,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.154},"t":58,"s":[171.943,193.241,0],"to":[0.028,-0.137,0],"ti":[-0.013,0.074,0]},{"t":59,"s":[172.02,192.796,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.05,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":6,"s":[113.755,94.294,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.873,0.226,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.031,-0.701,0]},"t":7,"s":[142.324,42.5,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.888,0.932,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.241,0.093,0]},"t":8,"s":[96.482,48.003,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.204,0.806,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.325,-0.366,0]},"t":9,"s":[72.3,93.591,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.921,0.58,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.059,0.146,0]},"t":10,"s":[63.951,85.144,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.459,0.97,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.529,0.104,0]},"t":11,"s":[92.765,73.936,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.995,0.876,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.088,-0.048,0]},"t":12,"s":[91.276,28.604,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.992,1.032,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.006,0.254,0]},"t":13,"s":[66.684,57.392,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.995,1.013,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.008,0.023,0]},"t":14,"s":[89.679,71.451,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.731,0.864,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.005,0.012,0]},"t":15,"s":[68.777,51.94,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.035,0.702,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.121,0.214,0]},"t":16,"s":[88.421,74.596,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.935,0.98,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,0.116,0]},"t":17,"s":[132.23,89.048,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.575,0.845,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.305,-0.027,0]},"t":18,"s":[69.974,126.315,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.931,0.98,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.104,0.18,0]},"t":19,"s":[83.338,98.135,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.313,0.783,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.391,-0.026,0]},"t":20,"s":[138.117,73.778,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.993,0.847,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.095,0.135,0]},"t":21,"s":[128.488,92.265,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.953,1.145,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.008,0.183,0]},"t":22,"s":[58.681,121.955,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.815,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.109,0.053,0]},"t":23,"s":[122.512,146.761,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,5.116,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.152,17.285,0]},"t":24,"s":[94.854,78.682,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.02,1.177,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,0.082,0]},"t":25,"s":[61.077,78.352,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,1.017,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,0.057,0]},"t":26,"s":[91.084,94.97,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.068,0.014,0]},"t":27,"s":[53.825,43.095,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.888,0.893,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.186,-0.051,0]},"t":28,"s":[74.356,105.62,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.814,1.146,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.324,0.371,0]},"t":29,"s":[91.09,66.721,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.498,0.904,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.151,0.053,0]},"t":30,"s":[96.876,55.445,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.992,1.823,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.645,0]},"t":31,"s":[104.026,86.476,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.986,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.009,0.076,0]},"t":32,"s":[54.137,91.081,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.906,0.944,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.017,-0.393,0]},"t":33,"s":[99.353,40.985,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.666,0.653,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.764,-0.167,0]},"t":34,"s":[61.794,49.744,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.993,0.855,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,0.11,0]},"t":35,"s":[57.194,46.834,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.011,1.186,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.008,0.195,0]},"t":36,"s":[98.549,37.625,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.007,1.031,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.009,0.058,0]},"t":37,"s":[60.683,30.783,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.882,0.935,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,0.023,0]},"t":38,"s":[103.351,52.891,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.098,0.657,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.282,-0.288,0]},"t":39,"s":[56.948,22.459,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.953,0.83,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,0.11,0]},"t":40,"s":[37.48,29.284,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.913,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.106,0.164,0]},"t":41,"s":[79.938,50.562,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-5.816,1.008,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.804,-0.017,0]},"t":42,"s":[61.287,72.597,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,1.042,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.084,0.008,0]},"t":43,"s":[60.383,54.282,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.87,0.901,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.155,0.028,0]},"t":44,"s":[-12.609,74.447,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.795,0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.231,0.53,0]},"t":45,"s":[12.933,44.106,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.094,1.748,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,-0.037,0]},"t":46,"s":[27.371,38.441,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,0.881,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,0.075,0]},"t":47,"s":[48.387,42.359,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.231,0.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.898,0.279,0]},"t":48,"s":[3.703,3.258,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.188,-0.77,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,-0.58,0]},"t":49,"s":[7.498,-13.404,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.899,0.864,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,0.087,0]},"t":50,"s":[-6.825,-11.313,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.382,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.485,0.214,0]},"t":51,"s":[39.727,31.017,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,1.572,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.068,0.386,0]},"t":52,"s":[49.384,58.012,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.795,0.914,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.055,0.073,0]},"t":53,"s":[-4.528,65.442,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,1.691,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,3.173,0]},"t":54,"s":[27.852,6.982,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.907,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.001,0.074,0]},"t":55,"s":[75.027,5.405,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,0.463,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.146,0.773,0]},"t":56,"s":[28.477,20.053,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.164,1.39,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.224,0.099,0]},"t":57,"s":[45.412,21.822,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.069,0]},"t":58,"s":[55.436,31.442,100]},{"t":59,"s":[25.653,-23.168,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":6,"op":60,"st":6,"bm":0},{"ddd":0,"ind":33,"ty":4,"nm":"Shape Layer 19","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":16,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":38,"s":[100]},{"t":54,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.51},"o":{"x":0.167,"y":0.167},"t":11,"s":[273.717,253.598,0],"to":[0.28,-0.028,0],"ti":[-1.664,-0.082,0]},{"i":{"x":0.833,"y":0.759},"o":{"x":0.167,"y":0.1},"t":12,"s":[275.395,253.428,0],"to":[1.664,0.082,0],"ti":[-3.922,-0.767,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.127},"t":13,"s":[283.7,254.091,0],"to":[3.922,0.767,0],"ti":[-5.158,-1.975,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":14,"s":[298.929,258.033,0],"to":[5.158,1.975,0],"ti":[-4.6,-2.946,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":15,"s":[314.649,265.939,0],"to":[4.6,2.946,0],"ti":[-3.328,-3.296,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":16,"s":[326.529,275.709,0],"to":[3.328,3.296,0],"ti":[-2.233,-3.252,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":17,"s":[334.618,285.714,0],"to":[2.233,3.252,0],"ti":[-1.455,-3.046,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.18},"t":18,"s":[339.929,295.223,0],"to":[1.455,3.046,0],"ti":[-0.931,-2.79,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":19,"s":[343.35,303.988,0],"to":[0.931,2.79,0],"ti":[-0.582,-2.533,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":20,"s":[345.513,311.964,0],"to":[0.582,2.533,0],"ti":[-0.353,-2.293,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[346.843,319.188,0],"to":[0.353,2.293,0],"ti":[-0.205,-2.077,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[347.63,325.723,0],"to":[0.205,2.077,0],"ti":[-0.115,-1.887,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[348.075,331.651,0],"to":[0.115,1.887,0],"ti":[-0.064,-1.719,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":24,"s":[348.319,337.046,0],"to":[0.064,1.719,0],"ti":[-0.041,-1.573,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":25,"s":[348.461,341.968,0],"to":[0.041,1.573,0],"ti":[-0.028,-1.452,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":26,"s":[348.564,346.482,0],"to":[0.028,1.452,0],"ti":[-0.019,-1.351,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":27,"s":[348.628,350.68,0],"to":[0.019,1.351,0],"ti":[-0.016,-1.256,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":28,"s":[348.676,354.586,0],"to":[0.016,1.256,0],"ti":[-0.018,-1.167,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":29,"s":[348.724,358.218,0],"to":[0.018,1.167,0],"ti":[-0.022,-1.082,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":30,"s":[348.783,361.589,0],"to":[0.022,1.082,0],"ti":[-0.029,-1,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":31,"s":[348.858,364.71,0],"to":[0.029,1,0],"ti":[-0.036,-0.921,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":32,"s":[348.955,367.59,0],"to":[0.036,0.921,0],"ti":[-0.043,-0.844,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":33,"s":[349.073,370.236,0],"to":[0.043,0.844,0],"ti":[-0.05,-0.768,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":34,"s":[349.212,372.652,0],"to":[0.05,0.768,0],"ti":[-0.056,-0.694,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":35,"s":[349.372,374.845,0],"to":[0.056,0.694,0],"ti":[-0.062,-0.622,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":36,"s":[349.55,376.819,0],"to":[0.062,0.622,0],"ti":[-0.067,-0.551,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":37,"s":[349.743,378.577,0],"to":[0.067,0.551,0],"ti":[-0.07,-0.481,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":38,"s":[349.95,380.124,0],"to":[0.07,0.481,0],"ti":[-0.073,-0.413,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":39,"s":[350.166,381.464,0],"to":[0.073,0.413,0],"ti":[-0.075,-0.348,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":40,"s":[350.389,382.603,0],"to":[0.075,0.348,0],"ti":[-0.075,-0.284,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.186},"t":41,"s":[350.614,383.549,0],"to":[0.075,0.284,0],"ti":[-0.075,-0.222,0]},{"i":{"x":0.833,"y":0.855},"o":{"x":0.167,"y":0.19},"t":42,"s":[350.84,384.306,0],"to":[0.075,0.222,0],"ti":[-0.073,-0.163,0]},{"i":{"x":0.833,"y":0.859},"o":{"x":0.167,"y":0.196},"t":43,"s":[351.062,384.882,0],"to":[0.073,0.163,0],"ti":[-0.071,-0.106,0]},{"i":{"x":0.833,"y":0.86},"o":{"x":0.167,"y":0.204},"t":44,"s":[351.279,385.285,0],"to":[0.071,0.106,0],"ti":[-0.068,-0.053,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.205},"t":45,"s":[351.487,385.521,0],"to":[0.068,0.053,0],"ti":[-0.064,-0.002,0]},{"i":{"x":0.833,"y":0.806},"o":{"x":0.167,"y":0.173},"t":46,"s":[351.685,385.601,0],"to":[0.064,0.002,0],"ti":[-0.059,0.045,0]},{"i":{"x":0.833,"y":0.804},"o":{"x":0.167,"y":0.146},"t":47,"s":[351.87,385.535,0],"to":[0.059,-0.045,0],"ti":[-0.054,0.088,0]},{"i":{"x":0.833,"y":0.811},"o":{"x":0.167,"y":0.145},"t":48,"s":[352.04,385.333,0],"to":[0.054,-0.088,0],"ti":[-0.049,0.127,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.149},"t":49,"s":[352.195,385.007,0],"to":[0.049,-0.127,0],"ti":[-0.043,0.161,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.153},"t":50,"s":[352.331,384.571,0],"to":[0.043,-0.161,0],"ti":[-0.035,0.185,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.161},"t":51,"s":[352.45,384.039,0],"to":[0.035,-0.185,0],"ti":[-0.024,0.192,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":52,"s":[352.541,383.461,0],"to":[0.024,-0.192,0],"ti":[-0.013,0.188,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.17},"t":53,"s":[352.596,382.887,0],"to":[0.013,-0.188,0],"ti":[-0.001,0.179,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.172},"t":54,"s":[352.617,382.333,0],"to":[0.001,-0.179,0],"ti":[0.007,0.165,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":55,"s":[352.603,381.814,0],"to":[-0.007,-0.165,0],"ti":[0.003,0.149,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":56,"s":[352.574,381.345,0],"to":[-0.003,-0.149,0],"ti":[-0.011,0.133,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":57,"s":[352.587,380.922,0],"to":[0.011,-0.133,0],"ti":[-0.023,0.117,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.177},"t":58,"s":[352.639,380.546,0],"to":[0.023,-0.117,0],"ti":[-0.014,0.055,0]},{"t":59,"s":[352.724,380.217,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[2.45,-1.622,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":11,"s":[64.676,29.575,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.851,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,0.086,0]},"t":12,"s":[63.619,31.654,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.096,0.849,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.189,-0.115,0]},"t":13,"s":[83.082,94.984,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.044,1.052,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,0.186,0]},"t":14,"s":[98.428,68.346,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.791,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.029,0.032,0]},"t":15,"s":[65.326,46.692,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.846,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.117,0.139,0]},"t":16,"s":[115.976,81.786,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.191,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.181,-0.001,0]},"t":17,"s":[94.867,134.538,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,1.096,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,0.505,0]},"t":18,"s":[76.907,82.463,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.592,0.774,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.576,0.045,0]},"t":19,"s":[136.087,72.163,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.95,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.132,0]},"t":20,"s":[146.093,94.38,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,1.06,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.126,0.245,0]},"t":21,"s":[64.998,132.461,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.163,0.877,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,0.035,0]},"t":22,"s":[97.241,152.104,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.936,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.259,0]},"t":23,"s":[126.81,118.253,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.87,0.449,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.273,0.246,0]},"t":24,"s":[39.419,102.155,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.013,0.912,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.231,0.098,0]},"t":25,"s":[59.845,93.89,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.068,2.263,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.011,1.504,0]},"t":26,"s":[71.34,47.538,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.87,0.951,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.037,0.078,0]},"t":27,"s":[58.098,44.818,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.848,0.781,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.234,-0.119,0]},"t":28,"s":[82.135,88.74,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.37,1.069,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.185,0.135,0]},"t":29,"s":[95.452,70.689,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,0.946,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.068,0.038,0]},"t":30,"s":[106.399,41.339,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.931,0.784,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,-0.156,0]},"t":31,"s":[46.837,95.066,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,0.989,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.417,0.136,0]},"t":32,"s":[100.241,76.371,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.202,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.224,-0.013,0]},"t":33,"s":[91.348,46.624,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.96,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.059,-0.017,0]},"t":34,"s":[86.082,72.273,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.193,0.684,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.069,-0.076,0]},"t":35,"s":[104.113,51.031,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.964,0.99,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,0.113,0]},"t":36,"s":[94.281,62.125,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.717,0.85,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.063,-0.012,0]},"t":37,"s":[126.942,93.089,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.99,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.118,0.187,0]},"t":38,"s":[108.331,65.944,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.969,0.494,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.012,-0.318,0]},"t":39,"s":[63.723,44.138,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.903,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.049,0.1,0]},"t":40,"s":[102.921,48.671,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.464,0.113,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.147,0.604,0]},"t":41,"s":[78.268,71.671,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.902,1.068,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.092,0]},"t":42,"s":[87.176,75.352,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.234,0.908,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.554,0.037,0]},"t":43,"s":[28.705,110.854,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.845,1.441,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,0.901,0]},"t":44,"s":[18.357,46.477,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.091,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.181,0.07,0]},"t":45,"s":[57.758,39.913,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.801,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,-0.023,0]},"t":46,"s":[91.489,81.222,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.953,0.932,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.137,0.143,0]},"t":47,"s":[20.878,48.853,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.976,0.994,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.108,-0.379,0]},"t":48,"s":[47.568,3.824,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.831,1.042,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.035,-0.006,0]},"t":49,"s":[35.966,11.947,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.412,0.592,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.165,0.028,0]},"t":50,"s":[44.168,4.41,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,0.978,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,0.105,0]},"t":51,"s":[52.575,15.778,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.454,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.254,-0.03,0]},"t":52,"s":[2.557,60.137,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.959,1.407,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.098,-0.624,0]},"t":53,"s":[14.899,27.604,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,1.066,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.082,0.069,0]},"t":54,"s":[83.445,31.436,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.046,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.186,0.037,0]},"t":55,"s":[48.803,8.908,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,0.694,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.03,-0.323,0]},"t":56,"s":[20.805,49.183,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.959,0.832,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.065,0.115,0]},"t":57,"s":[64.315,40.919,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.079,0.165,0]},"t":58,"s":[39.92,18.864,100]},{"t":59,"s":[52.411,-3.555,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.167325382607,0.716982433843,1,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":11,"op":60,"st":11,"bm":0},{"ddd":0,"ind":34,"ty":4,"nm":"Shape Layer 18","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":17,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":39,"s":[100]},{"t":55,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.544},"o":{"x":0.167,"y":0.167},"t":12,"s":[267.865,255.269,0],"to":[-0.161,-0.402,0],"ti":[1.026,2.138,0]},{"i":{"x":0.833,"y":0.761},"o":{"x":0.167,"y":0.102},"t":13,"s":[266.901,252.858,0],"to":[-1.026,-2.138,0],"ti":[2.899,4.739,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.128},"t":14,"s":[261.706,242.443,0],"to":[-2.899,-4.739,0],"ti":[4.94,5.799,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":15,"s":[249.506,224.427,0],"to":[-4.94,-5.799,0],"ti":[5.963,4.5,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":16,"s":[232.069,207.647,0],"to":[-5.963,-4.5,0],"ti":[5.888,2.374,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":17,"s":[213.727,197.425,0],"to":[-5.888,-2.374,0],"ti":[5.276,0.563,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":18,"s":[196.743,193.401,0],"to":[-5.276,-0.563,0],"ti":[4.452,-0.741,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":19,"s":[182.074,194.046,0],"to":[-4.452,0.741,0],"ti":[3.589,-1.577,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":20,"s":[170.029,197.845,0],"to":[-3.589,1.577,0],"ti":[2.794,-2.037,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":21,"s":[160.537,203.508,0],"to":[-2.794,2.037,0],"ti":[2.125,-2.229,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":22,"s":[153.265,210.066,0],"to":[-2.125,2.229,0],"ti":[1.595,-2.253,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":23,"s":[147.784,216.885,0],"to":[-1.595,2.253,0],"ti":[1.188,-2.18,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":24,"s":[143.692,223.586,0],"to":[-1.188,2.18,0],"ti":[0.88,-2.056,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":25,"s":[140.657,229.963,0],"to":[-0.88,2.056,0],"ti":[0.65,-1.912,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":26,"s":[138.413,235.924,0],"to":[-0.65,1.912,0],"ti":[0.477,-1.76,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":27,"s":[136.76,241.434,0],"to":[-0.477,1.76,0],"ti":[0.347,-1.607,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":28,"s":[135.553,246.481,0],"to":[-0.347,1.607,0],"ti":[0.25,-1.458,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":29,"s":[134.678,251.075,0],"to":[-0.25,1.458,0],"ti":[0.178,-1.316,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":30,"s":[134.053,255.23,0],"to":[-0.178,1.316,0],"ti":[0.125,-1.182,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":31,"s":[133.611,258.969,0],"to":[-0.125,1.182,0],"ti":[0.088,-1.057,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":32,"s":[133.302,262.321,0],"to":[-0.088,1.057,0],"ti":[0.062,-0.942,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":33,"s":[133.085,265.313,0],"to":[-0.062,0.942,0],"ti":[0.045,-0.835,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":34,"s":[132.931,267.971,0],"to":[-0.045,0.835,0],"ti":[0.035,-0.737,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":35,"s":[132.816,270.322,0],"to":[-0.035,0.737,0],"ti":[0.031,-0.648,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":36,"s":[132.72,272.392,0],"to":[-0.031,0.648,0],"ti":[0.031,-0.567,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":37,"s":[132.63,274.209,0],"to":[-0.031,0.567,0],"ti":[0.035,-0.496,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":38,"s":[132.533,275.797,0],"to":[-0.035,0.496,0],"ti":[0.04,-0.432,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":39,"s":[132.422,277.182,0],"to":[-0.04,0.432,0],"ti":[0.043,-0.377,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":40,"s":[132.291,278.392,0],"to":[-0.043,0.377,0],"ti":[0.036,-0.328,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.18},"t":41,"s":[132.166,279.447,0],"to":[-0.036,0.328,0],"ti":[0.026,-0.282,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.181},"t":42,"s":[132.075,280.359,0],"to":[-0.026,0.282,0],"ti":[0.017,-0.24,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.182},"t":43,"s":[132.011,281.139,0],"to":[-0.017,0.24,0],"ti":[0.01,-0.201,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.184},"t":44,"s":[131.972,281.799,0],"to":[-0.01,0.201,0],"ti":[0.003,-0.166,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.186},"t":45,"s":[131.954,282.347,0],"to":[-0.003,0.166,0],"ti":[-0.002,-0.133,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.189},"t":46,"s":[131.953,282.792,0],"to":[0.002,0.133,0],"ti":[-0.007,-0.102,0]},{"i":{"x":0.833,"y":0.859},"o":{"x":0.167,"y":0.195},"t":47,"s":[131.967,283.142,0],"to":[0.007,0.102,0],"ti":[-0.011,-0.073,0]},{"i":{"x":0.833,"y":0.866},"o":{"x":0.167,"y":0.204},"t":48,"s":[131.994,283.403,0],"to":[0.011,0.073,0],"ti":[-0.014,-0.046,0]},{"i":{"x":0.833,"y":0.869},"o":{"x":0.167,"y":0.222},"t":49,"s":[132.031,283.58,0],"to":[0.014,0.046,0],"ti":[-0.017,-0.02,0]},{"i":{"x":0.833,"y":0.811},"o":{"x":0.167,"y":0.23},"t":50,"s":[132.078,283.677,0],"to":[0.017,0.02,0],"ti":[-0.019,0.004,0]},{"i":{"x":0.833,"y":0.815},"o":{"x":0.167,"y":0.149},"t":51,"s":[132.132,283.7,0],"to":[0.019,-0.004,0],"ti":[-0.021,0.02,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.152},"t":52,"s":[132.193,283.651,0],"to":[0.021,-0.02,0],"ti":[-0.023,0.018,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.18},"t":53,"s":[132.259,283.583,0],"to":[0.023,-0.018,0],"ti":[-0.025,0.008,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.172},"t":54,"s":[132.331,283.544,0],"to":[0.025,-0.008,0],"ti":[-0.026,0,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.164},"t":55,"s":[132.408,283.533,0],"to":[0.026,0,0],"ti":[-0.027,-0.008,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.16},"t":56,"s":[132.487,283.545,0],"to":[0.027,0.008,0],"ti":[-0.027,-0.014,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":57,"s":[132.568,283.579,0],"to":[0.027,0.014,0],"ti":[-0.027,-0.019,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.162},"t":58,"s":[132.649,283.63,0],"to":[0.027,0.019,0],"ti":[-0.013,-0.011,0]},{"t":59,"s":[132.728,283.695,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.085,2.343,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":12,"s":[95.923,97.698,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.959,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.042,0.078,0]},"t":13,"s":[119.853,100.809,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.846,0.754,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.082,-0.083,0]},"t":14,"s":[71.61,47.548,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.108,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.182,0.126,0]},"t":15,"s":[95.909,74.187,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.971,0.88,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.047,-0.015,0]},"t":16,"s":[116.529,126.213,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.765,1.081,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.046,0.272,0]},"t":17,"s":[69.282,82.296,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.014,0.826,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.129,0.041,0]},"t":18,"s":[99.804,62.944,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.925,0.917,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.012,0.16,0]},"t":19,"s":[155.455,101.204,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.797,-29.781,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.787,622.842,0]},"t":20,"s":[90.335,142.892,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.16,1.74,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,0.084,0]},"t":21,"s":[96.569,142.898,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.834,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.075,0]},"t":22,"s":[105.516,144.953,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.994,0.309,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.168,0.547,0]},"t":23,"s":[79.375,124.659,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,0.77,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.007,0.095,0]},"t":24,"s":[53.599,121.011,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.006,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.187,0.131,0]},"t":25,"s":[77.44,94.418,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,0.812,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,-0.044,0]},"t":26,"s":[96.658,47.588,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.869,1.081,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,0.15,0]},"t":27,"s":[75.964,78.236,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.159,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.23,0.041,0]},"t":28,"s":[94.431,116.596,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.918,0.82,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,-0.058,0]},"t":29,"s":[104.908,41.132,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-2.162,1.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[-4.52,0.155,0]},"t":30,"s":[74.497,85.65,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.955,0.956,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.086,0.011,0]},"t":31,"s":[75.047,137.348,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,1.056,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.097,-0.092,0]},"t":32,"s":[95.389,78.111,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.564,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.248,0.034,0]},"t":33,"s":[85.987,106.338,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.138,0.755,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,-0.133,0]},"t":34,"s":[86.576,59.064,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,0.905,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.052,0.126,0]},"t":35,"s":[67.887,77.283,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.048,1.699,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.951,0.69,0]},"t":36,"s":[117.559,112.652,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.968,0.892,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,0.074,0]},"t":37,"s":[115.524,117.508,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.028,0.868,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.052,0.361,0]},"t":38,"s":[94.322,71.923,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.911,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.021,0.226,0]},"t":39,"s":[107.41,58.262,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-1.313,0.689,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.225,-0.022,0]},"t":40,"s":[89.918,50.26,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,0.965,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.086,0.114,0]},"t":41,"s":[88.641,56.58,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.028,0.486,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.303,-0.061,0]},"t":42,"s":[54.485,73.842,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.806,1.029,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.021,0.099,0]},"t":43,"s":[41.537,63.86,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.052,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.146,0.021,0]},"t":44,"s":[58.89,12.33,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,1.279,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.032,0.455,0]},"t":45,"s":[81.867,81.726,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.858,0.905,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.543,0.064,0]},"t":46,"s":[44.5,97.298,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.018,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.202,0.656,0]},"t":47,"s":[49.475,29.648,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.415,-65.882,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.015,34.7,0]},"t":48,"s":[52.978,19.812,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.126,1.189,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,0.083,0]},"t":49,"s":[48.737,19.789,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,0.939,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.05,0.058,0]},"t":50,"s":[74.095,0.809,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.788,1.031,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.229,-0.233,0]},"t":51,"s":[10.268,62.842,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.913,1.141,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.137,0.023,0]},"t":52,"s":[27.318,46.475,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.588,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.212,0.052,0]},"t":53,"s":[53.611,68.98,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.93,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,-0.069,0]},"t":54,"s":[54.64,8.452,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.974,0.937,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.428,-0.05,0]},"t":55,"s":[21.647,41.626,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.031,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.037,-0.251,0]},"t":56,"s":[27.025,20.873,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.011,0.234,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.023,0.727,0]},"t":57,"s":[23.296,26.051,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.01,0.093,0]},"t":58,"s":[28.415,26.722,100]},{"t":59,"s":[22.608,32.219,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.167325382607,0.716982433843,1,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":12,"op":60,"st":12,"bm":0},{"ddd":0,"ind":35,"ty":4,"nm":"Shape Layer 17","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":8,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":13,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":35,"s":[100]},{"t":51,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.628},"o":{"x":0.167,"y":0.167},"t":8,"s":[275.449,259.557,0],"to":[0.546,0.159,0],"ti":[-2.494,-0.483,0]},{"i":{"x":0.833,"y":0.767},"o":{"x":0.167,"y":0.107},"t":9,"s":[278.724,260.512,0],"to":[2.494,0.483,0],"ti":[-5.49,-0.572,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.13},"t":10,"s":[290.414,262.453,0],"to":[5.49,0.572,0],"ti":[-7.452,0.063,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.159},"t":11,"s":[311.663,263.943,0],"to":[7.452,-0.063,0],"ti":[-7.172,1.285,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":12,"s":[335.126,262.075,0],"to":[7.172,-1.285,0],"ti":[-5.684,2.429,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":13,"s":[354.697,256.231,0],"to":[5.684,-2.429,0],"ti":[-4.084,3.144,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":14,"s":[369.23,247.5,0],"to":[4.084,-3.144,0],"ti":[-2.734,3.404,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":15,"s":[379.199,237.368,0],"to":[2.734,-3.404,0],"ti":[-1.736,3.345,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":16,"s":[385.634,227.077,0],"to":[1.736,-3.345,0],"ti":[-1.056,3.132,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":17,"s":[389.616,217.298,0],"to":[1.056,-3.132,0],"ti":[-0.604,2.868,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":18,"s":[391.967,208.287,0],"to":[0.604,-2.868,0],"ti":[-0.305,2.608,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":19,"s":[393.24,200.09,0],"to":[0.305,-2.608,0],"ti":[-0.105,2.371,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":20,"s":[393.797,192.641,0],"to":[0.105,-2.371,0],"ti":[0.031,2.158,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":21,"s":[393.871,185.865,0],"to":[-0.031,-2.158,0],"ti":[0.123,1.968,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":22,"s":[393.614,179.691,0],"to":[-0.123,-1.968,0],"ti":[0.186,1.797,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[393.131,174.057,0],"to":[-0.186,-1.797,0],"ti":[0.228,1.644,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[392.497,168.908,0],"to":[-0.228,-1.644,0],"ti":[0.251,1.504,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[391.765,164.196,0],"to":[-0.251,-1.504,0],"ti":[0.261,1.377,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[390.988,159.882,0],"to":[-0.261,-1.377,0],"ti":[0.262,1.261,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":27,"s":[390.199,155.932,0],"to":[-0.262,-1.261,0],"ti":[0.256,1.155,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":28,"s":[389.418,152.313,0],"to":[-0.256,-1.155,0],"ti":[0.246,1.057,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":29,"s":[388.662,149,0],"to":[-0.246,-1.057,0],"ti":[0.233,0.967,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":30,"s":[387.941,145.969,0],"to":[-0.233,-0.967,0],"ti":[0.217,0.882,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":31,"s":[387.264,143.201,0],"to":[-0.217,-0.882,0],"ti":[0.199,0.803,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":32,"s":[386.64,140.677,0],"to":[-0.199,-0.803,0],"ti":[0.18,0.729,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":33,"s":[386.071,138.382,0],"to":[-0.18,-0.729,0],"ti":[0.159,0.659,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":34,"s":[385.563,136.303,0],"to":[-0.159,-0.659,0],"ti":[0.138,0.594,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":35,"s":[385.116,134.426,0],"to":[-0.138,-0.594,0],"ti":[0.117,0.531,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":36,"s":[384.732,132.741,0],"to":[-0.117,-0.531,0],"ti":[0.096,0.473,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":37,"s":[384.411,131.237,0],"to":[-0.096,-0.473,0],"ti":[0.076,0.417,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":38,"s":[384.153,129.905,0],"to":[-0.076,-0.417,0],"ti":[0.055,0.364,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":39,"s":[383.957,128.736,0],"to":[-0.055,-0.364,0],"ti":[0.036,0.314,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":40,"s":[383.821,127.721,0],"to":[-0.036,-0.314,0],"ti":[0.017,0.266,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.183},"t":41,"s":[383.742,126.852,0],"to":[-0.017,-0.266,0],"ti":[0,0.222,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.185},"t":42,"s":[383.718,126.122,0],"to":[0,-0.222,0],"ti":[-0.017,0.179,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.187},"t":43,"s":[383.745,125.523,0],"to":[0.017,-0.179,0],"ti":[-0.032,0.139,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.19},"t":44,"s":[383.819,125.048,0],"to":[0.032,-0.139,0],"ti":[-0.046,0.101,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.19},"t":45,"s":[383.937,124.69,0],"to":[0.046,-0.101,0],"ti":[-0.058,0.066,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.185},"t":46,"s":[384.093,124.442,0],"to":[0.058,-0.066,0],"ti":[-0.068,0.032,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.173},"t":47,"s":[384.283,124.297,0],"to":[0.068,-0.032,0],"ti":[-0.074,0.009,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.166},"t":48,"s":[384.5,124.247,0],"to":[0.074,-0.009,0],"ti":[-0.073,0.002,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":49,"s":[384.725,124.244,0],"to":[0.073,-0.002,0],"ti":[-0.068,0.003,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.174},"t":50,"s":[384.937,124.237,0],"to":[0.068,-0.003,0],"ti":[-0.061,0.006,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.178},"t":51,"s":[385.131,124.223,0],"to":[0.061,-0.006,0],"ti":[-0.052,0.011,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.182},"t":52,"s":[385.302,124.198,0],"to":[0.052,-0.011,0],"ti":[-0.041,0.017,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.184},"t":53,"s":[385.442,124.158,0],"to":[0.041,-0.017,0],"ti":[-0.028,0.024,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.179},"t":54,"s":[385.546,124.097,0],"to":[0.028,-0.024,0],"ti":[-0.026,0.021,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.171},"t":55,"s":[385.612,124.016,0],"to":[0.026,-0.021,0],"ti":[-0.034,0.006,0]},{"i":{"x":0.833,"y":0.813},"o":{"x":0.167,"y":0.153},"t":56,"s":[385.701,123.974,0],"to":[0.034,-0.006,0],"ti":[-0.043,-0.01,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.15},"t":57,"s":[385.818,123.981,0],"to":[0.043,0.01,0],"ti":[-0.048,-0.025,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.153},"t":58,"s":[385.957,124.035,0],"to":[0.048,0.025,0],"ti":[-0.025,-0.016,0]},{"t":59,"s":[386.107,124.13,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.944,0.679,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":8,"s":[49.219,103.811,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.111,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.166,0.113,0]},"t":9,"s":[96.918,92.8,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,29.641,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.048,19.517,0]},"t":10,"s":[80.991,61.448,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.606,1.004,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.607,0.083,0]},"t":11,"s":[118.185,61.313,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.989,0.964,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.004,0]},"t":12,"s":[124.101,107.653,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.917,0.896,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.013,-0.062,0]},"t":13,"s":[75.181,59.182,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-66.49,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[-62.917,0.413,0]},"t":14,"s":[117.68,86.992,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,0.5,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.083,-0.053,0]},"t":15,"s":[117.624,94.022,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.649,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.154,0.1,0]},"t":16,"s":[72.153,89.729,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.994,0.616,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.109,-0.07,0]},"t":17,"s":[88.106,68.252,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.006,0.106,0]},"t":18,"s":[139.441,79.894,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.246,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.681,-0.189,0]},"t":19,"s":[91.59,121.922,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.031,1.095,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,-0.001,0]},"t":20,"s":[96.806,109.051,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.909,0.978,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.023,0.044,0]},"t":21,"s":[76.174,121.699,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.265,0.951,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.035,-0.029,0]},"t":22,"s":[104.516,94.655,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.009,0.92,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.078,-0.118,0]},"t":23,"s":[106.997,114.689,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.918,7.262,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.008,-1.803,0]},"t":24,"s":[66.837,106.412,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-2.408,1.035,1]},"o":{"x":[0.167,0.167,0.167],"y":[-4.987,0.082,0]},"t":25,"s":[111.464,106.778,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,1.046,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.085,0.025,0]},"t":26,"s":[110.731,78.932,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.884,0.945,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,0.03,0]},"t":27,"s":[81.464,118.575,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.215,0.686,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.293,-0.158,0]},"t":28,"s":[101.82,56.948,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,0.967,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.06,0.113,0]},"t":29,"s":[109.895,78.255,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.916,0.979,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.066,-0.056,0]},"t":30,"s":[80.942,137.249,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-6.827,1.076,1]},"o":{"x":[0.167,0.167,0.167],"y":[12.816,-0.028,0]},"t":31,"s":[97.117,101.933,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.453,0.923,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.084,0.04,0]},"t":32,"s":[97.223,128.384,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.989,0.619,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,-0.935,0]},"t":33,"s":[107.061,77.679,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.924,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.013,0.107,0]},"t":34,"s":[43.764,81.828,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,-1.177,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.143,-0.844,0]},"t":35,"s":[98.569,96.663,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.435,0.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[-2.31,0.087,0]},"t":36,"s":[78.363,95.331,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.931,2.861,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,-3.479,0]},"t":37,"s":[79.067,61.85,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.227,0.912,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.407,0.08,0]},"t":38,"s":[66.248,62.633,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.092,-2.126,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.093,1.726,0]},"t":39,"s":[68.428,44.364,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.901,1.019,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,0.086,0]},"t":40,"s":[86.481,43.437,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.779,1.036,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.541,0.015,0]},"t":41,"s":[48.571,9.591,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.776,1.009,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.134,0.025,0]},"t":42,"s":[41.665,51.149,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.092,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.133,0.008,0]},"t":43,"s":[30.243,-8.194,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.935,9.657,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,20.432,0]},"t":44,"s":[10.976,57.43,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.606,0.758,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.287,0.083,0]},"t":45,"s":[51.537,57.699,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.918,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.106,0.127,0]},"t":46,"s":[42.412,29.51,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-8.496,0.903,1]},"o":{"x":[0.167,0.167,0.167],"y":[-5.253,-0.014,0]},"t":47,"s":[8.401,-24.232,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.961,0.669,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.084,0.582,0]},"t":48,"s":[8.932,21.756,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.072,0.111,0]},"t":49,"s":[68.92,29.447,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.049,0.445,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.094,1.927,0]},"t":50,"s":[36.715,52.274,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.899,1.631,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.031,0.098,0]},"t":51,"s":[51.861,53.306,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.136,0.908,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.481,0.074,0]},"t":52,"s":[27.738,59.151,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.062,0.322,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.052,0.928,0]},"t":53,"s":[22.685,9.056,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.882,0.979,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.035,0.095,0]},"t":54,"s":[35.966,4.115,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.028,0.812,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.285,-0.029,0]},"t":55,"s":[12.831,-31.117,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.846,0.873,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.021,0.15,0]},"t":56,"s":[3.287,-4.957,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.815,0.964,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.182,0.241,0]},"t":57,"s":[16.066,27.774,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.152,-0.063,0]},"t":58,"s":[26.832,45.045,100]},{"t":59,"s":[39.927,35.232,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.167325382607,0.716982433843,1,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":8,"op":60,"st":8,"bm":0},{"ddd":0,"ind":36,"ty":4,"nm":"Shape Layer 16","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":9,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":14,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":36,"s":[100]},{"t":52,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.587},"o":{"x":0.167,"y":0.167},"t":9,"s":[275.519,261.1,0],"to":[-0.307,-0.313,0],"ti":[1.698,1.185,0]},{"i":{"x":0.833,"y":0.774},"o":{"x":0.167,"y":0.104},"t":10,"s":[273.678,259.22,0],"to":[-1.698,-1.185,0],"ti":[1.745,-2.083,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.132},"t":11,"s":[265.331,253.99,0],"to":[-1.745,2.083,0],"ti":[-0.013,-6.364,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.156},"t":12,"s":[263.209,271.718,0],"to":[0.013,6.364,0],"ti":[-0.69,-6.377,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":13,"s":[265.411,292.176,0],"to":[0.69,6.377,0],"ti":[-0.533,-5.437,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":14,"s":[267.35,309.98,0],"to":[0.533,5.437,0],"ti":[-0.321,-4.543,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":15,"s":[268.609,324.8,0],"to":[0.321,4.543,0],"ti":[-0.144,-3.839,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":16,"s":[269.275,337.239,0],"to":[0.144,3.839,0],"ti":[-0.006,-3.291,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":17,"s":[269.475,347.837,0],"to":[0.006,3.291,0],"ti":[0.101,-2.855,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":18,"s":[269.314,356.986,0],"to":[-0.101,2.855,0],"ti":[0.186,-2.499,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":19,"s":[268.867,364.967,0],"to":[-0.186,2.499,0],"ti":[0.253,-2.201,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":20,"s":[268.196,371.978,0],"to":[-0.253,2.201,0],"ti":[0.306,-1.947,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":21,"s":[267.347,378.17,0],"to":[-0.306,1.947,0],"ti":[0.347,-1.728,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":22,"s":[266.36,383.661,0],"to":[-0.347,1.728,0],"ti":[0.377,-1.535,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":23,"s":[265.266,388.537,0],"to":[-0.377,1.535,0],"ti":[0.399,-1.366,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":24,"s":[264.095,392.873,0],"to":[-0.399,1.366,0],"ti":[0.413,-1.215,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":25,"s":[262.872,396.731,0],"to":[-0.413,1.215,0],"ti":[0.42,-1.082,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[261.617,400.166,0],"to":[-0.42,1.082,0],"ti":[0.421,-0.962,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[260.351,403.22,0],"to":[-0.421,0.962,0],"ti":[0.416,-0.854,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":28,"s":[259.092,405.935,0],"to":[-0.416,0.854,0],"ti":[0.407,-0.759,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":29,"s":[257.854,408.347,0],"to":[-0.407,0.759,0],"ti":[0.395,-0.675,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[256.649,410.491,0],"to":[-0.395,0.675,0],"ti":[0.379,-0.6,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":31,"s":[255.487,412.396,0],"to":[-0.379,0.6,0],"ti":[0.361,-0.535,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":32,"s":[254.375,414.092,0],"to":[-0.361,0.535,0],"ti":[0.343,-0.477,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":33,"s":[253.318,415.604,0],"to":[-0.343,0.477,0],"ti":[0.323,-0.427,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":34,"s":[252.319,416.955,0],"to":[-0.323,0.427,0],"ti":[0.304,-0.384,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":35,"s":[251.378,418.167,0],"to":[-0.304,0.384,0],"ti":[0.285,-0.347,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":36,"s":[250.495,419.259,0],"to":[-0.285,0.347,0],"ti":[0.268,-0.316,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":37,"s":[249.666,420.25,0],"to":[-0.268,0.316,0],"ti":[0.252,-0.289,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":38,"s":[248.887,421.154,0],"to":[-0.252,0.289,0],"ti":[0.235,-0.266,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.174},"t":39,"s":[248.153,421.986,0],"to":[-0.235,0.266,0],"ti":[0.208,-0.241,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":40,"s":[247.479,422.749,0],"to":[-0.208,0.241,0],"ti":[0.175,-0.214,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.18},"t":41,"s":[246.907,423.43,0],"to":[-0.175,0.214,0],"ti":[0.145,-0.189,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.181},"t":42,"s":[246.43,424.033,0],"to":[-0.145,0.189,0],"ti":[0.118,-0.164,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.182},"t":43,"s":[246.038,424.562,0],"to":[-0.118,0.164,0],"ti":[0.094,-0.141,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.184},"t":44,"s":[245.721,425.02,0],"to":[-0.094,0.141,0],"ti":[0.073,-0.118,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.186},"t":45,"s":[245.472,425.408,0],"to":[-0.073,0.118,0],"ti":[0.055,-0.096,0]},{"i":{"x":0.833,"y":0.856},"o":{"x":0.167,"y":0.19},"t":46,"s":[245.282,425.73,0],"to":[-0.055,0.096,0],"ti":[0.039,-0.074,0]},{"i":{"x":0.833,"y":0.862},"o":{"x":0.167,"y":0.197},"t":47,"s":[245.143,425.985,0],"to":[-0.039,0.074,0],"ti":[0.025,-0.053,0]},{"i":{"x":0.833,"y":0.853},"o":{"x":0.167,"y":0.211},"t":48,"s":[245.048,426.176,0],"to":[-0.025,0.053,0],"ti":[0.019,-0.036,0]},{"i":{"x":0.833,"y":0.822},"o":{"x":0.167,"y":0.193},"t":49,"s":[244.991,426.303,0],"to":[-0.019,0.036,0],"ti":[0.026,-0.027,0]},{"i":{"x":0.833,"y":0.815},"o":{"x":0.167,"y":0.157},"t":50,"s":[244.933,426.392,0],"to":[-0.026,0.027,0],"ti":[0.039,-0.021,0]},{"i":{"x":0.833,"y":0.814},"o":{"x":0.167,"y":0.152},"t":51,"s":[244.836,426.464,0],"to":[-0.039,0.021,0],"ti":[0.052,-0.016,0]},{"i":{"x":0.833,"y":0.816},"o":{"x":0.167,"y":0.151},"t":52,"s":[244.701,426.519,0],"to":[-0.052,0.016,0],"ti":[0.065,-0.01,0]},{"i":{"x":0.833,"y":0.818},"o":{"x":0.167,"y":0.152},"t":53,"s":[244.525,426.558,0],"to":[-0.065,0.01,0],"ti":[0.079,-0.004,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.154},"t":54,"s":[244.308,426.579,0],"to":[-0.079,0.004,0],"ti":[0.092,0.002,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.156},"t":55,"s":[244.051,426.582,0],"to":[-0.092,-0.002,0],"ti":[0.105,0.007,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.157},"t":56,"s":[243.755,426.569,0],"to":[-0.105,-0.007,0],"ti":[0.117,0.012,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.159},"t":57,"s":[243.422,426.54,0],"to":[-0.117,-0.012,0],"ti":[0.128,0.017,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.16},"t":58,"s":[243.055,426.497,0],"to":[-0.128,-0.017,0],"ti":[0.066,0.01,0]},{"t":59,"s":[242.656,426.439,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.563,1.02,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":9,"s":[92.322,93.922,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.109,0.948,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.103,0.016,0]},"t":10,"s":[95.253,42.092,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.054,0.89,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.047,-0.14,0]},"t":11,"s":[107.675,106.42,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,0.725,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.033,0.342,0]},"t":12,"s":[79.026,82.445,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.393,1.033,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.475,0.12,0]},"t":13,"s":[126.398,74.729,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.968,0.877,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.097,0.024,0]},"t":14,"s":[119.33,57.001,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.834,1.214,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.053,0.26,0]},"t":15,"s":[74.88,81.854,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.042,0.975,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.06,0]},"t":16,"s":[102.06,93.554,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.894,0.977,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.028,-0.035,0]},"t":17,"s":[129.135,51.875,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.863,1.026,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.389,-0.032,0]},"t":18,"s":[88.364,81.23,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.102,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.212,0.02,0]},"t":19,"s":[77.256,59.947,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.698,1.03,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.046,-0.051,0]},"t":20,"s":[70.037,87.966,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.995,1.049,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.115,0.022,0]},"t":21,"s":[86.104,70.53,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.904,1.07,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.006,0.031,0]},"t":22,"s":[128.201,94.177,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.376,0.967,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.627,0.038,0]},"t":23,"s":[88.836,56.66,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,0.977,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.068,-0.054,0]},"t":24,"s":[82.802,125.882,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.976,1.097,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.002,-0.031,0]},"t":25,"s":[116.041,83.757,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.9,0.955,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.033,0.045,0]},"t":26,"s":[83.397,114.487,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.069,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.508,-0.099,0]},"t":27,"s":[106.769,47.904,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.758,0.953,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.497,0]},"t":28,"s":[111.355,78.308,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.163,1.569,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.127,-0.106,0]},"t":29,"s":[102.953,84.434,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.9,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.073,0]},"t":30,"s":[86.963,81.742,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.429,1.529,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.493,1.851,0]},"t":31,"s":[134.26,102.822,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.944,1.156,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,0.072,0]},"t":32,"s":[143.885,103.815,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,1.116,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.171,0.054,0]},"t":33,"s":[84.672,96.517,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.547,0.897,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.171,0.048,0]},"t":34,"s":[104.079,117.449,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.772,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.088,0.439,0]},"t":35,"s":[102.79,67.476,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,1.47,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.131,2.037,0]},"t":36,"s":[80.144,55.767,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.822,1.082,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,0.071,0]},"t":37,"s":[40.802,55.268,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.966,0.556,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.157,0.041,0]},"t":38,"s":[68.335,58.584,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.795,1.044,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.058,0.103,0]},"t":39,"s":[99.544,51.985,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.792,1.038,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,0.029,0]},"t":40,"s":[81.176,23.398,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.896,0.994,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.139,0.026,0]},"t":41,"s":[54.468,66.977,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.334,0.99,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.427,-0.006,0]},"t":42,"s":[14.561,3.526,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.853,0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,-0.011,0]},"t":43,"s":[4.89,62.465,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.066,1.037,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.192,-0.037,0]},"t":44,"s":[53.289,10.271,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.882,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.037,0.026,0]},"t":45,"s":[90.493,46.486,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.078,0.738,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.284,-0.203,0]},"t":46,"s":[23.844,-5.784,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.862,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,0.122,0]},"t":47,"s":[-3.767,9.434,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.96,1.136,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.211,0.454,0]},"t":48,"s":[49.843,42.088,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.76,0.903,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.077,0.052,0]},"t":49,"s":[84.879,49.437,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,0.214,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.128,0.58,0]},"t":50,"s":[66.624,30.104,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.972,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.222,0.093,0]},"t":51,"s":[32.305,26.858,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.916,1.208,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.117,-0.043,0]},"t":52,"s":[11.683,-0.538,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.198,0.965,1]},"o":{"x":[0.167,0.167,0.167],"y":[10.387,0.06,0]},"t":53,"s":[20.245,17.535,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.475,0.823,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.078,-0.06,0]},"t":54,"s":[20.314,-45.662,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.442,0.954,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,0.157,0]},"t":55,"s":[19.25,-8.935,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,1.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,-0.101,0]},"t":56,"s":[13.603,32.461,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.007,0.766,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.071,0.01,0]},"t":57,"s":[49.18,13.788,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,0.129,0]},"t":58,"s":[29.913,35.117,100]},{"t":59,"s":[50.756,73.724,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.167325382607,0.716982433843,1,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":9,"op":60,"st":9,"bm":0},{"ddd":0,"ind":37,"ty":4,"nm":"Shape Layer 15","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":6,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":33,"s":[100]},{"t":49,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.63},"o":{"x":0.167,"y":0.167},"t":6,"s":[275.485,257.563,0],"to":[-0.11,-0.249,0],"ti":[0.689,1.001,0]},{"i":{"x":0.833,"y":0.767},"o":{"x":0.167,"y":0.108},"t":7,"s":[274.824,256.066,0],"to":[-0.689,-1.001,0],"ti":[1.729,2.01,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.13},"t":8,"s":[271.354,251.559,0],"to":[-1.729,-2.01,0],"ti":[2.505,2.573,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.159},"t":9,"s":[264.452,244.008,0],"to":[-2.505,-2.573,0],"ti":[2.58,2.412,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":10,"s":[256.321,236.124,0],"to":[-2.58,-2.412,0],"ti":[2.262,1.991,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":11,"s":[248.97,229.538,0],"to":[-2.262,-1.991,0],"ti":[1.91,1.643,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":12,"s":[242.747,224.176,0],"to":[-1.91,-1.643,0],"ti":[1.608,1.404,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":13,"s":[237.508,219.678,0],"to":[-1.608,-1.404,0],"ti":[1.343,1.257,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.179},"t":14,"s":[233.099,215.75,0],"to":[-1.343,-1.257,0],"ti":[1.062,1.209,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":15,"s":[229.447,212.138,0],"to":[-1.062,-1.209,0],"ti":[0.452,1.267,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.177},"t":16,"s":[226.724,208.498,0],"to":[-0.452,-1.267,0],"ti":[-0.52,1.026,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.171},"t":17,"s":[226.732,204.537,0],"to":[0.52,-1.026,0],"ti":[-1.055,0.562,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":18,"s":[229.844,202.343,0],"to":[1.055,-0.562,0],"ti":[-1.033,0.324,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":19,"s":[233.062,201.165,0],"to":[1.033,-0.324,0],"ti":[-0.951,0.215,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":20,"s":[236.043,200.396,0],"to":[0.951,-0.215,0],"ti":[-0.867,0.144,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[238.767,199.873,0],"to":[0.867,-0.144,0],"ti":[-0.789,0.088,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[241.247,199.533,0],"to":[0.789,-0.088,0],"ti":[-0.717,0.039,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":23,"s":[243.501,199.346,0],"to":[0.717,-0.039,0],"ti":[-0.654,0.005,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[245.548,199.298,0],"to":[0.654,-0.005,0],"ti":[-0.601,-0.009,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[247.425,199.317,0],"to":[0.601,0.009,0],"ti":[-0.552,-0.013,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":26,"s":[249.152,199.353,0],"to":[0.552,0.013,0],"ti":[-0.504,-0.015,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[250.734,199.397,0],"to":[0.504,0.015,0],"ti":[-0.457,-0.014,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":28,"s":[252.175,199.441,0],"to":[0.457,0.014,0],"ti":[-0.412,-0.011,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":29,"s":[253.479,199.479,0],"to":[0.412,0.011,0],"ti":[-0.367,-0.007,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":30,"s":[254.647,199.506,0],"to":[0.367,0.007,0],"ti":[-0.324,-0.001,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":31,"s":[255.684,199.519,0],"to":[0.324,0.001,0],"ti":[-0.281,0.005,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.18},"t":32,"s":[256.59,199.514,0],"to":[0.281,-0.005,0],"ti":[-0.238,0.012,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":33,"s":[257.369,199.488,0],"to":[0.238,-0.012,0],"ti":[-0.197,0.019,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.186},"t":34,"s":[258.021,199.442,0],"to":[0.197,-0.019,0],"ti":[-0.156,0.027,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.189},"t":35,"s":[258.548,199.373,0],"to":[0.156,-0.027,0],"ti":[-0.116,0.034,0]},{"i":{"x":0.833,"y":0.856},"o":{"x":0.167,"y":0.194},"t":36,"s":[258.956,199.282,0],"to":[0.116,-0.034,0],"ti":[-0.078,0.042,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.199},"t":37,"s":[259.247,199.167,0],"to":[0.078,-0.042,0],"ti":[-0.042,0.049,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.191},"t":38,"s":[259.427,199.031,0],"to":[0.042,-0.049,0],"ti":[-0.007,0.055,0]},{"i":{"x":0.833,"y":0.809},"o":{"x":0.167,"y":0.164},"t":39,"s":[259.498,198.875,0],"to":[0.007,-0.055,0],"ti":[0.027,0.061,0]},{"i":{"x":0.833,"y":0.808},"o":{"x":0.167,"y":0.148},"t":40,"s":[259.466,198.699,0],"to":[-0.027,-0.061,0],"ti":[0.059,0.066,0]},{"i":{"x":0.833,"y":0.813},"o":{"x":0.167,"y":0.147},"t":41,"s":[259.336,198.506,0],"to":[-0.059,-0.066,0],"ti":[0.089,0.071,0]},{"i":{"x":0.833,"y":0.818},"o":{"x":0.167,"y":0.15},"t":42,"s":[259.112,198.3,0],"to":[-0.089,-0.071,0],"ti":[0.117,0.074,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.153},"t":43,"s":[258.802,198.083,0],"to":[-0.117,-0.074,0],"ti":[0.142,0.075,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.156},"t":44,"s":[258.411,197.859,0],"to":[-0.142,-0.075,0],"ti":[0.165,0.075,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.158},"t":45,"s":[257.947,197.632,0],"to":[-0.165,-0.075,0],"ti":[0.182,0.074,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.163},"t":46,"s":[257.418,197.407,0],"to":[-0.182,-0.074,0],"ti":[0.188,0.072,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.167},"t":47,"s":[256.856,197.186,0],"to":[-0.188,-0.072,0],"ti":[0.188,0.068,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":48,"s":[256.29,196.974,0],"to":[-0.188,-0.068,0],"ti":[0.184,0.062,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.17},"t":49,"s":[255.73,196.777,0],"to":[-0.184,-0.062,0],"ti":[0.178,0.054,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":50,"s":[255.185,196.601,0],"to":[-0.178,-0.054,0],"ti":[0.168,0.043,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.174},"t":51,"s":[254.664,196.453,0],"to":[-0.168,-0.043,0],"ti":[0.154,0.03,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.177},"t":52,"s":[254.178,196.341,0],"to":[-0.154,-0.03,0],"ti":[0.136,0.014,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.181},"t":53,"s":[253.74,196.272,0],"to":[-0.136,-0.014,0],"ti":[0.114,-0.009,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.185},"t":54,"s":[253.361,196.256,0],"to":[-0.114,0.009,0],"ti":[0.088,-0.039,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.176},"t":55,"s":[253.059,196.325,0],"to":[-0.088,0.039,0],"ti":[0.062,-0.069,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.163},"t":56,"s":[252.835,196.49,0],"to":[-0.062,0.069,0],"ti":[0.038,-0.096,0]},{"i":{"x":0.833,"y":0.82},"o":{"x":0.167,"y":0.156},"t":57,"s":[252.687,196.74,0],"to":[-0.038,0.096,0],"ti":[0.015,-0.119,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.155},"t":58,"s":[252.609,197.066,0],"to":[-0.015,0.119,0],"ti":[0.002,-0.065,0]},{"t":59,"s":[252.597,197.456,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.755,1.005,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":6,"s":[82.502,83.365,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.13,0.857,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.126,0.005,0]},"t":7,"s":[92.567,121.915,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,1.043,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,0.199,0]},"t":8,"s":[112.053,80.824,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.879,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.103,0.029,0]},"t":9,"s":[62.128,51.3,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.612,1.126,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.269,0.247,0]},"t":10,"s":[84.48,96.186,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,0.869,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.106,0.05,0]},"t":11,"s":[94.531,118.987,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.7,1.007,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.155,0.229,0]},"t":12,"s":[131.266,61.724,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.907,0.775,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.115,0.006,0]},"t":13,"s":[118.417,29.003,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.355,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.826,0.132,0]},"t":14,"s":[85.075,64.302,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.982,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,-0.001,0]},"t":15,"s":[81.335,124.25,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.023,0.465,0]},"t":16,"s":[101.012,65.094,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-1.675,1.303,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.722,0.245,0]},"t":17,"s":[85.57,52.183,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.895,0.888,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.086,0.065,0]},"t":18,"s":[84.785,45.511,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.611,0.792,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.406,0.329,0]},"t":19,"s":[60.369,76.428,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.931,1.274,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.139,0]},"t":20,"s":[54.064,86.903,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.399,0.064,0]},"t":21,"s":[106.629,102.562,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.815,0.921,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.186,-0.033,0]},"t":22,"s":[97.556,35.47,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.903,2.194,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.152,-1.466,0]},"t":23,"s":[90.213,83.359,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.319,1.048,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.578,0.078,0]},"t":24,"s":[81.292,80.783,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.967,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.08,0.031,0]},"t":25,"s":[79.79,120.257,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.122,-0.054,0]},"t":26,"s":[123.086,57.838,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.232,0.873,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.07,-0.045,0]},"t":27,"s":[105.488,95.827,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.055,0.885,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,0.244,0]},"t":28,"s":[115.025,71.124,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,1.655,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.033,0.303,0]},"t":29,"s":[78.946,58.294,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.622,0.901,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.606,0.074,0]},"t":30,"s":[138.694,53.428,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,0.751,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.515,0]},"t":31,"s":[141.965,96.547,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.001,1.291,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,0.125,0]},"t":32,"s":[114.285,104.867,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.034,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.001,0.065,0]},"t":33,"s":[88.923,121.377,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.803,1.172,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.024,-0.633,0]},"t":34,"s":[114.574,47.291,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.958,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.145,0.056,0]},"t":35,"s":[78.466,55.906,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.8,1.065,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.086,-0.111,0]},"t":36,"s":[29.34,29.523,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.143,0.036,0]},"t":37,"s":[53.468,40.837,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.752,0.669,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.176,-0.016,0]},"t":38,"s":[87.347,20.728,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.766,1.02,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.126,0.111,0]},"t":39,"s":[76.456,37.629,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.931,0.959,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.129,0.016,0]},"t":40,"s":[54.995,87.783,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.044,1.088,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.403,-0.082,0]},"t":41,"s":[16.235,25.439,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.502,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.029,0.043,0]},"t":42,"s":[22.876,56.837,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,-0.033,0]},"t":43,"s":[12.722,-7.888,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.475,0.348,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.373,-1.253,0]},"t":44,"s":[84.051,38.375,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.949,0.884,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.099,0.096,0]},"t":45,"s":[71.021,35.491,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.795,1.365,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.133,0.294,0]},"t":46,"s":[1.946,15.801,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.93,1.015,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.14,0.068,0]},"t":47,"s":[28.567,8.027,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.248,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.452,0.012,0]},"t":48,"s":[67.532,49.862,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,1.749,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,-1.199,0]},"t":49,"s":[61.469,0.74,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.813,1.078,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.542,0.075,0]},"t":50,"s":[12.831,3.931,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.066,0.975,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.15,0.04,0]},"t":51,"s":[19.315,-27.951,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.028,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.037,-0.035,0]},"t":52,"s":[27.419,33.62,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.067,0.472,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.021,-0.242,0]},"t":53,"s":[12.92,-9.628,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.982,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.037,0.099,0]},"t":54,"s":[32.326,1.452,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.022,0.009,0]},"t":55,"s":[-2.724,60.543,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,-0.245,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.186,-1.156,0]},"t":56,"s":[24.945,-5.97,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.875,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.035,0.089,0]},"t":57,"s":[47.481,-1.498,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.25,-0.046,0]},"t":58,"s":[31.587,60.823,100]},{"t":59,"s":[23.622,20.517,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ty":"sr","sy":2,"d":1,"pt":{"a":0,"k":3,"ix":3},"p":{"a":0,"k":[6,32],"ix":4},"r":{"a":0,"k":0,"ix":5},"or":{"a":0,"k":6.599,"ix":7},"os":{"a":0,"k":0,"ix":9},"ix":1,"nm":"Polystar Path 1","mn":"ADBE Vector Shape - Star","hd":false},{"ty":"fl","c":{"a":0,"k":[0.167325382607,0.716982433843,1,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-7.382,-25.677],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Polystar 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":6,"op":60,"st":6,"bm":0},{"ddd":0,"ind":38,"ty":4,"nm":"Shape Layer 14","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":7,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":34,"s":[100]},{"t":50,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.588},"o":{"x":0.167,"y":0.167},"t":7,"s":[276.873,250.83,0],"to":[-0.189,-0.223,0],"ti":[1.057,0.986,0]},{"i":{"x":0.833,"y":0.763},"o":{"x":0.167,"y":0.104},"t":8,"s":[275.739,249.492,0],"to":[-1.057,-0.986,0],"ti":[2.736,1.782,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":9,"s":[270.529,244.915,0],"to":[-2.736,-1.782,0],"ti":[4.156,1.618,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":10,"s":[259.324,238.803,0],"to":[-4.156,-1.618,0],"ti":[4.345,0.759,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.179},"t":11,"s":[245.595,235.21,0],"to":[-4.345,-0.759,0],"ti":[3.769,0.09,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":12,"s":[233.251,234.251,0],"to":[-3.769,-0.09,0],"ti":[3.138,-0.25,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.183},"t":13,"s":[222.982,234.672,0],"to":[-3.138,0.25,0],"ti":[2.634,-0.412,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":14,"s":[214.421,235.749,0],"to":[-2.634,0.412,0],"ti":[2.24,-0.491,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":15,"s":[207.179,237.146,0],"to":[-2.24,0.491,0],"ti":[1.931,-0.529,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":16,"s":[200.979,238.697,0],"to":[-1.931,0.529,0],"ti":[1.683,-0.547,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":17,"s":[195.596,240.321,0],"to":[-1.683,0.547,0],"ti":[1.48,-0.555,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":18,"s":[190.881,241.977,0],"to":[-1.48,0.555,0],"ti":[1.309,-0.559,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":19,"s":[186.716,243.649,0],"to":[-1.309,0.559,0],"ti":[1.16,-0.566,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":20,"s":[183.025,245.334,0],"to":[-1.16,0.566,0],"ti":[1.025,-0.581,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":21,"s":[179.755,247.045,0],"to":[-1.025,0.581,0],"ti":[0.884,-0.616,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":22,"s":[176.877,248.822,0],"to":[-0.884,0.616,0],"ti":[0.678,-0.697,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.175},"t":23,"s":[174.448,250.742,0],"to":[-0.678,0.697,0],"ti":[0.181,-0.747,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.183},"t":24,"s":[172.806,253.001,0],"to":[-0.181,0.747,0],"ti":[-0.395,-0.538,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":25,"s":[173.36,255.224,0],"to":[0.395,0.538,0],"ti":[-0.6,-0.241,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":26,"s":[175.177,256.227,0],"to":[0.6,0.241,0],"ti":[-0.564,-0.106,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":27,"s":[176.961,256.669,0],"to":[0.564,0.106,0],"ti":[-0.502,-0.043,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":28,"s":[178.564,256.865,0],"to":[0.502,0.043,0],"ti":[-0.439,-0.008,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":29,"s":[179.975,256.928,0],"to":[0.439,0.008,0],"ti":[-0.38,0.014,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":30,"s":[181.2,256.911,0],"to":[0.38,-0.014,0],"ti":[-0.325,0.028,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.181},"t":31,"s":[182.253,256.843,0],"to":[0.325,-0.028,0],"ti":[-0.275,0.036,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.182},"t":32,"s":[183.148,256.743,0],"to":[0.275,-0.036,0],"ti":[-0.23,0.04,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.183},"t":33,"s":[183.9,256.626,0],"to":[0.23,-0.04,0],"ti":[-0.19,0.041,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.184},"t":34,"s":[184.526,256.502,0],"to":[0.19,-0.041,0],"ti":[-0.155,0.04,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.185},"t":35,"s":[185.04,256.378,0],"to":[0.155,-0.04,0],"ti":[-0.125,0.037,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.187},"t":36,"s":[185.458,256.261,0],"to":[0.125,-0.037,0],"ti":[-0.1,0.031,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.187},"t":37,"s":[185.791,256.158,0],"to":[0.1,-0.031,0],"ti":[-0.082,0.022,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.185},"t":38,"s":[186.06,256.075,0],"to":[0.082,-0.022,0],"ti":[-0.073,0.001,0]},{"i":{"x":0.833,"y":0.822},"o":{"x":0.167,"y":0.172},"t":39,"s":[186.284,256.027,0],"to":[0.073,-0.001,0],"ti":[-0.069,-0.03,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.157},"t":40,"s":[186.495,256.07,0],"to":[0.069,0.03,0],"ti":[-0.067,-0.059,0]},{"i":{"x":0.833,"y":0.818},"o":{"x":0.167,"y":0.153},"t":41,"s":[186.699,256.205,0],"to":[0.067,0.059,0],"ti":[-0.065,-0.084,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.154},"t":42,"s":[186.897,256.422,0],"to":[0.065,0.084,0],"ti":[-0.063,-0.106,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.156},"t":43,"s":[187.09,256.71,0],"to":[0.063,0.106,0],"ti":[-0.062,-0.125,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.158},"t":44,"s":[187.278,257.059,0],"to":[0.062,0.125,0],"ti":[-0.059,-0.141,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":45,"s":[187.46,257.46,0],"to":[0.059,0.141,0],"ti":[-0.057,-0.153,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.162},"t":46,"s":[187.635,257.904,0],"to":[0.057,0.153,0],"ti":[-0.059,-0.163,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.162},"t":47,"s":[187.802,258.381,0],"to":[0.059,0.163,0],"ti":[-0.07,-0.168,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":48,"s":[187.988,258.88,0],"to":[0.07,0.168,0],"ti":[-0.084,-0.171,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.164},"t":49,"s":[188.22,259.391,0],"to":[0.084,0.171,0],"ti":[-0.097,-0.171,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":50,"s":[188.492,259.907,0],"to":[0.097,0.171,0],"ti":[-0.108,-0.169,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":51,"s":[188.799,260.419,0],"to":[0.108,0.169,0],"ti":[-0.117,-0.165,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":52,"s":[189.138,260.923,0],"to":[0.117,0.165,0],"ti":[-0.125,-0.159,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.167},"t":53,"s":[189.502,261.41,0],"to":[0.125,0.159,0],"ti":[-0.13,-0.151,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":54,"s":[189.886,261.876,0],"to":[0.13,0.151,0],"ti":[-0.134,-0.141,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":55,"s":[190.284,262.315,0],"to":[0.134,0.141,0],"ti":[-0.135,-0.13,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.17},"t":56,"s":[190.69,262.722,0],"to":[0.135,0.13,0],"ti":[-0.135,-0.117,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":57,"s":[191.097,263.093,0],"to":[0.135,0.117,0],"ti":[-0.132,-0.103,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.173},"t":58,"s":[191.499,263.423,0],"to":[0.132,0.103,0],"ti":[-0.065,-0.048,0]},{"t":59,"s":[191.888,263.709,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.973,0.541,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":7,"s":[98.652,70.754,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.848,0.518,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.04,0.102,0]},"t":8,"s":[52.407,72.577,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.779,0.997,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.184,0.101,0]},"t":9,"s":[83.757,80.783,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.134,-0.003,0]},"t":10,"s":[109.776,120.093,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.877,0.951,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.028,0.222,0]},"t":11,"s":[152.905,82.158,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.679,0.626,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.257,-0.12,0]},"t":12,"s":[120.66,59.419,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.91,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,0.107,0]},"t":13,"s":[105.207,68.721,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.545,0.804,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.14,-0.083,0]},"t":14,"s":[61.164,101.207,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,1.003,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.072,0.145,0]},"t":15,"s":[57.69,84.913,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.643,0.859,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.256,0.003,0]},"t":16,"s":[83.888,62.79,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.821,0.798,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.109,0.204,0]},"t":17,"s":[77.451,85.785,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.087,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.156,0.142,0]},"t":18,"s":[56.288,101.636,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,1.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.042,0.454,0]},"t":19,"s":[32.111,124.298,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.091,0.877,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.178,0.076,0]},"t":20,"s":[81.428,129.386,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.009,0.967,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.043,0.259,0]},"t":21,"s":[65.685,67.723,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.955,0.799,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.008,-0.054,0]},"t":22,"s":[98.593,38.482,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.732,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.096,0.142,0]},"t":23,"s":[62.324,56.222,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.936,0.271,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.121,0.546,0]},"t":24,"s":[79.166,81.278,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.276,0.094,0]},"t":25,"s":[116.554,85.792,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.592,0.744,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.377,-0.236,0]},"t":26,"s":[107.89,120.793,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.083,0.766,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.124,0]},"t":27,"s":[109.46,111.669,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.998,0.945,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.042,0.129,0]},"t":28,"s":[96.752,92.8,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.958,0.687,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.002,-0.162,0]},"t":29,"s":[122.109,58.72,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.192,0.866,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.084,0.114,0]},"t":30,"s":[97.302,70.284,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.964,1.117,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,0.219,0]},"t":31,"s":[109.621,102.162,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.063,0.049,0]},"t":32,"s":[68.943,121.676,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.514,0.947,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.428,-0.071,0]},"t":33,"s":[92.184,74.87,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.944,0.654,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.072,-0.144,0]},"t":34,"s":[97.803,100.145,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.734,0.804,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.175,0.11,0]},"t":35,"s":[57.518,90.885,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.006,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.121,0.145,0]},"t":36,"s":[70.533,61.657,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.907,0.638,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,-0.187,0]},"t":37,"s":[99.011,22.279,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.045,0.875,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.773,0.108,0]},"t":38,"s":[68.391,34.404,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.907,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.029,0.249,0]},"t":39,"s":[64.692,74.939,100]},{"i":{"x":[0.833,0.833,0.833],"y":[8.194,0.523,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.82,-0.051,0]},"t":40,"s":[70.385,95.308,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.944,0.92,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.082,0.101,0]},"t":41,"s":[71.029,82.647,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.654,0.166,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.168,-2.039,0]},"t":42,"s":[14.793,22.846,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.026,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.11,0.093,0]},"t":43,"s":[33.445,25.195,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.908,1.479,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.754,0]},"t":44,"s":[92.281,46.336,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.981,0.951,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.884,0.071,0]},"t":45,"s":[15.049,48.962,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.761,1.318,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.024,-0.118,0]},"t":46,"s":[7.007,31.245,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.054,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.128,0.066,0]},"t":47,"s":[13.248,38.566,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.957,1.045,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.033,-0.241,0]},"t":48,"s":[24.888,3.264,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.6,1.009,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.09,0.029,0]},"t":49,"s":[5.738,12.341,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,0.936,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.105,0.008,0]},"t":50,"s":[14.963,-1.634,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,0.639,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.091,-0.275,0]},"t":51,"s":[50.015,13.874,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.185,1.318,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.46,0.108,0]},"t":52,"s":[33.27,10.27,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.869,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.078,0.066,0]},"t":53,"s":[35.84,-1.726,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.164,0.842,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.228,-0.049,0]},"t":54,"s":[-3.278,56.039,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,1.064,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.177,0]},"t":55,"s":[-25.859,19.601,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.938,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.139,0.036,0]},"t":56,"s":[41.265,-12.971,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.325,0.731,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.118,-0.247,0]},"t":57,"s":[16.064,44.564,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.066,0.121,0]},"t":58,"s":[26.47,30.06,100]},{"t":59,"s":[-24.557,-2.189,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.8941176470588236,0.11372549019607843,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":7,"op":60,"st":7,"bm":0},{"ddd":0,"ind":39,"ty":4,"nm":"Shape Layer 13","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":15,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":37,"s":[100]},{"t":53,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.569},"o":{"x":0.167,"y":0.167},"t":10,"s":[278.346,255.953,0],"to":[-0.206,-0.234,0],"ti":[0.991,1.248,0]},{"i":{"x":0.833,"y":0.794},"o":{"x":0.167,"y":0.103},"t":11,"s":[277.109,254.547,0],"to":[-0.991,-1.248,0],"ti":[0.533,2.822,0]},{"i":{"x":0.833,"y":0.801},"o":{"x":0.167,"y":0.14},"t":12,"s":[272.4,248.463,0],"to":[-0.533,-2.822,0],"ti":[-2.874,1.607,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.143},"t":13,"s":[273.909,237.613,0],"to":[2.874,-1.607,0],"ti":[-4.877,-0.657,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.18},"t":14,"s":[289.644,238.823,0],"to":[4.877,0.657,0],"ti":[-4.118,-0.868,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":15,"s":[303.171,241.554,0],"to":[4.118,0.868,0],"ti":[-3.424,-0.757,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":16,"s":[314.351,244.031,0],"to":[3.424,0.757,0],"ti":[-2.889,-0.625,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":17,"s":[323.717,246.095,0],"to":[2.889,0.625,0],"ti":[-2.477,-0.51,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":18,"s":[331.687,247.782,0],"to":[2.477,0.51,0],"ti":[-2.154,-0.414,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":19,"s":[338.58,249.155,0],"to":[2.154,0.414,0],"ti":[-1.892,-0.333,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":20,"s":[344.611,250.265,0],"to":[1.892,0.333,0],"ti":[-1.675,-0.265,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":21,"s":[349.933,251.155,0],"to":[1.675,0.265,0],"ti":[-1.493,-0.207,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.177},"t":22,"s":[354.663,251.856,0],"to":[1.493,0.207,0],"ti":[-1.344,-0.162,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.176},"t":23,"s":[358.894,252.395,0],"to":[1.344,0.162,0],"ti":[-1.227,-0.138,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[362.724,252.827,0],"to":[1.227,0.138,0],"ti":[-1.133,-0.127,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":25,"s":[366.255,253.222,0],"to":[1.133,0.127,0],"ti":[-1.051,-0.117,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":26,"s":[369.523,253.587,0],"to":[1.051,0.117,0],"ti":[-0.978,-0.109,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":27,"s":[372.561,253.926,0],"to":[0.978,0.109,0],"ti":[-0.912,-0.101,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":28,"s":[375.391,254.24,0],"to":[0.912,0.101,0],"ti":[-0.852,-0.093,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":29,"s":[378.033,254.529,0],"to":[0.852,0.093,0],"ti":[-0.796,-0.085,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":30,"s":[380.502,254.795,0],"to":[0.796,0.085,0],"ti":[-0.745,-0.076,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":31,"s":[382.812,255.037,0],"to":[0.745,0.076,0],"ti":[-0.697,-0.068,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":32,"s":[384.972,255.253,0],"to":[0.697,0.068,0],"ti":[-0.651,-0.059,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":33,"s":[386.993,255.443,0],"to":[0.651,0.059,0],"ti":[-0.608,-0.05,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":34,"s":[388.881,255.606,0],"to":[0.608,0.05,0],"ti":[-0.567,-0.04,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":35,"s":[390.642,255.741,0],"to":[0.567,0.04,0],"ti":[-0.527,-0.03,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":36,"s":[392.281,255.845,0],"to":[0.527,0.03,0],"ti":[-0.488,-0.019,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":37,"s":[393.801,255.919,0],"to":[0.488,0.019,0],"ti":[-0.45,-0.008,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":38,"s":[395.207,255.96,0],"to":[0.45,0.008,0],"ti":[-0.412,0.002,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":39,"s":[396.499,255.969,0],"to":[0.412,-0.002,0],"ti":[-0.375,0.014,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":40,"s":[397.679,255.945,0],"to":[0.375,-0.014,0],"ti":[-0.338,0.025,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":41,"s":[398.748,255.888,0],"to":[0.338,-0.025,0],"ti":[-0.301,0.036,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":42,"s":[399.706,255.797,0],"to":[0.301,-0.036,0],"ti":[-0.264,0.047,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":43,"s":[400.554,255.673,0],"to":[0.264,-0.047,0],"ti":[-0.227,0.057,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":44,"s":[401.291,255.517,0],"to":[0.227,-0.057,0],"ti":[-0.189,0.067,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":45,"s":[401.915,255.33,0],"to":[0.189,-0.067,0],"ti":[-0.151,0.077,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":46,"s":[402.427,255.112,0],"to":[0.151,-0.077,0],"ti":[-0.113,0.086,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.183},"t":47,"s":[402.824,254.867,0],"to":[0.113,-0.086,0],"ti":[-0.074,0.094,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.18},"t":48,"s":[403.104,254.596,0],"to":[0.074,-0.094,0],"ti":[-0.034,0.101,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.172},"t":49,"s":[403.266,254.302,0],"to":[0.034,-0.101,0],"ti":[0.002,0.108,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.162},"t":50,"s":[403.307,253.989,0],"to":[-0.002,-0.108,0],"ti":[0.028,0.114,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.16},"t":51,"s":[403.256,253.656,0],"to":[-0.028,-0.114,0],"ti":[0.05,0.119,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.159},"t":52,"s":[403.14,253.307,0],"to":[-0.05,-0.119,0],"ti":[0.071,0.117,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.165},"t":53,"s":[402.953,252.945,0],"to":[-0.071,-0.117,0],"ti":[0.081,0.103,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":54,"s":[402.716,252.603,0],"to":[-0.081,-0.103,0],"ti":[0.084,0.081,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":55,"s":[402.468,252.328,0],"to":[-0.084,-0.081,0],"ti":[0.087,0.059,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.175},"t":56,"s":[402.212,252.118,0],"to":[-0.087,-0.059,0],"ti":[0.089,0.038,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.172},"t":57,"s":[401.947,251.973,0],"to":[-0.089,-0.038,0],"ti":[0.091,0.018,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.168},"t":58,"s":[401.676,251.89,0],"to":[-0.091,-0.018,0],"ti":[0.046,0.004,0]},{"t":59,"s":[401.399,251.868,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.884,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":10,"s":[150.023,84.302,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.718,0.778,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.295,-0.001,0]},"t":11,"s":[115.536,94.772,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.873,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.118,0.133,0]},"t":12,"s":[101.947,84.391,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.035,1.291,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.243,0.465,0]},"t":13,"s":[69.48,67.085,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.989,0.813,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,0.065,0]},"t":14,"s":[52.556,63.303,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.856,0.84,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.012,0.15,0]},"t":15,"s":[76.674,80.31,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.805,0.955,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.197,0.174,0]},"t":16,"s":[55.698,101.414,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.152,1.045,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.145,-0.098,0]},"t":17,"s":[40.385,120.878,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,1.219,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,0.029,0]},"t":18,"s":[19.82,111.938,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.016,0.871,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.102,0.06,0]},"t":19,"s":[78.004,125.754,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.949,0.912,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.013,0.234,0]},"t":20,"s":[51.803,75.606,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.062,0.955,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.135,1.443,0]},"t":21,"s":[82.948,47.853,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,-0.218,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.035,-0.097,0]},"t":22,"s":[71.038,46.152,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.784,0.414,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.302,0.089,0]},"t":23,"s":[91.761,46.937,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,0.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.136,0.097,0]},"t":24,"s":[99.655,57.625,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.883,0.788,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.133,-3.004,0]},"t":25,"s":[112.222,122.076,100]},{"i":{"x":[0.833,0.833,0.833],"y":[4.25,0.248,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.289,0.137,0]},"t":26,"s":[111.361,120.336,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.223,0.834,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,0.094,0]},"t":27,"s":[111.012,117.65,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,1.003,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,0.167,0]},"t":28,"s":[124.959,96.101,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.488,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.648,0.003,0]},"t":29,"s":[73.624,74.745,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.933,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,-0.053,0]},"t":30,"s":[79.474,96.927,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.44,0.823,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.337,-0.086,0]},"t":31,"s":[39.363,83.33,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.955,0.354,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.098,0.158,0]},"t":32,"s":[47.32,90.031,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.968,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.099,0.096,0]},"t":33,"s":[92.847,97.529,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.719,0.771,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.051,-0.044,0]},"t":34,"s":[72.017,148.186,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.036,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.118,0.131,0]},"t":35,"s":[84.935,114.918,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.917,1.042,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,-0.658,0]},"t":36,"s":[115.633,56.769,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-3.33,1.406,1]},"o":{"x":[0.167,0.167,0.167],"y":[-14.6,0.028,0]},"t":37,"s":[71.733,63.301,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.842,0.897,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.085,0.069,0]},"t":38,"s":[71.982,53.5,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.296,1.262,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.177,0.435,0]},"t":39,"s":[84.68,111.004,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.873,0.865,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.065,0.063,0]},"t":40,"s":[96.026,124.642,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.082,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.241,0.218,0]},"t":41,"s":[44.359,68.117,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,1.947,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.041,1.94,0]},"t":42,"s":[16.99,33.186,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.848,0.777,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.028,0.077,0]},"t":43,"s":[71.31,31.618,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,1.037,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.185,0.133,0]},"t":44,"s":[30.563,51.009,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.958,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.066,0.026,0]},"t":45,"s":[-2.843,83.398,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.188,0.937,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.084,0.448,0]},"t":46,"s":[15.771,36.624,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,-0.025,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,-0.252,0]},"t":47,"s":[6.523,25.933,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.308,1.025,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.823,0.091,0]},"t":48,"s":[36.606,28.591,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.857,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.066,0.019,0]},"t":49,"s":[33.84,58.624,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.022,0.96,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.199,0.224,0]},"t":50,"s":[46.823,19.638,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.667,0.763,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.018,-0.078,0]},"t":51,"s":[56.147,-3.469,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.868,0.839,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.111,0.129,0]},"t":52,"s":[44.338,8.441,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.071,1.087,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.227,0.172,0]},"t":53,"s":[8.892,30.372,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.016,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.043,0]},"t":54,"s":[-11.629,50.878,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.012,0.823,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.013,-0.049,0]},"t":55,"s":[26.385,8.842,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,1.146,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.011,0.158,0]},"t":56,"s":[-18.969,35.342,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.128,0.932,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.244,0.053,0]},"t":57,"s":[33.051,65.059,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,-0.359,0]},"t":58,"s":[19.792,-16.798,100]},{"t":59,"s":[53.472,-1.378,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.8941176470588236,0.11372549019607843,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":10,"op":60,"st":10,"bm":0},{"ddd":0,"ind":40,"ty":4,"nm":"Shape Layer 12","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":16,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":38,"s":[100]},{"t":54,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.583},"o":{"x":0.167,"y":0.167},"t":11,"s":[278.886,263.108,0],"to":[-0.396,0.093,0],"ti":[2.046,0.295,0]},{"i":{"x":0.833,"y":0.762},"o":{"x":0.167,"y":0.104},"t":12,"s":[276.51,263.668,0],"to":[-2.046,-0.295,0],"ti":[4.357,2.028,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.128},"t":13,"s":[266.607,261.339,0],"to":[-4.357,-2.028,0],"ti":[5.491,3.808,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":14,"s":[250.369,251.499,0],"to":[-5.491,-3.808,0],"ti":[5.141,4.15,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":15,"s":[233.659,238.49,0],"to":[-5.141,-4.15,0],"ti":[4.303,3.645,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":16,"s":[219.52,226.601,0],"to":[-4.303,-3.645,0],"ti":[3.589,3.053,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.183},"t":17,"s":[207.84,216.618,0],"to":[-3.589,-3.053,0],"ti":[3.064,2.559,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":18,"s":[197.984,208.283,0],"to":[-3.064,-2.559,0],"ti":[2.675,2.164,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":19,"s":[189.458,201.262,0],"to":[-2.675,-2.164,0],"ti":[2.381,1.843,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":20,"s":[181.933,195.3,0],"to":[-2.381,-1.843,0],"ti":[2.153,1.575,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":21,"s":[175.17,190.206,0],"to":[-2.153,-1.575,0],"ti":[1.971,1.344,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[169.014,185.852,0],"to":[-1.971,-1.344,0],"ti":[1.824,1.14,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[163.346,182.143,0],"to":[-1.824,-1.14,0],"ti":[1.703,0.954,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":24,"s":[158.073,179.012,0],"to":[-1.703,-0.954,0],"ti":[1.601,0.778,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[153.127,176.419,0],"to":[-1.601,-0.778,0],"ti":[1.512,0.605,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[148.464,174.344,0],"to":[-1.512,-0.605,0],"ti":[1.427,0.431,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":27,"s":[144.054,172.787,0],"to":[-1.427,-0.431,0],"ti":[1.341,0.252,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":28,"s":[139.899,171.759,0],"to":[-1.341,-0.252,0],"ti":[1.243,0.072,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":29,"s":[136.007,171.274,0],"to":[-1.243,-0.072,0],"ti":[1.123,-0.099,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":30,"s":[132.443,171.328,0],"to":[-1.123,0.099,0],"ti":[0.987,-0.242,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":31,"s":[129.266,171.868,0],"to":[-0.987,0.242,0],"ti":[0.844,-0.342,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":32,"s":[126.522,172.779,0],"to":[-0.844,0.342,0],"ti":[0.71,-0.397,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":33,"s":[124.202,173.921,0],"to":[-0.71,0.397,0],"ti":[0.594,-0.415,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":34,"s":[122.26,175.163,0],"to":[-0.594,0.415,0],"ti":[0.499,-0.407,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":35,"s":[120.635,176.409,0],"to":[-0.499,0.407,0],"ti":[0.424,-0.386,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":36,"s":[119.263,177.604,0],"to":[-0.424,0.386,0],"ti":[0.362,-0.359,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":37,"s":[118.093,178.724,0],"to":[-0.362,0.359,0],"ti":[0.312,-0.329,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":38,"s":[117.089,179.757,0],"to":[-0.312,0.329,0],"ti":[0.271,-0.298,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":39,"s":[116.22,180.697,0],"to":[-0.271,0.298,0],"ti":[0.237,-0.267,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":40,"s":[115.461,181.543,0],"to":[-0.237,0.267,0],"ti":[0.208,-0.237,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":41,"s":[114.796,182.298,0],"to":[-0.208,0.237,0],"ti":[0.183,-0.208,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":42,"s":[114.21,182.964,0],"to":[-0.183,0.208,0],"ti":[0.161,-0.181,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":43,"s":[113.696,183.547,0],"to":[-0.161,0.181,0],"ti":[0.141,-0.155,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":44,"s":[113.244,184.051,0],"to":[-0.141,0.155,0],"ti":[0.122,-0.131,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.182},"t":45,"s":[112.85,184.48,0],"to":[-0.122,0.131,0],"ti":[0.105,-0.107,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":46,"s":[112.51,184.835,0],"to":[-0.105,0.107,0],"ti":[0.089,-0.086,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.186},"t":47,"s":[112.219,185.124,0],"to":[-0.089,0.086,0],"ti":[0.074,-0.066,0]},{"i":{"x":0.833,"y":0.855},"o":{"x":0.167,"y":0.19},"t":48,"s":[111.973,185.351,0],"to":[-0.074,0.066,0],"ti":[0.06,-0.046,0]},{"i":{"x":0.833,"y":0.861},"o":{"x":0.167,"y":0.197},"t":49,"s":[111.772,185.518,0],"to":[-0.06,0.046,0],"ti":[0.046,-0.026,0]},{"i":{"x":0.833,"y":0.855},"o":{"x":0.167,"y":0.208},"t":50,"s":[111.612,185.626,0],"to":[-0.046,0.026,0],"ti":[0.035,-0.013,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.196},"t":51,"s":[111.493,185.674,0],"to":[-0.035,0.013,0],"ti":[0.027,-0.015,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.171},"t":52,"s":[111.404,185.706,0],"to":[-0.027,0.015,0],"ti":[0.021,-0.022,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.164},"t":53,"s":[111.333,185.761,0],"to":[-0.021,0.022,0],"ti":[0.015,-0.028,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.161},"t":54,"s":[111.28,185.837,0],"to":[-0.015,0.028,0],"ti":[0.01,-0.033,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":55,"s":[111.243,185.929,0],"to":[-0.01,0.033,0],"ti":[0.006,-0.036,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.162},"t":56,"s":[111.219,186.034,0],"to":[-0.006,0.036,0],"ti":[0.003,-0.038,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.164},"t":57,"s":[111.207,186.148,0],"to":[-0.003,0.038,0],"ti":[0,-0.039,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.168},"t":58,"s":[111.203,186.265,0],"to":[0,0.039,0],"ti":[0,-0.019,0]},{"t":59,"s":[111.205,186.381,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.894,0.956,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":11,"s":[70.371,73.099,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.011,0.757,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.387,-0.094,0]},"t":12,"s":[81.451,49.764,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.535,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.009,0.127,0]},"t":13,"s":[84.489,60.747,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.814,1.157,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.102,-0.112,0]},"t":14,"s":[81.065,81.842,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.818,1.017,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.151,0.054,0]},"t":15,"s":[65.388,72.828,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.16,0.963,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.154,0.014,0]},"t":16,"s":[46.083,98.818,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.958,0.997,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,-0.066,0]},"t":17,"s":[23.236,67.65,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.086,-0.003,0]},"t":18,"s":[90.073,84.988,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.171,1.805,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.806,1.29,0]},"t":19,"s":[57.21,68.22,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,1.029,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.09,0.076,0]},"t":20,"s":[60.29,67.062,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.989,0.774,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.06,0.022,0]},"t":21,"s":[100.479,79.398,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.87,1.114,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.013,0.132,0]},"t":22,"s":[77.165,62.735,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.823,0.904,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.233,0.048,0]},"t":23,"s":[97.376,34.245,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.123,1.029,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.158,0.644,0]},"t":24,"s":[108.611,101.546,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,0.834,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.05,0.021,0]},"t":25,"s":[121.197,111.543,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.983,1.031,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.031,0.167,0]},"t":26,"s":[90.035,98.098,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.835,0.878,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.021,0.023,0]},"t":27,"s":[112.765,84.756,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.751,1.24,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.168,0.262,0]},"t":28,"s":[94.608,103.076,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.906,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.125,0.062,0]},"t":29,"s":[76.78,111.632,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.547,-0.27,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.74,2.198,0]},"t":30,"s":[41.278,78.465,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,1.188,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.072,0.089,0]},"t":31,"s":[36.77,77.159,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.441,0.893,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.649,0.058,0]},"t":32,"s":[70.888,58.554,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.52,1.21,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.098,0.377,0]},"t":33,"s":[71.996,119.193,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.101,0.06,0]},"t":34,"s":[78.321,136.39,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.766,1.004,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.031,-0.115,0]},"t":35,"s":[108.467,75.868,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.959,0.964,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.129,0.004,0]},"t":36,"s":[86.503,101.319,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,0.825,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.079,-0.062,0]},"t":37,"s":[46.776,74.608,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.98,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,0.159,0]},"t":38,"s":[67.187,89.94,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.699,3.529,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.027,1.019,0]},"t":39,"s":[85.904,106.857,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,0.895,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.115,0.081,0]},"t":40,"s":[71.727,108.364,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.929,0.834,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.984,0.401,0]},"t":41,"s":[34.691,61.105,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.084,1.299,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.08,0.168,0]},"t":42,"s":[33.627,48.718,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.001,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.042,0.065,0]},"t":43,"s":[59.325,36.458,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.944,0.928,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.001,-0.001,0]},"t":44,"s":[7.81,92.753,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.889,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.168,-0.538,0]},"t":45,"s":[59.941,37.215,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.33,1.334,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.335,-0.083,0]},"t":46,"s":[42.632,44.663,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.973,0.992,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,0.067,0]},"t":47,"s":[36.89,40.928,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.915,0.649,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.04,-0.009,0]},"t":48,"s":[65.36,59.633,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.366,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[3.347,0.109,0]},"t":49,"s":[46.071,42.784,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.284,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,-0.023,0]},"t":50,"s":[45.578,-11.282,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.976,-5.807,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,7.457,0]},"t":51,"s":[42.321,31.087,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.91,0.98,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.033,0.084,0]},"t":52,"s":[17.582,31.566,100]},{"i":{"x":[0.833,0.833,0.833],"y":[4.784,0.838,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.171,-0.026,0]},"t":53,"s":[35.243,70.197,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.95,1.095,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.082,0.171,0]},"t":54,"s":[36.597,40.776,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.762,1.02,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.123,0.044,0]},"t":55,"s":[-26.212,12.89,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.889,0.94,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.128,0.016,0]},"t":56,"s":[-0.792,72.669,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.804,0.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.335,-0.211,0]},"t":57,"s":[46.469,-1.813,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.145,1.163,0]},"t":58,"s":[62.111,19.267,100]},{"t":59,"s":[83.28,20.895,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.8941176470588236,0.11372549019607843,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":11,"op":60,"st":11,"bm":0},{"ddd":0,"ind":41,"ty":4,"nm":"Shape Layer 11","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":5,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":32,"s":[100]},{"t":48,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.595},"o":{"x":0.167,"y":0.167},"t":5,"s":[280.895,263.305,0],"to":[-0.567,0.175,0],"ti":[2.713,-0.992,0]},{"i":{"x":0.833,"y":0.764},"o":{"x":0.167,"y":0.105},"t":6,"s":[277.49,264.357,0],"to":[-2.713,0.992,0],"ti":[6.007,-2.449,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":7,"s":[264.618,269.257,0],"to":[-6.007,2.449,0],"ti":[8.091,-3.591,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":8,"s":[241.447,279.054,0],"to":[-8.091,3.591,0],"ti":[7.873,-3.782,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":9,"s":[216.07,290.801,0],"to":[-7.873,3.782,0],"ti":[6.648,-3.432,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":10,"s":[194.207,301.746,0],"to":[-6.648,3.432,0],"ti":[5.503,-3.025,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":11,"s":[176.185,311.391,0],"to":[-5.503,3.025,0],"ti":[4.615,-2.683,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":12,"s":[161.19,319.898,0],"to":[-4.615,2.683,0],"ti":[3.935,-2.407,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":13,"s":[148.493,327.489,0],"to":[-3.935,2.407,0],"ti":[3.399,-2.181,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":14,"s":[137.583,334.338,0],"to":[-3.399,2.181,0],"ti":[2.966,-1.993,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":15,"s":[128.098,340.575,0],"to":[-2.966,1.993,0],"ti":[2.607,-1.832,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":16,"s":[119.785,346.294,0],"to":[-2.607,1.832,0],"ti":[2.302,-1.693,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":17,"s":[112.455,351.568,0],"to":[-2.302,1.693,0],"ti":[2.037,-1.57,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":18,"s":[105.973,356.452,0],"to":[-2.037,1.57,0],"ti":[1.803,-1.461,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":19,"s":[100.233,360.99,0],"to":[-1.803,1.461,0],"ti":[1.593,-1.362,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":20,"s":[95.155,365.216,0],"to":[-1.593,1.362,0],"ti":[1.401,-1.271,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":21,"s":[90.677,369.16,0],"to":[-1.401,1.271,0],"ti":[1.224,-1.188,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[86.752,372.844,0],"to":[-1.224,1.188,0],"ti":[1.07,-1.108,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[83.332,376.285,0],"to":[-1.07,1.108,0],"ti":[0.937,-1.031,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":24,"s":[80.332,379.49,0],"to":[-0.937,1.031,0],"ti":[0.818,-0.958,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":25,"s":[77.708,382.471,0],"to":[-0.818,0.958,0],"ti":[0.71,-0.889,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":26,"s":[75.425,385.239,0],"to":[-0.71,0.889,0],"ti":[0.612,-0.822,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":27,"s":[73.45,387.804,0],"to":[-0.612,0.822,0],"ti":[0.523,-0.758,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":28,"s":[71.754,390.173,0],"to":[-0.523,0.758,0],"ti":[0.442,-0.696,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":29,"s":[70.313,392.354,0],"to":[-0.442,0.696,0],"ti":[0.369,-0.637,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":30,"s":[69.102,394.352,0],"to":[-0.369,0.637,0],"ti":[0.304,-0.579,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":31,"s":[68.096,396.175,0],"to":[-0.304,0.579,0],"ti":[0.245,-0.524,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":32,"s":[67.277,397.829,0],"to":[-0.245,0.524,0],"ti":[0.193,-0.471,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":33,"s":[66.624,399.32,0],"to":[-0.193,0.471,0],"ti":[0.147,-0.42,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":34,"s":[66.118,400.656,0],"to":[-0.147,0.42,0],"ti":[0.105,-0.37,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":35,"s":[65.744,401.84,0],"to":[-0.105,0.37,0],"ti":[0.069,-0.322,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":36,"s":[65.487,402.877,0],"to":[-0.069,0.322,0],"ti":[0.037,-0.276,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.183},"t":37,"s":[65.332,403.773,0],"to":[-0.037,0.276,0],"ti":[0.009,-0.231,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.185},"t":38,"s":[65.265,404.532,0],"to":[-0.009,0.231,0],"ti":[-0.014,-0.188,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.187},"t":39,"s":[65.276,405.16,0],"to":[0.014,0.188,0],"ti":[-0.034,-0.146,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.189},"t":40,"s":[65.351,405.66,0],"to":[0.034,0.146,0],"ti":[-0.051,-0.106,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.19},"t":41,"s":[65.481,406.037,0],"to":[0.051,0.106,0],"ti":[-0.064,-0.066,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.186},"t":42,"s":[65.655,406.293,0],"to":[0.064,0.066,0],"ti":[-0.074,-0.028,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.172},"t":43,"s":[65.864,406.433,0],"to":[0.074,0.028,0],"ti":[-0.081,0.01,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.158},"t":44,"s":[66.098,406.459,0],"to":[0.081,-0.01,0],"ti":[-0.083,0.038,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.161},"t":45,"s":[66.349,406.373,0],"to":[0.083,-0.038,0],"ti":[-0.078,0.048,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":46,"s":[66.596,406.231,0],"to":[0.078,-0.048,0],"ti":[-0.069,0.051,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":47,"s":[66.819,406.083,0],"to":[0.069,-0.051,0],"ti":[-0.057,0.054,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":48,"s":[67.01,405.928,0],"to":[0.057,-0.054,0],"ti":[-0.042,0.058,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.174},"t":49,"s":[67.16,405.761,0],"to":[0.042,-0.058,0],"ti":[-0.023,0.063,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.168},"t":50,"s":[67.259,405.58,0],"to":[0.023,-0.063,0],"ti":[-0.001,0.07,0]},{"i":{"x":0.833,"y":0.82},"o":{"x":0.167,"y":0.159},"t":51,"s":[67.298,405.38,0],"to":[0.001,-0.07,0],"ti":[0.023,0.076,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.155},"t":52,"s":[67.266,405.158,0],"to":[-0.023,-0.076,0],"ti":[0.046,0.073,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.166},"t":53,"s":[67.158,404.922,0],"to":[-0.046,-0.073,0],"ti":[0.062,0.06,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.166},"t":54,"s":[66.993,404.72,0],"to":[-0.062,-0.06,0],"ti":[0.076,0.046,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.165},"t":55,"s":[66.783,404.561,0],"to":[-0.076,-0.046,0],"ti":[0.086,0.031,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.164},"t":56,"s":[66.539,404.446,0],"to":[-0.086,-0.031,0],"ti":[0.093,0.016,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.164},"t":57,"s":[66.269,404.376,0],"to":[-0.093,-0.016,0],"ti":[0.097,0,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.164},"t":58,"s":[65.982,404.352,0],"to":[-0.097,0,0],"ti":[0.049,-0.004,0]},{"t":59,"s":[65.686,404.375,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.938,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":5,"s":[49.45,148.28,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.038,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.246,0.447,0]},"t":6,"s":[96.233,59.296,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.894,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.026,0.012,0]},"t":7,"s":[84.39,38.927,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.66,0.535,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.391,-0.32,0]},"t":8,"s":[101.568,62.764,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,0.102,0]},"t":9,"s":[106.222,57.834,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.162,0.853,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.472,-0.001,0]},"t":10,"s":[64.691,35.274,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.955,0.795,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.093,0.193,0]},"t":11,"s":[70.923,57.653,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.827,1.022,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.098,0.141,0]},"t":12,"s":[127.324,74.727,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.161,0.018,0]},"t":13,"s":[101.458,99.601,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.386,1.222,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.314,-0.197,0]},"t":14,"s":[73.693,68.044,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.064,0.905,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,0.061,0]},"t":15,"s":[79.522,77.427,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,1.413,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.036,0.681,0]},"t":16,"s":[46.671,43.016,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.358,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.543,0.069,0]},"t":17,"s":[104.84,38.219,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.006,1.25,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,-0.2,0]},"t":18,"s":[97.095,66.774,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.896,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,0.063,0]},"t":19,"s":[45.158,58.378,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,1.295,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.142,0.42,0]},"t":20,"s":[100.992,91.978,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.782,0.93,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.105,0.065,0]},"t":21,"s":[80.364,100.301,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.813,0.641,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,-0.451,0]},"t":22,"s":[89.494,62.471,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.187,0.991,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.15,0.109,0]},"t":23,"s":[104.25,68.373,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,0.748,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,-0.01,0]},"t":24,"s":[122.573,87.899,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.546,1.004,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.303,0.124,0]},"t":25,"s":[63.056,70.386,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,0.876,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.088,0.004,0]},"t":26,"s":[66.633,34.869,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.782,0.805,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.233,0.254,0]},"t":27,"s":[129.403,72.011,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.806,1.026,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,0.145,0]},"t":28,"s":[112.863,90.111,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,0.817,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.146,0.02,0]},"t":29,"s":[86.228,114.396,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,0.937,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.514,0.153,0]},"t":30,"s":[50.891,82.427,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.248,0.388,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,-0.251,0]},"t":31,"s":[55.82,44.045,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.895,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,0.096,0]},"t":32,"s":[49.632,53.61,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.992,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.404,-0.058,0]},"t":33,"s":[74.255,114.253,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.123,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.009,1.243,0]},"t":34,"s":[80.651,78.395,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.169,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,-0.398,0]},"t":35,"s":[74.861,75.817,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.802,1.723,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.176,0.093,0]},"t":36,"s":[19.702,76.263,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,1.043,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.144,0.075,0]},"t":37,"s":[37.393,80.265,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.507,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.243,0.028,0]},"t":38,"s":[61.705,41.546,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.85,0.881,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.1,-0.133,0]},"t":39,"s":[55.488,100.178,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.039,0.614,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.187,0.279,0]},"t":40,"s":[24.961,77.623,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.03,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.026,0.106,0]},"t":41,"s":[0.375,68.025,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.01,1.132,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.022,-0.04,0]},"t":42,"s":[36.365,33.155,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.865,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.009,0.051,0]},"t":43,"s":[-12.521,56.708,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.071,0.909,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.219,-0.051,0]},"t":44,"s":[42.47,-4.293,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.953,1.38,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.941,0]},"t":45,"s":[76.363,33.422,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.108,0.068,0]},"t":46,"s":[13.419,37.087,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,1.072,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.028,-0.014,0]},"t":47,"s":[40.835,16.71,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.911,1.007,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.176,0.039,0]},"t":48,"s":[20.376,34.131,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.32,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.367,0.007,0]},"t":49,"s":[26.948,1.687,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.176,0.438,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.08,0.755,0]},"t":50,"s":[27.375,36.914,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,0.868,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.057,0.098,0]},"t":51,"s":[15.071,41.282,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.862,1.428,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.019,0.226,0]},"t":52,"s":[53.354,66.388,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.873,0.951,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.21,0.07,0]},"t":53,"s":[22.228,81.077,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.145,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.242,-0.12,0]},"t":54,"s":[1.727,-9.051,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.874,0.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.053,-0.044,0]},"t":55,"s":[-9.032,27.824,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.791,-3.351,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.245,-3.045,0]},"t":56,"s":[20.502,3.731,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,0.93,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.139,0.085,0]},"t":57,"s":[35.73,4.373,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,-0.444,0]},"t":58,"s":[58.616,37.243,100]},{"t":59,"s":[30.021,32.047,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":5,"op":60,"st":5,"bm":0},{"ddd":0,"ind":42,"ty":4,"nm":"Shape Layer 10","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":8,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":13,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":35,"s":[100]},{"t":51,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.609},"o":{"x":0.167,"y":0.167},"t":8,"s":[272.114,255.071,0],"to":[-0.266,0.053,0],"ti":[1.216,-0.382,0]},{"i":{"x":0.833,"y":0.765},"o":{"x":0.167,"y":0.106},"t":9,"s":[270.517,255.387,0],"to":[-1.216,0.382,0],"ti":[2.644,-1.017,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":10,"s":[264.819,257.361,0],"to":[-2.644,1.017,0],"ti":[3.502,-1.605,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":11,"s":[254.654,261.49,0],"to":[-3.502,1.605,0],"ti":[3.128,-2.062,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.179},"t":12,"s":[243.809,266.989,0],"to":[-3.128,2.062,0],"ti":[0.929,-2.4,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.191},"t":13,"s":[235.889,273.86,0],"to":[-0.929,2.4,0],"ti":[-1.493,-1.834,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.174},"t":14,"s":[238.235,281.389,0],"to":[1.493,1.834,0],"ti":[-2.114,-0.944,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":15,"s":[244.846,284.864,0],"to":[2.114,0.944,0],"ti":[-1.909,-0.642,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":16,"s":[250.917,287.051,0],"to":[1.909,0.642,0],"ti":[-1.697,-0.51,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":17,"s":[256.301,288.717,0],"to":[1.697,0.51,0],"ti":[-1.516,-0.438,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":18,"s":[261.1,290.112,0],"to":[1.516,0.438,0],"ti":[-1.362,-0.392,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":19,"s":[265.397,291.343,0],"to":[1.362,0.392,0],"ti":[-1.229,-0.362,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":20,"s":[269.269,292.467,0],"to":[1.229,0.362,0],"ti":[-1.114,-0.341,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":21,"s":[272.774,293.517,0],"to":[1.114,0.341,0],"ti":[-1.012,-0.326,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":22,"s":[275.955,294.515,0],"to":[1.012,0.326,0],"ti":[-0.921,-0.314,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[278.847,295.472,0],"to":[0.921,0.314,0],"ti":[-0.839,-0.305,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[281.482,296.4,0],"to":[0.839,0.305,0],"ti":[-0.762,-0.298,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[283.879,297.304,0],"to":[0.762,0.298,0],"ti":[-0.69,-0.293,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":26,"s":[286.053,298.19,0],"to":[0.69,0.293,0],"ti":[-0.621,-0.288,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[288.016,299.06,0],"to":[0.621,0.288,0],"ti":[-0.554,-0.284,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":28,"s":[289.778,299.918,0],"to":[0.554,0.284,0],"ti":[-0.486,-0.279,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":29,"s":[291.338,300.763,0],"to":[0.486,0.279,0],"ti":[-0.419,-0.273,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":30,"s":[292.695,301.592,0],"to":[0.419,0.273,0],"ti":[-0.351,-0.266,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.178},"t":31,"s":[293.849,302.402,0],"to":[0.351,0.266,0],"ti":[-0.283,-0.258,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.18},"t":32,"s":[294.801,303.189,0],"to":[0.283,0.258,0],"ti":[-0.215,-0.245,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":33,"s":[295.549,303.949,0],"to":[0.215,0.245,0],"ti":[-0.147,-0.225,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.186},"t":34,"s":[296.09,304.662,0],"to":[0.147,0.225,0],"ti":[-0.082,-0.198,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.19},"t":35,"s":[296.429,305.301,0],"to":[0.082,0.198,0],"ti":[-0.024,-0.164,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.191},"t":36,"s":[296.583,305.847,0],"to":[0.024,0.164,0],"ti":[0.026,-0.131,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.179},"t":37,"s":[296.574,306.285,0],"to":[-0.026,0.131,0],"ti":[0.067,-0.108,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.164},"t":38,"s":[296.425,306.636,0],"to":[-0.067,0.108,0],"ti":[0.097,-0.091,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.163},"t":39,"s":[296.17,306.934,0],"to":[-0.097,0.091,0],"ti":[0.117,-0.076,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.163},"t":40,"s":[295.842,307.183,0],"to":[-0.117,0.076,0],"ti":[0.129,-0.065,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.165},"t":41,"s":[295.468,307.393,0],"to":[-0.129,0.065,0],"ti":[0.134,-0.057,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.167},"t":42,"s":[295.071,307.574,0],"to":[-0.134,0.057,0],"ti":[0.133,-0.051,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.169},"t":43,"s":[294.667,307.733,0],"to":[-0.133,0.051,0],"ti":[0.128,-0.046,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.171},"t":44,"s":[294.272,307.878,0],"to":[-0.128,0.046,0],"ti":[0.12,-0.044,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.174},"t":45,"s":[293.898,308.012,0],"to":[-0.12,0.044,0],"ti":[0.108,-0.042,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.176},"t":46,"s":[293.554,308.14,0],"to":[-0.108,0.042,0],"ti":[0.094,-0.041,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.179},"t":47,"s":[293.249,308.264,0],"to":[-0.094,0.041,0],"ti":[0.08,-0.044,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.174},"t":48,"s":[292.988,308.385,0],"to":[-0.08,0.044,0],"ti":[0.068,-0.055,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.166},"t":49,"s":[292.767,308.528,0],"to":[-0.068,0.055,0],"ti":[0.056,-0.069,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.164},"t":50,"s":[292.579,308.714,0],"to":[-0.056,0.069,0],"ti":[0.044,-0.082,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.162},"t":51,"s":[292.428,308.941,0],"to":[-0.044,0.082,0],"ti":[0.03,-0.094,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.161},"t":52,"s":[292.318,309.205,0],"to":[-0.03,0.094,0],"ti":[0.016,-0.105,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.161},"t":53,"s":[292.249,309.504,0],"to":[-0.016,0.105,0],"ti":[0.001,-0.114,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.161},"t":54,"s":[292.223,309.833,0],"to":[-0.001,0.114,0],"ti":[-0.013,-0.122,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.161},"t":55,"s":[292.242,310.189,0],"to":[0.013,0.122,0],"ti":[-0.028,-0.128,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.162},"t":56,"s":[292.304,310.564,0],"to":[0.028,0.128,0],"ti":[-0.042,-0.131,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.163},"t":57,"s":[292.408,310.955,0],"to":[0.042,0.131,0],"ti":[-0.055,-0.133,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.164},"t":58,"s":[292.554,311.353,0],"to":[0.055,0.133,0],"ti":[-0.031,-0.067,0]},{"t":59,"s":[292.738,311.753,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.866,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":8,"s":[127.438,90.591,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.119,0.935,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.22,-0.014,0]},"t":9,"s":[90.096,28.016,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,1.126,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.049,-0.3,0]},"t":10,"s":[67.312,81.384,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.746,0.942,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.141,0.05,0]},"t":11,"s":[122.538,69.77,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.94,1.165,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.124,-0.191,0]},"t":12,"s":[102.02,98.906,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.968,1.084,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.21,0.055,0]},"t":13,"s":[60.035,90.043,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.429,0.887,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.051,0.042,0]},"t":14,"s":[71.968,116.451,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.886,0.984,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,0.318,0]},"t":15,"s":[64.586,63.443,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.12,0.842,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.307,-0.02,0]},"t":16,"s":[109.991,44.631,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.887,0.933,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.049,0.176,0]},"t":17,"s":[126.937,59.824,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.971,1.735,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.316,-0.347,0]},"t":18,"s":[85.604,73.442,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.831,0.945,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.045,0.075,0]},"t":19,"s":[70.771,70.802,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.913,0.639,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.164,-0.162,0]},"t":20,"s":[80.389,96.73,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.179,0.905,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.79,0.108,0]},"t":21,"s":[90.289,87.923,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.589,1.199,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.093,0.673,0]},"t":22,"s":[90.772,58.569,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.878,1.108,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.073,0.059,0]},"t":23,"s":[95.049,54.421,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.33,0.947,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.262,0.047,0]},"t":24,"s":[60.54,68.48,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.901,0.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,-0.148,0]},"t":25,"s":[44.437,36.222,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,4.165,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.525,-0.607,0]},"t":26,"s":[124.279,47.844,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-1.352,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.486,0.081,0]},"t":27,"s":[139.353,46.441,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.883,2.129,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.086,1.924,0]},"t":28,"s":[137.147,101.148,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,0.914,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.29,0.078,0]},"t":29,"s":[77.097,103.625,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.16,2.38,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.222,2.472,0]},"t":30,"s":[52.92,67.602,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.876,0.839,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.079,0]},"t":31,"s":[38.391,66.346,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.058,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.255,0.173,0]},"t":32,"s":[80.849,88.417,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,0.752,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.034,-0.04,0]},"t":33,"s":[101.514,109.026,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.242,1.058,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.446,0.126,0]},"t":34,"s":[66.389,95.048,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.892,1.025,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,0.034,0]},"t":35,"s":[58.31,67.448,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.044,0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.364,0.019,0]},"t":36,"s":[89.867,114.12,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.587,0.887,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.029,-0.038,0]},"t":37,"s":[99.232,53.341,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,1.104,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.104,0.319,0]},"t":38,"s":[84.926,94.953,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.439,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.495,0.046,0]},"t":39,"s":[28.265,109.679,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.035,0.952,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,0.223,0]},"t":40,"s":[24.92,76.564,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,1.618,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,-0.113,0]},"t":41,"s":[45.873,56.751,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.603,0.985,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.138,0.073,0]},"t":42,"s":[16.102,65.143,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.028,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.105,-0.019,0]},"t":43,"s":[27.313,-5.503,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.923,1.561,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.021,1.347,0]},"t":44,"s":[69.528,52.086,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.395,0.972,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.056,0.073,0]},"t":45,"s":[13.347,55.885,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,0.96,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,-0.041,0]},"t":46,"s":[17.457,26.517,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.906,0.989,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,-0.077,0]},"t":47,"s":[-6.114,46.16,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.079,1.13,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.74,-0.013,0]},"t":48,"s":[14.941,35.961,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.225,0.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.041,0.051,0]},"t":49,"s":[17.614,44.753,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.151,-0.369,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,-0.61,0]},"t":50,"s":[12.402,22.217,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,0.089,0]},"t":51,"s":[31.658,24.926,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.879,0.864,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.115,-0.014,0]},"t":52,"s":[-22.38,66.727,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.835,0.983,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.269,0.215,0]},"t":53,"s":[0.27,30.796,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.829,1.068,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.168,-0.021,0]},"t":54,"s":[10.423,7.981,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.817,0.983,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.162,0.038,0]},"t":55,"s":[20.367,26.148,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.012,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.153,-0.021,0]},"t":56,"s":[30.864,-6.905,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.986,1.056,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.011,0.499,0]},"t":57,"s":[43.453,19.525,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.017,0.034,0]},"t":58,"s":[29.038,24.824,100]},{"t":59,"s":[41.021,15.933,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.8941176470588236,0.11372549019607843,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":8,"op":60,"st":8,"bm":0},{"ddd":0,"ind":43,"ty":4,"nm":"Shape Layer 9","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":15,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":37,"s":[100]},{"t":53,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10,"x":"var $bm_rt;\\n$bm_rt = $bm_mul(time, 300);"},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.607},"o":{"x":0.167,"y":0.167},"t":10,"s":[279.864,255.271,0],"to":[-0.343,0.177,0],"ti":[1.691,-0.679,0]},{"i":{"x":0.833,"y":0.764},"o":{"x":0.167,"y":0.106},"t":11,"s":[277.804,256.336,0],"to":[-1.691,0.679,0],"ti":[3.887,-1.09,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.129},"t":12,"s":[269.719,259.347,0],"to":[-3.887,1.09,0],"ti":[3.906,1.153,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.178},"t":13,"s":[254.482,262.875,0],"to":[-3.906,-1.153,0],"ti":[0.302,3.979,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.16},"t":14,"s":[246.284,252.432,0],"to":[-0.302,-3.979,0],"ti":[-2.124,4.016,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":15,"s":[252.67,238.999,0],"to":[2.124,-4.016,0],"ti":[-1.967,3.264,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":16,"s":[259.026,228.335,0],"to":[1.967,-3.264,0],"ti":[-1.669,2.774,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":17,"s":[264.471,219.412,0],"to":[1.669,-2.774,0],"ti":[-1.398,2.426,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":18,"s":[269.041,211.694,0],"to":[1.398,-2.426,0],"ti":[-1.168,2.169,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":19,"s":[272.862,204.857,0],"to":[1.168,-2.169,0],"ti":[-0.974,1.973,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":20,"s":[276.051,198.681,0],"to":[0.974,-1.973,0],"ti":[-0.806,1.82,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":21,"s":[278.705,193.017,0],"to":[0.806,-1.82,0],"ti":[-0.644,1.685,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[280.888,187.764,0],"to":[0.644,-1.685,0],"ti":[-0.482,1.558,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[282.57,182.907,0],"to":[0.482,-1.558,0],"ti":[-0.333,1.443,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":24,"s":[283.781,178.414,0],"to":[0.333,-1.443,0],"ti":[-0.197,1.336,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[284.566,174.251,0],"to":[0.197,-1.336,0],"ti":[-0.074,1.237,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[284.965,170.396,0],"to":[0.074,-1.237,0],"ti":[0.038,1.14,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":27,"s":[285.012,166.831,0],"to":[-0.038,-1.14,0],"ti":[0.139,1.044,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":28,"s":[284.739,163.555,0],"to":[-0.139,-1.044,0],"ti":[0.228,0.947,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":29,"s":[284.178,160.569,0],"to":[-0.228,-0.947,0],"ti":[0.304,0.848,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":30,"s":[283.368,157.874,0],"to":[-0.304,-0.848,0],"ti":[0.363,0.748,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":31,"s":[282.355,155.48,0],"to":[-0.363,-0.748,0],"ti":[0.405,0.649,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":32,"s":[281.189,153.386,0],"to":[-0.405,-0.649,0],"ti":[0.428,0.554,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":33,"s":[279.927,151.584,0],"to":[-0.428,-0.554,0],"ti":[0.433,0.464,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":34,"s":[278.623,150.062,0],"to":[-0.433,-0.464,0],"ti":[0.424,0.383,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":35,"s":[277.326,148.798,0],"to":[-0.424,-0.383,0],"ti":[0.403,0.31,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":36,"s":[276.078,147.765,0],"to":[-0.403,-0.31,0],"ti":[0.372,0.245,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":37,"s":[274.911,146.939,0],"to":[-0.372,-0.245,0],"ti":[0.335,0.19,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":38,"s":[273.847,146.293,0],"to":[-0.335,-0.19,0],"ti":[0.294,0.141,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.183},"t":39,"s":[272.9,145.802,0],"to":[-0.294,-0.141,0],"ti":[0.25,0.1,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.186},"t":40,"s":[272.082,145.445,0],"to":[-0.25,-0.1,0],"ti":[0.205,0.064,0]},{"i":{"x":0.833,"y":0.855},"o":{"x":0.167,"y":0.19},"t":41,"s":[271.398,145.204,0],"to":[-0.205,-0.064,0],"ti":[0.16,0.034,0]},{"i":{"x":0.833,"y":0.861},"o":{"x":0.167,"y":0.196},"t":42,"s":[270.85,145.06,0],"to":[-0.16,-0.034,0],"ti":[0.115,0.009,0]},{"i":{"x":0.833,"y":0.868},"o":{"x":0.167,"y":0.208},"t":43,"s":[270.438,144.999,0],"to":[-0.115,-0.009,0],"ti":[0.071,-0.011,0]},{"i":{"x":0.833,"y":0.86},"o":{"x":0.167,"y":0.226},"t":44,"s":[270.159,145.005,0],"to":[-0.071,0.011,0],"ti":[0.029,-0.027,0]},{"i":{"x":0.833,"y":0.798},"o":{"x":0.167,"y":0.205},"t":45,"s":[270.01,145.067,0],"to":[-0.029,0.027,0],"ti":[-0.01,-0.039,0]},{"i":{"x":0.833,"y":0.789},"o":{"x":0.167,"y":0.142},"t":46,"s":[269.983,145.17,0],"to":[0.01,0.039,0],"ti":[-0.046,-0.046,0]},{"i":{"x":0.833,"y":0.804},"o":{"x":0.167,"y":0.138},"t":47,"s":[270.07,145.3,0],"to":[0.046,0.046,0],"ti":[-0.08,-0.05,0]},{"i":{"x":0.833,"y":0.814},"o":{"x":0.167,"y":0.145},"t":48,"s":[270.262,145.447,0],"to":[0.08,0.05,0],"ti":[-0.111,-0.049,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.151},"t":49,"s":[270.551,145.599,0],"to":[0.111,0.049,0],"ti":[-0.133,-0.043,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.161},"t":50,"s":[270.927,145.744,0],"to":[0.133,0.043,0],"ti":[-0.141,-0.026,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.168},"t":51,"s":[271.347,145.854,0],"to":[0.141,0.026,0],"ti":[-0.142,-0.004,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.166},"t":52,"s":[271.77,145.903,0],"to":[0.142,0.004,0],"ti":[-0.143,0.021,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":53,"s":[272.2,145.876,0],"to":[0.143,-0.021,0],"ti":[-0.142,0.042,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":54,"s":[272.629,145.78,0],"to":[0.142,-0.042,0],"ti":[-0.137,0.061,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.166},"t":55,"s":[273.049,145.622,0],"to":[0.137,-0.061,0],"ti":[-0.13,0.077,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.166},"t":56,"s":[273.452,145.412,0],"to":[0.13,-0.077,0],"ti":[-0.12,0.091,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.167},"t":57,"s":[273.829,145.157,0],"to":[0.12,-0.091,0],"ti":[-0.109,0.102,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.168},"t":58,"s":[274.175,144.866,0],"to":[0.109,-0.102,0],"ti":[-0.051,0.054,0]},{"t":59,"s":[274.483,144.545,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.838,1.875,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":10,"s":[109.864,70.089,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.856,0.69,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.172,0.076,0]},"t":11,"s":[86.733,68.252,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.028,0.993,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.199,0.114,0]},"t":12,"s":[64.924,89.385,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.797,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.021,-0.008,0]},"t":13,"s":[49.171,146.99,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,0.755,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,0.509,0]},"t":14,"s":[70.147,94.506,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.944,1.08,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.941,0.126,0]},"t":15,"s":[100.345,84.223,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.003,0.981,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.08,0.041,0]},"t":16,"s":[99.102,64.322,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.843,0.917,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.002,-0.024,0]},"t":17,"s":[129.334,103.257,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.977,-0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.177,-21.258,0]},"t":18,"s":[98.181,73.031,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.969,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.032,0.087,0]},"t":19,"s":[70.521,73.149,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.819,0.396,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.048,-0.057,0]},"t":20,"s":[90.481,75.792,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.879,0.99,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.155,0.097,0]},"t":21,"s":[77.84,74.224,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.071,0.857,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.269,-0.012,0]},"t":22,"s":[63.067,64.426,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.698,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.2,0]},"t":23,"s":[56.422,72.999,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.807,0.455,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.115,-0.052,0]},"t":24,"s":[68.741,79.126,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.911,0.955,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.147,0.098,0]},"t":25,"s":[101.082,75.355,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.698,0.664,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.265,-0.1,0]},"t":26,"s":[143.609,54.475,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.73,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,0.111,0]},"t":27,"s":[146.608,63.972,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.888,1.027,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.121,0.012,0]},"t":28,"s":[118.472,92.79,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.05,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.324,0.02,0]},"t":29,"s":[55.394,59.194,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.866,0.94,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.031,-0.051,0]},"t":30,"s":[33.53,103.661,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.028,0.632,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.22,-0.21,0]},"t":31,"s":[68.551,76.112,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.9,1.084,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.021,0.108,0]},"t":32,"s":[89.966,83.951,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.007,1.005,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.497,0.042,0]},"t":33,"s":[61.396,110.752,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.966,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.077,0.004,0]},"t":34,"s":[55.64,56.936,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.958,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.058,-0.045,0]},"t":35,"s":[130.926,113.733,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.106,0.944,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.086,-0.085,0]},"t":36,"s":[86.439,76.969,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.866,0.598,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.047,-0.169,0]},"t":37,"s":[108.294,95.156,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.04,0.836,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.221,0.105,0]},"t":38,"s":[58.545,89.141,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.947,0.959,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,0.17,0]},"t":39,"s":[28.376,66.139,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.89,1.228,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.148,-0.079,0]},"t":40,"s":[72.983,43.989,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.128,1.021,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.339,0.061,0]},"t":41,"s":[56.935,55.362,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.199,0.907,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.05,0.017,0]},"t":42,"s":[51.706,12.909,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,1.369,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.059,0.774,0]},"t":43,"s":[64.946,65.932,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.252,1.001,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.238,0.068,0]},"t":44,"s":[20.13,72.328,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.97,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.063,0.001,0]},"t":45,"s":[31.738,37.588,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.893,1.019,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.046,-0.129,0]},"t":46,"s":[-14.966,72.869,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.07,1.207,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.373,0.016,0]},"t":47,"s":[15.041,59.042,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.075,0.894,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.059,0]},"t":48,"s":[23.68,76.032,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.071,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.039,0.396,0]},"t":49,"s":[7.79,16.858,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.941,0.564,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,-0.041,0]},"t":50,"s":[37.924,1.088,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.813,0.957,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.205,0.103,0]},"t":51,"s":[-17.733,11.662,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.814,1.008,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.15,-0.09,0]},"t":52,"s":[-1.626,56.352,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,1.062,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.151,0.008,0]},"t":53,"s":[18.416,34.888,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.817,0.918,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.441,0.035,0]},"t":54,"s":[43.027,58.529,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.877,3.279,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.153,-7.35,0]},"t":55,"s":[48.754,17.354,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.206,0.776,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.259,0.08,0]},"t":56,"s":[55.61,17.816,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.986,1.061,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.059,0.133,0]},"t":57,"s":[58.864,4.732,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.017,0.035,0]},"t":58,"s":[47.582,-17.422,100]},{"t":59,"s":[56.939,20.914,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"ind":0,"ty":"sh","ix":1,"ks":{"a":0,"k":{"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[0.012,10.621],[-5.695,5.34],[-2.557,-7.377],[3.15,-2.096]],"c":true},"ix":2},"nm":"Path 1","mn":"ADBE Vector Shape - Group","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.8941176470588236,0.11372549019607843,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[-3.421,4.043],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Rectangle 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":10,"op":60,"st":10,"bm":0},{"ddd":0,"ind":44,"ty":4,"nm":"Shape Layer 8","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":9,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":14,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":36,"s":[100]},{"t":52,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.626},"o":{"x":0.167,"y":0.167},"t":9,"s":[278.864,255.82,0],"to":[-0.509,0.141,0],"ti":[2.192,-0.891,0]},{"i":{"x":0.833,"y":0.768},"o":{"x":0.167,"y":0.107},"t":10,"s":[275.812,256.665,0],"to":[-2.192,0.891,0],"ti":[4.157,-2.88,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.13},"t":11,"s":[265.713,261.168,0],"to":[-4.157,2.88,0],"ti":[3.28,-5.54,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.161},"t":12,"s":[250.868,273.944,0],"to":[-3.28,5.54,0],"ti":[-0.211,-6.285,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.179},"t":13,"s":[246.03,294.41,0],"to":[0.211,6.285,0],"ti":[-2.493,-4.913,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.185},"t":14,"s":[252.134,311.655,0],"to":[2.493,4.913,0],"ti":[-2.968,-3.51,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.184},"t":15,"s":[260.991,323.886,0],"to":[2.968,3.51,0],"ti":[-2.895,-2.554,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":16,"s":[269.941,332.712,0],"to":[2.895,2.554,0],"ti":[-2.702,-1.885,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":17,"s":[278.362,339.211,0],"to":[2.702,1.885,0],"ti":[-2.495,-1.387,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":18,"s":[286.152,344.024,0],"to":[2.495,1.387,0],"ti":[-2.299,-0.994,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":19,"s":[293.332,347.535,0],"to":[2.299,0.994,0],"ti":[-2.119,-0.668,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":20,"s":[299.949,349.988,0],"to":[2.119,0.668,0],"ti":[-1.954,-0.387,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[306.048,351.543,0],"to":[1.954,0.387,0],"ti":[-1.795,-0.137,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":22,"s":[311.672,352.309,0],"to":[1.795,0.137,0],"ti":[-1.633,0.088,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":23,"s":[316.816,352.363,0],"to":[1.633,-0.088,0],"ti":[-1.471,0.288,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[321.472,351.78,0],"to":[1.471,-0.288,0],"ti":[-1.306,0.458,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":25,"s":[325.642,350.637,0],"to":[1.306,-0.458,0],"ti":[-1.135,0.591,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[329.308,349.031,0],"to":[1.135,-0.591,0],"ti":[-0.968,0.681,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":27,"s":[332.455,347.089,0],"to":[0.968,-0.681,0],"ti":[-0.817,0.731,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":28,"s":[335.117,344.942,0],"to":[0.817,-0.731,0],"ti":[-0.687,0.746,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":29,"s":[337.355,342.705,0],"to":[0.687,-0.746,0],"ti":[-0.581,0.735,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":30,"s":[339.24,340.468,0],"to":[0.581,-0.735,0],"ti":[-0.497,0.706,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":31,"s":[340.842,338.297,0],"to":[0.497,-0.706,0],"ti":[-0.432,0.666,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":32,"s":[342.224,336.234,0],"to":[0.432,-0.666,0],"ti":[-0.383,0.622,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":33,"s":[343.437,334.299,0],"to":[0.383,-0.622,0],"ti":[-0.345,0.576,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":34,"s":[344.521,332.502,0],"to":[0.345,-0.576,0],"ti":[-0.315,0.529,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":35,"s":[345.505,330.845,0],"to":[0.315,-0.529,0],"ti":[-0.292,0.487,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.173},"t":36,"s":[346.41,329.327,0],"to":[0.292,-0.487,0],"ti":[-0.279,0.457,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.17},"t":37,"s":[347.258,327.924,0],"to":[0.279,-0.457,0],"ti":[-0.273,0.434,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.17},"t":38,"s":[348.086,326.587,0],"to":[0.273,-0.434,0],"ti":[-0.265,0.411,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":39,"s":[348.894,325.317,0],"to":[0.265,-0.411,0],"ti":[-0.255,0.387,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":40,"s":[349.675,324.119,0],"to":[0.255,-0.387,0],"ti":[-0.243,0.361,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.172},"t":41,"s":[350.425,322.996,0],"to":[0.243,-0.361,0],"ti":[-0.229,0.334,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.173},"t":42,"s":[351.135,321.951,0],"to":[0.229,-0.334,0],"ti":[-0.212,0.306,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":43,"s":[351.799,320.989,0],"to":[0.212,-0.306,0],"ti":[-0.194,0.278,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":44,"s":[352.409,320.113,0],"to":[0.194,-0.278,0],"ti":[-0.173,0.249,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.177},"t":45,"s":[352.961,319.322,0],"to":[0.173,-0.249,0],"ti":[-0.15,0.219,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.18},"t":46,"s":[353.448,318.619,0],"to":[0.15,-0.219,0],"ti":[-0.126,0.188,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.183},"t":47,"s":[353.864,318.007,0],"to":[0.126,-0.188,0],"ti":[-0.101,0.156,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.188},"t":48,"s":[354.205,317.489,0],"to":[0.101,-0.156,0],"ti":[-0.075,0.131,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.184},"t":49,"s":[354.467,317.068,0],"to":[0.075,-0.131,0],"ti":[-0.049,0.118,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":50,"s":[354.652,316.704,0],"to":[0.049,-0.118,0],"ti":[-0.024,0.112,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.176},"t":51,"s":[354.762,316.358,0],"to":[0.024,-0.112,0],"ti":[0.001,0.104,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.174},"t":52,"s":[354.797,316.033,0],"to":[-0.001,-0.104,0],"ti":[0.025,0.094,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.17},"t":53,"s":[354.758,315.735,0],"to":[-0.025,-0.094,0],"ti":[0.049,0.084,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.165},"t":54,"s":[354.646,315.467,0],"to":[-0.049,-0.084,0],"ti":[0.071,0.072,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.162},"t":55,"s":[354.465,315.232,0],"to":[-0.071,-0.072,0],"ti":[0.091,0.06,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.16},"t":56,"s":[354.22,315.034,0],"to":[-0.091,-0.06,0],"ti":[0.109,0.047,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.16},"t":57,"s":[353.918,314.873,0],"to":[-0.109,-0.047,0],"ti":[0.124,0.033,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.161},"t":58,"s":[353.565,314.754,0],"to":[-0.124,-0.033,0],"ti":[0.066,0.013,0]},{"t":59,"s":[353.172,314.677,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.872,1.101,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":9,"s":[121.274,69.223,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.683,1.187,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.239,0.046,0]},"t":10,"s":[102.389,75.161,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.98,0.955,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,0.058,0]},"t":11,"s":[92.262,61.994,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.931,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.026,-0.097,0]},"t":12,"s":[63.863,104.769,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.274,0.951,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.401,0.012,0]},"t":13,"s":[85.432,85.001,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.092,1.206,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,-0.118,0]},"t":14,"s":[81.722,108.079,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.882,0.993,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,0.059,0]},"t":15,"s":[53.108,98.543,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.078,0.868,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.286,-0.008,0]},"t":16,"s":[113.303,131.662,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.91,0.819,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,0.226,0]},"t":17,"s":[138.039,101.322,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.209,0.154,0]},"t":18,"s":[90.083,83.561,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.383,0.796,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.061,0.009,0]},"t":19,"s":[86.533,62.73,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,1.025,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,0.141,0]},"t":20,"s":[88.578,85.983,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.633,0.981,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.181,0.019,0]},"t":21,"s":[52.598,119.725,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.913,0.912,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.108,-0.024,0]},"t":22,"s":[63.951,75.737,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.114,2.104,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.9,1.706,0]},"t":23,"s":[102.646,109.959,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.968,0.914,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.09,0.077,0]},"t":24,"s":[104.421,111.717,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.984,1.195,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.051,2.682,0]},"t":25,"s":[126.369,86.671,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.465,2.183,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.02,0.058,0]},"t":26,"s":[112.748,85.867,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.895,1.029,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,0.078,0]},"t":27,"s":[123.717,88.551,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.153,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.407,0.021,0]},"t":28,"s":[51.588,47.768,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,2.015,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,1.345,0]},"t":29,"s":[33.003,102.64,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.199,1.013,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.892,0.077,0]},"t":30,"s":[85.701,106.264,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.078,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.059,0.011,0]},"t":31,"s":[81.198,58.493,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.008,0.991,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,-0.058,0]},"t":32,"s":[96.43,113.732,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.02,0.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,-0.01,0]},"t":33,"s":[67.01,81.239,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.963,0.771,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,-0.605,0]},"t":34,"s":[99.306,110.284,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.823,0.301,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.067,0.131,0]},"t":35,"s":[59.341,106.769,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.189,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.158,0.095,0]},"t":36,"s":[81.457,100.615,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.974,1.274,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.058,-1.127,0]},"t":37,"s":[106.222,55.153,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,0.719,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.038,0.064,0]},"t":38,"s":[25.147,58.283,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.061,0.119,0]},"t":39,"s":[80.819,44.863,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.839,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.137,-0.083,0]},"t":40,"s":[48.72,13.108,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.058,0.872,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.173,0]},"t":41,"s":[60.869,29.019,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,1.357,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.034,0.237,0]},"t":42,"s":[73.011,43.835,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.288,1.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.054,0.068,0]},"t":43,"s":[52.393,51.855,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.942,0.906,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.065,0.011,0]},"t":44,"s":[64.871,9.492,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.756,0.775,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.193,0.731,0]},"t":45,"s":[9.336,58.19,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.038,1.229,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.127,0.132,0]},"t":46,"s":[26.111,64.455,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.014,0.878,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.026,0.061,0]},"t":47,"s":[58.4,75.08,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.98,0.77,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.012,0.262,0]},"t":48,"s":[11.51,35.267,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,1.06,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.026,0.131,0]},"t":49,"s":[66.25,16.716,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.825,0.866,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.377,0.035,0]},"t":50,"s":[24.642,-15.996,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.942,0.933,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.159,0.221,0]},"t":51,"s":[32.168,40.121,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.214,0.377,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.193,-0.341,0]},"t":52,"s":[40.464,74.155,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.906,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.06,0.096,0]},"t":53,"s":[37.96,67.475,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.566,1.064,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.711,-0.084,0]},"t":54,"s":[46.883,24.242,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.073,0.829,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.088,0.036,0]},"t":55,"s":[48.067,45.748,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.938,1.019,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.039,0.163,0]},"t":56,"s":[69.141,7.771,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,0.887,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.247,0.016,0]},"t":57,"s":[29.508,-32.064,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,0.316,0]},"t":58,"s":[39.499,17.057,100]},{"t":59,"s":[27.006,34.616,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":9,"op":60,"st":9,"bm":0},{"ddd":0,"ind":45,"ty":4,"nm":"Shape Layer 7","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":5,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":10,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":32,"s":[100]},{"t":48,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.638},"o":{"x":0.167,"y":0.167},"t":5,"s":[269.945,256.942,0],"to":[-0.448,-0.21,0],"ti":[1.991,0.807,0]},{"i":{"x":0.833,"y":0.768},"o":{"x":0.167,"y":0.108},"t":6,"s":[267.255,255.684,0],"to":[-1.991,-0.807,0],"ti":[4.389,1.346,0]},{"i":{"x":0.833,"y":0.825},"o":{"x":0.167,"y":0.13},"t":7,"s":[257.999,252.103,0],"to":[-4.389,-1.346,0],"ti":[6.039,1.363,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.159},"t":8,"s":[240.923,247.608,0],"to":[-6.039,-1.363,0],"ti":[5.978,1.14,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":9,"s":[221.767,243.923,0],"to":[-5.978,-1.14,0],"ti":[5.09,1.054,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":10,"s":[205.053,240.767,0],"to":[-5.09,-1.054,0],"ti":[4.207,1.095,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.183},"t":11,"s":[191.227,237.6,0],"to":[-4.207,-1.095,0],"ti":[3.481,1.188,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":12,"s":[179.812,234.195,0],"to":[-3.481,-1.188,0],"ti":[2.886,1.294,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.179},"t":13,"s":[170.338,230.471,0],"to":[-2.886,-1.294,0],"ti":[2.378,1.39,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":14,"s":[162.496,226.433,0],"to":[-2.378,-1.39,0],"ti":[1.933,1.462,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":15,"s":[156.068,222.132,0],"to":[-1.933,-1.462,0],"ti":[1.54,1.499,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":16,"s":[150.896,217.662,0],"to":[-1.54,-1.499,0],"ti":[1.198,1.498,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":17,"s":[146.827,213.136,0],"to":[-1.198,-1.498,0],"ti":[0.906,1.46,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":18,"s":[143.71,208.674,0],"to":[-0.906,-1.46,0],"ti":[0.665,1.393,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":19,"s":[141.391,204.376,0],"to":[-0.665,-1.393,0],"ti":[0.471,1.306,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":20,"s":[139.72,200.316,0],"to":[-0.471,-1.306,0],"ti":[0.313,1.216,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":21,"s":[138.565,196.541,0],"to":[-0.313,-1.216,0],"ti":[0.184,1.137,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":22,"s":[137.84,193.02,0],"to":[-0.184,-1.137,0],"ti":[0.081,1.063,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":23,"s":[137.463,189.72,0],"to":[-0.081,-1.063,0],"ti":[0.003,0.99,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":24,"s":[137.352,186.641,0],"to":[-0.003,-0.99,0],"ti":[-0.055,0.919,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":25,"s":[137.442,183.779,0],"to":[0.055,-0.919,0],"ti":[-0.097,0.849,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":26,"s":[137.68,181.129,0],"to":[0.097,-0.849,0],"ti":[-0.126,0.781,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":27,"s":[138.024,178.686,0],"to":[0.126,-0.781,0],"ti":[-0.145,0.716,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":28,"s":[138.439,176.442,0],"to":[0.145,-0.716,0],"ti":[-0.156,0.653,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":29,"s":[138.897,174.391,0],"to":[0.156,-0.653,0],"ti":[-0.159,0.592,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[139.374,172.526,0],"to":[0.159,-0.592,0],"ti":[-0.157,0.533,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":31,"s":[139.853,170.839,0],"to":[0.157,-0.533,0],"ti":[-0.15,0.477,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":32,"s":[140.317,169.325,0],"to":[0.15,-0.477,0],"ti":[-0.139,0.422,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":33,"s":[140.753,167.978,0],"to":[0.139,-0.422,0],"ti":[-0.125,0.37,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":34,"s":[141.152,166.791,0],"to":[0.125,-0.37,0],"ti":[-0.109,0.319,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":35,"s":[141.505,165.76,0],"to":[0.109,-0.319,0],"ti":[-0.091,0.27,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.183},"t":36,"s":[141.806,164.879,0],"to":[0.091,-0.27,0],"ti":[-0.072,0.222,0]},{"i":{"x":0.833,"y":0.853},"o":{"x":0.167,"y":0.187},"t":37,"s":[142.051,164.142,0],"to":[0.072,-0.222,0],"ti":[-0.052,0.176,0]},{"i":{"x":0.833,"y":0.858},"o":{"x":0.167,"y":0.192},"t":38,"s":[142.237,163.545,0],"to":[0.052,-0.176,0],"ti":[-0.032,0.132,0]},{"i":{"x":0.833,"y":0.865},"o":{"x":0.167,"y":0.201},"t":39,"s":[142.363,163.084,0],"to":[0.032,-0.132,0],"ti":[-0.012,0.09,0]},{"i":{"x":0.833,"y":0.873},"o":{"x":0.167,"y":0.218},"t":40,"s":[142.428,162.753,0],"to":[0.012,-0.09,0],"ti":[0.007,0.05,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.243},"t":41,"s":[142.433,162.544,0],"to":[-0.007,-0.05,0],"ti":[0.026,0.012,0]},{"i":{"x":0.833,"y":0.765},"o":{"x":0.167,"y":0.167},"t":42,"s":[142.383,162.452,0],"to":[-0.026,-0.012,0],"ti":[0.042,-0.024,0]},{"i":{"x":0.833,"y":0.791},"o":{"x":0.167,"y":0.129},"t":43,"s":[142.28,162.472,0],"to":[-0.042,0.024,0],"ti":[0.057,-0.059,0]},{"i":{"x":0.833,"y":0.816},"o":{"x":0.167,"y":0.139},"t":44,"s":[142.128,162.599,0],"to":[-0.057,0.059,0],"ti":[0.07,-0.085,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.152},"t":45,"s":[141.935,162.827,0],"to":[-0.07,0.085,0],"ti":[0.078,-0.097,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":46,"s":[141.71,163.111,0],"to":[-0.078,0.097,0],"ti":[0.083,-0.101,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.164},"t":47,"s":[141.466,163.41,0],"to":[-0.083,0.101,0],"ti":[0.085,-0.104,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.166},"t":48,"s":[141.211,163.718,0],"to":[-0.085,0.104,0],"ti":[0.084,-0.104,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.168},"t":49,"s":[140.955,164.031,0],"to":[-0.084,0.104,0],"ti":[0.078,-0.103,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.171},"t":50,"s":[140.709,164.343,0],"to":[-0.078,0.103,0],"ti":[0.08,-0.102,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.163},"t":51,"s":[140.484,164.649,0],"to":[-0.08,0.102,0],"ti":[0.095,-0.103,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.159},"t":52,"s":[140.231,164.956,0],"to":[-0.095,0.103,0],"ti":[0.112,-0.102,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.161},"t":53,"s":[139.917,165.266,0],"to":[-0.112,0.102,0],"ti":[0.125,-0.098,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.164},"t":54,"s":[139.556,165.569,0],"to":[-0.125,0.098,0],"ti":[0.134,-0.092,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.166},"t":55,"s":[139.164,165.856,0],"to":[-0.134,0.092,0],"ti":[0.138,-0.083,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.168},"t":56,"s":[138.753,166.12,0],"to":[-0.138,0.083,0],"ti":[0.139,-0.072,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.17},"t":57,"s":[138.335,166.353,0],"to":[-0.139,0.072,0],"ti":[0.135,-0.059,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.172},"t":58,"s":[137.921,166.55,0],"to":[-0.135,0.059,0],"ti":[0.067,-0.026,0]},{"t":59,"s":[137.522,166.706,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.747,0.81,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":5,"s":[67.084,129.75,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.927,0.854,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.075,0.148,0]},"t":6,"s":[62.33,109.198,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.16,1.108,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.593,0.195,0]},"t":7,"s":[109.714,82.895,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.95,1.011,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.047,0]},"t":8,"s":[103.874,63.18,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.652,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.126,0.009,0]},"t":9,"s":[120.913,108.471,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.996,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.11,-0.133,0]},"t":10,"s":[114.137,57.365,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,1.188,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.004,-0.05,0]},"t":11,"s":[92.637,77.058,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.717,0.892,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,0.058,0]},"t":12,"s":[113.058,64.766,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,0.862,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.118,0.369,0]},"t":13,"s":[94.823,104.736,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.686,1.222,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.095,0.21,0]},"t":14,"s":[51.043,116.383,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.935,0.877,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,0.061,0]},"t":15,"s":[71.52,124.057,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.826,0.869,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.292,0.259,0]},"t":16,"s":[128.168,95.925,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,1.124,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.16,0.23,0]},"t":17,"s":[115.579,82.608,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.512,0.751,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.223,0.05,0]},"t":18,"s":[101.896,75.048,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.973,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.1,0.125,0]},"t":19,"s":[93.758,93.897,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.898,0.923,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.041,0.012,0]},"t":20,"s":[54.25,131.368,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.724,0.144,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.465,-1.005,0]},"t":21,"s":[80.758,87.771,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.924,1.014,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.119,0.092,0]},"t":22,"s":[86.54,91.108,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.904,0.995,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.912,0.012,0]},"t":23,"s":[99.918,122.044,100]},{"i":{"x":[0.833,0.833,0.833],"y":[11.64,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.63,-0.006,0]},"t":24,"s":[98.798,85.895,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.179,0.764,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.083,-0.058,0]},"t":25,"s":[98.627,119.725,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,1,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.057,0.129,0]},"t":26,"s":[120.625,99.715,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.129,0.846,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.428,0,0]},"t":27,"s":[51.46,62.943,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.842,1.057,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,0.182,0]},"t":28,"s":[34.76,99.837,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.887,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.176,0.034,0]},"t":29,"s":[77.215,131.147,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.072,1,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.319,-0.044,0]},"t":30,"s":[115.216,78.51,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.783,0.915,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.039,0,0]},"t":31,"s":[128.632,112.897,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,4.711,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.135,4.172,0]},"t":32,"s":[103.693,78.444,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.008,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.662,0.082,0]},"t":33,"s":[63.665,77.742,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.628,0.82,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,-0.016,0]},"t":34,"s":[68.138,109.713,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.039,0.849,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.074,0.155,0]},"t":35,"s":[63.26,82.764,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.925,1.07,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,0.186,0]},"t":36,"s":[104.89,51.456,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.219,1.029,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.727,0.038,0]},"t":37,"s":[43.633,26.02,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.892,0.966,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.06,0.022,0]},"t":38,"s":[49.933,72.695,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.243,0.988,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.367,-0.056,0]},"t":39,"s":[27.075,9.722,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.832,0.92,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,-0.015,0]},"t":40,"s":[20.365,47.293,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.936,3.513,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.165,-2.021,0]},"t":41,"s":[46.638,15.298,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.362,0.978,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.284,0.081,0]},"t":42,"s":[73.418,16.565,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.887,0.789,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,-0.029,0]},"t":43,"s":[67.341,-22.905,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.174,0.978,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.315,0.138,0]},"t":44,"s":[26.919,6.334,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,0.904,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.056,-0.031,0]},"t":45,"s":[12.404,51.222,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.142,1.239,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.264,0.63,0]},"t":46,"s":[57.152,18.419,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.933,1.11,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.052,0.062,0]},"t":47,"s":[46.41,13.418,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,0.964,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.344,0.047,0]},"t":48,"s":[75.422,32.787,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.055,0.769,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.185,-0.063,0]},"t":49,"s":[69.769,-12.086,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.398,0.912,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.033,0.13,0]},"t":50,"s":[65.147,13.473,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.884,2.464,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,1.481,0]},"t":51,"s":[72.836,58.774,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.911,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.296,0.079,0]},"t":52,"s":[28.388,61.475,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.369,0.846,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.317,-0.176,0]},"t":53,"s":[10.975,11.341,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.072,1.254,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,0.182,0]},"t":54,"s":[9.799,27.481,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.039,0.063,0]},"t":55,"s":[44.403,41.158,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,-0.052,0]},"t":56,"s":[-20.175,-14.141,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.193,1.424,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.504,-0.402,0]},"t":57,"s":[25.048,19.928,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.093,0.07,0]},"t":58,"s":[27.7,14.076,100]},{"t":59,"s":[50.717,49.703,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":5,"op":60,"st":5,"bm":0},{"ddd":0,"ind":46,"ty":4,"nm":"Shape Layer 6","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":17,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":39,"s":[100]},{"t":55,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.603},"o":{"x":0.167,"y":0.167},"t":12,"s":[269.021,253.165,0],"to":[0.334,-0.343,0],"ti":[-1.472,1.743,0]},{"i":{"x":0.833,"y":0.765},"o":{"x":0.167,"y":0.105},"t":13,"s":[271.027,251.106,0],"to":[1.472,-1.743,0],"ti":[-3.01,4.091,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":14,"s":[277.851,242.705,0],"to":[3.01,-4.091,0],"ti":[-3.712,5.823,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.159},"t":15,"s":[289.088,226.561,0],"to":[3.712,-5.823,0],"ti":[-3.262,5.954,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":16,"s":[300.123,207.765,0],"to":[3.262,-5.954,0],"ti":[-2.491,5.217,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":17,"s":[308.66,190.839,0],"to":[2.491,-5.217,0],"ti":[-1.89,4.434,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":18,"s":[315.071,176.461,0],"to":[1.89,-4.434,0],"ti":[-1.471,3.789,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.181},"t":19,"s":[320.001,164.236,0],"to":[1.471,-3.789,0],"ti":[-1.175,3.273,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":20,"s":[323.898,153.727,0],"to":[1.175,-3.273,0],"ti":[-0.958,2.857,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":21,"s":[327.049,144.595,0],"to":[0.958,-2.857,0],"ti":[-0.794,2.513,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":22,"s":[329.644,136.588,0],"to":[0.794,-2.513,0],"ti":[-0.666,2.226,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":23,"s":[331.811,129.516,0],"to":[0.666,-2.226,0],"ti":[-0.563,1.981,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":24,"s":[333.638,123.235,0],"to":[0.563,-1.981,0],"ti":[-0.48,1.769,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":25,"s":[335.19,117.632,0],"to":[0.48,-1.769,0],"ti":[-0.41,1.586,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":26,"s":[336.516,112.618,0],"to":[0.41,-1.586,0],"ti":[-0.351,1.424,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":27,"s":[337.65,108.119,0],"to":[0.351,-1.424,0],"ti":[-0.3,1.282,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":28,"s":[338.62,104.073,0],"to":[0.3,-1.282,0],"ti":[-0.255,1.156,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":29,"s":[339.448,100.428,0],"to":[0.255,-1.156,0],"ti":[-0.216,1.044,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":30,"s":[340.152,97.139,0],"to":[0.216,-1.044,0],"ti":[-0.181,0.944,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":31,"s":[340.744,94.166,0],"to":[0.181,-0.944,0],"ti":[-0.149,0.856,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":32,"s":[341.237,91.474,0],"to":[0.149,-0.856,0],"ti":[-0.12,0.778,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":33,"s":[341.639,89.031,0],"to":[0.12,-0.778,0],"ti":[-0.094,0.709,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":34,"s":[341.96,86.808,0],"to":[0.094,-0.709,0],"ti":[-0.069,0.648,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":35,"s":[342.205,84.778,0],"to":[0.069,-0.648,0],"ti":[-0.038,0.59,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":36,"s":[342.374,82.918,0],"to":[0.038,-0.59,0],"ti":[-0.003,0.532,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":37,"s":[342.434,81.24,0],"to":[0.003,-0.532,0],"ti":[0.027,0.479,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":38,"s":[342.395,79.728,0],"to":[-0.027,-0.479,0],"ti":[0.051,0.433,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":39,"s":[342.275,78.364,0],"to":[-0.051,-0.433,0],"ti":[0.069,0.392,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":40,"s":[342.091,77.129,0],"to":[-0.069,-0.392,0],"ti":[0.082,0.356,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":41,"s":[341.861,76.01,0],"to":[-0.082,-0.356,0],"ti":[0.091,0.324,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":42,"s":[341.597,74.992,0],"to":[-0.091,-0.324,0],"ti":[0.096,0.296,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":43,"s":[341.314,74.063,0],"to":[-0.096,-0.296,0],"ti":[0.096,0.272,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":44,"s":[341.023,73.213,0],"to":[-0.096,-0.272,0],"ti":[0.094,0.25,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":45,"s":[340.736,72.433,0],"to":[-0.094,-0.25,0],"ti":[0.088,0.23,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":46,"s":[340.462,71.716,0],"to":[-0.088,-0.23,0],"ti":[0.08,0.212,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":47,"s":[340.209,71.055,0],"to":[-0.08,-0.212,0],"ti":[0.069,0.195,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":48,"s":[339.984,70.445,0],"to":[-0.069,-0.195,0],"ti":[0.056,0.18,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":49,"s":[339.795,69.882,0],"to":[-0.056,-0.18,0],"ti":[0.042,0.165,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":50,"s":[339.646,69.365,0],"to":[-0.042,-0.165,0],"ti":[0.027,0.151,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.176},"t":51,"s":[339.542,68.892,0],"to":[-0.027,-0.151,0],"ti":[0.01,0.143,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.168},"t":52,"s":[339.484,68.461,0],"to":[-0.01,-0.143,0],"ti":[-0.008,0.149,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.16},"t":53,"s":[339.48,68.033,0],"to":[0.008,-0.149,0],"ti":[-0.028,0.16,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.16},"t":54,"s":[339.535,67.568,0],"to":[0.028,-0.16,0],"ti":[-0.047,0.169,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.161},"t":55,"s":[339.647,67.072,0],"to":[0.047,-0.169,0],"ti":[-0.065,0.176,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.162},"t":56,"s":[339.816,66.553,0],"to":[0.065,-0.176,0],"ti":[-0.083,0.179,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.163},"t":57,"s":[340.039,66.018,0],"to":[0.083,-0.179,0],"ti":[-0.099,0.18,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.164},"t":58,"s":[340.312,65.476,0],"to":[0.099,-0.18,0],"ti":[-0.053,0.09,0]},{"t":59,"s":[340.633,64.937,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.887,0.858,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":12,"s":[119.548,103.77,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.117,0.413,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.32,0.202,0]},"t":13,"s":[75.487,97.516,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.839,0.857,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.049,0.097,0]},"t":14,"s":[59.951,93.123,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.919,1.065,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.172,0.2,0]},"t":15,"s":[97.388,66.554,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-2.744,0.826,1]},"o":{"x":[0.167,0.167,0.167],"y":[-2.464,0.037,0]},"t":16,"s":[132.453,47.628,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.896,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.085,0.16,0]},"t":17,"s":[131.306,81.38,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.123,0.94,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.42,-0.023,0]},"t":18,"s":[80.905,117.92,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.005,1.294,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.05,-0.217,0]},"t":19,"s":[68.413,89.26,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.919,1,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.004,0.065,0]},"t":20,"s":[99.306,97.223,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.384,0.975,1]},"o":{"x":[0.167,0.167,0.167],"y":[-3.282,0,0]},"t":21,"s":[66.686,61.177,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.766,1.087,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,-0.035,0]},"t":22,"s":[67.494,97.2,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.858,0.983,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.129,0.043,0]},"t":23,"s":[80.098,71.917,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.241,0.896,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.201,-0.022,0]},"t":24,"s":[102.945,123.537,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,1.252,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,0.427,0]},"t":25,"s":[119.07,82.641,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.957,0.883,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.378,0.063,0]},"t":26,"s":[56.225,72.735,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.764,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.089,0.289,0]},"t":27,"s":[67.587,112.547,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.911,1.01,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.075,-0.022,0]},"t":28,"s":[62.103,128.666,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.018,1.194,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.415,0.009,0]},"t":29,"s":[117.871,115.971,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.145,0.86,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.015,0.058,0]},"t":30,"s":[121.362,130.232,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.007,1.04,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,0.206,0]},"t":31,"s":[117.096,82.77,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.944,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,0.027,0]},"t":32,"s":[77.619,50.423,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.856,0.71,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.172,-1.15,0]},"t":33,"s":[120.562,98.344,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1,0.484,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.199,0.117,0]},"t":34,"s":[106.534,95.106,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.217,1.063,1]},"o":{"x":[0.167,0.167,0.167],"y":[0,0.099,0]},"t":35,"s":[96.391,87.062,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.954,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.06,0.036,0]},"t":36,"s":[106.476,45.285,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.104,0.996,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.179,-0.103,0]},"t":37,"s":[70.097,118.413,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.008,1.087,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.046,-0.004,0]},"t":38,"s":[81.643,85.768,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,0.043,0]},"t":39,"s":[55.641,116.896,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.011,0.466,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.675,1.402,0]},"t":40,"s":[83.987,53.173,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.17,0.898,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,0.099,0]},"t":41,"s":[85.471,49.147,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.916,1.762,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.056,0.465,0]},"t":42,"s":[102.003,27.383,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.131,1.022,1]},"o":{"x":[0.167,0.167,0.167],"y":[7.85,0.075,0]},"t":43,"s":[51.831,22.635,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-1.033,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,0.017,0]},"t":44,"s":[51.293,70.822,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.9,1.802,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.087,1.766,0]},"t":45,"s":[52.676,10.032,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.828,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.491,0.075,0]},"t":46,"s":[85.033,7.021,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.048,0.89,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.162,-0.017,0]},"t":47,"s":[91.644,38.996,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.854,1.128,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.031,0.341,0]},"t":48,"s":[98.641,12.455,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.425,0.857,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.195,0.05,0]},"t":49,"s":[87.583,3.883,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.84,0.985,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.097,0.199,0]},"t":50,"s":[79.333,25.603,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.932,0.756,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.174,-0.018,0]},"t":51,"s":[30.689,41.218,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.381,1.037,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.362,0.127,0]},"t":52,"s":[-14.018,28.37,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.031,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,0.026,0]},"t":53,"s":[-5.646,3.642,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.941,0.845,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.022,-0.306,0]},"t":54,"s":[48.164,39.412,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.726,0.492,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.197,0.18,0]},"t":55,"s":[-25.525,31.762,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,1.052,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.12,0.1,0]},"t":56,"s":[-3.614,25.17,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.387,0.994,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.415,0.032,0]},"t":57,"s":[46.515,-8.426,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.079,-0.006,0]},"t":58,"s":[48.307,46.093,100]},{"t":59,"s":[16.689,-4.541,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":12,"op":60,"st":12,"bm":0},{"ddd":0,"ind":47,"ty":4,"nm":"Shape Layer 5","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":8,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":13,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":35,"s":[100]},{"t":51,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.549},"o":{"x":0.167,"y":0.167},"t":8,"s":[272.244,257.087,0],"to":[-0.124,-0.32,0],"ti":[0.816,1.673,0]},{"i":{"x":0.833,"y":0.761},"o":{"x":0.167,"y":0.102},"t":9,"s":[271.5,255.164,0],"to":[-0.816,-1.673,0],"ti":[2.219,3.742,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.128},"t":10,"s":[267.351,247.049,0],"to":[-2.219,-3.742,0],"ti":[3.396,4.937,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":11,"s":[258.186,232.713,0],"to":[-3.396,-4.937,0],"ti":[3.5,4.76,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.179},"t":12,"s":[246.972,217.428,0],"to":[-3.5,-4.76,0],"ti":[2.913,4.097,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.184},"t":13,"s":[237.183,204.156,0],"to":[-2.913,-4.097,0],"ti":[2.265,3.524,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":14,"s":[229.496,192.846,0],"to":[-2.265,-3.524,0],"ti":[1.734,3.087,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":15,"s":[223.596,183.013,0],"to":[-1.734,-3.087,0],"ti":[1.325,2.741,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":16,"s":[219.089,174.323,0],"to":[-1.325,-2.741,0],"ti":[1.014,2.455,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":17,"s":[215.647,166.568,0],"to":[-1.014,-2.455,0],"ti":[0.78,2.215,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":18,"s":[213.008,159.593,0],"to":[-0.78,-2.215,0],"ti":[0.606,2.01,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":19,"s":[210.967,153.279,0],"to":[-0.606,-2.01,0],"ti":[0.48,1.83,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":20,"s":[209.371,147.533,0],"to":[-0.48,-1.83,0],"ti":[0.392,1.66,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[208.089,142.301,0],"to":[-0.392,-1.66,0],"ti":[0.329,1.501,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[207.021,137.573,0],"to":[-0.329,-1.501,0],"ti":[0.281,1.36,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[206.115,133.293,0],"to":[-0.281,-1.36,0],"ti":[0.243,1.235,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":24,"s":[205.336,129.411,0],"to":[-0.243,-1.235,0],"ti":[0.212,1.123,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":25,"s":[204.658,125.885,0],"to":[-0.212,-1.123,0],"ti":[0.187,1.022,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[204.062,122.676,0],"to":[-0.187,-1.022,0],"ti":[0.165,0.933,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[203.537,119.751,0],"to":[-0.165,-0.933,0],"ti":[0.145,0.852,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":28,"s":[203.075,117.081,0],"to":[-0.145,-0.852,0],"ti":[0.126,0.779,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":29,"s":[202.67,114.64,0],"to":[-0.126,-0.779,0],"ti":[0.108,0.713,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":30,"s":[202.32,112.406,0],"to":[-0.108,-0.713,0],"ti":[0.09,0.654,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":31,"s":[202.024,110.359,0],"to":[-0.09,-0.654,0],"ti":[0.072,0.599,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":32,"s":[201.781,108.484,0],"to":[-0.072,-0.599,0],"ti":[0.054,0.55,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":33,"s":[201.591,106.763,0],"to":[-0.054,-0.55,0],"ti":[0.036,0.505,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":34,"s":[201.455,105.183,0],"to":[-0.036,-0.505,0],"ti":[0.018,0.464,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":35,"s":[201.374,103.731,0],"to":[-0.018,-0.464,0],"ti":[-0.001,0.427,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":36,"s":[201.348,102.397,0],"to":[0.001,-0.427,0],"ti":[-0.019,0.391,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":37,"s":[201.377,101.171,0],"to":[0.019,-0.391,0],"ti":[-0.037,0.358,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":38,"s":[201.462,100.048,0],"to":[0.037,-0.358,0],"ti":[-0.055,0.327,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":39,"s":[201.6,99.021,0],"to":[0.055,-0.327,0],"ti":[-0.072,0.298,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":40,"s":[201.792,98.084,0],"to":[0.072,-0.298,0],"ti":[-0.089,0.269,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":41,"s":[202.034,97.235,0],"to":[0.089,-0.269,0],"ti":[-0.104,0.241,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":42,"s":[202.325,96.471,0],"to":[0.104,-0.241,0],"ti":[-0.118,0.213,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":43,"s":[202.66,95.791,0],"to":[0.118,-0.213,0],"ti":[-0.131,0.186,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":44,"s":[203.035,95.192,0],"to":[0.131,-0.186,0],"ti":[-0.142,0.158,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":45,"s":[203.446,94.675,0],"to":[0.142,-0.158,0],"ti":[-0.151,0.13,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.172},"t":46,"s":[203.886,94.242,0],"to":[0.151,-0.13,0],"ti":[-0.157,0.102,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.172},"t":47,"s":[204.349,93.894,0],"to":[0.157,-0.102,0],"ti":[-0.161,0.078,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.169},"t":48,"s":[204.827,93.632,0],"to":[0.161,-0.078,0],"ti":[-0.163,0.067,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.167},"t":49,"s":[205.315,93.423,0],"to":[0.163,-0.067,0],"ti":[-0.166,0.057,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.167},"t":50,"s":[205.808,93.233,0],"to":[0.166,-0.057,0],"ti":[-0.172,0.042,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.165},"t":51,"s":[206.31,93.079,0],"to":[0.172,-0.042,0],"ti":[-0.179,0.025,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":52,"s":[206.839,92.979,0],"to":[0.179,-0.025,0],"ti":[-0.182,0.01,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.167},"t":53,"s":[207.382,92.928,0],"to":[0.182,-0.01,0],"ti":[-0.181,-0.005,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.168},"t":54,"s":[207.929,92.922,0],"to":[0.181,0.005,0],"ti":[-0.176,-0.017,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":55,"s":[208.467,92.955,0],"to":[0.176,0.017,0],"ti":[-0.169,-0.028,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.17},"t":56,"s":[208.986,93.024,0],"to":[0.169,0.028,0],"ti":[-0.158,-0.038,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":57,"s":[209.478,93.124,0],"to":[0.158,0.038,0],"ti":[-0.145,-0.046,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.173},"t":58,"s":[209.934,93.25,0],"to":[0.145,0.046,0],"ti":[-0.069,-0.025,0]},{"t":59,"s":[210.348,93.398,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.87,0.772,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":8,"s":[119.453,55.55,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.231,0.131,0]},"t":9,"s":[106.941,69.697,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.287,0.579,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.135,-0.59,0]},"t":10,"s":[99.865,94.324,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,0.86,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.065,0.104,0]},"t":11,"s":[102.566,91.274,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.741,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.301,0.205,0]},"t":12,"s":[90.563,78.91,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.394,1.467,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.123,-0.044,0]},"t":13,"s":[85.972,70.463,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.896,0.973,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,0.071,0]},"t":14,"s":[76.313,75.971,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.334,0.741,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.413,-0.04,0]},"t":15,"s":[131.692,39.582,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.923,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.067,0.123,0]},"t":16,"s":[145.687,64.182,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.286,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.941,-0.085,0]},"t":17,"s":[75.572,116.009,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.026,3.589,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.094,-1.303,0]},"t":18,"s":[81.277,90.372,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.88,0.961,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.081,0]},"t":19,"s":[124.472,91.912,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.062,0.923,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.275,-0.072,0]},"t":20,"s":[67.544,42.493,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.861,1.459,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.036,-0.98,0]},"t":21,"s":[42.728,69.036,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,0.917,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.208,0.071,0]},"t":22,"s":[85.996,66.957,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-1.537,-47.561,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.187,74,0]},"t":23,"s":[114.947,80.491,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.566,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.086,0.083,0]},"t":24,"s":[113.049,80.506,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.925,0.917,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.054,0.103,0]},"t":25,"s":[57.15,89.382,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.098,5.726,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.754,-8.491,0]},"t":26,"s":[91.118,126.727,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.09,0.993,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.045,0.082,0]},"t":27,"s":[87.736,126.364,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.908,0.868,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.043,-0.008,0]},"t":28,"s":[95.097,147.312,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.56,0.516,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.913,0.225,0]},"t":29,"s":[79.819,128.237,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.128,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.088,0.101,0]},"t":30,"s":[78.286,116.992,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.612,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.051,-0.463,0]},"t":31,"s":[51.103,62.981,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.755,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.177,0.106,0]},"t":32,"s":[120.097,71.216,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.78,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.126,-0.07,0]},"t":33,"s":[97.985,101.356,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.958,0.93,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.054,0.134,0]},"t":34,"s":[55.144,84.965,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.921,0.999,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.086,-0.43,0]},"t":35,"s":[81.051,58.096,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.51,1.562,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.655,-0.001,0]},"t":36,"s":[68.289,62.454,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.243,1.041,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.072,0.073,0]},"t":37,"s":[68.9,58.129,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.116,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,0.028,0]},"t":38,"s":[64.548,91.624,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.007,0.943,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.048,-0.49,0]},"t":39,"s":[81.603,41.501,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.027,-0.787,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.007,-0.185,0]},"t":40,"s":[40.821,48.785,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,0.981,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,0.087,0]},"t":41,"s":[85.22,46.522,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.945,1.024,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.112,-0.025,0]},"t":42,"s":[26.513,0.273,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.915,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.158,0.018,0]},"t":43,"s":[51.537,35.891,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.358,1.609,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.318,4.617,0]},"t":44,"s":[42.897,-9.828,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,0.696,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,0.073,0]},"t":45,"s":[44.69,-10.668,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,0.946,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.095,0.115,0]},"t":46,"s":[72.117,-3.689,100]},{"i":{"x":[0.833,0.833,0.833],"y":[2.789,0.815,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.25,-0.156,0]},"t":47,"s":[59.287,14.778,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.827,0.946,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.08,0.152,0]},"t":48,"s":[60.089,8.351,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.698,-0.404,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.161,-0.154,0]},"t":49,"s":[42.069,0.529,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.95,0.986,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.115,0.089,0]},"t":50,"s":[22.609,3.273,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.627,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.126,-0.016,0]},"t":51,"s":[-28.548,46.76,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.957,0.182,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.107,-0.67,0]},"t":52,"s":[-8.217,10.367,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.796,0.936,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.089,0.093,0]},"t":53,"s":[62.57,14.39,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.988,0.546,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,-0.271,0]},"t":54,"s":[28.321,49.863,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.858,1,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.015,0.102,0]},"t":55,"s":[-21.183,41.528,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.006,0.962,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.202,0,0]},"t":56,"s":[20.981,4.416,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.939,0.899,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.005,-0.071,0]},"t":57,"s":[50.62,41.308,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.223,0.468,0]},"t":58,"s":[18.942,21.34,100]},{"t":59,"s":[27.547,17.011,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":8,"op":60,"st":8,"bm":0},{"ddd":0,"ind":48,"ty":4,"nm":"Shape Layer 4","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":2,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":7,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":29,"s":[100]},{"t":45,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.599},"o":{"x":0.167,"y":0.167},"t":2,"s":[273.94,260.349,0],"to":[-0.179,0.314,0],"ti":[0.915,-1.476,0]},{"i":{"x":0.833,"y":0.765},"o":{"x":0.167,"y":0.105},"t":3,"s":[272.864,262.231,0],"to":[-0.915,1.476,0],"ti":[2.375,-3.065,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":4,"s":[268.451,269.206,0],"to":[-2.375,3.065,0],"ti":[3.841,-3.636,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":5,"s":[258.614,280.62,0],"to":[-3.841,3.636,0],"ti":[4.317,-2.988,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.178},"t":6,"s":[245.406,291.021,0],"to":[-4.317,2.988,0],"ti":[3.939,-2.213,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":7,"s":[232.715,298.549,0],"to":[-3.939,2.213,0],"ti":[3.368,-1.76,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":8,"s":[221.77,304.296,0],"to":[-3.368,1.76,0],"ti":[2.855,-1.514,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":9,"s":[212.51,309.106,0],"to":[-2.855,1.514,0],"ti":[2.434,-1.366,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":10,"s":[204.641,313.381,0],"to":[-2.434,1.366,0],"ti":[2.092,-1.261,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":11,"s":[197.904,317.3,0],"to":[-2.092,1.261,0],"ti":[1.81,-1.177,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":12,"s":[192.09,320.946,0],"to":[-1.81,1.177,0],"ti":[1.575,-1.103,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":13,"s":[187.044,324.36,0],"to":[-1.575,1.103,0],"ti":[1.376,-1.035,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":14,"s":[182.643,327.564,0],"to":[-1.376,1.035,0],"ti":[1.205,-0.972,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":15,"s":[178.791,330.572,0],"to":[-1.205,0.972,0],"ti":[1.057,-0.911,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":16,"s":[175.413,333.393,0],"to":[-1.057,0.911,0],"ti":[0.928,-0.852,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[172.447,336.036,0],"to":[-0.928,0.852,0],"ti":[0.814,-0.797,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":18,"s":[169.845,338.508,0],"to":[-0.814,0.797,0],"ti":[0.712,-0.743,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":19,"s":[167.565,340.816,0],"to":[-0.712,0.743,0],"ti":[0.621,-0.692,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":20,"s":[165.574,342.967,0],"to":[-0.621,0.692,0],"ti":[0.539,-0.642,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":21,"s":[163.841,344.967,0],"to":[-0.539,0.642,0],"ti":[0.464,-0.595,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":22,"s":[162.342,346.821,0],"to":[-0.464,0.595,0],"ti":[0.397,-0.549,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":23,"s":[161.056,348.535,0],"to":[-0.397,0.549,0],"ti":[0.336,-0.504,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":24,"s":[159.962,350.113,0],"to":[-0.336,0.504,0],"ti":[0.28,-0.461,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":25,"s":[159.042,351.559,0],"to":[-0.28,0.461,0],"ti":[0.23,-0.419,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":26,"s":[158.282,352.878,0],"to":[-0.23,0.419,0],"ti":[0.185,-0.378,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":27,"s":[157.663,354.073,0],"to":[-0.185,0.378,0],"ti":[0.145,-0.339,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":28,"s":[157.172,355.149,0],"to":[-0.145,0.339,0],"ti":[0.11,-0.301,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.18},"t":29,"s":[156.793,356.108,0],"to":[-0.11,0.301,0],"ti":[0.08,-0.264,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.181},"t":30,"s":[156.511,356.955,0],"to":[-0.08,0.264,0],"ti":[0.054,-0.227,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.182},"t":31,"s":[156.313,357.691,0],"to":[-0.054,0.227,0],"ti":[0.034,-0.191,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.185},"t":32,"s":[156.184,358.318,0],"to":[-0.034,0.191,0],"ti":[0.018,-0.156,0]},{"i":{"x":0.833,"y":0.854},"o":{"x":0.167,"y":0.188},"t":33,"s":[156.11,358.839,0],"to":[-0.018,0.156,0],"ti":[0.007,-0.122,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.194},"t":34,"s":[156.077,359.255,0],"to":[-0.007,0.122,0],"ti":[-0.009,-0.098,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.177},"t":35,"s":[156.07,359.568,0],"to":[0.009,0.098,0],"ti":[-0.034,-0.089,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.162},"t":36,"s":[156.129,359.841,0],"to":[0.034,0.089,0],"ti":[-0.059,-0.084,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.16},"t":37,"s":[156.271,360.101,0],"to":[0.059,0.084,0],"ti":[-0.08,-0.078,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.16},"t":38,"s":[156.482,360.344,0],"to":[0.08,0.078,0],"ti":[-0.098,-0.071,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.161},"t":39,"s":[156.751,360.569,0],"to":[0.098,0.071,0],"ti":[-0.112,-0.063,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":40,"s":[157.068,360.771,0],"to":[0.112,0.063,0],"ti":[-0.124,-0.054,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.163},"t":41,"s":[157.425,360.948,0],"to":[0.124,0.054,0],"ti":[-0.129,-0.049,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.166},"t":42,"s":[157.811,361.097,0],"to":[0.129,0.049,0],"ti":[-0.126,-0.051,0]},{"i":{"x":0.833,"y":0.837},"o":{"x":0.167,"y":0.169},"t":43,"s":[158.201,361.241,0],"to":[0.126,0.051,0],"ti":[-0.118,-0.056,0]},{"i":{"x":0.833,"y":0.838},"o":{"x":0.167,"y":0.171},"t":44,"s":[158.569,361.404,0],"to":[0.118,0.056,0],"ti":[-0.109,-0.06,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.172},"t":45,"s":[158.91,361.58,0],"to":[0.109,0.06,0],"ti":[-0.098,-0.062,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":46,"s":[159.221,361.765,0],"to":[0.098,0.062,0],"ti":[-0.087,-0.063,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.174},"t":47,"s":[159.499,361.954,0],"to":[0.087,0.063,0],"ti":[-0.074,-0.063,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":48,"s":[159.741,362.144,0],"to":[0.074,0.063,0],"ti":[-0.062,-0.061,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.178},"t":49,"s":[159.945,362.33,0],"to":[0.062,0.061,0],"ti":[-0.049,-0.058,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.18},"t":50,"s":[160.111,362.509,0],"to":[0.049,0.058,0],"ti":[-0.036,-0.053,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.182},"t":51,"s":[160.237,362.677,0],"to":[0.036,0.053,0],"ti":[-0.023,-0.048,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.185},"t":52,"s":[160.325,362.83,0],"to":[0.023,0.048,0],"ti":[-0.011,-0.042,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.187},"t":53,"s":[160.376,362.965,0],"to":[0.011,0.042,0],"ti":[0.001,-0.034,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.187},"t":54,"s":[160.39,363.08,0],"to":[-0.001,0.034,0],"ti":[0.011,-0.026,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.178},"t":55,"s":[160.371,363.171,0],"to":[-0.011,0.026,0],"ti":[0.021,-0.017,0]},{"i":{"x":0.833,"y":0.82},"o":{"x":0.167,"y":0.164},"t":56,"s":[160.322,363.235,0],"to":[-0.021,0.017,0],"ti":[0.029,-0.006,0]},{"i":{"x":0.833,"y":0.816},"o":{"x":0.167,"y":0.155},"t":57,"s":[160.247,363.27,0],"to":[-0.029,0.006,0],"ti":[0.035,0.004,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.152},"t":58,"s":[160.149,363.273,0],"to":[-0.035,-0.004,0],"ti":[0.019,0.005,0]},{"t":59,"s":[160.035,363.244,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.052,0.912,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":2,"s":[91.411,73.902,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.827,0.39,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.032,1.61,0]},"t":3,"s":[65.83,91.918,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.046,1.571,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.161,0.097,0]},"t":4,"s":[107.448,92.901,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.942,0.923,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.03,0.073,0]},"t":5,"s":[152.078,99.113,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.847,0.674,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.195,-0.969,0]},"t":6,"s":[82.703,50.368,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.029,0.728,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.183,0.112,0]},"t":7,"s":[103.48,54.226,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.79,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.022,0.12,0]},"t":8,"s":[120.884,65.462,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.94,0.95,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.138,-0.135,0]},"t":9,"s":[97.362,90.861,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.855,1.74,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.209,-0.125,0]},"t":10,"s":[61.56,81.156,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.023,1.03,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.196,0.075,0]},"t":11,"s":[71.774,85.037,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.369,0.932,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.018,0.022,0]},"t":12,"s":[79.328,46.686,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.894,0.844,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.068,-0.362,0]},"t":13,"s":[69.652,98.9,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.223,1.075,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.384,0.179,0]},"t":14,"s":[122.192,89.119,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.92,0.879,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.061,0.04,0]},"t":15,"s":[136.768,80.556,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.268,1.123,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.772,0.269,0]},"t":16,"s":[83.231,96.867,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.041,0.715,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,0.05,0]},"t":17,"s":[85.636,104.189,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.882,0.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.028,0.118,0]},"t":18,"s":[119.843,86.028,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.101,-1.337,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.286,-2.378,0]},"t":19,"s":[68.608,42.162,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.862,0.996,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.046,0.086,0]},"t":20,"s":[47.52,43.647,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.979,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.21,-0.004,0]},"t":21,"s":[94.079,83.818,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.63,0.826,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.12,-0.028,0]},"t":22,"s":[124.614,45.446,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.959,0.867,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.108,0.16,0]},"t":23,"s":[112.121,74.09,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.798,1.043,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.082,0.224,0]},"t":24,"s":[69.17,105.139,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.989,1.056,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.142,0.028,0]},"t":25,"s":[90.804,123.544,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.836,0.968,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.013,0.034,0]},"t":26,"s":[121.588,95.744,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.88,0.908,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.17,-0.053,0]},"t":27,"s":[95.051,142.325,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.702,0.553,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.275,0.917,0]},"t":28,"s":[69.508,113.896,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.63,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.116,0.102,0]},"t":29,"s":[58.415,111.056,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,0.97,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.069,0.108,0]},"t":30,"s":[29.777,98.67,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.682,0.963,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,-0.046,0]},"t":31,"s":[45.407,56.133,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.99,0.982,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,-0.066,0]},"t":32,"s":[34.491,83.602,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,1.248,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.012,-0.022,0]},"t":33,"s":[3.789,68.283,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,0.862,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.224,0.062,0]},"t":34,"s":[30.69,80.365,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.11,1.012,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,0.211,0]},"t":35,"s":[46.645,32.371,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.919,0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.047,0.011,0]},"t":36,"s":[32.343,1.12,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.962,1.001,1]},"o":{"x":[0.167,0.167,0.167],"y":[-2.466,-0.037,0]},"t":37,"s":[65.552,37.014,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.026,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.087,0.001,0]},"t":38,"s":[64.467,12.19,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.059,-0.017,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,-0.473,0]},"t":39,"s":[40,37.362,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.936,0.93,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.034,0.091,0]},"t":40,"s":[72.228,33.594,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.744,0.949,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.269,-0.441,0]},"t":41,"s":[17.233,-8.641,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,1.149,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.124,-0.134,0]},"t":42,"s":[30.25,-1.935,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.994,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.017,0.053,0]},"t":43,"s":[57.17,-4.506,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.998,1.452,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.006,-0.034,0]},"t":44,"s":[23.549,2.656,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.864,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.002,0.07,0]},"t":45,"s":[54.911,-2.436,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.044,1.08,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.138,0.216,0]},"t":46,"s":[24.231,30.266,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.98,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.029,0.041,0]},"t":47,"s":[35.767,50.837,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.284,0.351,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.027,-0.49,0]},"t":48,"s":[18.177,10.486,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.93,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,0.096,0]},"t":49,"s":[31.443,16.354,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.369,0.815,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.427,-0.033,0]},"t":50,"s":[-27.024,56.172,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,0.996,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,0.152,0]},"t":51,"s":[-17.47,27.705,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.145,0.868,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.306,-0.005,0]},"t":52,"s":[45.271,-6.879,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.872,1.081,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.053,0.227,0]},"t":53,"s":[68.762,25.916,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.048,1.039,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.239,0.041,0]},"t":54,"s":[4.456,44.914,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,0.959,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.03,0.026,0]},"t":55,"s":[-30.068,7.408,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.082,0.922,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.101,-0.08,0]},"t":56,"s":[24.371,62.294,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.894,2.558,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.041,-1.19,0]},"t":57,"s":[-0.235,34.376,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.394,0.079,0]},"t":58,"s":[48.622,36.203,100]},{"t":59,"s":[61.716,0.231,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":2,"op":60,"st":2,"bm":0},{"ddd":0,"ind":49,"ty":4,"nm":"Shape Layer 3","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":7,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":12,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":34,"s":[100]},{"t":50,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.551},"o":{"x":0.167,"y":0.167},"t":7,"s":[272.377,258.087,0],"to":[-0.394,0.137,0],"ti":[2.139,-0.681,0]},{"i":{"x":0.833,"y":0.762},"o":{"x":0.167,"y":0.102},"t":8,"s":[270.01,258.906,0],"to":[-2.139,0.681,0],"ti":[5.098,-1.104,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.128},"t":9,"s":[259.541,262.172,0],"to":[-5.098,1.104,0],"ti":[7.13,-0.303,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":10,"s":[239.424,265.531,0],"to":[-7.13,0.303,0],"ti":[6.918,1.295,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":11,"s":[216.76,263.988,0],"to":[-6.918,-1.295,0],"ti":[5.553,2.383,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":12,"s":[197.916,257.762,0],"to":[-5.553,-2.383,0],"ti":[4.294,2.71,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":13,"s":[183.44,249.693,0],"to":[-4.294,-2.71,0],"ti":[3.412,2.643,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":14,"s":[172.149,241.499,0],"to":[-3.412,-2.643,0],"ti":[2.824,2.427,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":15,"s":[162.967,233.838,0],"to":[-2.824,-2.427,0],"ti":[2.43,2.167,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":16,"s":[155.204,226.937,0],"to":[-2.43,-2.167,0],"ti":[2.166,1.903,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":17,"s":[148.386,220.833,0],"to":[-2.166,-1.903,0],"ti":[1.988,1.643,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":18,"s":[142.207,215.519,0],"to":[-1.988,-1.643,0],"ti":[1.866,1.384,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":19,"s":[136.458,210.978,0],"to":[-1.866,-1.384,0],"ti":[1.768,1.113,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":20,"s":[131.01,207.214,0],"to":[-1.768,-1.113,0],"ti":[1.677,0.828,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":21,"s":[125.848,204.301,0],"to":[-1.677,-0.828,0],"ti":[1.589,0.546,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":22,"s":[120.949,202.245,0],"to":[-1.589,-0.546,0],"ti":[1.493,0.277,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":23,"s":[116.315,201.024,0],"to":[-1.493,-0.277,0],"ti":[1.38,0.034,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":24,"s":[111.993,200.584,0],"to":[-1.38,-0.034,0],"ti":[1.254,-0.168,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[108.033,200.82,0],"to":[-1.254,0.168,0],"ti":[1.122,-0.322,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.174},"t":26,"s":[104.469,201.591,0],"to":[-1.122,0.322,0],"ti":[0.991,-0.429,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":27,"s":[101.304,202.75,0],"to":[-0.991,0.429,0],"ti":[0.869,-0.496,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":28,"s":[98.52,204.163,0],"to":[-0.869,0.496,0],"ti":[0.759,-0.533,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.173},"t":29,"s":[96.088,205.725,0],"to":[-0.759,0.533,0],"ti":[0.662,-0.548,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.173},"t":30,"s":[93.965,207.359,0],"to":[-0.662,0.548,0],"ti":[0.577,-0.547,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":31,"s":[92.114,209.011,0],"to":[-0.577,0.547,0],"ti":[0.502,-0.535,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":32,"s":[90.501,210.642,0],"to":[-0.502,0.535,0],"ti":[0.436,-0.515,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":33,"s":[89.1,212.221,0],"to":[-0.436,0.515,0],"ti":[0.376,-0.488,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":34,"s":[87.887,213.729,0],"to":[-0.376,0.488,0],"ti":[0.324,-0.458,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":35,"s":[86.842,215.15,0],"to":[-0.324,0.458,0],"ti":[0.277,-0.424,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":36,"s":[85.945,216.475,0],"to":[-0.277,0.424,0],"ti":[0.234,-0.388,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":37,"s":[85.182,217.694,0],"to":[-0.234,0.388,0],"ti":[0.195,-0.349,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.178},"t":38,"s":[84.541,218.8,0],"to":[-0.195,0.349,0],"ti":[0.16,-0.31,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":39,"s":[84.009,219.79,0],"to":[-0.16,0.31,0],"ti":[0.129,-0.269,0]},{"i":{"x":0.833,"y":0.849},"o":{"x":0.167,"y":0.182},"t":40,"s":[83.578,220.658,0],"to":[-0.129,0.269,0],"ti":[0.099,-0.226,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.186},"t":41,"s":[83.238,221.401,0],"to":[-0.099,0.226,0],"ti":[0.072,-0.183,0]},{"i":{"x":0.833,"y":0.857},"o":{"x":0.167,"y":0.19},"t":42,"s":[82.984,222.016,0],"to":[-0.072,0.183,0],"ti":[0.048,-0.14,0]},{"i":{"x":0.833,"y":0.865},"o":{"x":0.167,"y":0.199},"t":43,"s":[82.806,222.502,0],"to":[-0.048,0.14,0],"ti":[0.026,-0.097,0]},{"i":{"x":0.833,"y":0.882},"o":{"x":0.167,"y":0.219},"t":44,"s":[82.697,222.858,0],"to":[-0.026,0.097,0],"ti":[0.005,-0.052,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.284},"t":45,"s":[82.651,223.082,0],"to":[-0.005,0.052,0],"ti":[-0.014,-0.008,0]},{"i":{"x":0.833,"y":0.751},"o":{"x":0.167,"y":0.174},"t":46,"s":[82.666,223.173,0],"to":[0.014,0.008,0],"ti":[-0.027,0.032,0]},{"i":{"x":0.833,"y":0.805},"o":{"x":0.167,"y":0.125},"t":47,"s":[82.736,223.128,0],"to":[0.027,-0.032,0],"ti":[-0.027,0.062,0]},{"i":{"x":0.833,"y":0.812},"o":{"x":0.167,"y":0.145},"t":48,"s":[82.826,222.979,0],"to":[0.027,-0.062,0],"ti":[-0.023,0.085,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.149},"t":49,"s":[82.9,222.757,0],"to":[0.023,-0.085,0],"ti":[-0.025,0.099,0]},{"i":{"x":0.833,"y":0.831},"o":{"x":0.167,"y":0.161},"t":50,"s":[82.964,222.469,0],"to":[0.025,-0.099,0],"ti":[-0.032,0.102,0]},{"i":{"x":0.833,"y":0.834},"o":{"x":0.167,"y":0.165},"t":51,"s":[83.05,222.164,0],"to":[0.032,-0.102,0],"ti":[-0.036,0.101,0]},{"i":{"x":0.833,"y":0.836},"o":{"x":0.167,"y":0.167},"t":52,"s":[83.155,221.857,0],"to":[0.036,-0.101,0],"ti":[-0.038,0.098,0]},{"i":{"x":0.833,"y":0.839},"o":{"x":0.167,"y":0.17},"t":53,"s":[83.269,221.556,0],"to":[0.038,-0.098,0],"ti":[-0.038,0.093,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.172},"t":54,"s":[83.385,221.268,0],"to":[0.038,-0.093,0],"ti":[-0.035,0.085,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.175},"t":55,"s":[83.497,221,0],"to":[0.035,-0.085,0],"ti":[-0.031,0.076,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.179},"t":56,"s":[83.598,220.758,0],"to":[0.031,-0.076,0],"ti":[-0.024,0.065,0]},{"i":{"x":0.833,"y":0.853},"o":{"x":0.167,"y":0.184},"t":57,"s":[83.682,220.545,0],"to":[0.024,-0.065,0],"ti":[-0.017,0.053,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.192},"t":58,"s":[83.745,220.367,0],"to":[0.017,-0.053,0],"ti":[-0.006,0.023,0]},{"t":59,"s":[83.781,220.227,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.852,3.785,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":7,"s":[121.49,88.737,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.91,0.921,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.19,0.081,0]},"t":8,"s":[91.076,88.159,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.429,-3.486,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.162,-1.415,0]},"t":9,"s":[67.412,108.062,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.033,0.953,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.098,0.085,0]},"t":10,"s":[65.583,106.955,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.841,0.807,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.024,-0.106,0]},"t":11,"s":[54.874,48.475,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.733,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.176,0.146,0]},"t":12,"s":[69.879,74.186,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.032,1.001,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.121,0.246,0]},"t":13,"s":[83.434,108.14,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.978,0.868,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.023,0.001,0]},"t":14,"s":[113.236,125.507,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.937,0.727,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.03,0.227,0]},"t":15,"s":[72.085,108.002,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.63,0.848,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.261,0.12,0]},"t":16,"s":[102.383,97.845,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.922,0.819,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.108,0.184,0]},"t":17,"s":[95.054,74.753,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.758,1.104,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.116,0.155,0]},"t":18,"s":[69.833,55.593,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.253,0.933,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.127,0.046,0]},"t":19,"s":[71.585,33.206,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.845,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,-0.336,0]},"t":20,"s":[74.924,83.413,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.895,1.179,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.069,0.18,0]},"t":21,"s":[121.778,73.433,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.929,0.903,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.41,0.057,0]},"t":22,"s":[96.063,64.842,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-2.344,1.503,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.493,0.61,0]},"t":23,"s":[89.506,91.864,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1,1.002,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.085,0.071,0]},"t":24,"s":[90.453,96.143,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.913,0.848,1]},"o":{"x":[0.167,0.167,0.167],"y":[0,0.002,0]},"t":25,"s":[127.532,66.021,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.203,1.019,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.831,0.185,0]},"t":26,"s":[90.276,96.965,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.094,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.059,0.015,0]},"t":27,"s":[88.5,122.317,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.722,1.067,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.044,-0.049,0]},"t":28,"s":[94.599,91.269,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.905,0.958,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.119,0.037,0]},"t":29,"s":[81.585,110.87,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.81,0.948,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.689,-0.085,0]},"t":30,"s":[51.164,75.568,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.817,1.401,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.148,-0.141,0]},"t":31,"s":[46.977,93.082,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.272,0.996,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.153,0.069,0]},"t":32,"s":[41.615,86.585,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,0.774,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.064,-0.004,0]},"t":33,"s":[35.186,124.337,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.499,0.946,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.532,0.132,0]},"t":34,"s":[62.591,88.269,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.951,0.921,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.071,-0.154,0]},"t":35,"s":[58.881,26.321,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.072,1.831,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.121,-1.508,0]},"t":36,"s":[84.818,48.109,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.946,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.038,0.076,0]},"t":37,"s":[74.254,46.968,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.594,35.497,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.15,12.359,0]},"t":38,"s":[93.883,59.486,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.867,0.926,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.105,0.083,0]},"t":39,"s":[86.872,59.571,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.076,0.594,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.224,-0.643,0]},"t":40,"s":[59.763,24.304,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.02,0.969,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,0.105,0]},"t":41,"s":[43.653,28.352,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,1.059,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,-0.049,0]},"t":42,"s":[74.399,44.017,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.401,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.523,0.034,0]},"t":43,"s":[36.198,34.152,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.943,0.234,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.069,0.557,0]},"t":44,"s":[41.448,50.974,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.109,1.139,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.183,0.094,0]},"t":45,"s":[10.923,53.931,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.042,0.95,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.047,0.052,0]},"t":46,"s":[20.467,78.157,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.003,0.801,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.028,-0.123,0]},"t":47,"s":[-1.575,13.482,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.927,0.974,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.003,0.144,0]},"t":48,"s":[31.467,39.632,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.201,0.662,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.568,-0.037,0]},"t":49,"s":[-2.832,75.796,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,0.948,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.093,0.111,0]},"t":50,"s":[1.556,50.773,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.966,0.83,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.302,-0.136,0]},"t":51,"s":[39.258,-25.561,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.108,1.032,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.057,0.164,0]},"t":52,"s":[53.602,3.457,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,1.046,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.092,0.023,0]},"t":53,"s":[45.062,33.613,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.964,0.941,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.07,0.03,0]},"t":54,"s":[-37.848,-8.287,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.116,0.998,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.063,-0.206,0]},"t":55,"s":[7.213,56.765,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.896,1.241,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.049,-0.003,0]},"t":56,"s":[-18.526,38.06,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.835,0.935,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.416,0.062,0]},"t":57,"s":[43.172,56.215,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.168,-0.296,0]},"t":58,"s":[58.626,-14.531,100]},{"t":59,"s":[73.773,0.993,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[0.3137254901960784,0.8901960784313725,0.7607843137254902,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":7,"op":60,"st":7,"bm":0},{"ddd":0,"ind":50,"ty":4,"nm":"Shape Layer 2","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":6,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":11,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":33,"s":[100]},{"t":49,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.592},"o":{"x":0.167,"y":0.167},"t":6,"s":[272.75,257.654,0],"to":[0.022,0.426,0],"ti":[-0.208,-2.079,0]},{"i":{"x":0.833,"y":0.764},"o":{"x":0.167,"y":0.105},"t":7,"s":[272.88,260.209,0],"to":[0.208,2.079,0],"ti":[-0.762,-4.636,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.129},"t":8,"s":[273.997,270.127,0],"to":[0.762,4.636,0],"ti":[-1.59,-6.205,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.158},"t":9,"s":[277.455,288.025,0],"to":[1.59,6.205,0],"ti":[-2.259,-5.901,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":10,"s":[283.537,307.355,0],"to":[2.259,5.901,0],"ti":[-2.568,-4.763,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":11,"s":[291.011,323.433,0],"to":[2.568,4.763,0],"ti":[-2.65,-3.689,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.182},"t":12,"s":[298.946,335.933,0],"to":[2.65,3.689,0],"ti":[-2.621,-2.842,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":13,"s":[306.911,345.569,0],"to":[2.621,2.842,0],"ti":[-2.532,-2.186,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.179},"t":14,"s":[314.674,352.986,0],"to":[2.532,2.186,0],"ti":[-2.408,-1.678,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.178},"t":15,"s":[322.1,358.684,0],"to":[2.408,1.678,0],"ti":[-2.266,-1.285,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":16,"s":[329.119,363.056,0],"to":[2.266,1.285,0],"ti":[-2.117,-0.981,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":17,"s":[335.695,366.397,0],"to":[2.117,0.981,0],"ti":[-1.968,-0.746,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":18,"s":[341.819,368.944,0],"to":[1.968,0.746,0],"ti":[-1.824,-0.565,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":19,"s":[347.502,370.875,0],"to":[1.824,0.565,0],"ti":[-1.688,-0.424,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":20,"s":[352.764,372.332,0],"to":[1.688,0.424,0],"ti":[-1.56,-0.315,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":21,"s":[357.63,373.419,0],"to":[1.56,0.315,0],"ti":[-1.442,-0.23,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":22,"s":[362.126,374.22,0],"to":[1.442,0.23,0],"ti":[-1.332,-0.163,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[366.28,374.797,0],"to":[1.332,0.163,0],"ti":[-1.23,-0.11,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[370.118,375.198,0],"to":[1.23,0.11,0],"ti":[-1.135,-0.068,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":25,"s":[373.661,375.459,0],"to":[1.135,0.068,0],"ti":[-1.047,-0.035,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":26,"s":[376.93,375.608,0],"to":[1.047,0.035,0],"ti":[-0.963,-0.007,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":27,"s":[379.942,375.666,0],"to":[0.963,0.007,0],"ti":[-0.884,0.016,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":28,"s":[382.71,375.649,0],"to":[0.884,-0.016,0],"ti":[-0.808,0.036,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":29,"s":[385.245,375.569,0],"to":[0.808,-0.036,0],"ti":[-0.736,0.053,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":30,"s":[387.559,375.434,0],"to":[0.736,-0.053,0],"ti":[-0.666,0.068,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":31,"s":[389.659,375.251,0],"to":[0.666,-0.068,0],"ti":[-0.598,0.082,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":32,"s":[391.553,375.025,0],"to":[0.598,-0.082,0],"ti":[-0.531,0.096,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":33,"s":[393.245,374.757,0],"to":[0.531,-0.096,0],"ti":[-0.471,0.106,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.176},"t":34,"s":[394.738,374.451,0],"to":[0.471,-0.106,0],"ti":[-0.425,0.111,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":35,"s":[396.069,374.119,0],"to":[0.425,-0.111,0],"ti":[-0.387,0.11,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":36,"s":[397.286,373.785,0],"to":[0.387,-0.11,0],"ti":[-0.348,0.107,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.176},"t":37,"s":[398.388,373.457,0],"to":[0.348,-0.107,0],"ti":[-0.31,0.102,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.177},"t":38,"s":[399.376,373.143,0],"to":[0.31,-0.102,0],"ti":[-0.272,0.094,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.179},"t":39,"s":[400.249,372.847,0],"to":[0.272,-0.094,0],"ti":[-0.233,0.086,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.181},"t":40,"s":[401.007,372.576,0],"to":[0.233,-0.086,0],"ti":[-0.195,0.076,0]},{"i":{"x":0.833,"y":0.851},"o":{"x":0.167,"y":0.184},"t":41,"s":[401.649,372.333,0],"to":[0.195,-0.076,0],"ti":[-0.156,0.064,0]},{"i":{"x":0.833,"y":0.856},"o":{"x":0.167,"y":0.189},"t":42,"s":[402.175,372.123,0],"to":[0.156,-0.064,0],"ti":[-0.118,0.052,0]},{"i":{"x":0.833,"y":0.864},"o":{"x":0.167,"y":0.198},"t":43,"s":[402.586,371.947,0],"to":[0.118,-0.052,0],"ti":[-0.079,0.039,0]},{"i":{"x":0.833,"y":0.881},"o":{"x":0.167,"y":0.215},"t":44,"s":[402.881,371.81,0],"to":[0.079,-0.039,0],"ti":[-0.041,0.026,0]},{"i":{"x":0.833,"y":0.888},"o":{"x":0.167,"y":0.276},"t":45,"s":[403.061,371.711,0],"to":[0.041,-0.026,0],"ti":[-0.01,0.014,0]},{"i":{"x":0.833,"y":0.805},"o":{"x":0.167,"y":0.327},"t":46,"s":[403.128,371.653,0],"to":[0.01,-0.014,0],"ti":[0.008,0.004,0]},{"i":{"x":0.833,"y":0.754},"o":{"x":0.167,"y":0.145},"t":47,"s":[403.122,371.629,0],"to":[-0.008,-0.004,0],"ti":[0.019,-0.005,0]},{"i":{"x":0.833,"y":0.789},"o":{"x":0.167,"y":0.126},"t":48,"s":[403.083,371.63,0],"to":[-0.019,0.005,0],"ti":[0.031,-0.013,0]},{"i":{"x":0.833,"y":0.805},"o":{"x":0.167,"y":0.138},"t":49,"s":[403.008,371.658,0],"to":[-0.031,0.013,0],"ti":[0.042,-0.022,0]},{"i":{"x":0.833,"y":0.813},"o":{"x":0.167,"y":0.145},"t":50,"s":[402.898,371.711,0],"to":[-0.042,0.022,0],"ti":[0.053,-0.03,0]},{"i":{"x":0.833,"y":0.818},"o":{"x":0.167,"y":0.15},"t":51,"s":[402.755,371.79,0],"to":[-0.053,0.03,0],"ti":[0.064,-0.038,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.154},"t":52,"s":[402.579,371.893,0],"to":[-0.064,0.038,0],"ti":[0.074,-0.045,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.156},"t":53,"s":[402.371,372.019,0],"to":[-0.074,0.045,0],"ti":[0.083,-0.052,0]},{"i":{"x":0.833,"y":0.826},"o":{"x":0.167,"y":0.158},"t":54,"s":[402.135,372.165,0],"to":[-0.083,0.052,0],"ti":[0.091,-0.057,0]},{"i":{"x":0.833,"y":0.828},"o":{"x":0.167,"y":0.16},"t":55,"s":[401.874,372.329,0],"to":[-0.091,0.057,0],"ti":[0.097,-0.062,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.162},"t":56,"s":[401.591,372.509,0],"to":[-0.097,0.062,0],"ti":[0.102,-0.065,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.163},"t":57,"s":[401.29,372.7,0],"to":[-0.102,0.065,0],"ti":[0.106,-0.067,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":58,"s":[400.977,372.899,0],"to":[-0.106,0.067,0],"ti":[0.053,-0.034,0]},{"t":59,"s":[400.657,373.102,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[0.867,1.088,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":6,"s":[119.144,105.827,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.795,0.877,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.222,0.043,0]},"t":7,"s":[108.917,89.684,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.672,1.179,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.141,0.257,0]},"t":8,"s":[102.765,122.894,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.908,0.844,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.112,0.057,0]},"t":9,"s":[93.801,138.844,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.347,0.934,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.904,0.179,0]},"t":10,"s":[67.511,88.612,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.047,0.005,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,-0.308,0]},"t":11,"s":[64.842,45.024,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.006,0.93,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.03,0.091,0]},"t":12,"s":[46.604,54.307,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.068,0.682,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,-0.449,0]},"t":13,"s":[75.236,155.919,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.917,0.86,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.037,0.113,0]},"t":14,"s":[44.513,140.008,100]},{"i":{"x":[0.833,0.833,0.833],"y":[152.322,0.967,1]},"o":{"x":[0.167,0.167,0.167],"y":[270.204,0.206,0]},"t":15,"s":[100.175,95.172,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.025,0.944,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.083,-0.056,0]},"t":16,"s":[100.192,64.696,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.006,0.462,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,-0.171,0]},"t":17,"s":[68.993,82.975,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.991,1.026,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.006,0.099,0]},"t":18,"s":[109.723,76.983,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.999,0.913,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.01,0.02,0]},"t":19,"s":[65.994,44.292,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.952,2.661,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.001,2.206,0]},"t":20,"s":[105.048,87.05,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.907,0.908,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.115,0.079,0]},"t":21,"s":[66.25,88.729,100]},{"i":{"x":[0.833,0.833,0.833],"y":[-0.395,1.035,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.774,0.85,0]},"t":22,"s":[82.513,53.595,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.013,0.706,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.089,0.024,0]},"t":23,"s":[84.475,49.774,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.914,0.563,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.011,0.116,0]},"t":24,"s":[115.346,55.183,100]},{"i":{"x":[0.833,0.833,0.833],"y":[3.517,0.971,1]},"o":{"x":[0.167,0.167,0.167],"y":[2.433,0.103,0]},"t":25,"s":[79.807,68.878,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.948,0.939,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.081,-0.045,0]},"t":26,"s":[78.547,127.052,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.885,1.131,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.137,-0.23,0]},"t":27,"s":[117.885,89.313,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.672,0.895,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.305,0.051,0]},"t":28,"s":[103.035,99.348,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.965,1.107,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.112,0.406,0]},"t":29,"s":[97.463,73.594,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.366,0.923,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.06,0.047,0]},"t":30,"s":[81.079,66.939,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.944,2.315,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.068,-1.035,0]},"t":31,"s":[90.595,82.121,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.794,1.043,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.173,0.078,0]},"t":32,"s":[39.327,80.989,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.967,0.857,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.14,0.028,0]},"t":33,"s":[55.998,99.978,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.774,1.028,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.055,0.199,0]},"t":34,"s":[80.566,71.181,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.053,0.916,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.132,0.021,0]},"t":35,"s":[65.735,50.454,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.93,2.464,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.032,17.741,0]},"t":36,"s":[40.364,78.153,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.919,-0.052,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.444,0.079,0]},"t":37,"s":[81.902,78.284,100]},{"i":{"x":[0.833,0.833,0.833],"y":[8.691,0.931,1]},"o":{"x":[0.167,0.167,0.167],"y":[-2.408,0.091,0]},"t":38,"s":[75.338,75.857,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,0.423,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.082,-0.405,0]},"t":39,"s":[75.557,47.651,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.151,1.036,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.069,0.097,0]},"t":40,"s":[55.074,52.465,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.832,0.956,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.054,0.025,0]},"t":41,"s":[66.306,80.999,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.912,0.908,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.165,-0.095,0]},"t":42,"s":[34.754,40.269,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.421,1.175,1]},"o":{"x":[0.167,0.167,0.167],"y":[1.584,0.921,0]},"t":43,"s":[2.594,59.324,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.174,-0.146,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.07,0.056,0]},"t":44,"s":[0.808,61.219,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.039,0.948,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.056,0.09,0]},"t":45,"s":[11.618,55.345,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.955,0.744,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.027,-0.14,0]},"t":46,"s":[-21.777,-19.555,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.083,0.932,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.096,0.123,0]},"t":47,"s":[27.422,8.427,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.896,0.349,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.042,-0.359,0]},"t":48,"s":[4.597,66.513,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.087,0.933,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.425,0.096,0]},"t":49,"s":[50.12,55.561,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.942,0.784,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.043,-0.354,0]},"t":50,"s":[61.217,-19.054,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.717,1.007,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.186,0.136,0]},"t":51,"s":[38.47,-4.836,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.956,1.072,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.075,0.006,0]},"t":52,"s":[45.499,17.824,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.926,0.911,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.092,0.039,0]},"t":53,"s":[-22.007,-6.623,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.362,0.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.679,1.237,0]},"t":54,"s":[10.024,38.847,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.873,-34.653,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.068,-2.841,0]},"t":55,"s":[6.522,42.131,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.708,0.925,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.243,0.084,0]},"t":56,"s":[25.242,42.037,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.976,-0.201,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.117,-0.791,0]},"t":57,"s":[35.027,2.106,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.034,0.09,0]},"t":58,"s":[59.478,5.911,100]},{"t":59,"s":[42.168,56.973,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":6,"op":60,"st":6,"bm":0},{"ddd":0,"ind":51,"ty":4,"nm":"Shape Layer 1","sr":1,"ks":{"o":{"a":1,"k":[{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":3,"s":[0]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":8,"s":[100]},{"i":{"x":[0.667],"y":[1]},"o":{"x":[0.333],"y":[0]},"t":30,"s":[100]},{"t":46,"s":[0]}],"ix":11},"r":{"a":0,"k":0,"ix":10},"p":{"a":1,"k":[{"i":{"x":0.833,"y":0.625},"o":{"x":0.167,"y":0.167},"t":3,"s":[273.808,259.665,0],"to":[0.527,-0.068,0],"ti":[-2.371,0.302,0]},{"i":{"x":0.833,"y":0.767},"o":{"x":0.167,"y":0.107},"t":4,"s":[276.968,259.259,0],"to":[2.371,-0.302,0],"ti":[-5.103,0.961,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.13},"t":5,"s":[288.032,257.854,0],"to":[5.103,-0.961,0],"ti":[-6.631,2.225,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.159},"t":6,"s":[307.589,253.495,0],"to":[6.631,-2.225,0],"ti":[-5.809,3.598,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.179},"t":7,"s":[327.818,244.501,0],"to":[5.809,-3.598,0],"ti":[-3.886,4.355,0]},{"i":{"x":0.833,"y":0.847},"o":{"x":0.167,"y":0.183},"t":8,"s":[342.441,231.905,0],"to":[3.886,-4.355,0],"ti":[-2.267,4.355,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.183},"t":9,"s":[351.136,218.373,0],"to":[2.267,-4.355,0],"ti":[-1.31,3.958,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.181},"t":10,"s":[356.042,205.774,0],"to":[1.31,-3.958,0],"ti":[-0.831,3.49,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.18},"t":11,"s":[358.996,194.624,0],"to":[0.831,-3.49,0],"ti":[-0.612,3.067,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.178},"t":12,"s":[361.029,184.836,0],"to":[0.612,-3.067,0],"ti":[-0.528,2.704,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":13,"s":[362.67,176.222,0],"to":[0.528,-2.704,0],"ti":[-0.514,2.391,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.177},"t":14,"s":[364.195,168.613,0],"to":[0.514,-2.391,0],"ti":[-0.538,2.117,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":15,"s":[365.753,161.877,0],"to":[0.538,-2.117,0],"ti":[-0.583,1.874,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":16,"s":[367.425,155.911,0],"to":[0.583,-1.874,0],"ti":[-0.636,1.65,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":17,"s":[369.252,150.634,0],"to":[0.636,-1.65,0],"ti":[-0.685,1.438,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.176},"t":18,"s":[371.238,146.01,0],"to":[0.685,-1.438,0],"ti":[-0.735,1.242,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.175},"t":19,"s":[373.362,142.007,0],"to":[0.735,-1.242,0],"ti":[-0.783,1.066,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":20,"s":[375.646,138.557,0],"to":[0.783,-1.066,0],"ti":[-0.813,0.905,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":21,"s":[378.057,135.612,0],"to":[0.813,-0.905,0],"ti":[-0.823,0.759,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":22,"s":[380.527,133.13,0],"to":[0.823,-0.759,0],"ti":[-0.814,0.633,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":23,"s":[382.995,131.056,0],"to":[0.814,-0.633,0],"ti":[-0.791,0.525,0]},{"i":{"x":0.833,"y":0.84},"o":{"x":0.167,"y":0.174},"t":24,"s":[385.412,129.334,0],"to":[0.791,-0.525,0],"ti":[-0.756,0.434,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.174},"t":25,"s":[387.738,127.909,0],"to":[0.756,-0.434,0],"ti":[-0.712,0.358,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":26,"s":[389.945,126.733,0],"to":[0.712,-0.358,0],"ti":[-0.664,0.295,0]},{"i":{"x":0.833,"y":0.841},"o":{"x":0.167,"y":0.175},"t":27,"s":[392.013,125.763,0],"to":[0.664,-0.295,0],"ti":[-0.613,0.243,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.175},"t":28,"s":[393.931,124.963,0],"to":[0.613,-0.243,0],"ti":[-0.561,0.201,0]},{"i":{"x":0.833,"y":0.842},"o":{"x":0.167,"y":0.176},"t":29,"s":[395.694,124.303,0],"to":[0.561,-0.201,0],"ti":[-0.508,0.165,0]},{"i":{"x":0.833,"y":0.843},"o":{"x":0.167,"y":0.177},"t":30,"s":[397.298,123.76,0],"to":[0.508,-0.165,0],"ti":[-0.456,0.135,0]},{"i":{"x":0.833,"y":0.844},"o":{"x":0.167,"y":0.178},"t":31,"s":[398.744,123.313,0],"to":[0.456,-0.135,0],"ti":[-0.404,0.11,0]},{"i":{"x":0.833,"y":0.845},"o":{"x":0.167,"y":0.179},"t":32,"s":[400.032,122.948,0],"to":[0.404,-0.11,0],"ti":[-0.352,0.089,0]},{"i":{"x":0.833,"y":0.846},"o":{"x":0.167,"y":0.18},"t":33,"s":[401.165,122.652,0],"to":[0.352,-0.089,0],"ti":[-0.302,0.071,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.182},"t":34,"s":[402.144,122.415,0],"to":[0.302,-0.071,0],"ti":[-0.253,0.055,0]},{"i":{"x":0.833,"y":0.85},"o":{"x":0.167,"y":0.184},"t":35,"s":[402.975,122.228,0],"to":[0.253,-0.055,0],"ti":[-0.207,0.041,0]},{"i":{"x":0.833,"y":0.853},"o":{"x":0.167,"y":0.188},"t":36,"s":[403.664,122.085,0],"to":[0.207,-0.041,0],"ti":[-0.163,0.029,0]},{"i":{"x":0.833,"y":0.859},"o":{"x":0.167,"y":0.193},"t":37,"s":[404.217,121.981,0],"to":[0.163,-0.029,0],"ti":[-0.119,0.018,0]},{"i":{"x":0.833,"y":0.868},"o":{"x":0.167,"y":0.203},"t":38,"s":[404.639,121.911,0],"to":[0.119,-0.018,0],"ti":[-0.078,0.008,0]},{"i":{"x":0.833,"y":0.887},"o":{"x":0.167,"y":0.226},"t":39,"s":[404.934,121.872,0],"to":[0.078,-0.008,0],"ti":[-0.038,-0.001,0]},{"i":{"x":0.833,"y":0.82},"o":{"x":0.167,"y":0.317},"t":40,"s":[405.107,121.862,0],"to":[0.038,0.001,0],"ti":[0,-0.01,0]},{"i":{"x":0.833,"y":0.714},"o":{"x":0.167,"y":0.155},"t":41,"s":[405.163,121.879,0],"to":[0,0.01,0],"ti":[0.036,-0.018,0]},{"i":{"x":0.833,"y":0.805},"o":{"x":0.167,"y":0.118},"t":42,"s":[405.107,121.921,0],"to":[-0.036,0.018,0],"ti":[0.064,-0.025,0]},{"i":{"x":0.833,"y":0.829},"o":{"x":0.167,"y":0.146},"t":43,"s":[404.945,121.988,0],"to":[-0.064,0.025,0],"ti":[0.075,-0.028,0]},{"i":{"x":0.833,"y":0.83},"o":{"x":0.167,"y":0.163},"t":44,"s":[404.725,122.069,0],"to":[-0.075,0.028,0],"ti":[0.078,-0.03,0]},{"i":{"x":0.833,"y":0.832},"o":{"x":0.167,"y":0.164},"t":45,"s":[404.495,122.157,0],"to":[-0.078,0.03,0],"ti":[0.079,-0.032,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.165},"t":46,"s":[404.258,122.252,0],"to":[-0.079,0.032,0],"ti":[0.08,-0.034,0]},{"i":{"x":0.833,"y":0.835},"o":{"x":0.167,"y":0.166},"t":47,"s":[404.019,122.352,0],"to":[-0.08,0.034,0],"ti":[0.078,-0.036,0]},{"i":{"x":0.833,"y":0.852},"o":{"x":0.167,"y":0.169},"t":48,"s":[403.781,122.458,0],"to":[-0.078,0.036,0],"ti":[0.065,-0.038,0]},{"i":{"x":0.833,"y":0.858},"o":{"x":0.167,"y":0.191},"t":49,"s":[403.552,122.57,0],"to":[-0.065,0.038,0],"ti":[0.039,-0.038,0]},{"i":{"x":0.833,"y":0.848},"o":{"x":0.167,"y":0.201},"t":50,"s":[403.392,122.685,0],"to":[-0.039,0.038,0],"ti":[0.013,-0.038,0]},{"i":{"x":0.833,"y":0.823},"o":{"x":0.167,"y":0.184},"t":51,"s":[403.315,122.8,0],"to":[-0.013,0.038,0],"ti":[-0.011,-0.037,0]},{"i":{"x":0.833,"y":0.81},"o":{"x":0.167,"y":0.157},"t":52,"s":[403.314,122.914,0],"to":[0.011,0.037,0],"ti":[-0.033,-0.036,0]},{"i":{"x":0.833,"y":0.812},"o":{"x":0.167,"y":0.148},"t":53,"s":[403.381,123.025,0],"to":[0.033,0.036,0],"ti":[-0.052,-0.035,0]},{"i":{"x":0.833,"y":0.817},"o":{"x":0.167,"y":0.15},"t":54,"s":[403.509,123.131,0],"to":[0.052,0.035,0],"ti":[-0.069,-0.033,0]},{"i":{"x":0.833,"y":0.821},"o":{"x":0.167,"y":0.153},"t":55,"s":[403.692,123.233,0],"to":[0.069,0.033,0],"ti":[-0.084,-0.031,0]},{"i":{"x":0.833,"y":0.824},"o":{"x":0.167,"y":0.156},"t":56,"s":[403.923,123.328,0],"to":[0.084,0.031,0],"ti":[-0.096,-0.028,0]},{"i":{"x":0.833,"y":0.827},"o":{"x":0.167,"y":0.159},"t":57,"s":[404.195,123.417,0],"to":[0.096,0.028,0],"ti":[-0.107,-0.025,0]},{"i":{"x":0.833,"y":0.833},"o":{"x":0.167,"y":0.16},"t":58,"s":[404.501,123.497,0],"to":[0.107,0.025,0],"ti":[-0.056,-0.012,0]},{"t":59,"s":[404.836,123.569,0]}],"ix":2,"l":2},"a":{"a":0,"k":[0,0,0],"ix":1,"l":2},"s":{"a":1,"k":[{"i":{"x":[0.833,0.833,0.833],"y":[1.013,0.991,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.167,0.167,0]},"t":3,"s":[94.174,88.68,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.854,0.871,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.011,-0.01,0]},"t":4,"s":[62.657,103.74,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1,0.874,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.195,0.236,0]},"t":5,"s":[98.913,90.256,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.962,1.852,1]},"o":{"x":[0.167,0.167,0.167],"y":[0,0.247,0]},"t":6,"s":[126.059,82.91,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.12,0.944,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.071,0.076,0]},"t":7,"s":[99.016,79.175,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.877,0.719,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.049,-0.167,0]},"t":8,"s":[113.623,121.094,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.808,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.258,0.118,0]},"t":9,"s":[77.907,107.132,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.024,1.089,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.147,0.166,0]},"t":10,"s":[60.895,73.957,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.02,0.864,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.019,0.043,0]},"t":11,"s":[38.746,40.656,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.997,1.04,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,0.216,0]},"t":12,"s":[67.397,109.363,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.787,0.879,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.003,0.027,0]},"t":13,"s":[31.725,152.383,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.983,0.979,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.137,0.27,0]},"t":14,"s":[66.258,88.662,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.996,0.751,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.021,-0.027,0]},"t":15,"s":[119.803,60.228,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.949,1.058,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.004,0.125,0]},"t":16,"s":[76.922,81.625,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.975,0.91,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.135,0.034,0]},"t":17,"s":[117.696,124.097,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.215,1.712,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.036,1.154,0]},"t":18,"s":[102.098,51.832,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.928,0.975,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.06,0.075,0]},"t":19,"s":[113.025,46.206,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.706,0.89,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.543,-0.036,0]},"t":20,"s":[73.843,99.907,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.753,0.976,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.116,0.341,0]},"t":21,"s":[79.054,62.467,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.064,0.498,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.126,-0.035,0]},"t":22,"s":[92.238,50.379,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.883,0.885,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.036,0.1,0]},"t":23,"s":[118.149,58.916,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.141,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.287,0.3,0]},"t":24,"s":[72.379,101.804,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.893,0.826,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.052,-0.015,0]},"t":25,"s":[53.668,118.311,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.925,1.044,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.378,0.16,0]},"t":26,"s":[103.965,104.335,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.059,1.06,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.756,0.029,0]},"t":27,"s":[118.191,89.162,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.036,0.944,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.091,0.035,0]},"t":28,"s":[116.779,112.396,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.158,1.215,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.025,-0.166,0]},"t":29,"s":[102.243,72.439,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,0.987,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.055,0.06,0]},"t":30,"s":[123.077,85.771,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.064,0.868,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.435,-0.015,0]},"t":31,"s":[62.652,37.969,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.949,1.055,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.036,0.226,0]},"t":32,"s":[48.316,78.523,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.545,0.919,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.131,0.033,0]},"t":33,"s":[73.718,102.173,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.979,-0.741,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.102,-2.641,0]},"t":34,"s":[63.843,62.973,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.921,1.048,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.028,0.088,0]},"t":35,"s":[19.791,64.173,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.758,0.937,1]},"o":{"x":[0.167,0.167,0.167],"y":[-1.621,0.03,0]},"t":36,"s":[52.853,88.034,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.306,0.9,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.127,-0.264,0]},"t":37,"s":[51.236,50.431,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.889,-0.258,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.065,0.495,0]},"t":38,"s":[48.161,59.44,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.538,1.041,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.334,0.089,0]},"t":39,"s":[62.531,61.263,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.759,0.854,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.072,0.027,0]},"t":40,"s":[67.316,86.954,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.954,1.042,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.127,0.194,0]},"t":41,"s":[31.655,48.638,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.875,0.994,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.1,0.028,0]},"t":42,"s":[-35.843,19.827,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.076,0.824,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.249,-0.007,0]},"t":43,"s":[-5.221,63.199,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.027,0.939,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.04,0.158,0]},"t":44,"s":[10.17,23.011,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,0.602,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.02,-0.226,0]},"t":45,"s":[-19.321,-21.801,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.313,0.928,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.321,0.105,0]},"t":46,"s":[19.702,-9.718,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.894,0.502,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.066,-0.533,0]},"t":47,"s":[11.661,35.883,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.021,0.956,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.392,0.1,0]},"t":48,"s":[49.934,29.715,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.685,0.902,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.016,-0.095,0]},"t":49,"s":[60.278,-1.013,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.849,1.927,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.113,0.549,0]},"t":50,"s":[47.382,13.308,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.934,1.073,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.187,0.076,0]},"t":51,"s":[11.559,15.87,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.35,0.905,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.311,0.039,0]},"t":52,"s":[-17.364,-15.186,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.953,1.167,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.096,0.662,0]},"t":53,"s":[-11.255,42.917,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.05,0.864,1]},"o":{"x":[0.167,0.167,0.167],"y":[-0.107,0.056,0]},"t":54,"s":[30.321,51.283,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.897,1.071,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.031,0.214,0]},"t":55,"s":[12.149,26.161,100]},{"i":{"x":[0.833,0.833,0.833],"y":[1.244,0.929,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.44,0.038,0]},"t":56,"s":[41.324,10.186,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.878,0.901,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.062,-0.468,0]},"t":57,"s":[48.144,39.687,100]},{"i":{"x":[0.833,0.833,0.833],"y":[0.833,0.833,1]},"o":{"x":[0.167,0.167,0.167],"y":[0.262,0.526,0]},"t":58,"s":[21.344,35.23,100]},{"t":59,"s":[8.869,34.392,100]}],"ix":6,"l":2}},"ao":0,"shapes":[{"ty":"gr","it":[{"d":1,"ty":"el","s":{"a":0,"k":[10,10],"ix":2},"p":{"a":0,"k":[0,0],"ix":3},"nm":"Ellipse Path 1","mn":"ADBE Vector Shape - Ellipse","hd":false},{"ty":"fl","c":{"a":0,"k":[1,0.10196078431372549,0.3803921568627451,1],"ix":4},"o":{"a":0,"k":100,"ix":5},"r":1,"bm":0,"nm":"Fill 1","mn":"ADBE Vector Graphic - Fill","hd":false},{"ty":"tr","p":{"a":0,"k":[0,0],"ix":2},"a":{"a":0,"k":[0,0],"ix":1},"s":{"a":0,"k":[100,100],"ix":3},"r":{"a":0,"k":0,"ix":6},"o":{"a":0,"k":100,"ix":7},"sk":{"a":0,"k":0,"ix":4},"sa":{"a":0,"k":0,"ix":5},"nm":"Transform"}],"nm":"Ellipse 1","np":3,"cix":2,"bm":0,"ix":1,"mn":"ADBE Vector Group","hd":false}],"ip":3,"op":60,"st":3,"bm":0}],"markers":[]}',
	};

	module.exports = { SyncWizard };
});
