/**
 * @module calendar/sync-page/icloud-dialog
 */
jn.define('calendar/sync-page/icloud-dialog', (require, exports, module) => {
	const AppTheme = require('apptheme');
	const { Loc } = require('loc');
	const { BottomSheet } = require('bottom-sheet');
	const { StringField } = require('calendar/layout/fields');
	const { Alert } = require('alert');
	const { NotifyManager } = require('notify-manager');
	const { inAppUrl } = require('in-app-url');
	const { Color } = require('tokens');

	const appleIdUrl = 'https://appleid.apple.com/';

	/**
	 * @class IcloudDialog
	 */
	class IcloudDialog extends LayoutComponent
	{
		constructor(props)
		{
			super(props);

			this.state = {
				isStepperClosed: false,
			};

			this.connectionModel = this.props.connectionModel;

			this.layoutWidget = null;
			this.stepperRef = null;
			this.formRef = null;
			this.appleIdFieldRef = null;
			this.appPassFieldRef = null;

			this.onLoginButtonClick = this.onLoginButtonClick.bind(this);
			this.onFormSave = this.onFormSave.bind(this);

			this.onFocusAppleId = this.onFocusAppleId.bind(this);
			this.onBlurAppleId = this.onBlurAppleId.bind(this);
			this.onFocusAppPass = this.onFocusAppPass.bind(this);
			this.onBlurAppPass = this.onBlurAppPass.bind(this);
		}

		show(parentWidget = PageManager)
		{
			const bottomSheet = new BottomSheet({ component: this });
			const helpdeskUrl = helpdesk.getArticleUrl('18429414');

			// eslint-disable-next-line promise/catch-or-return
			bottomSheet.setParentWidget(parentWidget)
				.setHelpUrl(helpdeskUrl)
				.setBackgroundColor(Color.bgNavigation.toHex())
				.setMediumPositionPercent(85)
				.disableContentSwipe()
				.open()
				.then((widget) => {
					this.layoutWidget = widget;
				});
		}

		render()
		{
			return View(
				{
					testId: 'sync_page_icloud_dialog_container',
				},
				this.renderHeader(),
				this.state.isStepperClosed ? this.renderForm() : this.renderStepper(),
				this.renderButtonContainer(),
			);
		}

		renderHeader()
		{
			return View(
				{
					style: {
						flexDirection: 'row',
						alignItems: 'center',
						paddingVertical: 15,
						paddingHorizontal: 20,
					},
					clickable: true,
					onClick: () => this.layoutWidget.close(),
					testId: 'sync_page_icloud_dialog_header',
				},
				this.renderLeftArrow(),
				this.renderTitle(),
			);
		}

		renderLeftArrow()
		{
			return View(
				{
					style: {
						width: 30,
					},
				},
				Image(
					{
						svg: {
							content: icons.leftArrow,
						},
						style: {
							width: 9,
							height: 16,
						},
					},
				),
			);
		}

		renderTitle()
		{
			return Text(
				{
					style: {
						fontSize: 19,
						fontWeight: '400',
					},
					text: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_TITLE'),
				},
			);
		}

		renderStepper()
		{
			return View(
				{
					style: {
						backgroundColor: AppTheme.colors.bgContentPrimary,
						borderRadius: 12,
					},
					ref: (ref) => {
						this.stepperRef = ref;
					},
					testId: `sync_page_icloud_stepper_container`,
				},
				View(
					{
						style: {
							paddingHorizontal: 20,
							paddingTop: 15,
							paddingBottom: 30,
						},
					},
					this.renderStepperTitle(),
					...stepperItems.map((text, index) => this.renderStepperItem(text, index + 1)),
				),
				this.renderLottie(),
			);
		}

		renderStepperTitle()
		{
			return Text(
				{
					style: {
						fontSize: 16,
						fontWeight: '400',
						marginBottom: 10,
					},
					text: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_STEPPER_TITLE'),
					testId: `sync_page_icloud_stepper_title`,
				},
			);
		}

		renderStepperItem(text, index)
		{
			return View(
				{
					style: {
						flexDirection: 'row',
						marginTop: index === 1 ? 12 : 16,
					},
					testId: `sync_page_icloud_stepper_item_${index}`,
				},
				View(
					{
						style: {
							alignItems: 'center',
							justifyContent: 'center',
							width: 19,
							height: 19,
							borderRadius: 50,
							marginRight: 5,
							backgroundColor: AppTheme.colors.accentSoftBlue1,
						},
					},
					Text(
						{
							style: {
								fontSize: 12,
								fontWeight: '600',
								color: AppTheme.colors.accentMainPrimary,
							},
							text: index.toString(),
						},
					),
				),
				Text(
					{
						style: {
							marginLeft: 5,
							fontSize: 14,
							fontWeight: '400',
							color: AppTheme.colors.base2,
						},
						text,
						testId: `sync_page_icloud_stepper_item_${index}_text`,
					},
				),
			);
		}

		renderLottie()
		{
			return View(
				{
					style: {
						alignItems: 'center',
					},
				},
				LottieView(
					{
						style: {
							width: 220,
							height: 220,
						},
						data: {
							content: lottie,
						},
						params: {
							loopMode: 'loop',
						},
						autoPlay: true,
					},
				),
			)
		}

		renderForm()
		{
			return View(
				{
					style: {
						backgroundColor: AppTheme.colors.bgContentPrimary,
						borderRadius: 12,
						paddingHorizontal: 20,
						paddingTop: 15,
						paddingBottom: 15,
					},
					ref: (ref) => {
						this.formRef = ref;
					},
					testId: `sync_page_icloud_form_container`,
				},
				this.renderFormTitle(),
				this.renderAppleIdField(),
				this.renderAppPassField(),
			);
		}

		renderFormTitle()
		{
			return View(
				{
					style: {
						flexDirection: 'row',
						alignItems: 'center',
						justifyContent: 'center',
					},
					testId: `sync_page_icloud_form_title`,
				},
				Image(
					{
						svg: {
							content: icons.apple,
						},
						style: {
							height: 22,
							width: 19,
						},
						tintColor: AppTheme.colors.base1,
					},
				),
				Text(
					{
						style: {
							fontWeight: '500',
							fontSize: 16,
							color: AppTheme.colors.base1,
							marginLeft: 5,
						},
						text: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_FORM_TITLE'),
						testId: `sync_page_icloud_form_title_text`,
					},
				),
			);
		}

		renderAppleIdField()
		{
			return new StringField({
				placeholder: 'email@address.com',
				label: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_FORM_FIELD_APPLE_ID'),
				keyboardType: 'email-address',
				errorMessage: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_FORM_FIELD_APPLE_ID_WARNING'),
				onBlur: this.onBlurAppleId,
				onFocus: this.onFocusAppleId,
				ref: (ref) => {
					this.appleIdFieldRef = ref;
				},
				testId: `sync_page_icloud_apple_id_field`,
			});
		}

		onFocusAppleId()
		{
			if (this.appleIdFieldRef)
			{
				this.appleIdFieldRef.clearErrorState();
			}
		}

		onBlurAppleId()
		{
			if (this.appleIdFieldRef && !this.validateEmail())
			{
				this.appleIdFieldRef.setErrorState();
			}
		}

		renderAppPassField()
		{
			return new StringField({
				placeholder: 'xxxx-xxxx-xxxx-xxxx',
				label: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_FORM_FIELD_PASSWORD'),
				errorMessage: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_FORM_FIELD_PASSWORD_WARNING'),
				maxLength: 19,
				isPassword: true,
				onBlur: this.onBlurAppPass,
				onFocus: this.onFocusAppPass,
				ref: (ref) => {
					this.appPassFieldRef = ref;
				},
				testId: `sync_page_icloud_app_pass_field`,
			});
		}

		onFocusAppPass()
		{
			if (this.appPassFieldRef)
			{
				this.appPassFieldRef.clearErrorState();
			}
		}

		onBlurAppPass()
		{
			if (this.appPassFieldRef && !this.validateAppPass())
			{
				this.appPassFieldRef.setErrorState();
			}
		}

		renderButtonContainer()
		{
			return View(
				{
					style: {
						backgroundColor: AppTheme.colors.bgContentPrimary,
						borderRadius: 12,
						marginTop: 10,
						paddingVertical: 20,
						paddingHorizontal: 40,
					},
				},
				!this.state.isStepperClosed && this.renderLoginButton(),
				this.state.isStepperClosed && this.renderContinueButton(),
			);
		}

		renderLoginButton()
		{
			return View(
				{
					style: {
						height: 42,
						borderRadius: 6,
						flexDirection: 'row',
						alignItems: 'center',
						justifyContent: 'center',
						backgroundColor: AppTheme.colors.accentMainPrimaryalt,
					},
					onClick: this.onLoginButtonClick,
					testId: `sync_page_icloud_login_button`,
				},
				Image(
					{
						svg: {
							content: icons.apple,
						},
						style: {
							width: 19,
							height: 22,
							color: AppTheme.colors.base8,
						},
					},
				),
				Text(
					{
						style: {
							fontSize: 17,
							fontWeight: '400',
							marginLeft: 5,
							color: AppTheme.colors.baseWhiteFixed,
						},
						text: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_LOGIN_BUTTON'),
					},
				),
			);
		}

		renderContinueButton()
		{
			return View(
				{
					style: {
						height: 42,
						borderRadius: 6,
						flexDirection: 'row',
						alignItems: 'center',
						justifyContent: 'center',
						backgroundColor: AppTheme.colors.accentMainPrimaryalt,
					},
					onClick: this.onFormSave,
					testId: `sync_page_icloud_continue_button`,
				},
				Text(
					{
						style: {
							fontSize: 17,
							fontWeight: '400',
							marginLeft: 5,
							color: AppTheme.colors.baseWhiteFixed,
						},
						text: Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_SYNC_BUTTON'),
					},
				),
			);
		}

		onLoginButtonClick()
		{
			inAppUrl.open(appleIdUrl);

			setTimeout(() => {
				this.setState({ isStepperClosed: true });
			}, 2000);
		}

		onFormSave()
		{
			if (this.appleIdFieldRef && this.appPassFieldRef)
			{
				if (this.appleIdFieldRef.isFocused())
				{
					this.appleIdFieldRef.blur();
				}

				if (this.appPassFieldRef.isFocused())
				{
					this.appPassFieldRef.blur();
				}

				const emailValidation = this.validateEmail();
				const passwordValidation = this.validateAppPass();

				if (emailValidation && passwordValidation)
				{
					this.handleConnectionCreate();
				}
				else
				{
					this.alertIncorrectFormData();
				}
			}
		}

		handleConnectionCreate()
		{
			NotifyManager.showLoadingIndicator();
			const appleId = this.appleIdFieldRef.value;
			const appPass = this.appPassFieldRef.value;

			// eslint-disable-next-line promise/catch-or-return
			this.connectionModel.createIcloudConnection(appleId, appPass).then((response) => {
				NotifyManager.hideLoadingIndicator();

				if (response && response.connectionId)
				{
					if (this.props.onIcloudConnectionCreated)
					{
						this.props.onIcloudConnectionCreated(response.connectionId, appleId);
					}

					this.layoutWidget.close();
				}
				else
				{
					this.alertIncorrectFormData();
				}
			});
		}

		validateEmail()
		{
			const value = this.appleIdFieldRef.value;
			const emailRegExp = /^[\w!#$%&'*+./=?^`{|}~-]+@[\dA-Za-z-]+(?:\.[\dA-Za-z-]+)*$/;

			return emailRegExp.test(value);
		}

		validateAppPass()
		{
			const value = this.appPassFieldRef.value;
			const appPasswordRegExp = /^(?:[a-z]{4}-){3}[a-z]{4}$/;

			return appPasswordRegExp.test(value);
		}

		alertIncorrectFormData()
		{
			Alert.alert(
				Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_INCORRECT_FORM_DATA_TITLE'),
				Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_INCORRECT_FORM_DATA_DESC'),
			);
		}
	}

	const icons = {
		leftArrow: '<svg width="9" height="16" viewBox="0 0 9 16" fill="none" xmlns="http://www.w3.org/2000/svg"> <path fill-rule="evenodd" clip-rule="evenodd" d="M4.3341 9.13027L8.86115 13.6573L7.26368 15.2547L0.00952148 8.0005L7.26368 0.746338L8.86115 2.34381L4.3341 6.87086L3.20009 8.0001L4.3341 9.13027Z" fill="#828B95"/></svg>',
		apple: '<svg width="19" height="22" viewBox="0 0 19 22" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M13.668 0.269531C13.1353 0.29696 12.569 0.457156 11.9703 0.746827C11.3703 1.04088 10.8755 1.40951 10.4865 1.85284C9.70723 2.72844 9.26843 3.79933 9.26843 4.79998C9.26843 4.93828 9.27737 5.06989 9.29279 5.19494C10.5031 5.29258 11.6191 4.67377 12.5418 3.60288C13.3064 2.70207 13.69 1.72883 13.69 0.686489C13.69 0.548241 13.6835 0.408873 13.668 0.269531ZM16.6123 18.5265C17.0477 17.9103 17.4236 17.24 17.7411 16.5119C17.872 16.2031 17.9951 15.8815 18.1117 15.5457C17.5739 15.3218 17.0905 15.0066 16.6577 14.5988C15.7026 13.7201 15.2166 12.6135 15.2023 11.2836C15.1868 9.57628 15.9682 8.24538 17.548 7.29517C16.6654 6.04983 15.3384 5.35858 13.5721 5.21707C12.9202 5.16213 12.1245 5.30159 11.1817 5.6384C10.1848 6.00158 9.59669 6.18375 9.42316 6.18375C9.19106 6.18375 8.66168 6.02795 7.83721 5.72074C7.01041 5.41455 6.34508 5.25875 5.83772 5.25875C4.91049 5.27406 4.04947 5.51335 3.25261 5.98293C2.45574 6.4525 1.81906 7.09216 1.34047 7.90304C0.731514 8.91257 0.42749 10.1173 0.42749 11.514C0.42749 12.7334 0.652982 13.9929 1.1028 15.2935C1.52281 16.4977 2.05893 17.5539 2.71104 18.4648C3.31883 19.3177 3.82619 19.9198 4.23181 20.271C4.86733 20.8577 5.50402 21.1369 6.14278 21.1098C6.56279 21.0957 7.11214 20.9529 7.79405 20.6789C8.47493 20.4061 9.11149 20.271 9.70503 20.271C10.271 20.271 10.8899 20.4061 11.5634 20.6789C12.2347 20.9529 12.8114 21.088 13.2896 21.088C13.9557 21.0725 14.5777 20.8011 15.1583 20.271C15.5328 19.9481 16.0188 19.3666 16.6123 18.5265Z" fill="white"/></svg>',
	};

	const stepperItems = [
		Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_STEPPER_ITEM_1'),
		Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_STEPPER_ITEM_2_MSGVER_1'),
		Loc.getMessage('M_CALENDAR_SYNC_ICLOUD_DIALOG_STEPPER_ITEM_3'),
	];

	const lottie = '{"assets":[{"id":"ohlJxtL4kpDaIeDiDWV39","layers":[{"ddd":0,"ind":7,"ty":2,"nm":"","ln":"image_oS94-ZyB0AXIFqsIFRe8j","sr":1,"ks":{"a":{"a":0,"k":[6,2.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[7,3.55]},"r":{"a":0,"k":0},"s":{"a":0,"k":[95.83,95.83]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"refId":"8qKZ3hP4bdKxe0qZdkOfg"},{"ddd":0,"ind":8,"ty":4,"ln":"layer_8","sr":1,"ks":{"a":{"a":0,"k":[0.75,1.125]},"o":{"a":0,"k":50},"p":{"a":0,"k":[14.25,3.55]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface11","it":[{"ty":"gr","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":false,"i":[[0,0],[0,0],[0,0.35],[0.32,0.14]],"o":[[0,0],[0.32,-0.13],[0,-0.35],[0,0]],"v":[[0.6,0.29],[0.6,1.88],[1.13,1.09],[0.6,0.29]]}}},{"ty":"fl","c":{"a":0,"k":[1,1,1,1]},"r":1,"o":{"a":0,"k":50}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":9,"ty":2,"nm":"","ln":"image_odNyzGQ2Noc4uQgUGrxlv","sr":1,"ks":{"a":{"a":0,"k":[7,3.5]},"o":{"a":0,"k":40},"p":{"a":0,"k":[7,3.55]},"r":{"a":0,"k":0},"s":{"a":0,"k":[96.43,96.43]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"refId":"RcXQKgdsFgbWJ3CC8kXh_"}]},{"id":"csoQDFE-ziE27p2xzvdFO","layers":[{"ddd":0,"ind":11,"ty":4,"ln":"layer_11","sr":1,"ks":{"a":{"a":0,"k":[3.75,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200.49,2200.18]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface16","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.71,-0.7],[-0.05,0.05],[0,0],[0,0.04],[0.03,0.03],[1.87,-1.83],[0,-0.04],[-0.03,-0.03],[0,0],[-0.05,0.05],[-0.98,0]],"o":[[0.98,0],[0.05,0.05],[0,0],[0.03,-0.03],[0,-0.04],[-1.87,-1.83],[-0.03,0.03],[0,0.04],[0,0],[0.05,0.05],[0.71,-0.7],[0,0]],"v":[[3.38,1.03],[6.02,2.11],[6.21,2.11],[6.72,1.58],[6.77,1.48],[6.72,1.38],[0.04,1.38],[0,1.48],[0.04,1.58],[0.55,2.11],[0.75,2.11],[3.38,1.03]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.4,-0.37],[-0.05,0.05],[0,0],[0,0.04],[0.03,0.03],[1.22,-1.16],[0,-0.04],[-0.03,-0.03],[0,0],[-0.05,0.05],[-0.54,0]],"o":[[0.54,0],[0.05,0.05],[0,0],[0.03,-0.03],[0,-0.04],[-1.21,-1.16],[-0.03,0.03],[0,0.04],[0,0],[0.05,0.05],[0.4,-0.37],[0,0]],"v":[[3.38,2.75],[4.84,3.33],[5.04,3.33],[5.55,2.8],[5.59,2.7],[5.54,2.59],[1.22,2.59],[1.18,2.7],[1.22,2.8],[1.73,3.33],[1.93,3.33],[3.38,2.75]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.03,-0.03],[0,0],[0.04,0],[0.03,0.03],[0,0],[0,0.04],[-0.03,0.03],[-0.57,-0.49],[0,-0.04]],"o":[[0,0.04],[0,0],[-0.02,0.03],[-0.04,0],[0,0],[-0.03,-0.03],[0,-0.04],[0.56,-0.49],[0.03,0.03],[0,0]],"v":[[4.41,3.91],[4.37,4.02],[3.48,4.93],[3.38,4.97],[3.29,4.93],[2.4,4.02],[2.36,3.91],[2.41,3.81],[4.36,3.81],[4.41,3.91]]}}},{"ty":"fl","c":{"a":0,"k":[1,1,1,1]},"r":2,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"Ee2T4y_f6HDWktIrz-gXt","layers":[{"ddd":0,"ind":13,"ty":4,"ln":"layer_13","sr":1,"ks":{"a":{"a":0,"k":[4.125,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200.2,2200.32]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface21","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,-0.25],[0,0],[-0.26,0],[0,0],[0,0.25],[0,0],[0.26,0]],"o":[[0,0],[-0.26,0],[0,0],[0,0.25],[0,0],[0.26,0],[0,0],[0,-0.25],[0,0]],"v":[[7.48,0],[7.02,0],[6.55,0.45],[6.55,4.33],[7.02,4.77],[7.48,4.77],[7.95,4.33],[7.95,0.45],[7.48,0]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,-0.25],[0,0],[0.25,0],[0,0],[0,0.25],[0,0],[-0.26,0]],"o":[[0,0],[0.25,0],[0,0],[0,0.25],[0,0],[-0.26,0],[0,0],[0,-0.25],[0,0]],"v":[[4.84,1.04],[5.3,1.04],[5.76,1.49],[5.76,4.33],[5.3,4.77],[4.84,4.77],[4.37,4.33],[4.37,1.49],[4.84,1.04]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,-0.25],[0,0],[-0.25,0],[0,0],[0,0.25],[0,0],[0.26,0]],"o":[[0,0],[-0.25,0],[0,0],[0,0.25],[0,0],[0.26,0],[0,0],[0,-0.25],[0,0]],"v":[[3.11,2.09],[2.65,2.09],[2.18,2.54],[2.18,4.33],[2.65,4.77],[3.11,4.77],[3.58,4.33],[3.58,2.54],[3.11,2.09]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,-0.25],[0,0],[-0.26,0],[0,0],[0,0.25],[0,0],[0.25,0]],"o":[[0,0],[-0.26,0],[0,0],[0,0.25],[0,0],[0.25,0],[0,0],[0,-0.25],[0,0]],"v":[[0.93,2.98],[0.46,2.98],[0,3.43],[0,4.33],[0.46,4.78],[0.93,4.78],[1.39,4.33],[1.39,3.43],[0.93,2.98]]}}},{"ty":"fl","c":{"a":0,"k":[1,1,1,1]},"r":2,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"PWEQRI8ZwpkjjlFoneA_K","layers":[{"ddd":0,"ind":6,"ty":0,"nm":"","ln":"precomp_yoqAa1nf5No1wQd8N5vjk6","sr":1,"ks":{"a":{"a":0,"k":[7.5,3.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2212.8,2199.95]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":15,"h":7,"ip":0,"op":899,"st":0,"bm":0,"refId":"ohlJxtL4kpDaIeDiDWV39"},{"ddd":0,"ind":10,"ty":0,"nm":"","ln":"precomp_rLK9GywpSqBqrs1wTq_xH10","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2197.85,2200.95]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"csoQDFE-ziE27p2xzvdFO"},{"ddd":0,"ind":12,"ty":0,"nm":"","ln":"precomp_CRVkKKyqBMDFNh6jCbfUc12","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2184.86,2200.95]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"Ee2T4y_f6HDWktIrz-gXt"}]},{"id":"3bfaA3x5K9sPSrOkYAjm5","layers":[{"ddd":0,"ind":4,"ty":4,"ln":"layer_4","sr":1,"ks":{"a":{"a":0,"k":[7.875,4.125]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2131.6,2200.46]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,121.21]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface6","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"v":[[1.84,6.72],[1.84,2.79],[1.79,2.79],[0.6,3.63],[0.6,2.83],[1.84,1.95],[2.7,1.95],[2.7,6.72]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"v":[[4.98,6.72],[4.98,2.79],[4.93,2.79],[3.74,3.63],[3.74,2.83],[4.99,1.95],[5.84,1.95],[5.84,6.72]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0.29],[-0.3,0],[0,-0.29],[0.3,0]],"o":[[-0.3,0],[0,-0.29],[0.3,0],[0,0.29],[0,0]],"v":[[7.58,3.63],[7.05,3.11],[7.58,2.59],[8.1,3.11],[7.58,3.63]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0.3],[-0.3,0],[0,-0.29],[0.3,0]],"o":[[-0.3,0],[0,-0.29],[0.3,0],[0,0.3],[0,0]],"v":[[7.58,6.08],[7.05,5.56],[7.58,5.04],[8.1,5.56],[7.58,6.08]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"v":[[10.16,6.72],[10.16,2.79],[10.1,2.79],[8.91,3.63],[8.91,2.83],[10.16,1.95],[11.01,1.95],[11.01,6.72]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"v":[[13.3,6.72],[13.3,2.79],[13.25,2.79],[12.06,3.63],[12.06,2.83],[13.3,1.95],[14.15,1.95],[14.15,6.72]]}}},{"ty":"fl","c":{"a":0,"k":[1,1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":5,"ty":0,"nm":"","ln":"precomp_V-wCwqBOaSnYRwJCRpsK15","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2266.5,2201]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"PWEQRI8ZwpkjjlFoneA_K"},{"ddd":0,"ind":14,"ty":4,"nm":"","ln":"bgvDuCbN-SGUs3LhmbbuD","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[196,25]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.176,0.176,0.176]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"Pr9xUEqkCHJf11QB0bjuh","layers":[{"ddd":0,"ind":2,"ty":4,"ln":"layer_2","sr":1,"ks":{"a":{"a":0,"k":[82.5,167.255]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface1","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-14.08,0],[0,0],[0,-14.08],[0,0],[0,-0.41],[0,0],[0.41,0],[0,0],[14.08,0],[0,0],[0,14.08],[0,0],[0,0.41],[0,0],[-0.41,0],[0,0],[0,0.41],[0,0],[-0.41,0],[0,0],[0,0.41],[0,0],[-0.41,0]],"o":[[0,-14.08],[0,0],[14.08,0],[0,0],[0.41,0],[0,0],[0,0.41],[0,0],[0,14.08],[0,0],[-14.08,0],[0,0],[-0.41,0],[0,0],[0,-0.41],[0,0],[-0.41,0],[0,0],[0,-0.41],[0,0],[-0.41,0],[0,0],[0,-0.41],[0,0]],"v":[[0.75,25.5],[26.25,0],[138.75,0],[164.25,25.5],[164.25,99.75],[165,100.5],[165,138.75],[164.25,139.5],[164.25,309],[138.75,334.5],[26.25,334.5],[0.75,309],[0.75,148.5],[0,147.75],[0,123.75],[0.75,123],[0.75,116.25],[0,115.5],[0,91.5],[0.75,90.75],[0.75,79.5],[0,78.75],[0,67.5],[0.75,66.75]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-10.36,0],[0,0],[-0.16,-0.65],[0,0],[-4.97,0],[0,0],[0,4.97],[0,0],[-0.44,0.2],[0,0],[0,-10.36],[0,0],[10.36,0],[0,0],[0,10.36]],"o":[[0,-10.36],[0,0],[0.44,0.2],[0,0],[0,4.97],[0,0],[4.97,0],[0,0],[0.16,-0.65],[0,0],[10.36,0],[0,0],[0,10.36],[0,0],[-10.36,0],[0,0]],"v":[[9,27],[27.75,8.25],[51,8.25],[51.75,9.38],[51.75,11.25],[60.75,20.25],[104.25,20.25],[113.25,11.25],[113.25,9.38],[114,8.25],[137.25,8.25],[156,27],[156,308.25],[137.25,327],[27.75,327],[9,308.25]]}}},{"ty":"fl","c":{"a":0,"k":[0,0,0,1]},"r":2,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":3,"ty":0,"nm":"","ln":"precomp_bgvDuCbN-SGUs3LhmbbuD3","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2000.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"3bfaA3x5K9sPSrOkYAjm5"},{"ddd":0,"ind":15,"ty":2,"nm":"","ln":"image_XUx-g5qId_zc5ANSmZe2Z","sr":1,"ks":{"a":{"a":0,"k":[220,446]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[50,50]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"refId":"G482sGnJF-hwxSHMfPSF5"}]},{"id":"H3ho1nZHxCztErVuQpnMS","layers":[{"ddd":0,"ind":22,"ty":4,"nm":"","ln":"yKiTzplo80sz94t8PnR0H","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":228,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":229,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.5,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[3.37,-1.86],[3.37,-1.86],[2.57,-2.08],[1.98,-2.67],[1.76,-3.48],[1.76,-3.48],[1.98,-4.29],[2.56,-4.87],[3.37,-5.09],[3.37,-5.09],[4.19,-4.87],[4.77,-4.29],[4.99,-3.48],[4.99,-3.48],[4.77,-2.67],[4.19,-2.08],[3.37,-1.86]],"i":[[0,0],[0,0],[0.24,0.15],[0.15,0.25],[0,0.3],[0,0],[-0.15,0.25],[-0.25,0.15],[-0.29,0],[0,0],[-0.25,-0.15],[-0.15,-0.24],[0,-0.29],[0,0],[0.15,-0.24],[0.24,-0.15],[0.3,0]],"o":[[0,0],[-0.29,0],[-0.25,-0.15],[-0.15,-0.24],[0,0],[0,-0.29],[0.14,-0.24],[0.25,-0.15],[0,0],[0.3,0],[0.24,0.15],[0.15,0.25],[0,0],[0,0.3],[-0.15,0.25],[-0.25,0.15],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"oV-fdUojghfcLPVJQq0TX","layers":[{"ddd":0,"ind":24,"ty":4,"nm":"","ln":"QgPILJcUCaNDYqy1GQS9k","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":218,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":219,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.5,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[3.37,-1.86],[3.37,-1.86],[2.57,-2.08],[1.98,-2.67],[1.76,-3.48],[1.76,-3.48],[1.98,-4.29],[2.56,-4.87],[3.37,-5.09],[3.37,-5.09],[4.19,-4.87],[4.77,-4.29],[4.99,-3.48],[4.99,-3.48],[4.77,-2.67],[4.19,-2.08],[3.37,-1.86]],"i":[[0,0],[0,0],[0.24,0.15],[0.15,0.25],[0,0.3],[0,0],[-0.15,0.25],[-0.25,0.15],[-0.29,0],[0,0],[-0.25,-0.15],[-0.15,-0.24],[0,-0.29],[0,0],[0.15,-0.24],[0.24,-0.15],[0.3,0]],"o":[[0,0],[-0.29,0],[-0.25,-0.15],[-0.15,-0.24],[0,0],[0,-0.29],[0.14,-0.24],[0.25,-0.15],[0,0],[0.3,0],[0.24,0.15],[0.15,0.25],[0,0],[0,0.3],[-0.15,0.25],[-0.25,0.15],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"34ofWT1AtgGZZycrcc4gn","layers":[{"ddd":0,"ind":26,"ty":4,"nm":"","ln":"YTj8L1n1Bd0EappO-uUHl","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":209,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":210,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.5,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[3.37,-1.86],[3.37,-1.86],[2.57,-2.08],[1.98,-2.67],[1.76,-3.48],[1.76,-3.48],[1.98,-4.29],[2.56,-4.87],[3.37,-5.09],[3.37,-5.09],[4.19,-4.87],[4.77,-4.29],[4.99,-3.48],[4.99,-3.48],[4.77,-2.67],[4.19,-2.08],[3.37,-1.86]],"i":[[0,0],[0,0],[0.24,0.15],[0.15,0.25],[0,0.3],[0,0],[-0.15,0.25],[-0.25,0.15],[-0.29,0],[0,0],[-0.25,-0.15],[-0.15,-0.24],[0,-0.29],[0,0],[0.15,-0.24],[0.24,-0.15],[0.3,0]],"o":[[0,0],[-0.29,0],[-0.25,-0.15],[-0.15,-0.24],[0,0],[0,-0.29],[0.14,-0.24],[0.25,-0.15],[0,0],[0.3,0],[0.24,0.15],[0.15,0.25],[0,0],[0,0.3],[-0.15,0.25],[-0.25,0.15],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"WZ5EcwRay0j_c7MKjoUSE","layers":[{"ddd":0,"ind":28,"ty":4,"nm":"","ln":"80lzCsduCre3AtKcIaIMl","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":199,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":200,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.5,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[3.37,-1.86],[3.37,-1.86],[2.57,-2.08],[1.98,-2.67],[1.76,-3.48],[1.76,-3.48],[1.98,-4.29],[2.56,-4.87],[3.37,-5.09],[3.37,-5.09],[4.19,-4.87],[4.77,-4.29],[4.99,-3.48],[4.99,-3.48],[4.77,-2.67],[4.19,-2.08],[3.37,-1.86]],"i":[[0,0],[0,0],[0.24,0.15],[0.15,0.25],[0,0.3],[0,0],[-0.15,0.25],[-0.25,0.15],[-0.29,0],[0,0],[-0.25,-0.15],[-0.15,-0.24],[0,-0.29],[0,0],[0.15,-0.24],[0.24,-0.15],[0.3,0]],"o":[[0,0],[-0.29,0],[-0.25,-0.15],[-0.15,-0.24],[0,0],[0,-0.29],[0.14,-0.24],[0.25,-0.15],[0,0],[0.3,0],[0.24,0.15],[0.15,0.25],[0,0],[0,0.3],[-0.15,0.25],[-0.25,0.15],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"WkyfiRQvmzfN8RfD551vE","layers":[{"ddd":0,"ind":30,"ty":4,"nm":"","ln":"t7NPhXuwKDIz0YfB590hZ","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":187,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":188,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.5,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[3.37,-1.86],[3.37,-1.86],[2.57,-2.08],[1.98,-2.67],[1.76,-3.48],[1.76,-3.48],[1.98,-4.29],[2.56,-4.87],[3.37,-5.09],[3.37,-5.09],[4.19,-4.87],[4.77,-4.29],[4.99,-3.48],[4.99,-3.48],[4.77,-2.67],[4.19,-2.08],[3.37,-1.86]],"i":[[0,0],[0,0],[0.24,0.15],[0.15,0.25],[0,0.3],[0,0],[-0.15,0.25],[-0.25,0.15],[-0.29,0],[0,0],[-0.25,-0.15],[-0.15,-0.24],[0,-0.29],[0,0],[0.15,-0.24],[0.24,-0.15],[0.3,0]],"o":[[0,0],[-0.29,0],[-0.25,-0.15],[-0.15,-0.24],[0,0],[0,-0.29],[0.14,-0.24],[0.25,-0.15],[0,0],[0.3,0],[0.24,0.15],[0.15,0.25],[0,0],[0,0.3],[-0.15,0.25],[-0.25,0.15],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"Gwtrg1LBZ-PxqGlhNhPK_","layers":[{"ddd":0,"ind":32,"ty":4,"nm":"","ln":"LWP54lC3na2JAFD9ZVlXc","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":181,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":182,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.5,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[3.37,-1.86],[3.37,-1.86],[2.57,-2.08],[1.98,-2.67],[1.76,-3.48],[1.76,-3.48],[1.98,-4.29],[2.56,-4.87],[3.37,-5.09],[3.37,-5.09],[4.19,-4.87],[4.77,-4.29],[4.99,-3.48],[4.99,-3.48],[4.77,-2.67],[4.19,-2.08],[3.37,-1.86]],"i":[[0,0],[0,0],[0.24,0.15],[0.15,0.25],[0,0.3],[0,0],[-0.15,0.25],[-0.25,0.15],[-0.29,0],[0,0],[-0.25,-0.15],[-0.15,-0.24],[0,-0.29],[0,0],[0.15,-0.24],[0.24,-0.15],[0.3,0]],"o":[[0,0],[-0.29,0],[-0.25,-0.15],[-0.15,-0.24],[0,0],[0,-0.29],[0.14,-0.24],[0.25,-0.15],[0,0],[0.3,0],[0.24,0.15],[0.15,0.25],[0,0],[0,0.3],[-0.15,0.25],[-0.25,0.15],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"_92GiMzjvOzAeV0GnP5wb","layers":[{"ddd":0,"ind":34,"ty":4,"nm":"","ln":"LNDkZKgL3CkQ60IAXSPMH","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":169,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":170,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.5,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[3.37,-1.86],[3.37,-1.86],[2.57,-2.08],[1.98,-2.67],[1.76,-3.48],[1.76,-3.48],[1.98,-4.29],[2.56,-4.87],[3.37,-5.09],[3.37,-5.09],[4.19,-4.87],[4.77,-4.29],[4.99,-3.48],[4.99,-3.48],[4.77,-2.67],[4.19,-2.08],[3.37,-1.86]],"i":[[0,0],[0,0],[0.24,0.15],[0.15,0.25],[0,0.3],[0,0],[-0.15,0.25],[-0.25,0.15],[-0.29,0],[0,0],[-0.25,-0.15],[-0.15,-0.24],[0,-0.29],[0,0],[0.15,-0.24],[0.24,-0.15],[0.3,0]],"o":[[0,0],[-0.29,0],[-0.25,-0.15],[-0.15,-0.24],[0,0],[0,-0.29],[0.14,-0.24],[0.25,-0.15],[0,0],[0.3,0],[0.24,0.15],[0.15,0.25],[0,0],[0,0.3],[-0.15,0.25],[-0.25,0.15],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"R-TnzhgvyVZNt-Uhm1f1T","layers":[{"ddd":0,"ind":36,"ty":4,"nm":"","ln":"MQ40MR2C2sWFwjlYr3vqB","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":175,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":176,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.5,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[3.37,-1.86],[3.37,-1.86],[2.57,-2.08],[1.98,-2.67],[1.76,-3.48],[1.76,-3.48],[1.98,-4.29],[2.56,-4.87],[3.37,-5.09],[3.37,-5.09],[4.19,-4.87],[4.77,-4.29],[4.99,-3.48],[4.99,-3.48],[4.77,-2.67],[4.19,-2.08],[3.37,-1.86]],"i":[[0,0],[0,0],[0.24,0.15],[0.15,0.25],[0,0.3],[0,0],[-0.15,0.25],[-0.25,0.15],[-0.29,0],[0,0],[-0.25,-0.15],[-0.15,-0.24],[0,-0.29],[0,0],[0.15,-0.24],[0.24,-0.15],[0.3,0]],"o":[[0,0],[-0.29,0],[-0.25,-0.15],[-0.15,-0.24],[0,0],[0,-0.29],[0.14,-0.24],[0.25,-0.15],[0,0],[0.3,0],[0.24,0.15],[0.15,0.25],[0,0],[0,0.3],[-0.15,0.25],[-0.25,0.15],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"FxbZyythxd4NTlDzJBNCM","layers":[{"ddd":0,"ind":39,"ty":4,"ln":"layer_39","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.25]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200.4,2199.9]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface36","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"v":[[2.85,1.62],[0.04,1.62],[0.04,2.85],[2.85,2.85],[2.85,4.26],[5.14,2.32],[5.14,2.15],[2.85,0.21]]}}},{"ty":"fl","c":{"a":0,"k":[0.66,0.68,0.71,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":40,"ty":4,"nm":"","ln":"d_ZK2AXOxx4uflutAbmQ3","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"el","hd":false,"p":{"a":0,"k":[0,0]},"s":{"a":0,"k":[12.2,12.2]},"d":1},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0.659,0.678,0.706]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0.47}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"HXN6dvOyn-3vSnSu1c0ws","layers":[{"ddd":0,"ind":21,"ty":0,"nm":"","ln":"precomp_yKiTzplo80sz94t8PnR0H21","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":228,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":229,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2185.5,2203.29]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"H3ho1nZHxCztErVuQpnMS"},{"ddd":0,"ind":23,"ty":0,"nm":"","ln":"precomp_QgPILJcUCaNDYqy1GQS9k23","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":218,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":219,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2178.5,2203.29]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"oV-fdUojghfcLPVJQq0TX"},{"ddd":0,"ind":25,"ty":0,"nm":"","ln":"precomp_YTj8L1n1Bd0EappO-uUHl25","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":209,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":210,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2171.5,2203.29]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"34ofWT1AtgGZZycrcc4gn"},{"ddd":0,"ind":27,"ty":0,"nm":"","ln":"precomp_80lzCsduCre3AtKcIaIMl27","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":199,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":200,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2164.5,2203.29]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"WZ5EcwRay0j_c7MKjoUSE"},{"ddd":0,"ind":29,"ty":0,"nm":"","ln":"precomp_t7NPhXuwKDIz0YfB590hZ29","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":187,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":188,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2157.5,2203.29]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"WkyfiRQvmzfN8RfD551vE"},{"ddd":0,"ind":31,"ty":0,"nm":"","ln":"precomp_LWP54lC3na2JAFD9ZVlXc31","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":181,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":182,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2150.5,2203.29]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"Gwtrg1LBZ-PxqGlhNhPK_"},{"ddd":0,"ind":33,"ty":0,"nm":"","ln":"precomp_LNDkZKgL3CkQ60IAXSPMH33","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":169,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":170,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2136.5,2203.29]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"_92GiMzjvOzAeV0GnP5wb"},{"ddd":0,"ind":35,"ty":0,"nm":"","ln":"precomp_MQ40MR2C2sWFwjlYr3vqB35","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":175,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":176,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2143.5,2203.29]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"R-TnzhgvyVZNt-Uhm1f1T"},{"ddd":0,"ind":37,"ty":4,"ln":"layer_37","sr":1,"ks":{"a":{"a":0,"k":[9,2.25]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2145.51,2194.42]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,111.11]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface31","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"v":[[0.71,3.47],[1.81,0.5],[2.18,0.5],[3.28,3.47],[2.89,3.47],[2.59,2.62],[1.4,2.62],[1.11,3.47]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[1.98,0.98],[1.51,2.3],[2.48,2.3],[2.01,0.98]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[-0.31,0],[0,-0.69],[0,0],[0.57,0],[0.12,0.24],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0.14,-0.25],[0.56,0],[0,0],[0,0.7],[-0.3,0],[0,0],[0,0],[0,0]],"v":[[3.75,4.21],[3.75,1.25],[4.11,1.25],[4.11,1.61],[4.14,1.61],[4.84,1.21],[5.77,2.36],[5.77,2.36],[4.84,3.51],[4.14,3.12],[4.11,3.12],[4.11,4.21]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0.52],[0,0],[0.41,0],[0,-0.51],[0,0],[-0.4,0]],"o":[[0.41,0],[0,0],[0,-0.52],[-0.4,0],[0,0],[0,0.52],[0,0]],"v":[[4.75,3.2],[5.4,2.36],[5.4,2.36],[4.75,1.53],[4.1,2.36],[4.1,2.36],[4.75,3.2]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[-0.31,0],[0,-0.69],[0,0],[0.57,0],[0.12,0.24],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0.13,-0.25],[0.56,0],[0,0],[0,0.7],[-0.3,0],[0,0],[0,0],[0,0]],"v":[[6.33,4.21],[6.33,1.25],[6.69,1.25],[6.69,1.61],[6.72,1.61],[7.41,1.21],[8.35,2.36],[8.35,2.36],[7.41,3.51],[6.72,3.12],[6.69,3.12],[6.69,4.21]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0.52],[0,0],[0.41,0],[0,-0.51],[0,0],[-0.41,0]],"o":[[0.41,0],[0,0],[0,-0.52],[-0.41,0],[0,0],[0,0.52],[0,0]],"v":[[7.33,3.2],[7.98,2.36],[7.98,2.36],[7.33,1.53],[6.68,2.36],[6.68,2.36],[7.33,3.2]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[8.93,3.47],[8.93,0.37],[9.29,0.37],[9.29,3.47]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0.7],[0,0],[-0.61,0],[0,-0.66],[0,0],[0,0],[-0.39,0],[-0.06,0.13],[0,0],[0,0],[0,0],[0.52,0]],"o":[[-0.64,0],[0,0],[0,-0.69],[0.61,0],[0,0],[0,0],[0.01,0.46],[0.3,0],[0,0],[0,0],[0,0],[-0.08,0.3],[0,0]],"v":[[10.88,3.51],[9.86,2.37],[9.86,2.37],[10.86,1.21],[11.83,2.32],[11.83,2.46],[10.23,2.46],[10.89,3.19],[11.44,2.92],[11.45,2.9],[11.8,2.9],[11.8,2.91],[10.88,3.51]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.04,-0.43],[0,0],[0.32,0]],"o":[[-0.33,0],[0,0],[-0.04,-0.44],[0,0]],"v":[[10.86,1.53],[10.24,2.18],[11.46,2.18],[10.86,1.53]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0]],"v":[[13.62,3.47],[13.62,0.5],[13.99,0.5],[13.99,3.47]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0,-0.93],[0,0],[0.89,0]],"o":[[0,0],[0,0],[0.88,0],[0,0],[0,0.93],[0,0]],"v":[[14.75,3.47],[14.75,0.5],[15.77,0.5],[17.18,1.98],[17.18,1.99],[15.77,3.47]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0.72],[0,0],[0.66,0],[0,0]],"o":[[0,0],[0.66,0],[0,0],[0,-0.73],[0,0],[0,0]],"v":[[15.12,3.14],[15.75,3.14],[16.8,1.99],[16.8,1.99],[15.75,0.83],[15.12,0.83]]}}},{"ty":"fl","c":{"a":0,"k":[0.51,0.55,0.58,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":38,"ty":0,"nm":"","ln":"precomp_U3jm5GO6RvoCIVyVsPuih38","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2262.27,2204.65]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"FxbZyythxd4NTlDzJBNCM"}]},{"id":"30vRG6suD_HlTDgtTZFHu","layers":[{"ddd":0,"ind":20,"ty":0,"nm":"","ln":"precomp_cQg2Y6Z1ffsOJMbZWr7B-20","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"HXN6dvOyn-3vSnSu1c0ws"},{"ddd":0,"ind":43,"ty":4,"nm":"","ln":"cQg2Y6Z1ffsOJMbZWr7B-","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":5},"s":{"a":0,"k":[149.88,27.31]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0.541,0.694,0.969]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":1.88}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":0}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":42,"ty":4,"nm":"","ln":"cQg2Y6Z1ffsOJMbZWr7B-","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":5},"s":{"a":0,"k":[148,25.43]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"e1blxH8-FLcN9exYqpMpy","layers":[{"ddd":0,"ind":45,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2115.4,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":46,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2124.97,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":47,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2134.54,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":48,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2144.12,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":49,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2153.03,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.82,-4.3],[4.8,-4.3],[4.8,-2.86],[0.82,-2.86],[0.82,-4.3]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":50,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2159.03,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":51,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2168.61,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":52,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2178.18,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":53,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2187.75,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":54,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2196.67,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.82,-4.3],[4.8,-4.3],[4.8,-2.86],[0.82,-2.86],[0.82,-4.3]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":55,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2202.67,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":56,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2212.25,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":57,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2221.82,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":58,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2231.39,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":59,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2240.31,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.82,-4.3],[4.8,-4.3],[4.8,-2.86],[0.82,-2.86],[0.82,-4.3]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":60,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2246.31,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":61,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2255.88,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":62,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2265.46,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":63,"ty":4,"nm":"","ln":"gVbPc150aaZLnCUFpaCID","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2275.03,2204.36]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"sh","hd":false,"ix":0,"ks":{"a":0,"k":{"v":[[0.4,-8.73],[2.49,-8.73],[4.25,-5.75],[4.32,-5.75],[6.09,-8.73],[8.17,-8.73],[5.51,-4.36],[8.23,0],[6.11,0],[4.32,-2.98],[4.25,-2.98],[2.46,0],[0.35,0],[3.08,-4.36],[0.4,-8.73]],"i":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]],"o":[[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0],[0,0]]}}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"WJuF6BioFt93Z3BD34mXO","layers":[{"ddd":0,"ind":67,"ty":4,"nm":"","ln":"Q0reP1O8OQ2xmjrOn3RvJ","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2153.66,2197.97]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[42.04,6.21]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":68,"ty":4,"nm":"","ln":"l17AcqoiIql3kTVgwqguf","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2179.22,2185.07]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[93.15,10.03]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":69,"ty":4,"ln":"layer_69","sr":1,"ks":{"a":{"a":0,"k":[4.875,4.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2259.5,2185.5]},"r":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":508,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":509,"s":[0.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":510,"s":[0.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":511,"s":[0.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":512,"s":[0.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":513,"s":[0.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":514,"s":[0.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":515,"s":[0.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":516,"s":[0.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":517,"s":[0.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":518,"s":[1.12],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":519,"s":[1.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":520,"s":[1.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":521,"s":[2.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":522,"s":[2.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":523,"s":[3.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":524,"s":[4.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":525,"s":[4.8],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":526,"s":[5.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":527,"s":[6.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":528,"s":[7.59],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":529,"s":[8.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":530,"s":[9.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":531,"s":[11.29],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":532,"s":[12.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":533,"s":[14.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":534,"s":[16.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":535,"s":[17.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":536,"s":[19.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":537,"s":[21.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":538,"s":[24.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":539,"s":[26.63],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":540,"s":[29.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":541,"s":[31.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":542,"s":[34.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":543,"s":[37.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":544,"s":[41.07],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":545,"s":[44.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":546,"s":[45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"s":{"a":1,"k":[{"t":0,"s":[133.33,133.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":508,"s":[133.33,133.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":509,"s":[133.31,133.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":510,"s":[133.26,133.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":511,"s":[133.16,133.16],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":512,"s":[133.01,133.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":513,"s":[132.78,132.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":514,"s":[132.46,132.46],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":515,"s":[132.04,132.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":516,"s":[131.5,131.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":517,"s":[130.84,130.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":518,"s":[130.03,130.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":519,"s":[129.06,129.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":520,"s":[127.92,127.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":521,"s":[126.59,126.59],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":522,"s":[125.06,125.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":523,"s":[123.31,123.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":524,"s":[121.33,121.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":525,"s":[119.11,119.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":526,"s":[116.64,116.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":527,"s":[113.88,113.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":528,"s":[110.85,110.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":529,"s":[107.51,107.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":530,"s":[103.85,103.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":531,"s":[99.87,99.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":532,"s":[95.54,95.54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":533,"s":[90.86,90.86],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":534,"s":[85.8,85.8],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":535,"s":[80.36,80.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":536,"s":[74.51,74.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":537,"s":[68.26,68.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":538,"s":[61.57,61.57],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":539,"s":[54.44,54.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":540,"s":[46.85,46.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":541,"s":[38.79,38.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":542,"s":[30.25,30.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":543,"s":[21.2,21.2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":544,"s":[11.64,11.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":545,"s":[1.56,1.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":546,"s":[0,0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface41","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.54],[0.44,-0.44],[0,0],[0.54,0],[0.44,0.43],[0,0],[0,0.55],[-0.43,0.44],[0,0],[-0.55,0],[-0.43,-0.43]],"o":[[0.45,0.45],[0,0.52],[0,0],[-0.46,0.46],[-0.53,0],[0,0],[-0.45,-0.45],[0,-0.53],[0,0],[0.45,-0.45],[0.53,0],[0,0]],"v":[[8.54,2.94],[9.34,4.36],[8.58,5.73],[6.46,7.86],[5.04,8.66],[3.66,7.91],[1.53,5.77],[0.73,4.35],[1.49,2.98],[3.62,0.85],[5.03,0.06],[6.4,0.8]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0.32,0],[0.29,-0.29],[0,0],[0,0],[0,-0.32],[-0.28,-0.28],[0,0],[0,0],[-0.31,0],[-0.29,0.29],[0,0],[0,0.32],[0.28,0.28]],"o":[[0,0],[0,0],[-0.28,-0.27],[-0.32,0],[0,0],[0,0],[-0.27,0.27],[0,0.33],[0,0],[0,0],[0.27,0.27],[0.33,0],[0,0],[0.27,-0.27],[0,-0.33],[0,0]],"v":[[8.2,3.42],[5.98,1.2],[5.94,1.16],[5.03,0.65],[4.09,1.2],[1.87,3.41],[1.83,3.45],[1.33,4.35],[1.87,5.29],[4.1,7.53],[4.14,7.56],[5.04,8.06],[5.98,7.52],[8.24,5.26],[8.74,4.36],[8.2,3.42]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[3.41,3.93],[3.88,4.41],[3.41,4.89],[2.94,4.41],[3.41,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.25,0]],"o":[[0.26,0],[0,0.27],[-0.26,0],[0,-0.26],[0,0]],"v":[[5.02,3.93],[5.48,4.41],[5.02,4.89],[4.55,4.41],[5.02,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[6.62,3.93],[7.09,4.41],[6.62,4.89],[6.15,4.41],[6.62,3.93]]}}},{"ty":"fl","c":{"a":0,"k":[0.21,0.44,0.86,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"GcK4r-hpimL7Z7K5ajIqW","layers":[{"ddd":0,"ind":66,"ty":0,"nm":"","ln":"precomp_YOzwnpwo118sXHojAUbTK66","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"WJuF6BioFt93Z3BD34mXO"},{"ddd":0,"ind":72,"ty":4,"nm":"","ln":"YOzwnpwo118sXHojAUbTK","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[149.52,58.52]}},{"ty":"st","hd":false,"bm":0,"c":{"a":1,"k":[{"t":0,"s":[0.835,0.843,0.859],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":423,"s":[0.835,0.843,0.859],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":424,"s":[0.851,0.855,0.871],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":425,"s":[0.878,0.886,0.894],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":426,"s":[0.914,0.918,0.925],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":427,"s":[0.941,0.945,0.949],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":428,"s":[0.961,0.965,0.969],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":429,"s":[0.976,0.976,0.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":430,"s":[0.984,0.988,0.988],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":431,"s":[0.992,0.992,0.992],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":432,"s":[0.996,0.996,0.996],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":433,"s":[1,1,1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0.48}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":0}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":71,"ty":4,"nm":"","ln":"YOzwnpwo118sXHojAUbTK","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[150,59]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"jv1MtMxxpqhMXSn611P71","layers":[{"ddd":0,"ind":75,"ty":4,"nm":"","ln":"7jxnGvZOO7TauyA45rPZ0","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2153.66,2197.97]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[42.04,6.21]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":76,"ty":4,"nm":"","ln":"-wARX8SHcsX80YA_i3ae7","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2260,2185.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[12,12]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":77,"ty":4,"nm":"","ln":"-ekyhZ8ryVQUHDntgp7ML","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2179.22,2185.07]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[93.15,10.03]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":78,"ty":4,"ln":"layer_78","sr":1,"ks":{"a":{"a":0,"k":[4.875,4.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2259.5,2185.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface46","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.54],[0.44,-0.44],[0,0],[0.54,0],[0.44,0.43],[0,0],[0,0.55],[-0.43,0.44],[0,0],[-0.55,0],[-0.43,-0.43]],"o":[[0.45,0.45],[0,0.52],[0,0],[-0.46,0.46],[-0.53,0],[0,0],[-0.45,-0.45],[0,-0.53],[0,0],[0.45,-0.45],[0.53,0],[0,0]],"v":[[8.54,2.94],[9.34,4.36],[8.58,5.73],[6.46,7.86],[5.04,8.66],[3.66,7.91],[1.53,5.77],[0.73,4.35],[1.49,2.98],[3.62,0.85],[5.03,0.06],[6.4,0.8]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0.32,0],[0.29,-0.29],[0,0],[0,0],[0,-0.32],[-0.28,-0.28],[0,0],[0,0],[-0.31,0],[-0.29,0.29],[0,0],[0,0.32],[0.28,0.28]],"o":[[0,0],[0,0],[-0.28,-0.27],[-0.32,0],[0,0],[0,0],[-0.27,0.27],[0,0.33],[0,0],[0,0],[0.27,0.27],[0.33,0],[0,0],[0.27,-0.27],[0,-0.33],[0,0]],"v":[[8.2,3.42],[5.98,1.2],[5.94,1.16],[5.03,0.65],[4.09,1.2],[1.87,3.41],[1.83,3.45],[1.33,4.35],[1.87,5.29],[4.1,7.53],[4.14,7.56],[5.04,8.06],[5.98,7.52],[8.24,5.26],[8.74,4.36],[8.2,3.42]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[3.41,3.93],[3.88,4.41],[3.41,4.89],[2.94,4.41],[3.41,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.25,0]],"o":[[0.26,0],[0,0.27],[-0.26,0],[0,-0.26],[0,0]],"v":[[5.02,3.93],[5.48,4.41],[5.02,4.89],[4.55,4.41],[5.02,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[6.62,3.93],[7.09,4.41],[6.62,4.89],[6.15,4.41],[6.62,3.93]]}}},{"ty":"fl","c":{"a":0,"k":[0.21,0.44,0.86,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"MB46byIoNLIvLNOn4t3SE","layers":[{"ddd":0,"ind":74,"ty":0,"nm":"","ln":"precomp_U_m_GZJIPkMpWJW26lh1L74","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"jv1MtMxxpqhMXSn611P71"},{"ddd":0,"ind":81,"ty":4,"nm":"","ln":"U_m_GZJIPkMpWJW26lh1L","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[149.52,58.52]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0.835,0.843,0.859]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0.48}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":0}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":80,"ty":4,"nm":"","ln":"U_m_GZJIPkMpWJW26lh1L","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[150,59]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"XWVSklTMgedpehWW5eXQP","layers":[{"ddd":0,"ind":84,"ty":4,"nm":"","ln":"G7kQlRW7r7QFw4Aa8oTjA","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2153.66,2197.97]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[42.04,6.21]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":85,"ty":4,"nm":"","ln":"3HXpXpbkfpustxDVHa-BU","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2260,2185.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[12,12]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":86,"ty":4,"nm":"","ln":"r-Fd5s_ukMbt6wU4zHFPW","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2179.22,2185.07]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[93.15,10.03]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":87,"ty":4,"ln":"layer_87","sr":1,"ks":{"a":{"a":0,"k":[4.875,4.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2259.5,2185.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface51","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.54],[0.44,-0.44],[0,0],[0.54,0],[0.44,0.43],[0,0],[0,0.55],[-0.43,0.44],[0,0],[-0.55,0],[-0.43,-0.43]],"o":[[0.45,0.45],[0,0.52],[0,0],[-0.46,0.46],[-0.53,0],[0,0],[-0.45,-0.45],[0,-0.53],[0,0],[0.45,-0.45],[0.53,0],[0,0]],"v":[[8.54,2.94],[9.34,4.36],[8.58,5.73],[6.46,7.86],[5.04,8.66],[3.66,7.91],[1.53,5.77],[0.73,4.35],[1.49,2.98],[3.62,0.85],[5.03,0.06],[6.4,0.8]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0.32,0],[0.29,-0.29],[0,0],[0,0],[0,-0.32],[-0.28,-0.28],[0,0],[0,0],[-0.31,0],[-0.29,0.29],[0,0],[0,0.32],[0.28,0.28]],"o":[[0,0],[0,0],[-0.28,-0.27],[-0.32,0],[0,0],[0,0],[-0.27,0.27],[0,0.33],[0,0],[0,0],[0.27,0.27],[0.33,0],[0,0],[0.27,-0.27],[0,-0.33],[0,0]],"v":[[8.2,3.42],[5.98,1.2],[5.94,1.16],[5.03,0.65],[4.09,1.2],[1.87,3.41],[1.83,3.45],[1.33,4.35],[1.87,5.29],[4.1,7.53],[4.14,7.56],[5.04,8.06],[5.98,7.52],[8.24,5.26],[8.74,4.36],[8.2,3.42]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[3.41,3.93],[3.88,4.41],[3.41,4.89],[2.94,4.41],[3.41,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.25,0]],"o":[[0.26,0],[0,0.27],[-0.26,0],[0,-0.26],[0,0]],"v":[[5.02,3.93],[5.48,4.41],[5.02,4.89],[4.55,4.41],[5.02,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[6.62,3.93],[7.09,4.41],[6.62,4.89],[6.15,4.41],[6.62,3.93]]}}},{"ty":"fl","c":{"a":0,"k":[0.21,0.44,0.86,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"G7B-83s-0GO8IaulhgMaz","layers":[{"ddd":0,"ind":83,"ty":0,"nm":"","ln":"precomp_f2DWH5sY7h-14gpWKHT_u83","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"XWVSklTMgedpehWW5eXQP"},{"ddd":0,"ind":90,"ty":4,"nm":"","ln":"f2DWH5sY7h-14gpWKHT_u","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[149.52,58.52]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0.835,0.843,0.859]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0.48}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":0}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":89,"ty":4,"nm":"","ln":"f2DWH5sY7h-14gpWKHT_u","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[150,59]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"h2KPtp5HZ61SRQiOmfUua","layers":[{"ddd":0,"ind":93,"ty":4,"nm":"","ln":"MjxFZFUlaEzr3CtUI8Hw5","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2153.66,2197.97]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[42.04,6.21]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":94,"ty":4,"nm":"","ln":"yexF7cRPTv_P7GLmue8F2","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2260,2185.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[12,12]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":95,"ty":4,"nm":"","ln":"hdowKOL4FBT2mvp34oeXC","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2179.22,2185.07]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[93.15,10.03]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":96,"ty":4,"ln":"layer_96","sr":1,"ks":{"a":{"a":0,"k":[4.875,4.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2259.5,2185.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface56","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.54],[0.44,-0.44],[0,0],[0.54,0],[0.44,0.43],[0,0],[0,0.55],[-0.43,0.44],[0,0],[-0.55,0],[-0.43,-0.43]],"o":[[0.45,0.45],[0,0.52],[0,0],[-0.46,0.46],[-0.53,0],[0,0],[-0.45,-0.45],[0,-0.53],[0,0],[0.45,-0.45],[0.53,0],[0,0]],"v":[[8.54,2.94],[9.34,4.36],[8.58,5.73],[6.46,7.86],[5.04,8.66],[3.66,7.91],[1.53,5.77],[0.73,4.35],[1.49,2.98],[3.62,0.85],[5.03,0.06],[6.4,0.8]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0.32,0],[0.29,-0.29],[0,0],[0,0],[0,-0.32],[-0.28,-0.28],[0,0],[0,0],[-0.31,0],[-0.29,0.29],[0,0],[0,0.32],[0.28,0.28]],"o":[[0,0],[0,0],[-0.28,-0.27],[-0.32,0],[0,0],[0,0],[-0.27,0.27],[0,0.33],[0,0],[0,0],[0.27,0.27],[0.33,0],[0,0],[0.27,-0.27],[0,-0.33],[0,0]],"v":[[8.2,3.42],[5.98,1.2],[5.94,1.16],[5.03,0.65],[4.09,1.2],[1.87,3.41],[1.83,3.45],[1.33,4.35],[1.87,5.29],[4.1,7.53],[4.14,7.56],[5.04,8.06],[5.98,7.52],[8.24,5.26],[8.74,4.36],[8.2,3.42]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[3.41,3.93],[3.88,4.41],[3.41,4.89],[2.94,4.41],[3.41,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.25,0]],"o":[[0.26,0],[0,0.27],[-0.26,0],[0,-0.26],[0,0]],"v":[[5.02,3.93],[5.48,4.41],[5.02,4.89],[4.55,4.41],[5.02,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[6.62,3.93],[7.09,4.41],[6.62,4.89],[6.15,4.41],[6.62,3.93]]}}},{"ty":"fl","c":{"a":0,"k":[0.21,0.44,0.86,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"WxOmXtI-s-IIB0w3ba7Lo","layers":[{"ddd":0,"ind":92,"ty":0,"nm":"","ln":"precomp_oAAGY5c0dHGbpyH_d2Y4X92","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"h2KPtp5HZ61SRQiOmfUua"},{"ddd":0,"ind":99,"ty":4,"nm":"","ln":"oAAGY5c0dHGbpyH_d2Y4X","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[149.52,58.52]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0.835,0.843,0.859]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0.48}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":0}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":98,"ty":4,"nm":"","ln":"oAAGY5c0dHGbpyH_d2Y4X","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[150,59]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"oEq9iwHqZT68h7cLXxDpT","layers":[{"ddd":0,"ind":102,"ty":4,"nm":"","ln":"5ob0WGJhTRLuQEqifULOO","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2153.66,2197.97]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[42.04,6.21]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":103,"ty":4,"nm":"","ln":"NQT2jbStDWcLbmcltmDEY","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2260,2185.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[12,12]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":104,"ty":4,"nm":"","ln":"jiSvuUd0ebMqxaQ8e1IXn","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2179.22,2185.07]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[93.15,10.03]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":105,"ty":4,"ln":"layer_105","sr":1,"ks":{"a":{"a":0,"k":[4.875,4.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2259.5,2185.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface61","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.54],[0.44,-0.44],[0,0],[0.54,0],[0.44,0.43],[0,0],[0,0.55],[-0.43,0.44],[0,0],[-0.55,0],[-0.43,-0.43]],"o":[[0.45,0.45],[0,0.52],[0,0],[-0.46,0.46],[-0.53,0],[0,0],[-0.45,-0.45],[0,-0.53],[0,0],[0.45,-0.45],[0.53,0],[0,0]],"v":[[8.54,2.94],[9.34,4.36],[8.58,5.73],[6.46,7.86],[5.04,8.66],[3.66,7.91],[1.53,5.77],[0.73,4.35],[1.49,2.98],[3.62,0.85],[5.03,0.06],[6.4,0.8]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0.32,0],[0.29,-0.29],[0,0],[0,0],[0,-0.32],[-0.28,-0.28],[0,0],[0,0],[-0.31,0],[-0.29,0.29],[0,0],[0,0.32],[0.28,0.28]],"o":[[0,0],[0,0],[-0.28,-0.27],[-0.32,0],[0,0],[0,0],[-0.27,0.27],[0,0.33],[0,0],[0,0],[0.27,0.27],[0.33,0],[0,0],[0.27,-0.27],[0,-0.33],[0,0]],"v":[[8.2,3.42],[5.98,1.2],[5.94,1.16],[5.03,0.65],[4.09,1.2],[1.87,3.41],[1.83,3.45],[1.33,4.35],[1.87,5.29],[4.1,7.53],[4.14,7.56],[5.04,8.06],[5.98,7.52],[8.24,5.26],[8.74,4.36],[8.2,3.42]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[3.41,3.93],[3.88,4.41],[3.41,4.89],[2.94,4.41],[3.41,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.25,0]],"o":[[0.26,0],[0,0.27],[-0.26,0],[0,-0.26],[0,0]],"v":[[5.02,3.93],[5.48,4.41],[5.02,4.89],[4.55,4.41],[5.02,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[6.62,3.93],[7.09,4.41],[6.62,4.89],[6.15,4.41],[6.62,3.93]]}}},{"ty":"fl","c":{"a":0,"k":[0.21,0.44,0.86,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"su_k_kfz4yaqO8jMMyzO3","layers":[{"ddd":0,"ind":101,"ty":0,"nm":"","ln":"precomp_FVwmrfkC3U1raGu0hogAE101","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"oEq9iwHqZT68h7cLXxDpT"},{"ddd":0,"ind":108,"ty":4,"nm":"","ln":"FVwmrfkC3U1raGu0hogAE","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[149.52,58.52]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0.835,0.843,0.859]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0.48}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":0}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":107,"ty":4,"nm":"","ln":"FVwmrfkC3U1raGu0hogAE","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[150,59]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"xuUVTJvq0qEnhU1Z-RNEm","layers":[{"ddd":0,"ind":65,"ty":0,"nm":"","ln":"precomp_YOzwnpwo118sXHojAUbTK65","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":1,"k":[{"t":0,"s":[2200,2338],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":422,"s":[2200,2338],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":423,"s":[2199.98,2338],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":424,"s":[2196.52,2338.86],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":425,"s":[2191.55,2340.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":426,"s":[2184.94,2341.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":427,"s":[2176.62,2343.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":428,"s":[2166.55,2346.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":429,"s":[2154.81,2349.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":430,"s":[2141.6,2352.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":431,"s":[2127.19,2355.9],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":432,"s":[2111.98,2359.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":433,"s":[2096.36,2363.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":434,"s":[2080.72,2367.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":435,"s":[2065.37,2371.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":436,"s":[2050.54,2374.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":437,"s":[2036.38,2378.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":438,"s":[2022.99,2381.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":439,"s":[2010.39,2384.61],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":440,"s":[1998.6,2387.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":441,"s":[1987.6,2390.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":442,"s":[1977.35,2392.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":443,"s":[1967.82,2395.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":444,"s":[1958.97,2397.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":445,"s":[1950.75,2399.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":446,"s":[1943.14,2401.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":447,"s":[1936.08,2402.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":448,"s":[1929.55,2404.49],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":449,"s":[1923.51,2405.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":450,"s":[1917.93,2407.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":451,"s":[1912.78,2408.61],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":452,"s":[1908.03,2409.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":453,"s":[1903.67,2410.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":454,"s":[1899.66,2411.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":455,"s":[1895.99,2412.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":456,"s":[1892.65,2413.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":457,"s":[1889.6,2414.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":458,"s":[1886.84,2414.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":459,"s":[1884.35,2415.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":460,"s":[1882.12,2416.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":461,"s":[1880.14,2416.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":462,"s":[1878.38,2417.07],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":463,"s":[1876.85,2417.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":464,"s":[1875.54,2417.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":465,"s":[1874.42,2418.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":466,"s":[1873.5,2418.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":467,"s":[1872.77,2418.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":468,"s":[1872.21,2418.59],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":469,"s":[1871.83,2418.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":470,"s":[1871.61,2418.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":471,"s":[1871.55,2418.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"r":{"a":0,"k":0},"s":{"a":1,"k":[{"t":0,"s":[100,100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":422,"s":[100,100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":423,"s":[100.02,100.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":424,"s":[104.77,104.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":425,"s":[111.58,111.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":426,"s":[120.63,120.63],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":427,"s":[132.04,132.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":428,"s":[145.83,145.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":429,"s":[161.91,161.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":430,"s":[180.02,180.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":431,"s":[199.75,199.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":432,"s":[220.59,220.59],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":433,"s":[241.99,241.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":434,"s":[263.42,263.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":435,"s":[284.45,284.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":436,"s":[304.77,304.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":437,"s":[324.16,324.16],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":438,"s":[342.52,342.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":439,"s":[359.77,359.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":440,"s":[375.93,375.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":441,"s":[391.01,391.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":442,"s":[405.05,405.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":443,"s":[418.1,418.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":444,"s":[430.23,430.23],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":445,"s":[441.48,441.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":446,"s":[451.92,451.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":447,"s":[461.58,461.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":448,"s":[470.53,470.53],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":449,"s":[478.81,478.81],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":450,"s":[486.46,486.46],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":451,"s":[493.51,493.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":452,"s":[500.01,500.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":453,"s":[505.99,505.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":454,"s":[511.48,511.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":455,"s":[516.51,516.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":456,"s":[521.09,521.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":457,"s":[525.27,525.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":458,"s":[529.05,529.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":459,"s":[532.46,532.46],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":460,"s":[535.52,535.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":461,"s":[538.24,538.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":462,"s":[540.64,540.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":463,"s":[542.73,542.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":464,"s":[544.54,544.54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":465,"s":[546.06,546.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":466,"s":[547.32,547.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":467,"s":[548.33,548.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":468,"s":[549.09,549.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":469,"s":[549.61,549.61],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":470,"s":[549.91,549.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":471,"s":[550,550],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"GcK4r-hpimL7Z7K5ajIqW"},{"ddd":0,"ind":73,"ty":0,"nm":"","ln":"precomp_U_m_GZJIPkMpWJW26lh1L73","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":422,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":423,"s":[99.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":424,"s":[90.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":425,"s":[81.39],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":426,"s":[72.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":427,"s":[62.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":428,"s":[53.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":429,"s":[44.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":430,"s":[35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":431,"s":[25.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":432,"s":[16.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":433,"s":[7.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":434,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[2200,2269]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"MB46byIoNLIvLNOn4t3SE"},{"ddd":0,"ind":82,"ty":0,"nm":"","ln":"precomp_f2DWH5sY7h-14gpWKHT_u82","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"G7B-83s-0GO8IaulhgMaz"},{"ddd":0,"ind":91,"ty":0,"nm":"","ln":"precomp_oAAGY5c0dHGbpyH_d2Y4X91","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2131]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"WxOmXtI-s-IIB0w3ba7Lo"},{"ddd":0,"ind":100,"ty":0,"nm":"","ln":"precomp_FVwmrfkC3U1raGu0hogAE100","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2062]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"su_k_kfz4yaqO8jMMyzO3"}]},{"id":"0AqroLC21-w_W7fP1p_se","layers":[{"ddd":0,"ind":111,"ty":4,"nm":"","ln":"O_GqK3SiqVM6uxuhRp1PI","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2153.66,2197.85]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[42.04,6.21]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":112,"ty":4,"nm":"","ln":"HSR7Yw7zo8WLITyLzac-r","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2260,2185.38]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[12,12]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":113,"ty":4,"nm":"","ln":"09JiQcAkaHK83F2j-FfBo","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2179.22,2184.95]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":6},"s":{"a":0,"k":[93.15,10.03]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0.851,0.851,0.851]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":114,"ty":4,"ln":"layer_114","sr":1,"ks":{"a":{"a":0,"k":[4.875,4.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2259.5,2185.38]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface66","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.54],[0.44,-0.44],[0,0],[0.54,0],[0.44,0.43],[0,0],[0,0.55],[-0.43,0.44],[0,0],[-0.55,0],[-0.43,-0.43]],"o":[[0.45,0.45],[0,0.52],[0,0],[-0.46,0.46],[-0.53,0],[0,0],[-0.45,-0.45],[0,-0.53],[0,0],[0.45,-0.45],[0.53,0],[0,0]],"v":[[8.54,2.94],[9.34,4.36],[8.58,5.73],[6.46,7.86],[5.04,8.66],[3.66,7.91],[1.53,5.77],[0.73,4.35],[1.49,2.98],[3.62,0.85],[5.03,0.06],[6.4,0.8]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,0],[0,0],[0.32,0],[0.29,-0.29],[0,0],[0,0],[0,-0.32],[-0.28,-0.28],[0,0],[0,0],[-0.31,0],[-0.29,0.29],[0,0],[0,0.32],[0.28,0.28]],"o":[[0,0],[0,0],[-0.28,-0.27],[-0.32,0],[0,0],[0,0],[-0.27,0.27],[0,0.33],[0,0],[0,0],[0.27,0.27],[0.33,0],[0,0],[0.27,-0.27],[0,-0.33],[0,0]],"v":[[8.2,3.42],[5.98,1.2],[5.94,1.16],[5.03,0.65],[4.09,1.2],[1.87,3.41],[1.83,3.45],[1.33,4.35],[1.87,5.29],[4.1,7.53],[4.14,7.56],[5.04,8.06],[5.98,7.52],[8.24,5.26],[8.74,4.36],[8.2,3.42]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[3.41,3.93],[3.88,4.41],[3.41,4.89],[2.94,4.41],[3.41,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.25,0]],"o":[[0.26,0],[0,0.27],[-0.26,0],[0,-0.26],[0,0]],"v":[[5.02,3.93],[5.48,4.41],[5.02,4.89],[4.55,4.41],[5.02,3.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.26],[0.26,0],[0,0.27],[-0.26,0]],"o":[[0.26,0],[0,0.27],[-0.27,0],[0,-0.26],[0,0]],"v":[[6.62,3.93],[7.09,4.41],[6.62,4.89],[6.15,4.41],[6.62,3.93]]}}},{"ty":"fl","c":{"a":0,"k":[0.21,0.44,0.86,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"x12A7yLE8RyjcXqdG-fcR","layers":[{"ddd":0,"ind":110,"ty":0,"nm":"","ln":"precomp_qFh6ju5aDkV3SpOBIZAJQ110","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"0AqroLC21-w_W7fP1p_se"},{"ddd":0,"ind":117,"ty":4,"nm":"","ln":"qFh6ju5aDkV3SpOBIZAJQ","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[149.52,58.76]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0.835,0.843,0.859]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0.48}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"r":1,"o":{"a":0,"k":0}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":116,"ty":4,"nm":"","ln":"qFh6ju5aDkV3SpOBIZAJQ","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2200]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":4},"s":{"a":0,"k":[150,59.24]}},{"ty":"st","hd":false,"bm":0,"c":{"a":0,"k":[0,0,0]},"lc":2,"lj":1,"o":{"a":0,"k":100},"w":{"a":0,"k":0}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"id":"tdeEH77cuXLwRjKVaLJaQ","layers":[{"ddd":0,"ind":119,"ty":4,"ln":"layer_119","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2166,2156]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface71","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.81,2.75],[3.07,0.01],[0.34,2.75],[3.07,5.48],[5.81,2.75]]}}},{"ty":"fl","c":{"a":0,"k":[0.94,0.73,0.51,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":120,"ty":4,"ln":"layer_120","sr":1,"ks":{"a":{"a":0,"k":[3,3.375]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2179,2148.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface76","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.53,3.3],[2.79,0.56],[0.06,3.3],[2.79,6.04],[5.53,3.3]]}}},{"ty":"fl","c":{"a":0,"k":[0.81,0.73,0.59,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":121,"ty":4,"ln":"layer_121","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2193,2145]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface81","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.59,2.73],[2.86,0],[0.12,2.73],[2.86,5.47],[5.59,2.73]]}}},{"ty":"fl","c":{"a":0,"k":[0.62,0.73,0.69,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":122,"ty":4,"ln":"layer_122","sr":1,"ks":{"a":{"a":0,"k":[3.375,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2207.5,2145]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface86","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6.02,2.73],[3.28,0],[0.55,2.73],[3.28,5.47],[6.02,2.73]]}}},{"ty":"fl","c":{"a":0,"k":[0.41,0.74,0.82,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":123,"ty":4,"ln":"layer_123","sr":1,"ks":{"a":{"a":0,"k":[3.375,3.375]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2221.5,2148.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface91","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6.07,3.34],[3.33,0.6],[0.59,3.34],[3.33,6.07],[6.07,3.34]]}}},{"ty":"fl","c":{"a":0,"k":[0.2,0.75,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":124,"ty":4,"ln":"layer_124","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2234,2156]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface96","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.79,2.81],[3.05,0.07],[0.31,2.81],[3.05,5.55],[5.79,2.81]]}}},{"ty":"fl","c":{"a":0,"k":[0.17,0.75,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":125,"ty":4,"ln":"layer_125","sr":1,"ks":{"a":{"a":0,"k":[3.375,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2244.5,2166]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface101","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6,3.07],[3.27,0.34],[0.53,3.07],[3.27,5.81],[6,3.07]]}}},{"ty":"fl","c":{"a":0,"k":[0.19,0.71,0.94,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":126,"ty":4,"ln":"layer_126","sr":1,"ks":{"a":{"a":0,"k":[3.375,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2251.5,2179]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface106","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6.19,2.79],[3.45,0.05],[0.71,2.79],[3.45,5.52],[6.19,2.79]]}}},{"ty":"fl","c":{"a":0,"k":[0.24,0.65,0.94,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":127,"ty":4,"ln":"layer_127","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2156,2166]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface111","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.58,2.96],[2.84,0.23],[0.11,2.96],[2.84,5.7],[5.58,2.96]]}}},{"ty":"fl","c":{"a":0,"k":[1,0.68,0.47,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":128,"ty":4,"ln":"layer_128","sr":1,"ks":{"a":{"a":0,"k":[3.375,3.375]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2148.5,2178.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface116","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6.08,3.41],[3.35,0.68],[0.61,3.41],[3.35,6.15],[6.08,3.41]]}}},{"ty":"fl","c":{"a":0,"k":[0.99,0.6,0.46,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":129,"ty":4,"ln":"layer_129","sr":1,"ks":{"a":{"a":0,"k":[3,3.375]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2145,2192.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface121","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.47,3.46],[2.73,0.72],[0,3.46],[2.73,6.19],[5.47,3.46]]}}},{"ty":"fl","c":{"a":0,"k":[0.98,0.54,0.44,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":130,"ty":4,"ln":"layer_130","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2145,2207]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface126","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.47,3.14],[2.73,0.41],[0,3.14],[2.73,5.88],[5.47,3.14]]}}},{"ty":"fl","c":{"a":0,"k":[0.98,0.4,0.41,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":131,"ty":4,"ln":"layer_131","sr":1,"ks":{"a":{"a":0,"k":[3.375,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2148.5,2221]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface131","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6.03,3.21],[3.29,0.48],[0.55,3.21],[3.29,5.95],[6.03,3.21]]}}},{"ty":"fl","c":{"a":0,"k":[0.96,0.32,0.41,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":132,"ty":4,"ln":"layer_132","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2156,2234]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface136","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.5,2.93],[2.76,0.19],[0.02,2.93],[2.76,5.66],[5.5,2.93]]}}},{"ty":"fl","c":{"a":0,"k":[0.95,0.21,0.42,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":133,"ty":4,"ln":"layer_133","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2166,2244]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface141","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.7,3.16],[2.96,0.43],[0.23,3.16],[2.96,5.9],[5.7,3.16]]}}},{"ty":"fl","c":{"a":0,"k":[0.93,0.13,0.54,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":134,"ty":4,"ln":"layer_134","sr":1,"ks":{"a":{"a":0,"k":[3.375,3.375]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2178.5,2251.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface146","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6.15,3.39],[3.41,0.65],[0.68,3.39],[3.41,6.13],[6.15,3.39]]}}},{"ty":"fl","c":{"a":0,"k":[0.93,0.13,0.55,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":135,"ty":4,"ln":"layer_135","sr":1,"ks":{"a":{"a":0,"k":[3.375,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2192.5,2255]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface151","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6.21,3.25],[3.47,0.51],[0.73,3.25],[3.47,5.98],[6.21,3.25]]}}},{"ty":"fl","c":{"a":0,"k":[0.9,0.12,0.66,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":136,"ty":4,"ln":"layer_136","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2207,2255]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface156","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.89,3.27],[3.15,0.53],[0.42,3.27],[3.15,6],[5.89,3.27]]}}},{"ty":"fl","c":{"a":0,"k":[0.87,0.13,0.78,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":137,"ty":4,"ln":"layer_137","sr":1,"ks":{"a":{"a":0,"k":[3,3.375]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2221,2251.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface161","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.95,3.45],[3.21,0.71],[0.48,3.45],[3.21,6.18],[5.95,3.45]]}}},{"ty":"fl","c":{"a":0,"k":[0.81,0.13,0.87,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":138,"ty":4,"ln":"layer_138","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2234,2244]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface166","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.68,3.26],[2.94,0.52],[0.2,3.26],[2.94,6],[5.68,3.26]]}}},{"ty":"fl","c":{"a":0,"k":[0.78,0.16,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":139,"ty":4,"ln":"layer_139","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2244,2234]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface171","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.91,3.05],[3.18,0.31],[0.44,3.05],[3.18,5.78],[5.91,3.05]]}}},{"ty":"fl","c":{"a":0,"k":[0.68,0.21,0.96,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":140,"ty":4,"ln":"layer_140","sr":1,"ks":{"a":{"a":0,"k":[3.375,3.375]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2251.5,2221.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface176","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6.14,3.33],[3.41,0.59],[0.67,3.33],[3.41,6.06],[6.14,3.33]]}}},{"ty":"fl","c":{"a":0,"k":[0.52,0.34,0.94,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":141,"ty":4,"ln":"layer_141","sr":1,"ks":{"a":{"a":0,"k":[3,3.375]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2255,2207.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface181","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[5.98,3.28],[3.25,0.54],[0.51,3.28],[3.25,6.02],[5.98,3.28]]}}},{"ty":"fl","c":{"a":0,"k":[0.54,0.34,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":142,"ty":4,"ln":"layer_142","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2255,2193]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface186","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.51,0],[0,-1.51],[-1.51,0],[0,1.51]],"o":[[0,-1.51],[-1.51,0],[0,1.51],[1.51,0],[0,0]],"v":[[6,2.86],[3.27,0.12],[0.53,2.86],[3.27,5.59],[6,2.86]]}}},{"ty":"fl","c":{"a":0,"k":[0.41,0.46,0.94,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":143,"ty":4,"ln":"layer_143","sr":1,"ks":{"a":{"a":0,"k":[3,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2167,2167.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface191","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.63,0.08],[0.46,-0.29],[0,0],[0.07,-0.63],[-0.39,-0.5],[-0.63,-0.08],[-0.1,0],[-0.43,0.33],[0.81,1.04]],"o":[[-0.39,-0.5],[-0.54,-0.07],[0,0],[-0.5,0.39],[-0.08,0.63],[0.39,0.5],[0.1,0.01],[0.53,0],[1.04,-0.81],[0,0]],"v":[[4.93,1.12],[3.34,0.22],[1.8,0.55],[1.58,0.71],[0.68,2.3],[1.17,4.06],[2.76,4.96],[3.05,4.98],[4.52,4.47],[4.93,1.12]]}}},{"ty":"fl","c":{"a":0,"k":[1,0.75,0.54,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":144,"ty":4,"ln":"layer_144","sr":1,"ks":{"a":{"a":0,"k":[2.625,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2159.5,2177]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface196","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.79,-1.05],[-0.09,-0.63],[-0.51,-0.38],[-0.52,0],[-0.11,0.02],[-0.38,0.51],[1.05,0.79]],"o":[[-1.05,-0.79],[-0.38,0.51],[0.09,0.63],[0.42,0.32],[0.11,0],[0.63,-0.09],[0.79,-1.05],[0,0]],"v":[[4.26,1.19],[0.91,1.66],[0.46,3.43],[1.38,5],[2.82,5.48],[3.15,5.46],[4.73,4.53],[4.26,1.19]]}}},{"ty":"fl","c":{"a":0,"k":[1,0.75,0.54,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":145,"ty":4,"ln":"layer_145","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2155.5,2188.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface201","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.49,-1.22],[-1.22,-0.49],[-0.3,0],[-0.3,0.13],[-0.24,0.59],[1.22,0.49]],"o":[[-1.22,-0.49],[-0.49,1.22],[0.29,0.12],[0.31,0],[0.59,-0.25],[0.49,-1.22],[0,0]],"v":[[3.29,0.29],[0.18,1.61],[1.5,4.72],[2.4,4.89],[3.33,4.7],[4.61,3.4],[3.29,0.29]]}}},{"ty":"fl","c":{"a":0,"k":[0.99,0.58,0.5,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":146,"ty":4,"ln":"layer_146","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2153.5,2200.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface206","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.3,0.16],[0.5,-0.39],[0.08,-0.63],[-0.39,-0.5],[-0.63,-0.08],[-0.09,0],[-0.15,1.2]],"o":[[0.16,-1.3],[-0.64,-0.08],[-0.5,0.39],[-0.08,0.63],[0.39,0.5],[0.1,0.01],[1.18,0],[0,0]],"v":[[5.04,2.92],[2.97,0.29],[1.22,0.77],[0.33,2.35],[0.82,4.09],[2.39,4.98],[2.68,5],[5.04,2.92]]}}},{"ty":"fl","c":{"a":0,"k":[0.98,0.52,0.51,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":147,"ty":4,"ln":"layer_147","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2155.5,2212.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface211","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.3,-0.18],[0.38,-0.51],[-0.09,-0.63],[-1.17,0],[-0.11,0.02],[0.18,1.3]],"o":[[-0.18,-1.3],[-0.63,0.09],[-0.38,0.51],[0.16,1.19],[0.11,0],[1.3,-0.18],[0,0]],"v":[[4.9,2.2],[2.21,0.16],[0.63,1.09],[0.18,2.86],[2.54,4.92],[2.87,4.89],[4.9,2.2]]}}},{"ty":"fl","c":{"a":0,"k":[0.98,0.45,0.48,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":148,"ty":4,"ln":"layer_148","sr":1,"ks":{"a":{"a":0,"k":[3,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2160,2223.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface216","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.59,-0.25],[0.24,-0.59],[-0.25,-0.59],[-0.59,-0.24],[-0.3,0],[-0.3,0.13],[-0.24,0.59],[0.25,0.59],[0.59,0.24]],"o":[[-0.59,-0.24],[-0.59,0.25],[-0.24,0.59],[0.25,0.59],[0.29,0.12],[0.32,0],[0.59,-0.25],[0.24,-0.59],[-0.25,-0.59],[0,0]],"v":[[3.97,0.44],[2.14,0.46],[0.86,1.76],[0.88,3.59],[2.18,4.87],[3.08,5.04],[4.01,4.85],[5.29,3.55],[5.27,1.72],[3.97,0.44]]}}},{"ty":"fl","c":{"a":0,"k":[0.97,0.38,0.49,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":149,"ty":4,"ln":"layer_149","sr":1,"ks":{"a":{"a":0,"k":[2.625,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2167.5,2233]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface221","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.63,0.08],[0.46,-0.28],[0,0],[-0.81,-1.04],[-0.63,-0.08],[-0.1,0],[-0.43,0.33],[0.81,1.04]],"o":[[-0.39,-0.5],[-0.54,-0.07],[0,0],[-1.04,0.81],[0.39,0.5],[0.1,0.01],[0.53,0],[1.04,-0.81],[0,0]],"v":[[4.54,1.61],[2.95,0.71],[1.41,1.04],[1.19,1.2],[0.78,4.55],[2.37,5.45],[2.67,5.46],[4.13,4.96],[4.54,1.61]]}}},{"ty":"fl","c":{"a":0,"k":[0.96,0.34,0.53,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":150,"ty":4,"ln":"layer_150","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2177.5,2240.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface226","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.63,-0.08],[0.38,-0.51],[-0.09,-0.63],[-0.51,-0.38],[-0.52,0],[-0.11,0.02],[-0.38,0.51],[0.09,0.63],[0.51,0.38]],"o":[[-0.51,-0.38],[-0.63,0.09],[-0.38,0.51],[0.09,0.63],[0.41,0.32],[0.11,0],[0.63,-0.09],[0.38,-0.51],[-0.09,-0.63],[0,0]],"v":[[3.83,0.65],[2.06,0.2],[0.49,1.12],[0.03,2.89],[0.96,4.46],[2.39,4.95],[2.73,4.92],[4.3,4],[4.76,2.23],[3.83,0.65]]}}},{"ty":"fl","c":{"a":0,"k":[0.95,0.24,0.61,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":151,"ty":4,"ln":"layer_151","sr":1,"ks":{"a":{"a":0,"k":[2.625,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2188.5,2245]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface231","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.5,-1.22],[-1.22,-0.49],[-0.3,0],[-0.37,0.93],[1.22,0.5]],"o":[[-1.22,-0.49],[-0.49,1.22],[0.29,0.12],[0.95,0],[0.5,-1.22],[0,0]],"v":[[3.47,0.77],[0.36,2.09],[1.68,5.2],[2.57,5.38],[4.79,3.88],[3.47,0.77]]}}},{"ty":"fl","c":{"a":0,"k":[0.92,0.19,0.69,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":152,"ty":4,"ln":"layer_152","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200.5,2246.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface236","it":[{"ty":"gr","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.16,-1.3],[0.5,-0.39],[0.53,0],[0.1,0.01],[0.39,0.5],[-0.08,0.63],[-1.2,-0.03],[-0.08,-0.01]],"o":[[1.3,0.16],[-0.08,0.63],[-0.42,0.33],[-0.1,0],[-0.63,-0.08],[-0.39,-0.5],[0.15,-1.21],[0.08,0],[0,0]],"v":[[2.95,0.2],[5.01,2.84],[4.11,4.42],[2.66,4.92],[2.37,4.91],[0.79,4.01],[0.3,2.27],[2.7,0.18],[2.95,0.2]]}}},{"ty":"fl","c":{"a":0,"k":[0.91,0.19,0.78,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":153,"ty":4,"ln":"layer_153","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2212.5,2244.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface241","it":[{"ty":"gr","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.64,-0.09],[0.38,-0.51],[-0.09,-0.63],[-0.51,-0.39],[-0.52,0],[-0.11,0.02],[-0.38,0.51],[0.09,0.63],[0.51,0.38]],"o":[[-0.51,-0.39],[-0.63,0.09],[-0.39,0.51],[0.09,0.63],[0.42,0.31],[0.11,0],[0.63,-0.09],[0.39,-0.51],[-0.09,-0.63],[0,0]],"v":[[4.04,0.92],[2.27,0.46],[0.7,1.39],[0.24,3.16],[1.17,4.73],[2.6,5.21],[2.93,5.19],[4.51,4.26],[4.97,2.49],[4.04,0.92]]}}},{"ty":"fl","c":{"a":0,"k":[0.87,0.19,0.86,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":154,"ty":4,"ln":"layer_154","sr":1,"ks":{"a":{"a":0,"k":[2.625,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2223.5,2240]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface246","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.21,-0.52],[-0.51,-1.21],[-0.93,0],[-0.3,0.13],[0.52,1.21]],"o":[[-0.51,-1.21],[-1.21,0.51],[0.39,0.91],[0.31,0],[1.21,-0.52],[0,0]],"v":[[4.91,2.12],[1.79,0.86],[0.52,3.98],[2.72,5.44],[3.65,5.25],[4.91,2.12]]}}},{"ty":"fl","c":{"a":0,"k":[0.84,0.22,0.93,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":155,"ty":4,"ln":"layer_155","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2233.5,2232.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface251","it":[{"ty":"gr","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.63,0.07],[0.5,-0.39],[-0.81,-1.04],[-0.63,-0.07],[-0.1,0],[-0.37,0.23],[0,0],[0.81,1.04]],"o":[[-0.39,-0.5],[-0.64,-0.08],[-1.04,0.81],[0.39,0.5],[0.1,0.02],[0.44,0],[0,0],[1.04,-0.81],[0,0]],"v":[[4.28,1.26],[2.69,0.36],[0.93,0.85],[0.52,4.2],[2.11,5.09],[2.41,5.11],[3.65,4.76],[3.87,4.61],[4.28,1.26]]}}},{"ty":"fl","c":{"a":0,"k":[0.78,0.25,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":156,"ty":4,"ln":"layer_156","sr":1,"ks":{"a":{"a":0,"k":[2.625,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2240.5,2223]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface256","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.63,-0.09],[0.38,-0.51],[-0.09,-0.63],[-0.51,-0.38],[-0.52,0],[-0.11,0.02],[-0.38,0.51],[1.05,0.79]],"o":[[-0.51,-0.38],[-0.63,0.09],[-0.38,0.51],[0.09,0.63],[0.42,0.32],[0.11,0],[0.63,-0.09],[0.79,-1.05],[0,0]],"v":[[4.07,1.06],[2.3,0.6],[0.73,1.53],[0.27,3.3],[1.2,4.87],[2.63,5.36],[2.96,5.33],[4.54,4.4],[4.07,1.06]]}}},{"ty":"fl","c":{"a":0,"k":[0.65,0.35,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":157,"ty":4,"ln":"layer_157","sr":1,"ks":{"a":{"a":0,"k":[3,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2245,2211.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface261","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.59,-0.25],[0.24,-0.59],[-0.25,-0.59],[-0.59,-0.24],[-0.3,0],[-0.37,0.93],[0.25,0.59],[0.59,0.24]],"o":[[-0.59,-0.24],[-0.59,0.25],[-0.24,0.59],[0.25,0.59],[0.29,0.12],[0.94,0],[0.24,-0.59],[-0.25,-0.59],[0,0]],"v":[[3.95,0.59],[2.12,0.61],[0.84,1.91],[0.86,3.73],[2.16,5.02],[3.05,5.19],[5.27,3.7],[5.25,1.87],[3.95,0.59]]}}},{"ty":"fl","c":{"a":0,"k":[0.51,0.44,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":158,"ty":4,"ln":"layer_158","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2246.5,2199.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface266","it":[{"ty":"gr","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.63,0.08],[0.5,-0.39],[0.08,-0.63],[-0.39,-0.5],[-0.62,-0.08],[-0.09,0],[-0.42,0.33],[-0.08,0.63],[0.39,0.5]],"o":[[-0.39,-0.5],[-0.63,-0.08],[-0.5,0.39],[-0.07,0.63],[0.39,0.5],[0.1,0.01],[0.53,0],[0.5,-0.39],[0.08,-0.63],[0,0]],"v":[[4.43,1.42],[2.85,0.52],[1.11,1.01],[0.21,2.59],[0.7,4.33],[2.27,5.23],[2.57,5.24],[4.02,4.74],[4.91,3.16],[4.43,1.42]]}}},{"ty":"fl","c":{"a":0,"k":[0.43,0.54,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":159,"ty":4,"ln":"layer_159","sr":1,"ks":{"a":{"a":0,"k":[3,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2245,2187.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface271","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.3,-0.18],[-0.18,-1.3],[-1.17,0],[-0.11,0.02],[0.18,1.3]],"o":[[-0.18,-1.3],[-1.3,0.19],[0.16,1.19],[0.11,0],[1.3,-0.18],[0,0]],"v":[[5.29,2.46],[2.6,0.42],[0.57,3.12],[2.93,5.18],[3.26,5.15],[5.29,2.46]]}}},{"ty":"fl","c":{"a":0,"k":[0.35,0.62,0.94,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":160,"ty":4,"ln":"layer_160","sr":1,"ks":{"a":{"a":0,"k":[3,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2240,2176.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface276","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.59,-0.25],[0.24,-0.59],[-0.25,-0.59],[-0.59,-0.24],[-0.3,0],[-0.3,0.13],[0.52,1.21],[0.59,0.24]],"o":[[-0.59,-0.24],[-0.59,0.25],[-0.24,0.59],[0.25,0.59],[0.29,0.12],[0.32,0],[1.21,-0.52],[-0.25,-0.59],[0,0]],"v":[[4.03,0.44],[2.2,0.46],[0.92,1.76],[0.94,3.59],[2.24,4.87],[3.14,5.04],[4.07,4.85],[5.33,1.72],[4.03,0.44]]}}},{"ty":"fl","c":{"a":0,"k":[0.29,0.69,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":161,"ty":4,"ln":"layer_161","sr":1,"ks":{"a":{"a":0,"k":[2.625,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2232.5,2167]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface281","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.39,-0.5],[1.04,-0.81],[0,0],[0.44,0],[0.1,0.01],[0.39,0.5],[-0.08,0.64],[-0.5,0.39],[-0.54,-0.01],[-0.09,-0.01]],"o":[[0.63,0.08],[0.81,1.04],[0,0],[-0.38,0.23],[-0.1,0],[-0.63,-0.08],[-0.39,-0.5],[0.07,-0.63],[0.43,-0.34],[0.09,0],[0,0]],"v":[[3.1,0.62],[4.69,1.52],[4.28,4.87],[4.06,5.02],[2.82,5.38],[2.52,5.36],[0.93,4.46],[0.44,2.7],[1.34,1.11],[2.83,0.6],[3.1,0.62]]}}},{"ty":"fl","c":{"a":0,"k":[0.25,0.77,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":162,"ty":4,"ln":"layer_162","sr":1,"ks":{"a":{"a":0,"k":[3,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2223,2159.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface286","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.63,-0.09],[0.37,-0.39],[0,0],[-1.05,-0.8],[-0.52,0],[-0.11,0.02],[-0.38,0.51],[0.09,0.63],[0.51,0.39]],"o":[[-0.51,-0.38],[-0.54,0.08],[0,0],[-0.79,1.05],[0.41,0.31],[0.11,0],[0.63,-0.09],[0.39,-0.51],[-0.09,-0.63],[0,0]],"v":[[4.48,0.85],[2.71,0.39],[1.31,1.12],[1.13,1.32],[1.61,4.67],[3.04,5.15],[3.37,5.12],[4.95,4.2],[5.41,2.43],[4.48,0.85]]}}},{"ty":"fl","c":{"a":0,"k":[0.27,0.78,0.95,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":163,"ty":4,"ln":"layer_163","sr":1,"ks":{"a":{"a":0,"k":[3,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2212,2155]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface291","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.5,-1.22],[-1.07,-0.56],[0,0],[-0.3,0],[-0.37,0.93],[0.25,0.59],[0.59,0.23]],"o":[[-1.23,-0.5],[-0.46,1.14],[0,0],[0.3,0.12],[0.95,0],[0.24,-0.59],[-0.25,-0.59],[0,0]],"v":[[3.78,0.86],[0.67,2.18],[1.77,5.18],[1.99,5.29],[2.88,5.46],[5.1,3.96],[5.08,2.14],[3.78,0.86]]}}},{"ty":"fl","c":{"a":0,"k":[0.38,0.77,0.89,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":164,"ty":4,"ln":"layer_164","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2199.5,2153.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface296","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.39,-0.5],[0.08,-0.62],[0.5,-0.39],[0.53,0],[0.1,0.01],[0.39,0.5],[-0.08,0.63],[-0.5,0.39],[-0.54,-0.01],[-0.09,-0.01]],"o":[[0.63,0.08],[0.39,0.5],[-0.08,0.63],[-0.42,0.33],[-0.09,0],[-0.63,-0.08],[-0.39,-0.5],[0.08,-0.63],[0.43,-0.33],[0.09,0],[0,0]],"v":[[3.16,0.34],[4.74,1.23],[5.23,2.97],[4.33,4.55],[2.88,5.05],[2.59,5.04],[1.01,4.14],[0.52,2.4],[1.42,0.82],[2.89,0.32],[3.16,0.34]]}}},{"ty":"fl","c":{"a":0,"k":[0.57,0.76,0.78,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":165,"ty":4,"ln":"layer_165","sr":1,"ks":{"a":{"a":0,"k":[2.625,2.625]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2187.5,2155.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface301","it":[{"ty":"gr","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.63,-0.09],[0.39,-0.51],[-0.09,-0.63],[-1.17,0],[-0.11,0.02],[-0.39,0.51],[0.09,0.63],[0.51,0.39]],"o":[[-0.51,-0.38],[-0.63,0.09],[-0.38,0.51],[0.16,1.19],[0.11,0],[0.63,-0.09],[0.38,-0.51],[-0.09,-0.63],[0,0]],"v":[[4.3,0.56],[2.53,0.11],[0.95,1.03],[0.5,2.8],[2.86,4.86],[3.2,4.83],[4.77,3.91],[5.23,2.14],[4.3,0.56]]}}},{"ty":"fl","c":{"a":0,"k":[0.73,0.76,0.69,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":166,"ty":4,"ln":"layer_166","sr":1,"ks":{"a":{"a":0,"k":[3.75,4.5]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2204,2181]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface306","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.05,0.2],[0.92,-0.48],[0.61,-0.71],[0.38,-0.81],[0,-0.86],[-0.02,-0.12],[-0.01,-0.05],[-0.16,0],[-0.86,0.45],[-0.56,0.68],[-0.33,0.82],[0,0.86]],"o":[[0,-0.25],[-0.84,0.04],[-0.92,0.48],[-0.48,0.54],[-0.37,0.82],[0,0.13],[0.02,0.11],[0.14,0.03],[0.77,0],[0.85,-0.45],[0.51,-0.64],[0.34,-0.82],[0,0]],"v":[[7.49,1.31],[7.42,0.63],[4.79,1.4],[2.49,3.19],[1.21,5.21],[0.65,7.72],[0.67,8.09],[0.72,8.33],[1.17,8.37],[3.59,7.7],[5.72,6],[6.99,3.82],[7.49,1.31]]}}},{"ty":"fl","c":{"a":0,"k":[0,0,0,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":167,"ty":4,"ln":"layer_167","sr":1,"ks":{"a":{"a":0,"k":[14.25,13.125]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2200,2203.5]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface311","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.24,0],[0.73,-0.25],[0.59,-0.26],[0.43,0],[0.56,0.25],[0.65,0.24],[0.74,0],[1.29,-0.76],[0.82,-1.48],[0,-2.16],[-0.31,-1.36],[-0.53,-1.21],[-0.64,-0.92],[-0.8,-0.84],[-1.05,0],[-0.52,0.23],[-0.57,0.23],[-0.86,0],[-0.52,-0.22],[-0.49,-0.23],[-0.76,0],[-0.61,0.41],[-0.51,0.64],[-0.48,0.68],[-0.33,0.71],[-0.16,0.46],[-0.02,0.06],[0,0],[0,0.02],[0.46,0.25],[0.58,0.53],[0.44,0.87],[0,1.24],[-0.34,0.79],[-0.46,0.52],[-0.37,0.27],[-0.05,0.04],[0.67,0.45],[0.64,0.21],[0.51,0.05]],"o":[[-0.51,-0.05],[-0.84,0],[-0.73,0.26],[-0.6,0.26],[-0.48,0],[-0.56,-0.24],[-0.65,-0.24],[-1.39,0],[-1.29,0.76],[-0.82,1.48],[0,1.36],[0.31,1.36],[0.53,1.21],[0.84,1.17],[0.79,0.83],[0.7,0],[0.51,-0.23],[0.57,-0.23],[0.87,0],[0.52,0.22],[0.49,0.23],[0.77,0],[0.6,-0.41],[0.52,-0.63],[0.54,-0.79],[0.33,-0.71],[0.02,-0.07],[0,0],[0.05,-0.16],[-0.02,-0.01],[-0.46,-0.25],[-0.58,-0.52],[-0.44,-0.86],[0,-1.08],[0.35,-0.79],[0.47,-0.51],[0.37,-0.27],[-0.56,-0.81],[-0.67,-0.46],[-0.64,-0.2],[0,0]],"v":[[21.5,0.58],[20.38,0.51],[18.02,0.89],[16.03,1.67],[14.48,2.05],[12.93,1.68],[11.11,0.96],[9.02,0.6],[5.01,1.74],[1.84,5.1],[0.62,10.56],[1.09,14.64],[2.35,18.49],[4.11,21.69],[6.56,24.7],[9.32,25.95],[11.14,25.6],[12.76,24.91],[14.9,24.57],[16.98,24.9],[18.5,25.57],[20.38,25.91],[22.43,25.3],[24.11,23.75],[25.59,21.78],[26.9,19.53],[27.64,17.77],[27.71,17.58],[27.81,17.29],[27.89,17.01],[27.16,16.63],[25.6,15.47],[24.07,13.38],[23.41,10.22],[23.93,7.43],[25.14,5.47],[26.4,4.3],[27.03,3.84],[25.18,1.95],[23.22,0.96],[21.5,0.58]]}}},{"ty":"fl","c":{"a":0,"k":[0,0,0,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":168,"ty":4,"ln":"layer_168","sr":1,"ks":{"a":{"a":0,"k":[2.625,3]},"o":{"a":0,"k":100},"p":{"a":0,"k":[2176.5,2160]},"r":{"a":0,"k":0},"s":{"a":0,"k":[133.33,133.33]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface316","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[1.21,-0.52],[-0.51,-1.21],[-0.93,0],[-0.3,0.13],[0.52,1.21]],"o":[[-0.52,-1.21],[-1.21,0.52],[0.39,0.91],[0.31,0],[1.21,-0.51],[0,0]],"v":[[4.92,2.07],[1.79,0.81],[0.53,3.94],[2.73,5.39],[3.66,5.2],[4.92,2.07]]}}},{"ty":"fl","c":{"a":0,"k":[0.88,0.76,0.61,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]}]},{"id":"dPC5W2E-7zMda_jZz4Ys2","layers":[{"ddd":0,"ind":17,"ty":4,"nm":"","ln":"RjSBkrotjCjlHXiimVxcq","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":1,"k":[{"t":0,"s":[13.35,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":680,"s":[13.35,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":681,"s":[14.24,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":682,"s":[15.51,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":683,"s":[17.2,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":684,"s":[19.34,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":685,"s":[21.92,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":686,"s":[24.94,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":687,"s":[28.33,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":688,"s":[32.02,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":689,"s":[35.93,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":690,"s":[39.93,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":691,"s":[43.94,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":692,"s":[47.87,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":693,"s":[51.67,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":694,"s":[55.3,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":695,"s":[58.73,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":696,"s":[61.95,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":697,"s":[64.96,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":698,"s":[67.78,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":699,"s":[70.4,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":700,"s":[72.83,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":701,"s":[75.1,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":702,"s":[77.19,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":703,"s":[79.14,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":704,"s":[80.94,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":705,"s":[82.61,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":706,"s":[84.15,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":707,"s":[85.57,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":708,"s":[86.89,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":709,"s":[88.1,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":710,"s":[89.21,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":711,"s":[90.23,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":712,"s":[91.17,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":713,"s":[92.02,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":714,"s":[92.79,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":715,"s":[93.49,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":716,"s":[94.13,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":717,"s":[94.69,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":718,"s":[95.2,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":719,"s":[95.64,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":720,"s":[96.03,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":721,"s":[96.36,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":722,"s":[96.64,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":723,"s":[96.87,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":724,"s":[97.06,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":725,"s":[97.19,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":726,"s":[97.29,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":727,"s":[97.34,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":728,"s":[97.51,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":729,"s":[98.48,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":730,"s":[99.84,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":731,"s":[101.63,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":732,"s":[103.86,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":733,"s":[106.54,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":734,"s":[109.64,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":735,"s":[113.1,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":736,"s":[116.85,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":737,"s":[120.79,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":738,"s":[124.8,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":739,"s":[128.8,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":740,"s":[132.71,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":741,"s":[136.47,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":742,"s":[140.05,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":743,"s":[143.44,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":744,"s":[146.62,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":745,"s":[149.59,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":746,"s":[152.36,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":747,"s":[154.94,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":748,"s":[157.34,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":749,"s":[159.56,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":750,"s":[161.63,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":751,"s":[163.54,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":752,"s":[165.31,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":753,"s":[166.95,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":754,"s":[168.47,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":755,"s":[169.87,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":756,"s":[171.16,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":757,"s":[172.35,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":758,"s":[173.44,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":759,"s":[174.44,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":760,"s":[175.36,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":761,"s":[176.19,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":762,"s":[176.95,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":763,"s":[177.64,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":764,"s":[178.25,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":765,"s":[178.81,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":766,"s":[179.3,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":767,"s":[179.73,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":768,"s":[180.1,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":769,"s":[180.43,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":770,"s":[180.7,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":771,"s":[180.92,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":772,"s":[181.09,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":773,"s":[181.22,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":774,"s":[181.3,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":775,"s":[181.34,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":776,"s":[181.35,113.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":3},"s":{"a":1,"k":[{"t":0,"s":[0,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":680,"s":[0,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":681,"s":[1.77,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":682,"s":[4.32,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":683,"s":[7.71,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":684,"s":[11.98,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":685,"s":[17.15,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":686,"s":[23.17,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":687,"s":[29.95,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":688,"s":[37.35,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":689,"s":[45.15,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":690,"s":[53.16,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":691,"s":[61.18,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":692,"s":[69.05,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":693,"s":[76.64,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":694,"s":[83.89,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":695,"s":[90.75,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":696,"s":[97.19,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":697,"s":[103.23,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":698,"s":[108.85,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":699,"s":[114.09,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":700,"s":[118.97,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":701,"s":[123.49,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":702,"s":[127.69,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":703,"s":[131.58,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":704,"s":[135.18,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":705,"s":[138.51,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":706,"s":[141.6,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":707,"s":[144.45,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":708,"s":[147.07,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":709,"s":[149.49,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":710,"s":[151.72,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":711,"s":[153.76,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":712,"s":[155.63,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":713,"s":[157.33,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":714,"s":[158.88,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":715,"s":[160.29,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":716,"s":[161.55,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":717,"s":[162.69,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":718,"s":[163.69,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":719,"s":[164.58,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":720,"s":[165.35,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":721,"s":[166.02,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":722,"s":[166.58,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":723,"s":[167.04,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":724,"s":[167.41,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":725,"s":[167.68,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":726,"s":[167.87,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":727,"s":[167.98,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":728,"s":[167.68,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":729,"s":[165.75,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":730,"s":[163.02,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":731,"s":[159.45,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":732,"s":[154.98,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":733,"s":[149.63,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":734,"s":[143.43,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":735,"s":[136.5,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":736,"s":[129,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":737,"s":[121.13,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":738,"s":[113.1,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":739,"s":[105.11,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":740,"s":[97.29,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":741,"s":[89.76,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":742,"s":[82.6,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":743,"s":[75.83,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":744,"s":[69.47,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":745,"s":[63.53,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":746,"s":[57.98,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":747,"s":[52.83,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":748,"s":[48.03,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":749,"s":[43.58,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":750,"s":[39.45,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":751,"s":[35.62,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":752,"s":[32.08,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":753,"s":[28.8,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":754,"s":[25.77,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":755,"s":[22.97,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":756,"s":[20.39,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":757,"s":[18.01,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":758,"s":[15.83,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":759,"s":[13.82,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":760,"s":[11.99,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":761,"s":[10.32,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":762,"s":[8.8,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":763,"s":[7.43,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":764,"s":[6.19,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":765,"s":[5.09,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":766,"s":[4.11,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":767,"s":[3.24,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":768,"s":[2.49,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":769,"s":[1.85,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":770,"s":[1.31,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":771,"s":[0.87,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":772,"s":[0.52,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":773,"s":[0.27,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":774,"s":[0.1,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":775,"s":[0.01,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":776,"s":[0,2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[0,0.82,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]},{"ddd":0,"ind":18,"ty":4,"ln":"layer_18","sr":1,"ks":{"a":{"a":0,"k":[21,20.625]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":545,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":546,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":780,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":781,"s":[99.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":782,"s":[99.97],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":783,"s":[99.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":784,"s":[99.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":785,"s":[99.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":786,"s":[99.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":787,"s":[99.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":788,"s":[99.29],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":789,"s":[99.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":790,"s":[98.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":791,"s":[98.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":792,"s":[97.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":793,"s":[97.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":794,"s":[96.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":795,"s":[96.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":796,"s":[95.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":797,"s":[94.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":798,"s":[93.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":799,"s":[92.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":800,"s":[91.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":801,"s":[89.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":802,"s":[88.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":803,"s":[87.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":804,"s":[85.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":805,"s":[83.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":806,"s":[81.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":807,"s":[79.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":808,"s":[77.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":809,"s":[74.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":810,"s":[72.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":811,"s":[69.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":812,"s":[66.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":813,"s":[63.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":814,"s":[60.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":815,"s":[56.53],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":816,"s":[52.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":817,"s":[48.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":818,"s":[44.8],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":819,"s":[40.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":820,"s":[35.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":821,"s":[31.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":822,"s":[26.12],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":823,"s":[20.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":824,"s":[15.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":825,"s":[9.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":826,"s":[3.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":827,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[98.35,51.75]},"r":{"a":1,"k":[{"t":0,"s":[-45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":545,"s":[-45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":546,"s":[-44.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":547,"s":[-44.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":548,"s":[-43.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":549,"s":[-42.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":550,"s":[-42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":551,"s":[-40.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":552,"s":[-39.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":553,"s":[-38.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":554,"s":[-36.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":555,"s":[-35.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":556,"s":[-33.81],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":557,"s":[-32.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":558,"s":[-30.43],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":559,"s":[-28.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":560,"s":[-27.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":561,"s":[-25.43],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":562,"s":[-23.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":563,"s":[-22.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":564,"s":[-20.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":565,"s":[-19.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":566,"s":[-18.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":567,"s":[-16.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":568,"s":[-15.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":569,"s":[-14.65],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":570,"s":[-13.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":571,"s":[-12.61],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":572,"s":[-11.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":573,"s":[-10.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":574,"s":[-9.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":575,"s":[-9.18],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":576,"s":[-8.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":577,"s":[-7.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":578,"s":[-7.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":579,"s":[-6.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":580,"s":[-5.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":581,"s":[-5.39],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":582,"s":[-4.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":583,"s":[-4.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":584,"s":[-3.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":585,"s":[-3.57],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":586,"s":[-3.18],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":587,"s":[-2.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":588,"s":[-2.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":589,"s":[-2.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":590,"s":[-1.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":591,"s":[-1.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":592,"s":[-1.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":593,"s":[-1.2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":594,"s":[-1.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":595,"s":[-0.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":596,"s":[-0.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":597,"s":[-0.54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":598,"s":[-0.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":599,"s":[-0.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":600,"s":[-0.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":601,"s":[-0.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":602,"s":[-0.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":603,"s":[-0.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":604,"s":[-0.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"s":{"a":1,"k":[{"t":0,"s":[0,0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":545,"s":[0,0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":546,"s":[0.88,0.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":547,"s":[2.26,2.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":548,"s":[4.04,4.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":549,"s":[6.25,6.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":550,"s":[8.89,8.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":551,"s":[11.98,11.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":552,"s":[15.5,15.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":553,"s":[19.43,19.43],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":554,"s":[23.73,23.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":555,"s":[28.33,28.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":556,"s":[33.16,33.16],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":557,"s":[38.13,38.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":558,"s":[43.17,43.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":559,"s":[48.2,48.2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":560,"s":[53.15,53.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":561,"s":[57.99,57.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":562,"s":[62.66,62.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":563,"s":[67.16,67.16],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":564,"s":[71.46,71.46],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":565,"s":[75.56,75.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":566,"s":[79.45,79.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":567,"s":[83.14,83.14],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":568,"s":[86.62,86.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":569,"s":[89.92,89.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":570,"s":[93.04,93.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":571,"s":[95.97,95.97],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":572,"s":[98.74,98.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":573,"s":[101.35,101.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":574,"s":[103.81,103.81],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":575,"s":[106.12,106.12],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":576,"s":[108.3,108.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":577,"s":[110.35,110.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":578,"s":[112.27,112.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":579,"s":[114.08,114.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":580,"s":[115.78,115.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":581,"s":[117.37,117.37],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":582,"s":[118.86,118.86],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":583,"s":[120.25,120.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":584,"s":[121.55,121.55],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":585,"s":[122.77,122.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":586,"s":[123.9,123.9],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":587,"s":[124.95,124.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":588,"s":[125.93,125.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":589,"s":[126.83,126.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":590,"s":[127.66,127.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":591,"s":[128.43,128.43],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":592,"s":[129.13,129.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":593,"s":[129.77,129.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":594,"s":[130.35,130.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":595,"s":[130.87,130.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":596,"s":[131.33,131.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":597,"s":[131.75,131.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":598,"s":[132.11,132.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":599,"s":[132.42,132.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":600,"s":[132.68,132.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":601,"s":[132.9,132.9],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":602,"s":[133.07,133.07],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":603,"s":[133.2,133.2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":604,"s":[133.28,133.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[133.33,133.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":781,"s":[133.33,133.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":782,"s":[133.31,133.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":783,"s":[133.29,133.29],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":784,"s":[133.25,133.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":785,"s":[133.19,133.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":786,"s":[133.11,133.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":787,"s":[133,133],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":788,"s":[132.86,132.86],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":789,"s":[132.69,132.69],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":790,"s":[132.48,132.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":791,"s":[132.22,132.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":792,"s":[131.93,131.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":793,"s":[131.58,131.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":794,"s":[131.19,131.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":795,"s":[130.74,130.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":796,"s":[130.23,130.23],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":797,"s":[129.65,129.65],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":798,"s":[129.01,129.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":799,"s":[128.3,128.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":800,"s":[127.51,127.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":801,"s":[126.65,126.65],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":802,"s":[125.71,125.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":803,"s":[124.68,124.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":804,"s":[123.56,123.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":805,"s":[122.35,122.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":806,"s":[121.04,121.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":807,"s":[119.63,119.63],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":808,"s":[118.12,118.12],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":809,"s":[116.51,116.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":810,"s":[114.78,114.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":811,"s":[112.94,112.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":812,"s":[110.98,110.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":813,"s":[108.9,108.9],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":814,"s":[106.69,106.69],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":815,"s":[104.35,104.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":816,"s":[101.89,101.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":817,"s":[99.28,99.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":818,"s":[96.54,96.54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":819,"s":[93.65,93.65],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":820,"s":[90.61,90.61],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":821,"s":[87.42,87.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":822,"s":[84.08,84.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":823,"s":[80.58,80.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":824,"s":[76.92,76.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":825,"s":[73.09,73.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":826,"s":[69.09,69.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":827,"s":[66.67,66.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","nm":"surface26","it":[{"ty":"gr","it":[{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.7],[0.7,0],[0,0.7],[-0.7,0]],"o":[[0.7,0],[0,0.7],[-0.7,0],[0,-0.7],[0,0]],"v":[[23.52,38.7],[24.79,39.98],[23.52,41.25],[22.24,39.98],[23.52,38.7]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.68,-0.18],[0,0],[0.17,-0.64],[0.56,0],[0.11,0.03],[-0.18,0.68]],"o":[[0.18,-0.68],[0,0],[0.61,0.23],[-0.15,0.57],[-0.11,0],[-0.68,-0.18],[0,0]],"v":[[17.19,39.64],[18.75,38.73],[18.87,38.77],[19.65,40.3],[18.42,41.24],[18.09,41.2],[17.19,39.64]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.29,-0.17],[-0.09,-0.33],[0.17,-0.29],[0.33,-0.09],[0.11,0],[0.2,0.11],[0.09,0.33],[-0.17,0.29],[-0.33,0.09]],"o":[[0.33,-0.09],[0.3,0.17],[0.09,0.33],[-0.17,0.3],[-0.11,0.03],[-0.22,0],[-0.3,-0.17],[-0.09,-0.33],[0.17,-0.3],[0,0]],"v":[[28.11,37.43],[29.07,37.56],[29.67,38.34],[29.54,39.3],[28.77,39.89],[28.44,39.94],[27.8,39.77],[27.21,38.99],[27.33,38.03],[28.11,37.43]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.29,-0.17],[-0.09,-0.33],[0.17,-0.29],[0.33,-0.09],[0.11,0],[0.2,0.11],[0.09,0.33],[-0.17,0.29],[-0.33,0.09]],"o":[[0.33,-0.09],[0.3,0.17],[0.09,0.33],[-0.17,0.3],[-0.11,0.03],[-0.22,0],[-0.3,-0.17],[-0.09,-0.33],[0.17,-0.3],[0,0]],"v":[[13.17,37.41],[14.14,37.54],[14.73,38.31],[14.61,39.28],[13.83,39.87],[13.5,39.91],[12.87,39.75],[12.27,38.97],[12.4,38],[13.17,37.41]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.07,-0.6],[0.23,-0.18],[0.24,0],[0.05,0],[0.18,0.23],[-0.04,0.29],[-0.61,-0.08]],"o":[[0.6,0.07],[-0.04,0.29],[-0.2,0.15],[-0.05,0],[-0.29,-0.04],[-0.18,-0.23],[0.07,-0.6],[0,0]],"v":[[21.32,35.76],[22.28,36.98],[21.86,37.72],[21.19,37.96],[21.05,37.95],[20.31,37.53],[20.09,36.71],[21.32,35.76]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.57,-0.23],[0.23,-0.57],[0.44,0],[0.14,0.05],[-0.23,0.57]],"o":[[0.23,-0.57],[0.57,0.23],[-0.18,0.43],[-0.14,0],[-0.57,-0.23],[0,0]],"v":[[15.93,35.93],[17.38,35.32],[17.99,36.77],[16.95,37.46],[16.54,37.38],[15.93,35.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.24,-0.18],[-0.04,-0.3],[0.18,-0.24],[0.3,-0.04],[0.05,0],[0.2,0.14],[0.04,0.29],[-0.18,0.24],[-0.29,0.04]],"o":[[0.3,-0.04],[0.24,0.18],[0.04,0.29],[-0.18,0.23],[-0.05,0.01],[-0.24,0],[-0.24,-0.18],[-0.04,-0.29],[0.18,-0.24],[0,0]],"v":[[25.2,35.19],[26.03,35.4],[26.46,36.14],[26.25,36.96],[25.51,37.39],[25.36,37.4],[24.69,37.18],[24.26,36.45],[24.47,35.62],[25.2,35.19]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.33,-0.09],[-0.17,-0.3],[0.61,-0.36],[0.22,0],[0.11,0.03],[0.17,0.29],[-0.09,0.33],[-0.29,0.17]],"o":[[0.3,-0.17],[0.33,0.09],[0.35,0.61],[-0.2,0.11],[-0.11,0],[-0.33,-0.09],[-0.17,-0.29],[0.09,-0.33],[0,0]],"v":[[32.21,35.02],[33.18,34.89],[33.96,35.49],[33.49,37.23],[32.86,37.4],[32.52,37.36],[31.75,36.76],[31.62,35.79],[32.21,35.02]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.45,-0.45],[0.5,-0.5],[0.32,0],[0.25,0.25],[0,0.34],[-0.24,0.24]],"o":[[0.49,-0.39],[0.5,0.5],[-0.25,0.25],[-0.33,0],[-0.24,-0.24],[0,-0.34],[0,0]],"v":[[8.3,35.09],[10,35.18],[10,36.99],[9.1,37.36],[8.2,36.99],[7.82,36.09],[8.2,35.18]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.23,-0.18],[-0.04,-0.29],[0.18,-0.24],[0.29,-0.04],[0.05,0],[0.19,0.15],[0.04,0.29],[-0.18,0.24],[-0.29,0.04]],"o":[[0.29,-0.04],[0.24,0.18],[0.04,0.29],[-0.18,0.24],[-0.05,0.01],[-0.24,0],[-0.24,-0.18],[-0.04,-0.29],[0.18,-0.24],[0,0]],"v":[[12.88,33.65],[13.7,33.87],[14.13,34.6],[13.92,35.43],[13.19,35.86],[13.03,35.87],[12.36,35.64],[11.93,34.91],[12.14,34.09],[12.88,33.65]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.24,-0.57],[0.57,-0.24],[0.14,0],[0.18,0.43],[-0.57,0.24]],"o":[[0.56,-0.24],[0.24,0.56],[-0.14,0.06],[-0.43,0],[-0.24,-0.56],[0,0]],"v":[[28.83,33.62],[30.29,34.21],[29.7,35.66],[29.26,35.75],[28.24,35.07],[28.83,33.62]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.45,-0.45],[0,-0.34],[0.24,-0.24],[0.33,0],[0.25,0.25],[0,0.34],[-0.24,0.24]],"o":[[0.49,-0.4],[0.24,0.24],[0,0.34],[-0.25,0.25],[-0.33,0],[-0.24,-0.24],[0,-0.34],[0,0]],"v":[[35.65,31.54],[37.36,31.63],[37.74,32.53],[37.36,33.43],[36.46,33.8],[35.56,33.43],[35.19,32.53],[35.56,31.63]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.33,-0.09],[-0.17,-0.29],[0.61,-0.35],[0.21,0],[0.24,0.41],[-0.09,0.33],[-0.29,0.17]],"o":[[0.29,-0.17],[0.33,0.09],[0.35,0.61],[-0.2,0.12],[-0.44,0],[-0.17,-0.3],[0.09,-0.33],[0,0]],"v":[[4.86,31.38],[5.83,31.25],[6.61,31.84],[6.14,33.58],[5.5,33.75],[4.39,33.12],[4.27,32.15],[4.86,31.38]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.18,-0.23],[0.48,-0.38],[0.25,0],[0.05,0.01],[0.18,0.23],[-0.48,0.38],[0,0],[-0.25,-0.04]],"o":[[0.3,0.04],[0.38,0.48],[-0.2,0.15],[-0.04,0],[-0.3,-0.04],[-0.37,-0.48],[0,0],[0.21,-0.13],[0,0]],"v":[[9.79,31.1],[10.53,31.52],[10.34,33.08],[9.65,33.31],[9.52,33.3],[8.77,32.89],[8.96,31.32],[9.07,31.25],[9.79,31.1]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.18,-0.23],[0.48,-0.38],[0,0],[0.21,0],[0.05,0.01],[0.18,0.23],[-0.48,0.38],[-0.29,-0.04]],"o":[[0.3,0.04],[0.38,0.48],[0,0],[-0.18,0.11],[-0.05,0],[-0.29,-0.04],[-0.37,-0.48],[0.23,-0.18],[0,0]],"v":[[32.74,30.93],[33.48,31.35],[33.29,32.91],[33.19,32.98],[32.61,33.15],[32.47,33.14],[31.73,32.72],[31.92,31.16],[32.74,30.93]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.27,-0.11],[-0.11,-0.28],[0.11,-0.27],[0.28,-0.12],[0.15,0],[0.13,0.05],[0.12,0.27],[-0.11,0.27],[-0.27,0.12]],"o":[[0.27,-0.11],[0.28,0.11],[0.12,0.27],[-0.11,0.28],[-0.14,0.06],[-0.14,0],[-0.28,-0.11],[-0.12,-0.27],[0.11,-0.28],[0,0]],"v":[[6.61,27.83],[7.46,27.83],[8.07,28.43],[8.08,29.27],[7.48,29.88],[7.05,29.97],[6.63,29.89],[6.02,29.29],[6.02,28.44],[6.61,27.83]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.23,-0.18],[0.37,-0.49],[0.29,-0.04],[0.05,0],[0.2,0.14],[0.04,0.29],[-0.18,0.24],[-0.29,0.04]],"o":[[0.29,-0.04],[0.49,0.37],[-0.18,0.24],[-0.05,0],[-0.24,0],[-0.23,-0.18],[-0.04,-0.29],[0.18,-0.24],[0,0]],"v":[[35,27.55],[35.83,27.77],[36.05,29.33],[35.32,29.76],[35.16,29.77],[34.49,29.55],[34.06,28.81],[34.27,27.99],[35,27.55]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.29,-0.17],[-0.09,-0.33],[0.17,-0.29],[0.33,-0.09],[0.11,0],[0.2,0.11],[0.09,0.33],[-0.17,0.29],[-0.33,0.09]],"o":[[0.33,-0.09],[0.3,0.17],[0.09,0.33],[-0.17,0.3],[-0.11,0.03],[-0.22,0],[-0.29,-0.17],[-0.09,-0.33],[0.17,-0.3],[0,0]],"v":[[38.68,26.89],[39.65,27.02],[40.25,27.79],[40.12,28.76],[39.34,29.35],[39.01,29.4],[38.38,29.23],[37.78,28.45],[37.91,27.48],[38.68,26.89]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.29,-0.17],[-0.09,-0.33],[0.17,-0.29],[0.33,-0.09],[0.11,0],[0.2,0.11],[0.09,0.33],[-0.17,0.29],[-0.33,0.09]],"o":[[0.33,-0.09],[0.3,0.17],[0.09,0.33],[-0.17,0.3],[-0.11,0.03],[-0.22,0],[-0.3,-0.17],[-0.09,-0.33],[0.17,-0.3],[0,0]],"v":[[2.63,26.83],[3.6,26.96],[4.19,27.73],[4.07,28.7],[3.29,29.29],[2.96,29.34],[2.32,29.17],[1.73,28.39],[1.86,27.43],[2.63,26.83]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.24,-0.02],[-0.3,-0.09],[-0.31,-0.21],[-0.26,-0.38],[0.18,-0.12],[0.22,-0.24],[0.16,-0.37],[0,-0.5],[-0.21,-0.4],[-0.27,-0.25],[-0.21,-0.12],[-0.01,-0.01],[0.02,-0.07],[0,0],[0.01,-0.03],[0.16,-0.33],[0.25,-0.37],[0.24,-0.29],[0.28,-0.19],[0.36,0],[0.23,0.11],[0.24,0.11],[0.41,0],[0.26,-0.11],[0.24,-0.11],[0.32,0],[0.37,0.39],[0.39,0.54],[0.25,0.57],[0.14,0.63],[0,0.64],[-0.38,0.69],[-0.6,0.36],[-0.64,0],[-0.3,-0.11],[-0.26,-0.11],[-0.22,0],[-0.28,0.12],[-0.34,0.12],[-0.39,0]],"o":[[0.11,0],[0.23,0.02],[0.3,0.1],[0.31,0.21],[-0.02,0.02],[-0.17,0.13],[-0.22,0.24],[-0.16,0.37],[0,0.58],[0.2,0.4],[0.27,0.25],[0.21,0.11],[0,0.01],[0,0],[-0.01,0.03],[-0.07,0.21],[-0.15,0.33],[-0.22,0.32],[-0.24,0.3],[-0.28,0.19],[-0.36,0],[-0.23,-0.11],[-0.24,-0.1],[-0.4,0],[-0.27,0.11],[-0.24,0.11],[-0.49,0],[-0.37,-0.39],[-0.3,-0.43],[-0.25,-0.57],[-0.14,-0.63],[0,-1.01],[0.38,-0.69],[0.6,-0.35],[0.34,0],[0.3,0.11],[0.26,0.11],[0.2,0],[0.28,-0.12],[0.34,-0.12],[0,0]],"v":[[23.86,15.96],[24.38,15.99],[25.18,16.17],[26.09,16.63],[26.95,17.52],[26.66,17.73],[26.08,18.27],[25.51,19.18],[25.27,20.49],[25.58,21.96],[26.29,22.93],[27.02,23.47],[27.36,23.65],[27.32,23.78],[27.27,23.91],[27.24,24],[26.89,24.82],[26.29,25.87],[25.59,26.79],[24.81,27.52],[23.86,27.8],[22.98,27.64],[22.27,27.33],[21.3,27.18],[20.3,27.33],[19.55,27.66],[18.7,27.82],[17.41,27.23],[16.27,25.83],[15.45,24.34],[14.86,22.55],[14.64,20.64],[15.22,18.1],[16.69,16.53],[18.56,16],[19.54,16.17],[20.38,16.51],[21.11,16.68],[21.83,16.5],[22.76,16.14],[23.86,15.96]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.09,-0.61],[0.61,-0.08],[0.05,0],[0.08,0.55],[-0.18,0.24],[-0.3,0.04]],"o":[[0.61,-0.08],[0.09,0.61],[-0.05,0.01],[-0.55,0],[-0.04,-0.29],[0.18,-0.24],[0,0]],"v":[[5.25,23.85],[6.5,24.8],[5.55,26.05],[5.4,26.07],[4.3,25.11],[4.51,24.29],[5.25,23.85]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.27,-0.11],[-0.12,-0.27],[0.11,-0.27],[0.44,0],[0.14,0.05],[0.11,0.27],[-0.11,0.27],[-0.27,0.12]],"o":[[0.28,-0.11],[0.28,0.11],[0.12,0.27],[-0.18,0.43],[-0.14,0],[-0.27,-0.11],[-0.12,-0.27],[0.11,-0.28],[0,0]],"v":[[36.32,23.71],[37.17,23.7],[37.78,24.3],[37.79,25.15],[36.75,25.84],[36.34,25.77],[35.73,25.17],[35.72,24.32],[36.32,23.71]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.68,-0.18],[0,0],[-0.15,-0.26],[0.09,-0.33],[0.56,0],[0.11,0.03],[-0.18,0.68]],"o":[[0.18,-0.68],[0,0],[0.27,0.1],[0.17,0.29],[-0.15,0.57],[-0.11,0],[-0.68,-0.18],[0,0]],"v":[[39.11,22.88],[40.67,21.97],[40.79,22.01],[41.45,22.57],[41.57,23.54],[40.34,24.48],[40.01,24.43],[39.11,22.88]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.7],[0.7,0],[0,0.7],[-0.7,0]],"o":[[0.7,0],[0,0.7],[-0.7,0],[0,-0.7],[0,0]],"v":[[1.65,21.86],[2.93,23.14],[1.65,24.41],[0.38,23.14],[1.65,21.86]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0.07,-0.6],[0.55,0],[0.04,0],[0.18,0.23],[-0.04,0.29],[-0.23,0.18],[-0.29,-0.04]],"o":[[0.61,0.07],[-0.07,0.56],[-0.04,0],[-0.29,-0.04],[-0.18,-0.23],[0.04,-0.29],[0.23,-0.18],[0,0]],"v":[[4.91,19.71],[5.87,20.94],[4.77,21.91],[4.64,21.91],[3.9,21.49],[3.68,20.68],[4.09,19.94],[4.91,19.71]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.18,-0.23],[0.04,-0.29],[0.23,-0.18],[0.24,0],[0.04,0],[0.18,0.23],[-0.04,0.29],[-0.23,0.18],[-0.29,-0.04]],"o":[[0.29,0.04],[0.18,0.23],[-0.04,0.29],[-0.2,0.15],[-0.05,0],[-0.29,-0.04],[-0.18,-0.23],[0.04,-0.29],[0.23,-0.18],[0,0]],"v":[[37.36,19.47],[38.1,19.88],[38.32,20.7],[37.91,21.43],[37.23,21.67],[37.09,21.66],[36.36,21.24],[36.13,20.43],[36.55,19.7],[37.36,19.47]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.7],[0.7,0],[0,0.7],[-0.7,0]],"o":[[0.7,0],[0,0.7],[-0.7,0],[0,-0.7],[0,0]],"v":[[40.35,16.84],[41.63,18.11],[40.35,19.39],[39.07,18.11],[40.35,16.84]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.68,-0.18],[0.18,-0.68],[0.56,0],[0.11,0.03],[0.17,0.3],[-0.09,0.33]],"o":[[0.18,-0.68],[0.68,0.18],[-0.15,0.57],[-0.11,0],[-0.33,-0.09],[-0.17,-0.29],[0,0]],"v":[[0.43,17.72],[1.99,16.82],[2.89,18.38],[1.66,19.32],[1.33,19.28],[0.55,18.68],[0.43,17.72]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.57,-0.23],[0.23,-0.57],[0.27,-0.12],[0.14,0],[0.13,0.05],[-0.23,0.57]],"o":[[0.23,-0.57],[0.57,0.23],[-0.11,0.28],[-0.14,0.06],[-0.14,0],[-0.57,-0.23],[0,0]],"v":[[4.31,16.14],[5.76,15.52],[6.37,16.97],[5.77,17.58],[5.34,17.67],[4.92,17.59],[4.31,16.14]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.09,-0.61],[0.61,-0.09],[0.05,0],[0.08,0.55],[-0.61,0.09]],"o":[[0.61,-0.09],[0.09,0.61],[-0.05,0.01],[-0.54,0],[-0.08,-0.61],[0,0]],"v":[[36.54,15.23],[37.8,16.18],[36.85,17.43],[36.69,17.45],[35.59,16.48],[36.54,15.23]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.12],[0.16,-0.38],[0.24,-0.3],[0.4,-0.21],[0.36,0],[0.07,0.01],[0.01,0.05],[0,0.06],[-0.17,0.38],[-0.22,0.25],[-0.43,0.22],[-0.39,0.02]],"o":[[0.02,0.09],[0,0.4],[-0.16,0.38],[-0.26,0.32],[-0.4,0.21],[-0.07,0],[-0.01,-0.02],[-0.01,-0.05],[0,-0.4],[0.18,-0.38],[0.29,-0.33],[0.43,-0.22],[0,0]],"v":[[24.1,12.18],[24.13,12.5],[23.9,13.67],[23.3,14.68],[22.32,15.47],[21.18,15.79],[20.98,15.77],[20.95,15.66],[20.94,15.48],[21.2,14.32],[21.8,13.37],[22.87,12.54],[24.1,12.18]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.18,-0.68],[0.68,-0.18],[0.11,0],[0.15,0.57],[-0.17,0.29],[-0.33,0.09]],"o":[[0.68,-0.18],[0.18,0.68],[-0.11,0.03],[-0.56,0],[-0.09,-0.33],[0.17,-0.3],[0,0]],"v":[[38.71,11.96],[40.27,12.86],[39.37,14.42],[39.04,14.46],[37.81,13.52],[37.93,12.55],[38.71,11.96]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.29,-0.17],[0.35,-0.61],[0.45,0],[0.2,0.11],[0.09,0.33],[-0.17,0.29],[-0.33,0.09]],"o":[[0.33,-0.09],[0.61,0.35],[-0.23,0.41],[-0.21,0],[-0.3,-0.17],[-0.09,-0.33],[0.17,-0.3],[0,0]],"v":[[2.66,11.9],[3.62,12.02],[4.09,13.77],[2.98,14.4],[2.35,14.23],[1.75,13.46],[1.88,12.49],[2.66,11.9]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.49,-0.37],[0.37,-0.49],[0.3,-0.04],[0.05,0],[0.2,0.15],[0.04,0.3],[-0.18,0.24]],"o":[[0.37,-0.49],[0.49,0.37],[-0.18,0.24],[-0.05,0.01],[-0.24,0],[-0.23,-0.18],[-0.04,-0.29],[0,0]],"v":[[6.04,11.96],[7.6,11.74],[7.82,13.3],[7.09,13.73],[6.93,13.74],[6.26,13.52],[5.83,12.78],[6.04,11.96]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.28,-0.11],[-0.12,-0.27],[0.56,-0.24],[0.14,0],[0.13,0.05],[0.12,0.27],[-0.11,0.27],[-0.27,0.12]],"o":[[0.27,-0.11],[0.28,0.11],[0.24,0.57],[-0.14,0.06],[-0.14,0],[-0.28,-0.11],[-0.11,-0.27],[0.11,-0.28],[0,0]],"v":[[34.61,11.4],[35.46,11.4],[36.07,11.99],[35.48,13.45],[35.05,13.54],[34.63,13.46],[34.02,12.86],[34.01,12.01],[34.61,11.4]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.18,-0.23],[0.48,-0.38],[0.25,0],[0.05,0.01],[0.18,0.23],[-0.04,0.29],[-0.23,0.18],[0,0],[-0.25,-0.04]],"o":[[0.29,0.04],[0.38,0.48],[-0.2,0.15],[-0.05,0],[-0.29,-0.04],[-0.18,-0.23],[0.04,-0.3],[0,0],[0.21,-0.13],[0,0]],"v":[[9.63,8.14],[10.36,8.56],[10.17,10.13],[9.49,10.36],[9.35,10.35],[8.61,9.93],[8.38,9.11],[8.8,8.37],[8.91,8.3],[9.63,8.14]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.18,-0.23],[0.48,-0.38],[0,0],[0.2,0],[0.05,0.01],[0.18,0.23],[-0.04,0.3],[-0.23,0.18],[-0.29,-0.04]],"o":[[0.29,0.04],[0.38,0.48],[0,0],[-0.18,0.11],[-0.05,0],[-0.29,-0.04],[-0.18,-0.23],[0.04,-0.29],[0.23,-0.18],[0,0]],"v":[[32.58,7.98],[33.32,8.4],[33.13,9.96],[33.02,10.03],[32.45,10.2],[32.3,10.19],[31.57,9.77],[31.34,8.95],[31.76,8.21],[32.58,7.98]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.33,-0.09],[-0.17,-0.29],[0.09,-0.33],[0.3,-0.17],[0,0],[0.17,0],[0.24,0.41],[-0.09,0.33],[-0.3,0.17]],"o":[[0.29,-0.17],[0.33,0.09],[0.17,0.3],[-0.09,0.33],[0,0],[-0.17,0.07],[-0.44,0],[-0.17,-0.3],[0.09,-0.33],[0,0]],"v":[[35.86,7.67],[36.83,7.54],[37.61,8.13],[37.73,9.1],[37.14,9.88],[37.02,9.94],[36.5,10.05],[35.39,9.41],[35.27,8.44],[35.86,7.67]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.5,-0.5],[0.5,-0.5],[0.33,0],[0.25,0.25],[-0.5,0.5]],"o":[[0.5,-0.5],[0.5,0.5],[-0.25,0.25],[-0.32,0],[-0.5,-0.5],[0,0]],"v":[[4.64,7.82],[6.44,7.82],[6.44,9.63],[5.54,10],[4.64,9.63],[4.64,7.82]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.24,-0.57],[0.57,-0.24],[0.14,0],[0.18,0.43],[-0.56,0.24]],"o":[[0.57,-0.24],[0.24,0.56],[-0.14,0.06],[-0.43,0],[-0.24,-0.56],[0,0]],"v":[[12.39,5.62],[13.86,6.21],[13.27,7.67],[12.83,7.76],[11.81,7.08],[12.39,5.62]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.24,-0.18],[-0.04,-0.29],[0.18,-0.24],[0.3,-0.04],[0.05,0],[0.2,0.15],[-0.37,0.49],[0,0],[-0.25,0.04]],"o":[[0.29,-0.04],[0.24,0.18],[0.04,0.29],[-0.18,0.24],[-0.05,0],[-0.24,0],[-0.49,-0.37],[0,0],[0.17,-0.18],[0,0]],"v":[[28.91,5.43],[29.73,5.64],[30.16,6.38],[29.95,7.2],[29.21,7.63],[29.06,7.64],[28.39,7.42],[28.17,5.86],[28.25,5.76],[28.91,5.43]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.5,-0.5],[0.5,-0.5],[0.33,0],[0.25,0.25],[-0.5,0.5]],"o":[[0.5,-0.5],[0.5,0.5],[-0.25,0.25],[-0.32,0],[-0.5,-0.5],[0,0]],"v":[[32,4.26],[33.8,4.26],[33.8,6.07],[32.9,6.44],[32,6.07],[32,4.26]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.35,-0.61],[0.61,-0.35],[0.22,0],[0.11,0.03],[0.17,0.3],[-0.61,0.35]],"o":[[0.61,-0.35],[0.35,0.61],[-0.2,0.11],[-0.11,0],[-0.33,-0.09],[-0.35,-0.61],[0,0]],"v":[[8.51,4.02],[10.25,4.49],[9.79,6.23],[9.15,6.4],[8.82,6.36],[8.04,5.76],[8.51,4.02]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.23,-0.18],[-0.04,-0.29],[0.18,-0.24],[0.29,-0.04],[0.05,0],[0.08,0.55],[-0.18,0.24],[-0.29,0.04]],"o":[[0.3,-0.04],[0.24,0.18],[0.04,0.3],[-0.18,0.24],[-0.05,0.01],[-0.55,0],[-0.04,-0.3],[0.18,-0.23],[0,0]],"v":[[16.58,3.89],[17.4,4.11],[17.84,4.84],[17.62,5.66],[16.89,6.1],[16.73,6.11],[15.63,5.15],[15.84,4.32],[16.58,3.89]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.57,-0.23],[-0.12,-0.27],[0.11,-0.27],[0.44,0],[0.14,0.05],[0,0],[-0.22,0.53]],"o":[[0.23,-0.57],[0.27,0.11],[0.12,0.27],[-0.18,0.43],[-0.14,0],[0,0],[-0.5,-0.26],[0,0]],"v":[[24.11,4.52],[25.55,3.91],[26.16,4.5],[26.17,5.35],[25.13,6.05],[24.72,5.97],[24.62,5.92],[24.11,4.52]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.18,-0.23],[0.04,-0.29],[0.23,-0.18],[0.25,0],[0.05,0.01],[0.18,0.23],[-0.04,0.29],[-0.23,0.18],[-0.29,-0.04]],"o":[[0.29,0.04],[0.18,0.23],[-0.04,0.29],[-0.2,0.15],[-0.05,0],[-0.29,-0.04],[-0.18,-0.23],[0.04,-0.29],[0.23,-0.18],[0,0]],"v":[[21.07,3.3],[21.81,3.72],[22.04,4.53],[21.62,5.27],[20.94,5.5],[20.8,5.49],[20.07,5.08],[19.84,4.27],[20.26,3.53],[21.07,3.3]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.29,-0.17],[-0.09,-0.33],[0.17,-0.3],[0.33,-0.09],[0.11,0],[0.2,0.11],[0.09,0.33],[-0.17,0.3],[-0.33,0.09]],"o":[[0.33,-0.09],[0.3,0.17],[0.09,0.33],[-0.17,0.29],[-0.11,0.03],[-0.22,0],[-0.3,-0.17],[-0.09,-0.33],[0.17,-0.29],[0,0]],"v":[[28.17,1.38],[29.13,1.51],[29.73,2.28],[29.6,3.25],[28.83,3.84],[28.5,3.89],[27.86,3.71],[27.27,2.94],[27.39,1.97],[28.17,1.38]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.25,-0.14],[-0.09,-0.33],[0.68,-0.18],[0.11,0],[0.2,0.11],[0.09,0.33],[-0.17,0.29],[-0.33,0.09]],"o":[[0.28,-0.04],[0.3,0.17],[0.18,0.68],[-0.11,0.03],[-0.22,0],[-0.3,-0.17],[-0.09,-0.33],[0.17,-0.3],[0,0]],"v":[[13.38,1.32],[14.2,1.48],[14.79,2.26],[13.89,3.82],[13.56,3.86],[12.93,3.69],[12.33,2.91],[12.46,1.95],[13.23,1.36]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[-0.33,-0.09],[-0.17,-0.29],[0.09,-0.33],[0.56,0],[0.11,0.03],[0,0],[-0.17,0.64],[-0.3,0.17]],"o":[[0.29,-0.17],[0.33,0.09],[0.17,0.3],[-0.15,0.57],[-0.11,0],[0,0],[-0.61,-0.23],[0.09,-0.33],[0,0]],"v":[[22.94,0.18],[23.91,0.05],[24.68,0.64],[24.81,1.61],[23.58,2.56],[23.25,2.52],[23.13,2.48],[22.35,0.95],[22.94,0.18]]}}},{"ty":"sh","d":1,"ks":{"a":0,"k":{"c":true,"i":[[0,0],[0,-0.7],[0.7,0],[0,0.71],[-0.7,0]],"o":[[0.7,0],[0,0.71],[-0.7,0],[0,-0.7],[0,0]],"v":[[18.48,0],[19.76,1.27],[18.48,2.55],[17.21,1.27],[18.48,0]]}}},{"ty":"fl","c":{"a":0,"k":[0,0,0,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}]}]},{"ddd":0,"ind":19,"ty":0,"nm":"","ln":"precomp_cQg2Y6Z1ffsOJMbZWr7B-19","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":149,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":150,"s":[1.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":151,"s":[3.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":152,"s":[6.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":153,"s":[8.14],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":154,"s":[10.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":155,"s":[12.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":156,"s":[14.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":157,"s":[16.49],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":158,"s":[18.57],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":159,"s":[20.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":160,"s":[22.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":161,"s":[24.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":162,"s":[26.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":163,"s":[29.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":164,"s":[31.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":165,"s":[33.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":166,"s":[35.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":167,"s":[37.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":168,"s":[39.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":169,"s":[41.54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":170,"s":[43.63],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":171,"s":[45.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":172,"s":[47.8],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":173,"s":[49.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":174,"s":[51.97],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":175,"s":[54.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":176,"s":[56.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":177,"s":[58.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":178,"s":[60.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":179,"s":[62.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":180,"s":[64.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":181,"s":[66.59],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":182,"s":[68.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":183,"s":[70.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":184,"s":[72.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":185,"s":[74.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":186,"s":[77.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":187,"s":[79.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":188,"s":[81.2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":189,"s":[83.29],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":190,"s":[85.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":191,"s":[87.46],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":192,"s":[89.55],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":193,"s":[91.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":194,"s":[93.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":195,"s":[95.81],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":196,"s":[97.9],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":197,"s":[99.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":198,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":1,"k":[{"t":0,"s":[98.35,71.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":262,"s":[98.35,71.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":263,"s":[89.63,72.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":264,"s":[78.91,72.07],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":265,"s":[68.63,72.12],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":266,"s":[58.77,72.18],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":267,"s":[49.33,72.23],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":268,"s":[40.3,72.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":269,"s":[31.67,72.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":270,"s":[23.43,72.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":271,"s":[15.57,72.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":272,"s":[8.09,72.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":273,"s":[0.96,72.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":274,"s":[-5.8,72.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":275,"s":[-12.22,72.55],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":276,"s":[-18.31,72.59],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":277,"s":[-24.07,72.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":278,"s":[-29.5,72.65],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":279,"s":[-34.63,72.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":280,"s":[-39.46,72.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":281,"s":[-44,72.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":282,"s":[-48.26,72.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":283,"s":[-52.24,72.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":284,"s":[-55.96,72.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":285,"s":[-59.42,72.8],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":286,"s":[-62.64,72.82],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":287,"s":[-65.62,72.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":288,"s":[-68.37,72.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":289,"s":[-70.91,72.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":290,"s":[-73.23,72.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":291,"s":[-75.36,72.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":292,"s":[-77.29,72.9],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":293,"s":[-79.04,72.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":294,"s":[-80.62,72.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":295,"s":[-82.03,72.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":296,"s":[-83.28,72.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":297,"s":[-84.39,72.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":298,"s":[-85.37,72.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":299,"s":[-86.21,72.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":300,"s":[-86.94,72.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":301,"s":[-87.55,72.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":302,"s":[-88.07,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":303,"s":[-88.49,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":304,"s":[-88.83,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":305,"s":[-89.1,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":306,"s":[-89.3,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":307,"s":[-89.44,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":308,"s":[-89.54,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":309,"s":[-89.6,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":310,"s":[-89.63,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":311,"s":[-89.65,72.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"30vRG6suD_HlTDgtTZFHu"},{"ddd":0,"ind":44,"ty":0,"nm":"","ln":"precomp_gVbPc150aaZLnCUFpaCID44","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":605,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":606,"s":[1.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":607,"s":[4.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":608,"s":[7.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":609,"s":[11.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":610,"s":[14.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":611,"s":[17.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":612,"s":[21.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":613,"s":[24.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":614,"s":[27.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":615,"s":[31.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":616,"s":[34.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":617,"s":[37.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":618,"s":[41.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":619,"s":[44.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":620,"s":[48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":621,"s":[51.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":622,"s":[54.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":623,"s":[58.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":624,"s":[61.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":625,"s":[64.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":626,"s":[68.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":627,"s":[71.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":628,"s":[74.72],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":629,"s":[78.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":630,"s":[81.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":631,"s":[84.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":632,"s":[88.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":633,"s":[91.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":634,"s":[94.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":635,"s":[98.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":636,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":780,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":781,"s":[99.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":782,"s":[99.97],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":783,"s":[99.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":784,"s":[99.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":785,"s":[99.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":786,"s":[99.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":787,"s":[99.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":788,"s":[99.29],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":789,"s":[99.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":790,"s":[98.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":791,"s":[98.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":792,"s":[97.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":793,"s":[97.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":794,"s":[96.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":795,"s":[96.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":796,"s":[95.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":797,"s":[94.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":798,"s":[93.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":799,"s":[92.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":800,"s":[91.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":801,"s":[89.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":802,"s":[88.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":803,"s":[87.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":804,"s":[85.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":805,"s":[83.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":806,"s":[81.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":807,"s":[79.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":808,"s":[77.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":809,"s":[74.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":810,"s":[72.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":811,"s":[69.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":812,"s":[66.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":813,"s":[63.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":814,"s":[60.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":815,"s":[56.53],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":816,"s":[52.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":817,"s":[48.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":818,"s":[44.8],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":819,"s":[40.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":820,"s":[35.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":821,"s":[31.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":822,"s":[26.12],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":823,"s":[20.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":824,"s":[15.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":825,"s":[9.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":826,"s":[3.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":827,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[98.35,104.25]},"r":{"a":0,"k":0},"s":{"a":1,"k":[{"t":0,"s":[100,100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":781,"s":[100,100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":782,"s":[99.99,99.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":783,"s":[99.97,99.97],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":784,"s":[99.94,99.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":785,"s":[99.89,99.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":786,"s":[99.83,99.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":787,"s":[99.75,99.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":788,"s":[99.64,99.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":789,"s":[99.51,99.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":790,"s":[99.36,99.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":791,"s":[99.17,99.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":792,"s":[98.95,98.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":793,"s":[98.69,98.69],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":794,"s":[98.39,98.39],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":795,"s":[98.05,98.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":796,"s":[97.67,97.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":797,"s":[97.24,97.24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":798,"s":[96.76,96.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":799,"s":[96.22,96.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":800,"s":[95.64,95.64],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":801,"s":[94.99,94.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":802,"s":[94.28,94.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":803,"s":[93.51,93.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":804,"s":[92.67,92.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":805,"s":[91.76,91.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":806,"s":[90.78,90.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":807,"s":[89.73,89.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":808,"s":[88.59,88.59],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":809,"s":[87.38,87.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":810,"s":[86.09,86.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":811,"s":[84.7,84.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":812,"s":[83.23,83.23],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":813,"s":[81.67,81.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":814,"s":[80.02,80.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":815,"s":[78.27,78.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":816,"s":[76.41,76.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":817,"s":[74.46,74.46],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":818,"s":[72.4,72.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":819,"s":[70.23,70.23],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":820,"s":[67.96,67.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":821,"s":[65.57,65.57],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":822,"s":[63.06,63.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":823,"s":[60.44,60.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":824,"s":[57.69,57.69],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":825,"s":[54.82,54.82],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":826,"s":[51.82,51.82],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":827,"s":[50,50],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"e1blxH8-FLcN9exYqpMpy"},{"ddd":0,"ind":64,"ty":0,"nm":"","ln":"precomp_aoTsCff9CXvpSJ6ylOjZk64","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":1,"k":[{"t":0,"s":[98.35,316.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":357,"s":[98.35,316.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":358,"s":[98.36,313.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":359,"s":[98.36,307.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":360,"s":[98.37,300.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":361,"s":[98.38,290.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":362,"s":[98.4,279.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":363,"s":[98.42,265.97],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":364,"s":[98.44,250.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":365,"s":[98.46,234.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":366,"s":[98.48,216.49],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":367,"s":[98.51,198.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":368,"s":[98.53,180.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":369,"s":[98.55,162.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":370,"s":[98.58,144.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":371,"s":[98.6,128.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":372,"s":[98.62,112.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":373,"s":[98.64,97.54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":374,"s":[98.66,83.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":375,"s":[98.67,70.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":376,"s":[98.69,58.65],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":377,"s":[98.7,47.43],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":378,"s":[98.72,37],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":379,"s":[98.73,27.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":380,"s":[98.74,18.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":381,"s":[98.75,10.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":382,"s":[98.76,2.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":383,"s":[98.77,-4.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":384,"s":[98.78,-11.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":385,"s":[98.79,-17.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":386,"s":[98.79,-23],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":387,"s":[98.8,-28.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":388,"s":[98.81,-32.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":389,"s":[98.81,-37.2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":390,"s":[98.82,-41.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":391,"s":[98.82,-44.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":392,"s":[98.83,-48.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":393,"s":[98.83,-50.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":394,"s":[98.83,-53.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":395,"s":[98.84,-55.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":396,"s":[98.84,-58.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":397,"s":[98.84,-59.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":398,"s":[98.84,-61.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":399,"s":[98.85,-62.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":400,"s":[98.85,-63.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":401,"s":[98.85,-64.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":402,"s":[98.85,-65.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":403,"s":[98.85,-65.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":404,"s":[98.85,-66.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":405,"s":[98.85,-66.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"xuUVTJvq0qEnhU1Z-RNEm"},{"ddd":0,"ind":109,"ty":0,"nm":"","ln":"precomp_qFh6ju5aDkV3SpOBIZAJQ109","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":1,"k":[{"t":0,"s":[277.35,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":255,"s":[277.35,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":256,"s":[275.67,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":257,"s":[271.85,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":258,"s":[266.45,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":259,"s":[260.84,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":260,"s":[256,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":261,"s":[252.13,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":262,"s":[249.1,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":263,"s":[246.75,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":264,"s":[244.95,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":265,"s":[243.61,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":266,"s":[242.63,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":267,"s":[241.95,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":268,"s":[241.55,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":269,"s":[241.37,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":270,"s":[241.35,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":275,"s":[241.35,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":276,"s":[240.56,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":277,"s":[238.61,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":278,"s":[235.92,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":279,"s":[232.47,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":280,"s":[228.22,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":281,"s":[223.18,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":282,"s":[217.42,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":283,"s":[211.05,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":284,"s":[204.23,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":285,"s":[197.13,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":286,"s":[189.95,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":287,"s":[182.84,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":288,"s":[175.92,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":289,"s":[169.29,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":290,"s":[162.99,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":291,"s":[157.05,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":292,"s":[151.48,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":293,"s":[146.28,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":294,"s":[141.42,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":295,"s":[136.91,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":296,"s":[132.72,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":297,"s":[128.82,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":298,"s":[125.21,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":299,"s":[121.86,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":300,"s":[118.76,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":301,"s":[115.89,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":302,"s":[113.24,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":303,"s":[110.79,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":304,"s":[108.54,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":305,"s":[106.46,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":306,"s":[104.55,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":307,"s":[102.8,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":308,"s":[101.2,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":309,"s":[99.74,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":310,"s":[98.42,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":311,"s":[97.22,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":312,"s":[96.15,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":313,"s":[95.19,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":314,"s":[94.33,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":315,"s":[93.59,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":316,"s":[92.94,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":317,"s":[92.39,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":318,"s":[91.93,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":319,"s":[91.55,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":320,"s":[91.26,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":321,"s":[91.05,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":322,"s":[90.92,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":323,"s":[90.86,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":324,"s":[90.85,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":357,"s":[90.85,71.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":358,"s":[90.85,69.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":359,"s":[90.85,66.69],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":360,"s":[90.85,62.43],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":361,"s":[90.85,57.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":362,"s":[90.85,50.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":363,"s":[90.85,42.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":364,"s":[90.85,34.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":365,"s":[90.85,24.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":366,"s":[90.85,14.49],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":367,"s":[90.85,4.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":368,"s":[90.85,-6.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":369,"s":[90.85,-16.61],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":370,"s":[90.85,-26.55],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":371,"s":[90.85,-36.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":372,"s":[90.85,-45.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":373,"s":[90.85,-53.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":374,"s":[90.85,-61.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":375,"s":[90.85,-68.86],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":376,"s":[90.85,-75.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":377,"s":[90.85,-82.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":378,"s":[90.85,-88.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":379,"s":[90.85,-93.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":380,"s":[90.85,-98.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":381,"s":[90.85,-103.55],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":382,"s":[90.85,-107.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":383,"s":[90.85,-112.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":384,"s":[90.85,-115.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":385,"s":[90.85,-119.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":386,"s":[90.85,-122.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":387,"s":[90.85,-125.39],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":388,"s":[90.85,-128.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":389,"s":[90.85,-130.57],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":390,"s":[90.85,-132.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":391,"s":[90.85,-134.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":392,"s":[90.85,-136.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":393,"s":[90.85,-138.43],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":394,"s":[90.85,-139.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":395,"s":[90.85,-141.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":396,"s":[90.85,-142.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":397,"s":[90.85,-143.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":398,"s":[90.85,-144.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":399,"s":[90.85,-145.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":400,"s":[90.85,-145.8],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":401,"s":[90.85,-146.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":402,"s":[90.85,-146.69],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":403,"s":[90.85,-146.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":404,"s":[90.85,-147.12],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":405,"s":[90.85,-147.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"x12A7yLE8RyjcXqdG-fcR"},{"ddd":0,"ind":118,"ty":0,"nm":"","ln":"precomp_4NPcpRsUakgVBjW7opIYg118","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":1,"k":[{"t":0,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":103,"s":[100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":104,"s":[99.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":105,"s":[97.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":106,"s":[94.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":107,"s":[92.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":108,"s":[90.8],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":109,"s":[88.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":110,"s":[86.63],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":111,"s":[84.54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":112,"s":[82.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":113,"s":[80.36],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":114,"s":[78.28],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":115,"s":[76.19],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":116,"s":[74.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":117,"s":[72.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":118,"s":[69.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":119,"s":[67.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":120,"s":[65.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":121,"s":[63.66],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":122,"s":[61.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":123,"s":[59.49],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":124,"s":[57.4],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":125,"s":[55.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":126,"s":[53.23],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":127,"s":[51.14],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":128,"s":[49.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":129,"s":[46.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":130,"s":[44.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":131,"s":[42.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":132,"s":[40.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":133,"s":[38.61],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":134,"s":[36.53],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":135,"s":[34.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":136,"s":[32.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":137,"s":[30.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":138,"s":[28.18],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":139,"s":[26.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":140,"s":[24],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":141,"s":[21.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":142,"s":[19.82],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":143,"s":[17.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":144,"s":[15.65],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":145,"s":[13.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":146,"s":[11.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":147,"s":[9.39],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":148,"s":[7.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":149,"s":[5.21],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":150,"s":[3.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":151,"s":[1.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":152,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"p":{"a":0,"k":[98.35,74.25]},"r":{"a":1,"k":[{"t":0,"s":[-45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":1,"s":[-44.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":2,"s":[-44.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":3,"s":[-44.14],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":4,"s":[-43.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":5,"s":[-43.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":6,"s":[-42.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":7,"s":[-42.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":8,"s":[-41.42],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":9,"s":[-40.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":10,"s":[-39.86],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":11,"s":[-38.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":12,"s":[-38.04],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":13,"s":[-37.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":14,"s":[-36.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":15,"s":[-34.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":16,"s":[-33.82],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":17,"s":[-32.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":18,"s":[-31.53],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":19,"s":[-30.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":20,"s":[-29.23],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":21,"s":[-28.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":22,"s":[-26.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":23,"s":[-25.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":24,"s":[-24.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":25,"s":[-23.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":26,"s":[-22.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":27,"s":[-21.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":28,"s":[-20.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":29,"s":[-19.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":30,"s":[-18.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":31,"s":[-17.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":32,"s":[-17.14],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":33,"s":[-16.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":34,"s":[-15.54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":35,"s":[-14.79],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":36,"s":[-14.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":37,"s":[-13.37],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":38,"s":[-12.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":39,"s":[-12.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":40,"s":[-11.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":41,"s":[-10.84],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":42,"s":[-10.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":43,"s":[-9.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":44,"s":[-9.2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":45,"s":[-8.7],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":46,"s":[-8.21],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":47,"s":[-7.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":48,"s":[-7.31],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":49,"s":[-6.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":50,"s":[-6.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":51,"s":[-6.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":52,"s":[-5.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":53,"s":[-5.35],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":54,"s":[-5.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":55,"s":[-4.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":56,"s":[-4.37],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":57,"s":[-4.07],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":58,"s":[-3.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":59,"s":[-3.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":60,"s":[-3.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":61,"s":[-3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":62,"s":[-2.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":63,"s":[-2.55],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":64,"s":[-2.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":65,"s":[-2.13],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":66,"s":[-1.94],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":67,"s":[-1.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":68,"s":[-1.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":69,"s":[-1.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":70,"s":[-1.29],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":71,"s":[-1.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":72,"s":[-1.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":73,"s":[-0.89],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":74,"s":[-0.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":75,"s":[-0.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":76,"s":[-0.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":77,"s":[-0.49],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":78,"s":[-0.41],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":79,"s":[-0.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":80,"s":[-0.27],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":81,"s":[-0.22],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":82,"s":[-0.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":83,"s":[-0.12],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":84,"s":[-0.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":85,"s":[-0.06],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":86,"s":[-0.03],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":87,"s":[-0.02],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":88,"s":[0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"s":{"a":1,"k":[{"t":0,"s":[0,0],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":1,"s":[0.51,0.51],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":2,"s":[1.15,1.15],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":3,"s":[1.92,1.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":4,"s":[2.83,2.83],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":5,"s":[3.88,3.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":6,"s":[5.09,5.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":7,"s":[6.44,6.44],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":8,"s":[7.95,7.95],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":9,"s":[9.61,9.61],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":10,"s":[11.43,11.43],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":11,"s":[13.38,13.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":12,"s":[15.46,15.46],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":13,"s":[17.67,17.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":14,"s":[19.98,19.98],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":15,"s":[22.38,22.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":16,"s":[24.85,24.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":17,"s":[27.37,27.37],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":18,"s":[29.92,29.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":19,"s":[32.49,32.49],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":20,"s":[35.05,35.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":21,"s":[37.59,37.59],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":22,"s":[40.09,40.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":23,"s":[42.56,42.56],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":24,"s":[44.97,44.97],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":25,"s":[47.33,47.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":26,"s":[49.62,49.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":27,"s":[51.85,51.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":28,"s":[54,54],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":29,"s":[56.09,56.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":30,"s":[58.1,58.1],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":31,"s":[60.05,60.05],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":32,"s":[61.92,61.92],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":33,"s":[63.73,63.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":34,"s":[65.47,65.47],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":35,"s":[67.14,67.14],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":36,"s":[68.75,68.75],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":37,"s":[70.3,70.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":38,"s":[71.78,71.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":39,"s":[73.21,73.21],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":40,"s":[74.58,74.58],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":41,"s":[75.9,75.9],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":42,"s":[77.17,77.17],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":43,"s":[78.38,78.38],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":44,"s":[79.55,79.55],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":45,"s":[80.67,80.67],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":46,"s":[81.74,81.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":47,"s":[82.77,82.77],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":48,"s":[83.76,83.76],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":49,"s":[84.71,84.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":50,"s":[85.62,85.62],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":51,"s":[86.48,86.48],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":52,"s":[87.32,87.32],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":53,"s":[88.11,88.11],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":54,"s":[88.87,88.87],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":55,"s":[89.6,89.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":56,"s":[90.3,90.3],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":57,"s":[90.96,90.96],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":58,"s":[91.6,91.6],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":59,"s":[92.2,92.2],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":60,"s":[92.78,92.78],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":61,"s":[93.33,93.33],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":62,"s":[93.85,93.85],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":63,"s":[94.34,94.34],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":64,"s":[94.81,94.81],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":65,"s":[95.26,95.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":66,"s":[95.68,95.68],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":67,"s":[96.08,96.08],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":68,"s":[96.45,96.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":69,"s":[96.81,96.81],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":70,"s":[97.14,97.14],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":71,"s":[97.45,97.45],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":72,"s":[97.74,97.74],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":73,"s":[98.01,98.01],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":74,"s":[98.26,98.26],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":75,"s":[98.5,98.5],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":76,"s":[98.71,98.71],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":77,"s":[98.91,98.91],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":78,"s":[99.09,99.09],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":79,"s":[99.25,99.25],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":80,"s":[99.39,99.39],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":81,"s":[99.52,99.52],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":82,"s":[99.63,99.63],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":83,"s":[99.73,99.73],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":84,"s":[99.81,99.81],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":85,"s":[99.88,99.88],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":86,"s":[99.93,99.93],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":87,"s":[99.97,99.97],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":88,"s":[99.99,99.99],"i":{"x":0,"y":0},"o":{"x":1,"y":1}},{"t":89,"s":[100,100],"i":{"x":0,"y":0},"o":{"x":1,"y":1}}]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"tdeEH77cuXLwRjKVaLJaQ"},{"ddd":0,"ind":169,"ty":4,"nm":"","ln":"J5u5ra1BFuvCfgAmKm1AS","sr":1,"ks":{"a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[98.85,81.75]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"ip":0,"op":899,"st":0,"bm":0,"shapes":[{"ty":"gr","hd":false,"bm":0,"it":[{"ty":"rc","hd":false,"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[198,164]}},{"ty":"fl","hd":false,"bm":0,"c":{"a":0,"k":[1,1,1]},"r":1,"o":{"a":0,"k":100}},{"ty":"tr","nm":"Transform","a":{"a":0,"k":[0,0]},"o":{"a":0,"k":100},"p":{"a":0,"k":[0,0]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}}],"np":0}]}]},{"h":5,"id":"8qKZ3hP4bdKxe0qZdkOfg","p":"data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAwAAAAFCAYAAABxeg0vAAAAAXNSR0IArs4c6QAAADNJREFUGFdj/P///2IGBgYfBsLgNwMDQz3j////3zEwMAgSVg9WsYo+GtoZGBg8iHRSMwAVPBetOUvCkQAAAABJRU5ErkJggg==","u":"","w":12,"e":1},{"h":7,"id":"RcXQKgdsFgbWJ3CC8kXh_","p":"data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAA4AAAAHCAYAAAA4R3wZAAAAAXNSR0IArs4c6QAAAFRJREFUKFPF0LsRQFAUBNDzGvAJRAoRqMPoQJkyHUgUIFGEa14oQmTjPTM7myKiRY/Kc04cmFNEjCifza2xZTh9RLm+/wMH1B/nrnlqgw7FS5xfXS55RSBi7PnRBgAAAABJRU5ErkJggg==","u":"","w":14,"e":1},{"h":892,"id":"G482sGnJF-hwxSHMfPSF5","p":"data:image/png;base64,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","u":"","w":440,"e":1}],"ddd":0,"fr":60,"h":185,"ip":0,"layers":[{"ddd":0,"ind":1,"ty":0,"nm":"","ln":"precomp_Z-XI-q05IqUdYEM98SYXk1","sr":1,"ks":{"a":{"a":0,"k":[2200,2200]},"o":{"a":0,"k":100},"p":{"a":0,"k":[110,223]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":4400,"h":4400,"ip":0,"op":899,"st":0,"bm":0,"refId":"Pr9xUEqkCHJf11QB0bjuh"},{"ddd":0,"ind":16,"ty":0,"nm":"","ln":"precomp_zhyQlJ5BP1rlfnw49ORik16","sr":1,"ks":{"a":{"a":0,"k":[98.5,242]},"o":{"a":0,"k":100},"p":{"a":0,"k":[110.15,277.75]},"r":{"a":0,"k":0},"s":{"a":0,"k":[100,100]},"sk":{"a":0,"k":0},"sa":{"a":0,"k":0}},"ao":0,"w":197,"h":484,"ip":0,"op":899,"st":0,"bm":0,"refId":"dPC5W2E-7zMda_jZz4Ys2"}],"meta":{"g":"https://jitter.video"},"nm":"New-file","op":899,"v":"5.7.4","w":220}';

	module.exports = { IcloudDialog };
});
